/*
 * Distributed V Kernel
 * Copyright (c) 1982 by David Cheriton, Tim Mann, Willy Zwaenepoel
 * (Transliterated from Zed and Verex Kernel)
 *
 * Machine-dependent kernel routines
 */

#include "process.h"
#include "asmprocess.h"
#include "memory.h"
#include "Vquerykernel.h"
#include "timer.h"

extern Process *Active, *ProcessDescriptors, *EndPds;
extern Team *TeamDescriptors;
extern unsigned long MaxProcesses, MaxTeams;
extern SystemCode AllocateMemory();

/* Saved configuration information */
MachineConfigurationReply MachineConfig;
PeripheralConfigurationReply PeripheralConfig;
KernelConfigurationReply KernelConfig;
unsigned char *LastPeripheral = &PeripheralConfig.peripheral[0];;
short FramebufferType = PRF_NONE;
short MouseType = PRF_NONE;
short ConsoleInputDevice = PRF_CONSOLE_TERMINAL;
short ConsoleOutputDevice = PRF_CONSOLE_TERMINAL; 
extern short MultibusHack;
extern unsigned char ExceptionStackFrameSizes[];

/* Basic Process Implementation */

dummySwitch()

  /* Save the state of the active process and
   * activate the process specified by Readyq.head,
   * loading its state from its process descriptor.
   * It is assumed to be called with interrupts disabled.
   * Switch is never called as a function; it is jumped
   * to.  The code is written inside this dummy function
   * body to allow C statements as well as asms to be
   * used.
   *
   * Warning: Do not use any automatic variables in this code, and
   * remember not to declare more register variables than there are 
   * registers!
   */
  {
    extern Unspec a6;
    extern SyncQueue Readyq;
    register Process *pd;
    register short prio;
    register Unspec (*finish_up)();

    asm("    .text");
    asm("    .globl    Switch");
    asm("Switch:");

    asm("    movl    a0, sp@-");    		/* Temp save a0 */
    asm("    movl    Active, a0");
    asm("    moveml  #/7FFF, a0@(_D0)");	/* Save registers in pd */
    asm("    movl    sp@+, a0@(_A0)");		/* ...including a0 */
    /* Save user SP as register A7--kernel SP not saved */
    asm("    movl    usp, a1");
    asm("    movl    a1, a0@(_USP)");
    /* Save user's sr and pc */
    asm("    movw    sp@, a0@(_SR)");		/* Save sr */
    asm("    movl    sp@(2), a0@(_PC)");	/* Save pc */
#ifdef MC68010
    asm("    movw    sp@(6), a0@(_FORMAT_OFFSET)");
#endif

    asm("    .globl  ActivateReadyqHead");
    asm("ActivateReadyqHead:");
newprocess:
    Active = pd = Readyq.head;    /* Remove new active process readyq */
    Readyq.head = pd->link;
    pd->queuePtr = NULL;

    /* Change to new team space */
    SetAddressableTeam(pd->team);

    /* Call a function to finish up if necessary */
    if( finish_up = pd->finish_up )
      {
        asm( "    movw    #/2300, sr" );
				/* Ensure running at kernel disable level */
        pd->finish_up = NULL;	/* This has to be in front of the call to
				   finish_up because finish_up may set it to
				   a new value. */
        /* Return value in d0 */
        pd->proc_state.regs[0] = (*finish_up)(pd);
        /* Check in case we are no longer ready */
        disable;
        if( pd != Readyq.head ) goto newprocess;
	Readyq.head = pd->link;
	pd->queuePtr = NULL;
      }

    /* Set the per-process data area pointer */
    if (pd->proc_state.perProcessLoc)
	*pd->proc_state.perProcessLoc = pd->proc_state.perProcess;
    
    a6 = (Unspec) pd;

#ifdef MC68020
    asm("    movl    #9, d0");		/* Flush the internal cache */
    asm("    movc    d0, cacr");
#endif MC68020
#ifdef MC68010
    /* Check whether we need to move a long exception stack frame */
    asm("    movw    a6@(_FORMAT_OFFSET), d0");
    asm("    movw    d0, sp@(6)");
    asm("    andw    #/f000, d0");
    asm("    bne     longframe");	/* Go if so (unusual case) */
#endif MC68010

    asm("    movl    a6@(_PC), sp@(2)");/* Restore pc */
    asm("    movw    a6@(_SR), sp@");
    asm("    movl    a6@(_USP), a1");   /* Restore usp */
    asm("    movl    a1, usp");
    asm("    bclr    #1, a6@(_PD_FLAGS)");
    asm("    beq     nomessage");
    asm("    moveml  a6@(_TRAP_PARAMS), #/0003");
    asm("    moveml  a6@(_MSG), #/07F8");
/* We restore the frame pointer just to make sure that a zero is put into the
 * first stack frame. Later traps into the kernel will save (and restore) the
 * current a6 value on the stack before the trap is executed.
 */
    asm("    movl    a6@(_A6), a6");
    asm("    rte");
    asm("nomessage:");
    asm("    moveml  a6@(_D0), #/7FFF");/* Restore registers.  A6 is restored
					 *  last, overwriting its old value */
    asm("    rte");             	/* Switch to user process */

#ifdef MC68010
    /* Set up and move long exception stack frame */
    asm("longframe:");
    asm("    rolw    #4, d0");		/* Frame type into low order bits by
    					 * rotating left instead of shifting
					 * right. All other bits have been
					 * masked off previously. We do this
					 * because you can't shift more than
					 * eight bits with a constant operand.
    					 */
#ifdef faster_but_the_compiler_thinks_the_colon_is_a_label
    asm("    movl    #ExceptionStackFrameSizes, a0"); /* Offset into table */
    asm("    movb    a0@(0, d0:W), d0");/* Grab frame size from table */
#endif

    asm("    addl    #ExceptionStackFrameSizes, d0"); /* Offset into table */
    asm("    movl    d0, a0");
    asm("    clrl    d0");
    asm("    movb    a0@, d0");
    asm("    lea     a6@(_FRAME), a0"); /* Source */
    asm("    movl    sp, a1");	        /* Destination */
    asm("copyframe:");
    asm("    movw    a0@+, a1@+");
    asm("    dbra    d0, copyframe");

    asm("    movl    a6@(_USP), a1");   /* Restore usp */
    asm("    movl    a1, usp");
    asm("    moveml  a6@(_D0), #/7FFF");/* Restore registers.  A6 is restored
					 *  last, overwriting its old value */
    asm("    rte");             	/* Switch to user process */
#endif MC68010

  }

AllocatePds()
  {
    unsigned long pdMemory;
    
#ifdef SUN3
    pdMemory = min( ( MachineConfig.memory >> 6 ), 0x10000 );
#else
#ifdef SUN2
    pdMemory = min( ( MachineConfig.memory >> 7 ), 0x8000 );
#else SUN1
    pdMemory = sizeof(Team) * 16;
#endif SUN2
#endif SUN3

    if ( AllocateMemory( TEAM_SEG, pdMemory ) != OK )
	Kabort("Couldn't allocate team descriptor memory.");
    TeamDescriptors = (Team *) TEAM_SEG;
    KernelConfig.maxTeams = MaxTeams = pdMemory/sizeof( Team );
printx("Allocating %d team descriptors.\n", MaxTeams);

    /* Allocate 16k per megabyte of physical memory, max 64k */
    pdMemory = min( ( MachineConfig.memory >> 6 ), 0x10000 );
    if ( AllocateMemory( PD_SEG, pdMemory ) != OK )
	Kabort("Couldn't allocate process descriptor memory.");
    ProcessDescriptors = (Process *) PD_SEG;
    KernelConfig.maxProcesses = MaxProcesses = pdMemory/sizeof( Process );
printx("Allocating %d process descriptors.\n", MaxProcesses);
    EndPds = (Process *) ( PD_SEG + ( MaxProcesses * sizeof( Process ) ) );
  }

/* Process Management */

Ms_CreateProcess( pd, entry, stack ) register Process *pd;
                    Unspec entry; Unspec stack;

  /* Perform the machine-specific portion of process creation,
   * namely initializing the state and such like.
   */
  {
    register MsgStruct *msg;
    register KProcessor_state *state;

    state = &(pd->proc_state);
    state->USER_STACK_POINTER = (Unspec) stack;
    state->regs[14] = 0;  /* Set initial frame pointer to 0 */
#ifdef MC68010
    state->frame.sr = 0;
    state->frame.pc = (unsigned long) entry;
    state->frame.format = 0;
    state->frame.vector = TRAP0;
#else
    state->sr = 0;        /* Set initial value of status register */
    state->pc = (unsigned long) entry;    /* Set entry point */
#endif

    /* Fill up message buffer */
    msg = &(pd->msg);
    msg->sysCode = PUT_ON_GREEN_HAT;
    msg->segmentPtr = (char *) TEAM_START;
    msg->segmentSize = TEAM_LIMIT - TEAM_START;

    /* Grant creator full access to team space */
    pd->dataSegmentPro = READ_ACCESS + WRITE_ACCESS;
    pd->dataSegmentPtr = (Unspec *) TEAM_START;
    pd->dataSegmentSize = TEAM_LIMIT - TEAM_START;
  }

Ms_CreateTeam( td, pd ) register Process *pd; register Team *td;

  /* Perform the machine-specific portion of team creation.
   *
   */
  {
    register Team_space *space;
    register KProcessor_state *state;

    space = &(td->team_space);
    state = &(pd->proc_state);

    ReclaimMemory(td);	/* Be sure team space is null */

  }


/*** Kernel Initialization ***/


#include <b.out.h> /* Initially use the standard b.out header */

Init_root_team()

  /* Create the first or root team and first process
   */
  {
    extern Team *TdFreeList, *FirstTeam;
    extern LhnRec *LhnFreeList;
    extern Process *IdleProcessPtr;
    extern SystemCode CreateTeam();
    register Process_id pid;
    extern struct bhdr TeamBheader;
    register KernelRequest *req;
    register char *teamsize;
    register Process *pd, *active;
    extern Team *AddressableTeam;

    active = IdleProcessPtr;
    AddressableTeam = active->team;
    /* Assume first team is already loaded at TEAM_START */
    teamsize = (char *) TEAM_START + TeamBheader.tsize
         + TeamBheader.dsize + TeamBheader.bsize + INIT_STACK;

    TdFreeList->lhn = LhnFreeList;
				/* We need this so that the first process has
				   an lhost available. */
    LhnFreeList->lhn = 0;

    req = (KernelRequest *) &(active->msg);
    req->unspecified[0] = ROOT_PRIORITY;
    req->unspecified[1] = TeamBheader.entry;
    req->unspecified[2] = (unsigned) teamsize-16;
    req->unspecified[3] = 0;
    if( (req->opcode=CreateTeam(active)) != OK )
      {
	printx( "Failed to create first team: %x\n", req->opcode );
        Kabort( "first team" );
      }
    pid = req->pid;
    pd = MapPid( pid );
    RemoveQueue( pd );
    pd->team->team_space.size = teamsize;
    pd->team->team_priority = DEFAULT_TEAM_PRIORITY;
    pd->priority = PROCESS_PRIORITY_MASK | pd->team->team_priority;
    pd->userNumber = SYSTEM_USER;
    FirstTeam = pd->team; /* First team is used for permission checking. */

    /* We'll use the V_BOOT_LHN while checking for lhn collisions, then
     * change our lhn when we find a unique value.
     */
#ifdef LITTLE_ENDIAN
    pd->localPid = pd->pid |= ((V_BOOT_LHN << 16) | LITTLE_ENDIAN_HOST);
#else
    pd->localPid = pd->pid |= (V_BOOT_LHN << 16);
#endif
    NewLhn(pd);
  }

Ms_td_init( td, i ) register Team *td; register unsigned i;

  /*
   * Perform machine-specific initialization of team descriptors
   */
  {
#ifdef SUN2
#ifdef SUN3
    if (i < NUM_CONTEXTS)
      {
	td->team_space.context = i;
      }
    else
      {
	SetContext(1);
	SaveSegMap(td->team_space.segmap);
	SetContext(0);
	td->team_space.context = MAP_NOT_RESIDENT;
      }
#else
    if (i < NUM_CONTEXTS)
      {
    	td->team_space.context.f.supervisor = i;
	td->team_space.context.f.user = i;
      }
    else
      {
	SetUserContext(1);
	SaveSegMap(td->team_space.segmap);
	SetUserContext(0);
	td->team_space.context.u = MAP_NOT_RESIDENT;
      }
#endif SUN3
#else
    td->team_space.context.f.context = i;
#endif SUN2

    td->team_space.size = (char *) TEAM_START; /* Set the initial size */
  }

/* See machine.h for Ms_pd_init */

/* Powerup() moved to config.c since the exact routines called are
 *   configuration-dependent, not just machine-dependent */


char *StackBottom;    /* Used to clean up stack */

Idle()

  /* Idle process function. Wait for an interrupt and
   * loop as soon as redispatched.
   */
  {
    /* Initialization */
    /* Add to ready queue. */
    Readyq.head = Active;
    Active->queuePtr = &Readyq;
    Active->state = READY;

#ifdef MC68010
    asm("    addl    #-128, sp");	/* Space for long excp stack frames */
    asm("    movw    #/80, sp@-");	/* Put fake format/offset on stack */
#endif MC68010
    asm("    movl    #loop, sp@-");	/* Set up for rte to loop */
    asm("    movw    #/2000, sp@-");
    asm("    movl    sp, StackBottom");
    asm("    jmp    Switch");		/* Switch to head of readyq */
    asm("loop:");

loop:    asm("    stop    #/2000");
    goto loop;
  }

FindConfiguration()
    /*
     * Find configuration of workstation.  Information used by device routines
     *   and QueryKernel operation.  We make a few assumptions here instead
     *   of actually looking for everything.  Note that not all of the
     *   configuration is done here; some of it happens in device
     *   powerup routines, etc.
     */
  {
    short version, config;
    extern unsigned versionnum;  /* from vers.c */

    /* Fill in kernel configuration information */
#ifdef ENET3MEG
    KernelConfig.ikcType = IKC_3MBIT;
#else
#ifdef ENET10MEG
    KernelConfig.ikcType = IKC_10MBIT;
#else
    KernelConfig.ikcType = IKC_NONE;
#endif
#endif
    KernelConfig.maxProcesses = MaxProcesses;
    KernelConfig.maxTeams = MaxTeams;
    KernelConfig.maxLogicalId = MAX_LOGICAL_ID;
    KernelConfig.rootPriority = ROOT_PRIORITY;
    KernelConfig.initStack = INIT_STACK;
    KernelConfig.vmConfig = VM_NONE;
    KernelConfig.versionNumber = versionnum;

#ifdef SUN3
    /* Sun-3 processor */
    printx("Sun-3 processor\n");
    MachineConfig.processor = PROC_MC68020;
    MachineConfig.machine = MACH_SMI_SUN3;
    MachineConfig.confreg = -1;		/* no config register */
    *LastPeripheral++ = PRF_SERIAL_SUNONBOARD;

    /* Look for frame buffer */
    if (Probe(V_SUN3_FRAMEBUFFER, PROBE_PRESERVE))
      {
	printx("Found Sun-3 style frame buffer\n");
	*LastPeripheral++ = PRF_FRAMEBUFFER_SUN3;
	FramebufferType = PRF_FRAMEBUFFER_SUN3;
	ConsoleOutputDevice = PRF_FRAMEBUFFER_SUN3;
      }

    if (FramebufferType == PRF_FRAMEBUFFER_SUN3)
      {
	printx("Assuming serial keyboard and mouse\n");
        *LastPeripheral++ = PRF_CONSOLE_SMI120;
        *LastPeripheral++ = PRF_MOUSE_SMI120;
	ConsoleInputDevice = PRF_CONSOLE_SMI120;
	MouseType = PRF_MOUSE_SMI120;
      }
    else
      {
	/* Assume we have a terminal as our console */
	printx("Assuming terminal on serial port 0\n");
        *LastPeripheral++ = PRF_CONSOLE_TERMINAL;
      }
#else
#ifdef SUN2
    /* Sun-2 processor */
    printx("Sun-2 processor\n");
    MachineConfig.processor = PROC_MC68010;
    MachineConfig.machine = MACH_SMI_SUN2;
    MachineConfig.confreg = -1;		/* no config register */
    *LastPeripheral++ = PRF_SERIAL_SUNONBOARD;

    /* Look for Sun-1 style frame buffer */
    if (Probe(V_SUN1_FRAMEBUFFER, PROBE_READ))
      {
	printx("Found Sun-1 style frame buffer\n");
	*LastPeripheral++ = PRF_FRAMEBUFFER_SUN1;
	FramebufferType = PRF_FRAMEBUFFER_SUN1;
	ConsoleOutputDevice = PRF_FRAMEBUFFER_SUN1;
      }

    /* Look for Sun-2 style frame buffer */
    if (Probe(V_SUN2_FRAMEBUFFER, PROBE_PRESERVE))
      {
	printx("Found Sun-2 style frame buffer\n");
	*LastPeripheral++ = PRF_FRAMEBUFFER_SUN2;
	FramebufferType = PRF_FRAMEBUFFER_SUN2;
	ConsoleOutputDevice = PRF_FRAMEBUFFER_SUN2;
      }

#ifndef SUN50
    if ((*(short *)(V_PARALLEL_PORT)) != 0xFFFF)
      {
	/* Something is plugged into the parallel port.  Assume it
	 *   is a keyboard and mouse.  We can't do much better since an
         *   uninitialized mouse looks just like no mouse at all */
	printx("Found parallel keyboard/mouse\n");
        *LastPeripheral++ = PRF_CONSOLE_SMI100;
        *LastPeripheral++ = PRF_MOUSE_SMI100;
	ConsoleInputDevice = PRF_CONSOLE_SMI100;
	MouseType = PRF_MOUSE_SMI100;
      }
    else
#endif SUN50
	if (FramebufferType == PRF_FRAMEBUFFER_SUN2)
      {
	printx("Assuming serial keyboard and mouse\n");
        *LastPeripheral++ = PRF_CONSOLE_SMI120;
        *LastPeripheral++ = PRF_MOUSE_SMI120;
	ConsoleInputDevice = PRF_CONSOLE_SMI120;
	MouseType = PRF_MOUSE_SMI120;
      }
    else
      {
	/* Assume we have a terminal as our console */
	printx("Assuming terminal on serial port 0\n");
        *LastPeripheral++ = PRF_CONSOLE_TERMINAL;
      }


#else
#ifdef SUN15
    /* Sun-1.5 */
    printx("Sun-1.5 processor\n");
    MachineConfig.processor = PROC_MC68010;
    MachineConfig.machine = MACH_SMI_SUN15;
    MachineConfig.confreg = -1;		/* no config register */
    *LastPeripheral++ = PRF_SERIAL_SUNONBOARD;

    /* Look for Sun-1 style frame buffer */
    if (Probe(V_SUN1_FRAMEBUFFER, PROBE_READ))
      {
	printx("Found Sun-1 style frame buffer\n");
	*LastPeripheral++ = PRF_FRAMEBUFFER_SUN1;
	FramebufferType = PRF_FRAMEBUFFER_SUN1;
	ConsoleOutputDevice = PRF_FRAMEBUFFER_SUN1;
      }

    if ((*(short *)(V_PARALLEL_PORT)) != 0xFFFF)
      {
	/* Something is plugged into the parallel port.  Assume it
	 *   is a keyboard and mouse.  We can't do much better since an
         *   uninitialized mouse looks just like no mouse at all */
	printx("Found parallel keyboard/mouse\n");
        *LastPeripheral++ = PRF_CONSOLE_SMI100;
        *LastPeripheral++ = PRF_MOUSE_SMI100;
	ConsoleInputDevice = PRF_CONSOLE_SMI100;
	MouseType = PRF_MOUSE_SMI100;
      }
    else
      {
	/* Assume we have a terminal as our console */
	printx("Assuming terminal on serial port 0\n");
        *LastPeripheral++ = PRF_CONSOLE_TERMINAL;
      }

#else
    /* Sun-1.  Need to identify manufacturer. */

    printx("Sun-1 processor\n");
    MachineConfig.processor = PROC_MC68000;
    *LastPeripheral++ = PRF_SERIAL_SUNONBOARD;

    /* Look for Sun-1 style frame buffer */
    if (Probe(V_SUN1_FRAMEBUFFER, PROBE_READ))
      {
	printx("Found Sun-1 style frame buffer\n");
	*LastPeripheral++ = PRF_FRAMEBUFFER_SUN1;
	FramebufferType = PRF_FRAMEBUFFER_SUN1;
	ConsoleOutputDevice = PRF_FRAMEBUFFER_SUN1;
      }

    version = K_version();
    if ( version == 101 || version == 0x200 )
      {
	/* SMI Sun 1.0  -- PROMs are version 0.101 or 2.0 */
	printx("SMI Model 100\n");
	MachineConfig.machine = MACH_SMI_SUN1;

	config = *(short *) V_PARALLEL_PORT;

	/* Kludge: assume the parallel port has a keyboard and mouse
	 *  plugged in if any of the high-order 14 bits are 0.
	 *  Otherwise, treat it as a config register.  This allows us
	 *  to strap bit 1 to indicate the "multibus hack" should be
	 *  turned on.
	 */
        if ( (config & 0xFFFC) != 0xFFFC )
          {
	    /* Something is plugged into the parallel port.  Assume it
	     *   is a keyboard and mouse.  We can't do much better since an
             *   uninitialized mouse looks just like no mouse at all */
	    MachineConfig.confreg = -1;	/* no config register */
	    printx("Found parallel keyboard/mouse\n");
            *LastPeripheral++ = PRF_CONSOLE_SMI100;
            *LastPeripheral++ = PRF_MOUSE_SMI100;
	    ConsoleInputDevice = PRF_CONSOLE_SMI100;
	    MouseType = PRF_MOUSE_SMI100;
          }
        else
          {
	    /* Assume we have a terminal as our console */
	    printx("Assuming terminal on serial port 0\n");
	    MachineConfig.confreg = config;
            *LastPeripheral++ = PRF_CONSOLE_TERMINAL;
          }
      }
    else if ( (((config = GetConfig())>>11) & 3) == 2 )
      {
	/* Cadlinc -- uses 811 baud keyboard */
	printx("Found Cadlinc keyboard/mouse\n");
	MachineConfig.machine = MACH_CADLINC_SUN;
	MachineConfig.confreg = config;
	*LastPeripheral++ = PRF_CONFREG;

	*LastPeripheral++ = PRF_CONSOLE_CADLINC;
	*LastPeripheral++ = PRF_MOUSE_CADLINC;
	ConsoleInputDevice = PRF_CONSOLE_CADLINC;
	MouseType = PRF_CADLINC_MOUSE;
      }
    else if (K_proctype() == 2)
      {
	/* Uses a Forward Tech Gateway Series (10 MHz) processor */
	printx("Forward Tech 10 MHz\n");
	MachineConfig.machine = MACH_FORWARD_SUN;
	MachineConfig.confreg = config;
	*LastPeripheral++ = PRF_CONFREG;

	printx("Assuming terminal on serial port 0\n");
	*LastPeripheral++ = PRF_CONSOLE_TERMINAL;
      }
    else
      {
	/* otherwise, one of the old homebrew Suns (8 MHz) */
	MachineConfig.machine = MACH_STANFORD_SUN;
	MachineConfig.confreg = config;
	*LastPeripheral++ = PRF_CONFREG;

	printx("Assuming terminal on serial port 0\n");
	*LastPeripheral++ = PRF_CONSOLE_TERMINAL;
      }

    if ((MachineConfig.confreg & 2) == 0) 
      {
	printx("Enabling user access to Multibus memory space.\n");
        MultibusHack++;
      }
#endif
#endif
#endif
  }


/* Copy between interal and external processor state representations.
 *   Also protects the system from attempts to assign invalid states
 *   to processes.
 */
SystemCode CopyInProcessorState(internal, external, modify_self)
    register KProcessor_state *internal;
    register Processor_state *external;
  {
    register short i;

    /* Integrity checks */
    if (external->perProcessLoc != NULL && 
    	BadUserPtr(external->perProcessLoc))
      {
	return (BAD_ADDRESS);
      }
    if (external->sr & (SUPERVISOR_STATE|INTERRUPT_MASK))
      {
	return (NO_PERMISSION);
      }

    /* PerProcess variable location and value */
    internal->perProcessLoc = external->perProcessLoc;
    internal->perProcess = external->perProcess;

    /* Don't let a process modify its own registers */
    if (modify_self) return( OK );
    
    /* Registers */
    for (i=0; i<16; i++)
        internal->regs[i] = external->regs[i];
#ifdef MC68010
    internal->frame.sr = external->sr;
    internal->frame.pc = (unsigned long) external->pc;
    internal->frame.format = 0;
    internal->frame.vector = TRAP0;
#else
    internal->sr = external->sr;
    internal->pc = (unsigned long) external->pc;
#endif
    return (OK);
  }


CopyOutProcessorState(external, internal)
    register Processor_state *external;
    register KProcessor_state *internal;
  {
    register short i;

    /* PerProcess variable location and value */
    external->perProcessLoc = internal->perProcessLoc;
    external->perProcess = internal->perProcess;

    /* Registers */
    for (i=0; i<16; i++)
        external->regs[i] = internal->regs[i];
#ifdef MC68010
    external->sr = internal->frame.sr;
    external->pc = (Unspec) internal->frame.pc;
#else
    external->sr = internal->sr;
    external->pc = (Unspec) internal->pc;
#endif
  }

#ifdef SUN3
Reboot()
  {
    printx("Please reboot your machine.\n");
    asm("trap #14");
  }
#else
#ifdef SUN2
Reboot()
  {
    PageMapEntry pg;
    
    TMRLoadCmd(TCReset);  /* Reset the timer chip so the PROM monitor will
    			   *  think this is a powerup reset */
    pg.f.valid = 0;	  /* Cause a double bus fault */
    SetPageMap(0, pg.u);
    asm("movl sp, 0");
    asm("movl sp@, 0");
  }
#else
Reboot()
  {
    int i;

    StopMouse();	/* for Cadlincs */

    /* First change the state of the timer chip sufficiently that the 
     *  PROM monitor will think this is a powerup reset.  Then make
     *  the watchdog timer bark loudly by shutting off memory refresh.
     *  NOTE: This doesn't work for Cadlincs and other machines with 
     *  pre-rev-D CPU boards, which don't have watchdog timers.
     */
    TMRLoadCmd(TCLdDataPtr(TIMWatchdog));
    LoadReg(TCMNoGa + TCMFall + TCMGate1 + TCMDiSpG + TCMRldLd +
	    TCMCntRep + TCMDecCnt + TCMDownCnt + TCMHiPulse);

    TMRLoadCmd(TCLdDataPtr(TIMRefresh));
    LoadReg(TCMOutLo);

    /* If we get here, it didn't work.  Exit to the monitor. */
    asm("trap #14");
  }
#endif
#endif
