/*
 * Distributed V Kernel
 * Copyright (c) 1982 by David Cheriton 
 *
 * Kernel Interphase disk driver, Paul Roy
 */

#include "interrupt.h"
#include "dm.h"
#include "Vexceptions.h"
#include "Vioprotocol.h"
#include "memory.h"
#include "idk.h"

/* Imports */
extern SystemCode NotSupported();
#ifdef SUN2
extern void AllocateDvmaSpace();
extern SystemCode AllocatePage();
#else /* not SUN2 */
extern void AllocateMultibusMemory();
#endif SUN2

/* Exports */
extern SystemCode IdkCreate();
extern SystemCode IdkRead();
extern SystemCode IdkWrite();
extern SystemCode IdkRelease();
extern SystemCode IdkCheckRequest();
extern SystemCode IdkPowerup();
extern SystemCode IdkFormat();

DeviceInstance    *DiskInstance = NULL;  /* ptr to DeviceInstance for disk */   


char    *dkspace_va;   /* virtual address of disk's space in multibus memory */
char    *dkspace_pa;   /* physical multibus address of disk's space */
char    *dkbuffer_va;  /* virtual address of data buffer */
char    *dkbuffer_pa;  /* physical multibus address of data buffer */

Iopb    *ip;           /* virtual ptr to i/o parameter block */
Iopb    *ip_pa;        /* physical ptr to i/o parameter block */

Uib     *up;           /* virtual ptr to unit initialization block */
Uib     *up_pa;        /* physical ptr to unit initialization block */



/* Macro expansion to interrupt-invoked C call to IdkInterrupt */

Call_inthandler(IdkInterrupt)


SystemCode IdkCreate( req, desc )
  CreateInstanceRequest *req;
  DeviceInstance *desc;

  /* Create an instance for the Interphase disk interface.
   */
 {
    if( req->filemode != FCREATE ) return( MODE_NOT_SUPPORTED );

    if( DiskInstance != NULL && 
        MapPid(DiskInstance->owner) )
      {
	return( BUSY );
      }

    /*
     * Initialize the device instance descriptor, can assume all fields
     * are zero except for owner and id.
     */

    desc->readfunc = IdkRead;
    desc->writefunc = IdkWrite;
    desc->modifyfunc = NotSupported;
    desc->queryfunc = NotSupported;
    desc->releasefunc = IdkRelease;
    desc->reader = 0;
    desc->writer = 0;

    desc->owner =  Active->pid;  

    desc->type = (READABLE+WRITEABLE+MULTI_BLOCK);
    desc->blocksize = SECSIZE;
    desc->lastblock = NCYL * NSECT * NTRACKS;
				/* cylinders * blocks per cylinder */

    DiskInstance = desc; /* Record the instance for interrupt routine */


    return( OK );
  }

 
SystemCode IdkPowerup()
  
  /*  Powerup and initialize the Interphase disk Interface Board.
   */
  {
    extern int Asm_IdkInterrupt();
    register int (**intvec)() = (int(**)()) INT2; /* Level 2 */
    IoRequest  req;
    unsigned buffersize;
 
    /* Plug interrupt location */
    *intvec = Asm_IdkInterrupt;

#ifdef SUN2
    /* Sun-2:  Allocate DVMA space for parameter blocks and for mapping
     *  in user buffers */
    buffersize = uptopage(DISK_MAX_BYTES) + PAGE_SIZE;
    AllocateDvmaSpace( buffersize + sizeof(Iopb) + sizeof(Uib),
	&dkspace_pa, &dkspace_va ); 
    
    /* Map in a page of physical memory for the parameter blocks */
    AllocatePage( dkspace_va + buffersize );
#else /* not Sun-2 */
    /* Sun-1:  Allocate contiguous Multibus memory for parameter
     *  blocks and the kernel disk buffer */
    buffersize = (unsigned) uptopage(DISK_MAX_BYTES);
    AllocateMultibusMemory( buffersize + sizeof(Iopb) + sizeof(Uib),
	&dkspace_pa, &dkspace_va ); 
#endif SUN2

    /* disk buffer */
    dkbuffer_va = dkspace_va;
    dkbuffer_pa = dkspace_pa;

    /* i/o parameter block */
    ip = (Iopb *) (dkspace_va + buffersize);
    ip_pa = (Iopb *) (dkspace_pa + buffersize);

    /* uib follows iopb */
    up = (Uib *) (dkspace_va + buffersize + sizeof(Iopb));
    up_pa = (Uib *) (dkspace_pa + buffersize + sizeof(Iopb));

    /* Now initialize controller. */

    IdkBuild( &req, RESET, 0 );
    IdkStart();

    while (( ip->status == DBUSY ) || (ip->status == 0) )
      ;
    while ( (R0(devaddr) & 0x0f) == 0x01 )
      ;                     /* wait for status reg to reflect oper. done int. 
                             * This is needed because an interrupt is not
                             * generated until 170 microseconds after the
                             * iopb is updated.
                             */

    if ( ip->status == ERROR )
      return( DEVICE_ERROR );

    IdkInit();
    IdkBuild( &req, INIT, up_pa );
    IdkStart();

    while (( ip->status == DBUSY ) || (ip->status == 0) )
      ;
    while ( (R0(devaddr) & 0x0f) == 0x01 )
      ;

    if ( ip->status == ERROR )
      return( DEVICE_ERROR );

    return( OK );
  }



SystemCode IdkRelease( req, desc )
  IoRequest *req;
  DeviceInstance *desc;
   /*
    * Release the disk file instance.
    */
  {
    desc->owner = 0; /* Free the descriptor */
    return( OK );
  }



SystemCode IdkRead( req, desc )
  IoRequest *req;
  DeviceInstance *desc;

   /* Handle a read instance request to the Interphase Disk interface.
    */
  {
    extern Process *Active;
    SystemCode	   r;
    register Process *pd;
#ifdef SUN2
    register char *userbuffer, *p, *q;
    PageMapEntry pme;
#endif SUN2

    if( (r=IdkCheckRequest(req)) != OK ) return( r );

    /*
     * Reply for RETRY if already a reader or writer process.
     * This effectively provides busy-wait queuing for reading.
     */

    if( MAP_TO_RPD(pd, desc->reader) || 
        (MAP_TO_RPD(pd, desc->writer)) )
      {
        return( RETRY );
      }

    /* Client waits for operation to complete */ 
    desc->reader = Active->pid;
    Active->state = AWAITING_INT;

#ifdef SUN2
    /* Map the client's buffer into DVMA space so the disk controller
     *   can DMA into it */
    if (req->bytecount <= IO_MSG_BUFFER)
        userbuffer = (char *) req->shortbuffer;
    else
        userbuffer = (char *) req->bufferptr;

    /* Copy page map entries from user buffer to DVMA space.
     *   p points to the source page and q to the destination */
    q = dkspace_va;
    for (p = (char *) downtopage(userbuffer);
    	 p < (char *) uptopage(userbuffer + req->bytecount);
	 p += PAGE_SIZE)
      {
	pme.u = GetPageMap(p);
	SetPageMap(q, pme.u);
	q += PAGE_SIZE;
      }

    /* Compute the Multibus address of the user buffer's image in
     *   DVMA space */
    dkbuffer_pa = dkspace_pa + (userbuffer - (char *) downtopage(userbuffer));
#endif SUN2

    /* Issue the read operation. */
    IdkBuild( req, READ, dkbuffer_pa );
    IdkStart();
  
    return( NO_REPLY );  /* interrupt routine will reply */
  }



SystemCode IdkWrite( req, desc )
  IoRequest *req; 
  DeviceInstance *desc;

   /* Handle a write instance request to the Interphase disk interface.
    */
  {
    extern	Process *Active;
    SystemCode  r;
    register Process *pd;
#ifdef SUN2
    register char *userbuffer, *p, *q;
    PageMapEntry pme;
#endif SUN2

    if( (r = IdkCheckRequest(req)) != OK ) return( r );

    /*
     * Reply for RETRY if currently writing or reading.
     * This effectively provides busy-wait queuing for writing.
     */
    if (MAP_TO_RPD(pd, desc->writer) || 
       (MAP_TO_RPD(pd, desc->reader)) )
      {
        return( RETRY );
      }

#ifdef SUN2
    /* Sun-2: Map the client's buffer into DVMA space so the disk controller
     *   can DMA from it */
    if (req->bytecount <= IO_MSG_BUFFER)
        userbuffer = (char *) req->shortbuffer;
    else
        userbuffer = (char *) req->bufferptr;

    /* Copy page map entries from user buffer to DVMA space.
     *   p points to the source page and q to the destination */
    q = dkspace_va;
    for (p = (char *) downtopage(userbuffer);
    	 p < (char *) uptopage(userbuffer + req->bytecount);
	 p += PAGE_SIZE)
      {
	pme.u = GetPageMap(p);
	SetPageMap(q, pme.u);
	q += PAGE_SIZE;
      }

    /* Compute the Multibus address of the user buffer's image in
     *   DVMA space */
    dkbuffer_pa = dkspace_pa + (userbuffer - (char *) downtopage(userbuffer));
#else /* not SUN2 */
    /* Sun-1: copy to multibus memory from client's memory */

    if (req->bytecount <= IO_MSG_BUFFER)
      Copy_bytes( dkbuffer_va, req->shortbuffer, req->bytecount );
    else
      Copy_bytes( dkbuffer_va, req->bufferptr, req->bytecount );
#endif SUN2

    desc->writer = Active->pid;
    Active->state = AWAITING_INT; 

    /* Now issue a write operation to the controller. */
    IdkBuild( req, WRITE, dkbuffer_pa );
    IdkStart();

    return( NO_REPLY );  /* interrupt routine will reply */
  }

 

SystemCode IdkCheckRequest( req ) 
  register IoRequest *req;

  /* Check that the read or write request has a legitimate buffer, etc.
   */
  {
    register unsigned count;
    register SystemCode r;

    /*  Check length */
    count = req->bytecount;

    req->bytecount = 0; /* To be left zero if a check fails */

    if( count > DISK_MAX_BYTES )
	r = BAD_BYTE_COUNT;
    else  /* Make sure data pointer is valid.
	   *  Check that on a word boundary and not in the kernel area.
	   */
    if( (count > IO_MSG_BUFFER) && ( (BadUserPtr(req->bufferptr)) ||
      (Active->team->team_space.size < (req->bufferptr + count)) ||
      (((int) req->bufferptr) & 1) ) )
	r = BAD_BUFFER;
    else
      {
	req->bytecount = count;
	r = OK;
      }
    return( r );
  }



IdkInterrupt()

   /* Handle an interrupt from the disk interface.
    */
 {
    extern   DeviceInstance *DiskInstance;
    register Process        *pd;
    register IoRequest      *req;
    register IoReply        *reply;
    register DeviceInstance *desc;
    unsigned count;
    Team *oldteam;
 
    desc = DiskInstance;

    R0(devaddr) = CLR_INT;   /* clear disk interrupt bit */

    /* Handle disk interrupt. */


    if (ip->status == SUCCESS) 
      {
        if (MAP_TO_RPD(pd, desc->reader))   
				/* operation was a read */
          {
            MAP_TO_RPD(pd, desc->reader);
            req = (IoRequest *) (pd->msg);

#ifndef SUN2
            /* Sun-1: copy to client's memory from multibus memory */

            count = req->bytecount;

	    oldteam = GetAddressableTeam();
	    SetAddressableTeam(pd->team);

            if ( count <= IO_MSG_BUFFER)
              Copy_bytes( req->shortbuffer, dkbuffer_va, count);
            else
              Copy_bytes( req->bufferptr, dkbuffer_va, count);
            
	    SetAddressableTeam(oldteam);
#endif /* not SUN2 */

            reply = (IoReply *) (pd->msg);
            reply->replycode = OK;
            desc->reader = 0;
            Addready( pd );
          }
        else if (MAP_TO_RPD(pd, desc->writer))  
				/* operation was a write */
          {
            MAP_TO_RPD(pd, desc->writer);
            reply = (IoReply *) (pd->msg);
            reply->replycode = OK;
            desc->writer = 0;
            Addready( pd );
          }
        }
    else if (ip->status == ERROR)
      {
            if (MAP_TO_RPD(pd, desc->writer))
              {
                MAP_TO_RPD(pd, desc->writer);
                desc->writer = 0;
                reply = (IoReply *) (pd->msg);
                reply->bytecount = 0;
                reply->replycode = DEVICE_ERROR;
                Addready( pd );
              }
            else if (MAP_TO_RPD(pd, desc->reader))
              {
                MAP_TO_RPD(pd, desc->reader);
                desc->reader = 0;
                reply = (IoReply *) (pd->msg);
                reply->bytecount = 0;
                reply->replycode = DEVICE_ERROR;
                Addready( pd );
              }
            else
	      Kabort("Bogus idk interrupt");

       }

  }



IdkInit()
  
  {  
    /* up references the unit initialization block */

    up->heads = 10;
    up->sectors = NSECT;
    up->secsize_lsb = SECSIZE;
    up->secsize_msb = SECSIZE >> 8;
    up->gap1 = 18;
    up->gap2 = 20;
    up->interleave = 2;
    up->retry = 3;
    up->ecc = 1;
    up->reseek = 1;
    up->movebad = 0;
    up->incrhead = 1;
    up->dualport = 0;
    up->intr = 0;
    up->skew = 9;
  

  }



IdkBuild( req, cmd, physaddr )
  register IoRequest *req;
  uschar   cmd;
  char     *physaddr;
  {
    /* build an I/O parameter block */
    
    register int blk, cylinder, head, sector, sectcnt, cmdopt;
  
    /* no linked iopb's and DMA transfers are 16 bits(word mode) */
    cmdopt = 0x01;
    
    /* calculate cylinder, head, and sector from physical block number */

    blk = req->blocknumber;
    cylinder = blk / (NHEAD * NSECT);
    head = blk % (NHEAD * NSECT) / NSECT;
    sector = blk % NSECT;
    sectcnt = ( req->bytecount % SECSIZE == 0 ) ? req->bytecount/SECSIZE :
                                                  req->bytecount/SECSIZE + 1;

    /* fill in I/O parameter block */

    ip->cmd = cmd;
    ip->cmd_opt = cmdopt;
    ip->status = 0;
    ip->error = 0;
    ip->unit = UNIT;
    ip->head = head;
    ip->cyl_msb = cylinder >> 8;
    ip->cyl_lsb = cylinder;
    ip->sector_msb = sector >> 8;
    ip->sector_lsb = sector;
    ip->sectcnt_msb = sectcnt >> 8;
    ip->sectcnt_lsb = sectcnt;
    ip->dmacnt = DMA_CNT;
    ip->buffaddr_xmb = (int) physaddr >> 16;
    ip->buffaddr_msb = (int) physaddr >> 8;
    ip->buffaddr_lsb = (int) physaddr;
    ip->ioaddr_msb = DEVADDR >> 8;
    ip->ioaddr_lsb = DEVADDR;
   
    /*
     * insert relative and linked iopb addresses below if the
     * command option field specifies linked iopb's.
     */
    ip->reladdr_msb = 0;
    ip->reladdr_lsb = 0;
    ip->linkaddr_xmb = 0;
    ip->linkaddr_msb = 0;
    ip->linkaddr_lsb = 0;
    ip->dummy = 0;

  }



IdkStart()
  {

    /* point controller at iopb multibus address and issue GO */
    
    R1(devaddr) = (int) ip_pa >> 16;
    R2(devaddr) = (int) ip_pa >> 8;
    R3(devaddr) = (int) ip_pa;
    R0(devaddr) = CLR_INT | GO | BUS;               

  }





