/*
 * V Kernel - Copyright (c) 1981 by David Cheriton
 * (Transliterated from Zed and Verex Kernel)
 * Copyright (c) 1982 Stanford University, all rights reserved.
 *
 * Exception handling for MC68000 and MC68010
 */

#include "Venviron.h"
#include "Vexceptions.h"
#include "process.h"
#include "naming.h"

/* Imports */
extern char *StackBottom;
extern Logical_id_entry Logical_id_map[];
extern short KernelInterrupted, ProcessInterrupted;
extern char version[];
extern unsigned long StackStart, StackSize;

/* Forward declarations */
extern short *KPrintException();
static short ExceptionType;

/* We subtract one from the stack frame sizes in order to use the dbra
 * instruction in the stack frame copy loops. Sizes are in words (shorts).
 */
unsigned char ExceptionStackFrameSizes[] =
  {
    4-1,	/* 0000: Short format		*/
    4-1,	/* 0001: Throwaway		*/
    6-1,	/* 0010: Instruction exception	*/
    0,0,0,0,0,
    29-1,	/* 1000: 68010 bus fault	*/
    10-1,	/* 1001: Co-processor mid-ins	*/
    16-1,	/* 1010: 68020 short bus fault	*/
    46-1,	/* 1011: 68020 long bus fault	*/
    0,0,0,0
  };

#define exvec(vecname,vecnum) 					\
asm("	.globl	vecname");					\
asm("vecname:");						\
	ExceptionType = vecnum; /* Put vecnum in message */	\
asm("	jmp	callException");

#define setexvec(vecname,vecnum)  /* Initialize a vector */	\
*( (int (**)()) (vecnum + VECTOR_BASE) ) = vecname

#ifndef MC68010
DummyExceptionHandler()
  {
   /*
    * Establish locations for each exception
    *  to vector to (needed for MC68000 only).
    *  Special code is needed to handle address and bus errors
    *  since the sr and pc are at a different offset on the stack (see below).
    */
    exvec(IllInst,ILLINST);
    exvec(ZeroDiv,ZERODIV);
    exvec(ChkInst,CHKINST);
    exvec(TrapvInst,TRAPVINST);
    exvec(PrivViol,PRIVVIOL);
    exvec(TraceTrap,TRACETRAP);
    exvec(Emu1010,EMU1010);
    exvec(Emu1111,EMU1111);
    exvec(FormatError,FORMATERR);
    exvec(UninitInt,UNINITINT);

    exvec(SpurInt,SPURINT);
    exvec(Int1,INT1);
    exvec(Int2,INT2);
    exvec(Int3,INT3);

    exvec(Trap0,TRAP0);
    exvec(Trap1,TRAP1);
    exvec(Trap2,TRAP2);
    exvec(Trap3,TRAP3);
    exvec(Trap4,TRAP4);
    exvec(Trap5,TRAP5);
    exvec(Trap6,TRAP6);
    exvec(Trap7,TRAP7);
    exvec(Trap8,TRAP8);
    exvec(Trap9,TRAP9);
    exvec(Trap10,TRAP10);
    exvec(Trap11,TRAP11);
    exvec(Trap12,TRAP12);
    exvec(Trap13,TRAP13);
  }
#endif MC68010

#include "asmprocess.h"

/*
 * Asm routine to call the HandleException() function
 *  Registers are saved and the stack is cleaned up
 *  so that the process state record will
 *  be accurate as of the time of the exception.
 */

asm("	.text");
asm("	.globl    callException");
asm("callException:");
disable;
asm("	movl	a1, sp@-" );
asm("	movl	Active, a1" );
asm("	moveml	#/7FFF, a1@(_D0)");	    /* Save registers. */
asm("	movl	sp@+, a1@(_A1)" );	    /* Save A1 from stack. */
asm("	movl	usp, a0" );		    /* Save user SP */
asm("	movl	a0, a1@(_USP)" );
asm("	movl	sp@(2), a1@(_PC)" );	    /* Save PC */
asm("	movw	sp@, a1@(_SR)" ); 	    /* Save SR */
asm("	movl	a1, sp@-" );		    /* Push Active on stack
					     *   as 1st arg. */
asm("	jsr	HandleException");
asm("	movl    StackBottom, sp");    	    /* Flush stack */
asm("	jmp	ActivateReadyqHead");

#ifdef MC68010
/* Special code to handle 68010 and 68020 stack frames for address and bus
 *   errors */
asm("BusError:");
disable;
/* Save registers (as above) */
asm("	movl	a1, sp@-" );		/* Push A1 */
asm("	movl	Active, a1" );
asm("	moveml	#/7FFF, a1@(_D0)");	/* Save registers. */
asm("	movl	sp@+, a1@(_A1)" );	/* Save A1 from stack. */
asm("	movl	usp, a0" );		/* Save user SP */
asm("	movl	a0, a1@(_USP)" );

/* Save exception stack frame */
asm("   movl    sp, a2");		/* Source */
asm("	lea	a1@(_FRAME), a3");	/* Destination */
/* We use the format type as an offset into the frame size array. First we
 * have to get the four format type bits into the four LSBs of a register.
 * We use the rolw instruction rather than a lsrw instruction because these
 * instructions accept eight as the maximum immediate operand.
 */
asm("   movw    sp@(6), d0");		/* Grab format & vector */
asm("   andl	#/f000, d0");		/* Mask off vector */
asm("	rolw	#4, d0");		/* Move format to low order 4 bits */
asm("	addl	#ExceptionStackFrameSizes, d0");	/* Offset into table */
asm("	movl	d0, a0");
asm("	clrl	d0");
asm("	movb	a0@, d0");		/* Frame size - 1 into d0 */
asm("copyframe:");
asm("	movw	a2@+, a3@+");
asm("	dbra	d0, copyframe");

/* Call exception handler */
asm("	movl	a1, sp@-" );		/* Push Active as 1st arg */
asm("	jsr	HandleException");
asm("	movl    StackBottom, sp");	/* Flush stack */
asm("	jmp	ActivateReadyqHead");

#else /* MC68000 */
/* Special code to handle 68000 long stack frames for address and bus
 *   errors */
asm("	.globl	BusError");
asm("	.globl	AddrError");
asm("AddrError:");
asm("	movw	#/00C, ExceptionType");		/* 0x00C = ADDRERROR */
asm("	bra	AddrOrBusError");
asm("BusError:");
asm("	movw	#/008, ExceptionType");		/* 0x008 = BUSERROR */
asm("AddrOrBusError:");
disable;
asm("	movl	a1, sp@-" );			/* Save a1 */
asm("	movl	Active, a1" );
asm("	moveml	#/7FFF, a1@(_D0)");		/* Save registers. */
asm("	movl	sp@+, a1@(_A1)" );		/* Save a1 from stack. */
asm("	movl	usp, a0" );
asm("	movl	a0, a1@(_USP)" );
asm("	movl	sp@(10), a1@(_PC)" );
asm("	movw	sp@(8), a1@(_SR)" ); 
asm("	movl	a1, sp@-" );			/* Push active as 1st arg. */
asm("	jsr	HandleException");
asm("	movl    StackBottom, sp");    		/* Flush stack */
asm("	jmp	ActivateReadyqHead");

#endif

InitException()

/*
 * Function to initialize exception vectors and kernel traps
 */

  {
#ifdef MC68010
    extern callException(), BusError();
#else !MC68010
    extern BusError(), AddrError(), IllInst(), ZeroDiv(), ChkInst(),
	   TrapvInst(), PrivViol(), TraceTrap(), Emu1010(), Emu1111(),
	   SpurInt(), Int1(), Int2(), Int3(), Trap0(), Trap1(), Trap2(),
	   Trap3(), Trap4(), Trap5(), Trap6(), Trap7(), Trap8(), Trap9(),
	   Trap10(), Trap11(), Trap12(), Trap13(), UninitInt(), FormatError();
#endif MC68010

/* Message register traps */

extern Asm_SendReg_trap(), Asm_ReceiveReg_trap(), Asm_ReplyReg_trap(),
	Asm_ReplyWithSeg_trap(), Asm_ForwardReg_trap(), Asm_GetReplyReg_trap();

#ifdef MC68010

/* Most exception vectors go to the same place because the 68010
 *  puts the vector number on the stack for us.  We use a different
 *  routine to handle address (?) and bus errors so we can save the
 *  longer exception stack frames these generate.  */


    setexvec(BusError,BUSERROR);
    setexvec(BusError,ADDRERROR);

    setexvec(callException,ILLINST);
    setexvec(callException,ZERODIV);
    setexvec(callException,CHKINST);
    setexvec(callException,TRAPVINST);
    setexvec(callException,PRIVVIOL);
    setexvec(callException,TRACETRAP);
    setexvec(callException,EMU1010);
    setexvec(callException,EMU1111);
    setexvec(callException,FORMATERR);
    setexvec(callException,UNINITINT);

    setexvec(callException,SPURINT);
    setexvec(callException,INT1);
    setexvec(callException,INT2);
    setexvec(callException,INT3);

    setexvec(callException,TRAP0);
/* TRAP1 is used for breakpoints on some PROM monitor versions */
    setexvec(callException,TRAP2);
    setexvec(callException,TRAP3);
    setexvec(callException,TRAP4);
    setexvec(callException,TRAP5);
    setexvec(callException,TRAP6);
    setexvec(callException,TRAP7);
    setexvec(callException,TRAP8);
    setexvec(callException,TRAP9);
    setexvec(callException,TRAP10);
    setexvec(callException,TRAP11);
    setexvec(callException,TRAP12);
    setexvec(callException,TRAP13);
/* TRAP14 is the Sun return-to-monitor trap */
    setexvec(callException,TRAP15);  /* Sun-1.5/2 workstations have no EMTs */

#else !MC68010

/* The 68000 doesn't put the vector number on the stack, so we have a
 *  separate routine for each exception to vector to, which loads the
 *  vector number into a global variable and jumps to the common routine */

    setexvec(BusError,BUSERROR);
    setexvec(AddrError,ADDRERROR);
    setexvec(IllInst,ILLINST);
    setexvec(ZeroDiv,ZERODIV);
    setexvec(ChkInst,CHKINST);
    setexvec(TrapvInst,TRAPVINST);
    setexvec(PrivViol,PRIVVIOL);
    setexvec(TraceTrap,TRACETRAP);
    setexvec(Emu1010,EMU1010);
    setexvec(Emu1111,EMU1111);
    setexvec(FormatError,FORMATERR);
    setexvec(UninitInt,UNINITINT);

    setexvec(SpurInt,SPURINT);
    setexvec(Int1,INT1);
    setexvec(Int2,INT2);
    setexvec(Int3,INT3);

    setexvec(Trap0,TRAP0);
/* TRAP1 is used for breakpoints on some PROM monitor versions */
    setexvec(Trap2,TRAP2);
    setexvec(Trap3,TRAP3);
    setexvec(Trap4,TRAP4);
    setexvec(Trap5,TRAP5);
    setexvec(Trap6,TRAP6);
    setexvec(Trap7,TRAP7);
    setexvec(Trap8,TRAP8);
    setexvec(Trap9,TRAP9);
    setexvec(Trap10,TRAP10);
    setexvec(Trap11,TRAP11);
    setexvec(Trap12,TRAP12);
    setexvec(Trap13,TRAP13);
/* TRAP14 returns to monitor */
/* TRAP15 is the SUN-1 emulator trap */    
#endif MC68010

/* Initialize the message register traps. */

    setexvec(Asm_SendReg_trap,TRAP0);
    setexvec(Asm_ForwardReg_trap,TRAP1); /* smashes PROM breakpoint trap;
					  *  should use another. */
    setexvec(Asm_ReceiveReg_trap,TRAP2);
    setexvec(Asm_ReplyWithSeg_trap,TRAP3);
    setexvec(Asm_GetReplyReg_trap,TRAP12);
  }


HandleException( active, frame )
    register Process *active;
    ExceptionStackFrame frame;
/*
 * Kernel exception handler.  Packages up the information
 *  about the exception and causes the offending process to send
 *  a message off to the exception handler process, if one is
 *  registered.  Otherwise, the process sends the message to itself.
 *
 * This function sees the exception information automatically pushed
 *   on the stack by the processor as its arguments.
 */

  {
    extern Process Idle_process;
    extern ProcessId KernelServerPid;
    ExceptionRequest *req;

#ifdef MC68010
    ExceptionType = frame.x10.vector; /* same as on the MC68020 */
#endif MC68010

    req = (ExceptionRequest *) &(active->msg);

    /* Build msg to send to exception handler */

    req->requestcode = EXCEPTION_REQUEST;
    req->type = ExceptionType;

    if( ExceptionType == ADDRERROR || ExceptionType == BUSERROR )
      {
#ifdef MC68010
#ifdef MC68020
        req->code = frame.x20.frame.f3.sw.word;
    	req->accaddr = frame.x20.frame.f3.faultaddr;
	/* Figuring out which instruction casued the fault is much too
	 * crazy in the pipelined world of the 68020. For now just
	 * return a zero.
	 */
	req->instruction = 0;
#else
        req->code = frame.x10.sw.word;
    	req->accaddr = frame.x10.faultAddress;
	/* Kludge: we return micro-register 14 in the instruction field,
	 *  as this appears to be the instruction register (the register
	 *  containing the opcode of the faulting instruction). */
	req->instruction = frame.x10.microRegisters[14];
#endif
    	req->status = frame.x10.sr;	/* same as on 20 */
    	req->errpc = frame.x10.pc;
#else !MC68010
        req->code = frame.x00.ab.code.word;
    	req->accaddr = frame.x00.ab.accaddr;
    	req->instruction = frame.x00.ab.ir;
    	req->status = frame.x00.ab.sr;
    	req->errpc = frame.x00.ab.pc;
#endif MC68010
      }
    else
      {
#ifdef MC68010
    	req->status = frame.x10.sr;
    	req->errpc = frame.x10.pc;
    	req->code = req->accaddr = req->instruction = 0;
#else !MC68010
    	req->status = frame.x00.oe.sr;
    	req->errpc = frame.x00.oe.pc;
    	req->code = req->accaddr = req->instruction = 0;
#endif MC68010
      }

    /* Diagnose cause of bus error -- see memory.c */
    if (ExceptionType == BUSERROR)
	req->buserrortype = DiagnoseBusError(req);

    req->segment = (char *) TEAM_START; /* Include access to the team space */
    req->segmentsize = ((unsigned) (active->team->team_space.size))
	 - TEAM_START;

    if( (req->status & SUPERVISOR_STATE) && (ExceptionType != TRACETRAP) )
      {
	/* Fatal kernel error */
	KPrintException( active, req );
    	Kabort( "Exception in kernel" );
      }

#ifdef VM
    /* See if exception was a page fault */
    if (PageFault(active, req) != BAD_ADDRESS) return;
#endif VM

    /* Send to the exception server and send to self if that fails */
    active->blocked_on = Logical_id_map[EXCEPTION_SERVER].pid;
    active->returnMessage = NULL;
    active->forwarder = KernelServerPid;
    active->finish_up = NULL; /* We don't want to get the reply */
    /* Setup data segment fields in PD - not set on forward. */
    active->dataSegmentPro = (READ_ACCESS+WRITE_ACCESS);
    active->dataSegmentPtr = (Unspec *) req->segment;
    active->dataExpected = NULL;
    active->dataSegmentSize = req->segmentsize;

    if( !MapPid(active->blocked_on) )
      {
	printx("No exception server\r\n" );
        KPrintException( active, req );
	active->blocked_on = active->pid;
	KSend( active );
      }
    else if ( active->blocked_on == active->pid )
      {
	printx("Exception in exception server\r\n" );
        KPrintException( active, req );
	active->blocked_on = active->pid;
	KSend( active );
      }
    else
      {
	/* Non-fatal exception */
	KSend( active );
      }

  }


#define MaxLookAhead 10
#define ABORT_INSTR	0x4afc
#define BKPT_INSTR	0x484f

short *KPrintException( pd, req )
    register Process *pd;
    register ExceptionRequest *req;

    /* Kernel exception handling routine.  Prints out some information
     * about the process incurring the exception and returns the pc at
     * which the exception occurred.
     */
  {
    register short *i;
    short codeBuffer[MaxLookAhead];
    register int instCnt, index = 0;
    KProcessor_state *state;
    unsigned stackSize;

    /* Print information about the state of the process */
    printx("\r\n");

    i = (short *) req->errpc;

    switch( req->type )
      {
      case ZERODIV:
	printx( " Divide by zero" );
	break;

      case CHKINST:
	printx( " Check instruction" );
	break;

      case TRAPVINST:
	printx( " Overflow" );
	break;

      case TRACETRAP:
	printx( " Trace bit on" );
	break;

      case PRIVVIOL:
	printx( " Privileged instruction" );
	req->instruction = *i;
	break;

      case ILLINST:
      case EMU1010:
      case EMU1111:
	/* Load instruction */
	req->instruction = *i;

	if( req->instruction == ABORT_INSTR )
	    printx(" Abort instruction" );
	else if ( req->instruction == BKPT_INSTR )
	    printx(" Illegal instruction--Breakpoint?" );
	else
	    printx(" Illegal instruction" );
	break;

      case ADDRERROR:
      case BUSERROR:
        /* Check if error was on instruction fetch */
	if (!InstructionFetch(req->code))
#ifdef MC68020
	    i = (short *) req->errpc;
#else
	  {
	    /* Error was not on the instruction fetch */
	    /* find actual address of erroneous instruction */
	    if ((i - MaxLookAhead*2-(TEAM_ORIGIN)) < 0)
		instCnt = (int)(i - (TEAM_ORIGIN))>>1;
	    else
		instCnt = MaxLookAhead;
			/* get code */
	    Copy(codeBuffer,i-instCnt, (short *)0 + instCnt);
			/* find instruction */
	    for( index=instCnt; (index > 0) &&
		(codeBuffer[index]!= req->instruction);
		index--)
		   ; 	/* empty */

	    if (codeBuffer[index] != req->instruction)
		i = (short *) req->errpc;
	      else
		i -= MaxLookAhead - index;
	  }
#endif
	if( req->type == BUSERROR )
	  {
	    switch (req->buserrortype)
	      {
	      case OUT_OF_RANGE:
	        printx( " Address out of range");
	        break;
	      case SPURIOUS:
	        printx( " Spurious bus error");
		break;
	      case SYSTEM_SPACE:
		printx( " Access to system space");
		break;
	      case PROTECTION:
		printx( " Memory protection violation");
		break;
	      case PAGE_INVALID:
	        printx( " Access to invalid page");
	        break;
	      case SEG_INVALID:
	        printx( " Access to invalid segment");
	        break;
	      case MB_TIMEOUT:
	        printx( " Multibus access timeout");
		break;
	      case PARITY:
		printx( " Memory parity error");
		break;
	      case BUS_TIMEOUT:
		printx( " System bus timeout ");
		break;
	      case VMEBERR:
		printx( " VME bus error");
		break;
	      case FPABERR:
		printx( " FPA bus error");
		break;
	      case FPAENERR:
		printx( " FPA enable error");
		break;
	      case WATCHDOG:
		printx( " Watchdog or user reset");
		break;
	      default:
		printx( " unknown bus error");
		break;
	      }
	  }
	else
	  {
	    printx( " Odd address access" );
	  }

	if (InstructionFetch(req->code))
	  {
	    printx( " on instruction fetch from");
	    req->instruction = 0;
	  }
	else
	  {
	    if (ReadAccess(req->code))
	        printx(" on read from");
	    else
        	printx(" on write to");
	  }

	printx( " address %x", req->accaddr );
	break;

      case TRAP0:
      case TRAP1:	case TRAP2:	case TRAP3:
      case TRAP4:	case TRAP5:	case TRAP6:
      case TRAP7:	case TRAP8:	case TRAP9:
      case TRAP10:	case TRAP11:	case TRAP12:
      case TRAP13:	case TRAP14:	case TRAP15:
	printx( " Unexpected trap #%d", (req->type - TRAP0)/sizeof(long) );
	break;

      case SPURINT:
	printx( " Spurious interrupt");
	break;

      case INT1:	case INT2:	case INT3:
      case INT4:	case INT5:	case INT6:
      case INT7:
	printx( " Unexpected level %d interrupt", 
	    ( req->type - ( INT1 + sizeof(long) ) / sizeof(long) ) );
	break;

      case SUSPENDED:
        printx( " Forced exception");
	break;

      default: 
	printx( " Unknown exception" );
	break;
      }

    printx( " in process %x [ 0x%x ]\r\n",pd->pid, pd );
    if( req->instruction != 0 ) 
        printx( " Instruction  " );
    printx( "  Program counter    Status register\r\n" );
    if( req->instruction != 0 )
        printx( "     %x     ", req->instruction );
    printx( "       %x              %x\r\n", i, req->status );

    state = &(pd->proc_state);
    if (req->status & SUPERVISOR_STATE) 
      {
	printx( " Last kernel trap invoked from address %x\r\n", 
		*(unsigned long *)(StackBottom+2));
      }

#ifdef NOTDEF
    if (state->perProcess != 0)
      {
        /* Check for user stack overflow */
	stackSize = ((PerProcessArea *) state->perProcess)->stackSize;

	if ((unsigned) state->USER_STACK_POINTER <= 
		(unsigned) state->perProcess ||
	    (unsigned) state->USER_STACK_POINTER > 
	    	(unsigned) state->perProcess + stackSize)
	    printx( " Probable stack overflow\r\n");
      }
#endif NOTDEF

    return(i);
  }


int InstructionFetch(code)
    int code;
  {
    StatusWord stat;

    stat.s00.word = code;
#ifdef MC68010
#ifdef MC68020
	return (stat.s20.field.fnc == FC_USER_PROGRAM ||
		stat.s20.field.fnc == FC_SUPER_PROGRAM);
#else
	return (stat.s10.field.fc == FC_USER_PROGRAM ||
		stat.s10.field.fc == FC_SUPER_PROGRAM);
#endif
#else
	return (stat.s00.field.fc == FC_USER_PROGRAM ||
		stat.s00.field.fc == FC_SUPER_PROGRAM);
#endif MC68010
  }

int ReadAccess(code)
    int code;
  {
    StatusWord stat;

    stat.s00.word = code;
#ifdef MC68010
#ifdef MC68020
    return (stat.s20.field.rw == 1); /* MC68020: 1 = read */
#else
    return (stat.s10.field.rwf == 1); /* MC68010: 1 = read */
#endif MC68020
#else MC68010
    return (stat.s00.field.rw == 1);  /* MC68000: 1 = read */
#endif MC68010
  }



/*
 * Primitive stack dump package
 */

extern *a6, *sp;


StackDump()
  {
    unsigned *currentFp;
    unsigned *returnPc;
    unsigned *stackTop;
    int i, nparams;

    stackTop = (unsigned *) (StackStart - StackSize);
    printx("\r\nStackDump:\r\n");
    currentFp = (unsigned *) a6;
    while (currentFp != 0)
      {
	if ( (currentFp < stackTop) || (currentFp > (unsigned *) StackStart))
	  {
	    printx("Invalid frame pointer (%x).\r\n", currentFp);
	    return;
          }
	returnPc = (unsigned *) (*(currentFp + 1));
	printx("    %x (", returnPc);
	nparams = GetNumberOfParams(returnPc);
	for (i = 0; i < nparams; i++)
	  {
	    printx("%x", *(currentFp + 2 + i));
	    if (i < (nparams - 1))
	      {
		printx(", ");
	      }
	  }
	printx(")\n");
	if ((((unsigned) returnPc) < KERNEL_START) || 
	    (((unsigned) returnPc) > TEAM_START))
	  {
	    return;
	  }
	if ( (unsigned *) *currentFp < currentFp )
	  {
	    printx("Stack trash beyond this point (%x)\r\n", *currentFp);
	    return;
	  }
	currentFp = (unsigned *) (*currentFp);
      }
  }


#define ADDQL4		0x588f	/* instruction to add 4 to stack pointer */
#define ADDQL8		0x508f	/* instruction to add 8 to stack pointer */
#define ADDL		0xdffc 	/* instruction to add contents of next long to
				    stack pointer */
#define ADDW		0xdefc 	/* instruction to add contents of next word to
				    stack pointer */
#define LEA		0x4fef	/* a faster version of ADDW */

#define MAX_NUM_ARGS 10

static int GetNumberOfParams( returnpc )
    unsigned short *returnpc;
  {
    /* Return the number of parameters in the call to abort, and the address
     * of the first (topmost, lowest addressed) parameter.
     */
    unsigned short instruction;
    unsigned short *addr;
    int count;
    int nparams;

    instruction = *returnpc;

    /* The instruction at the return address tells (at least in our compiler's
     * calling convention) how much to add to the stack pointer upon return
     * from the call.  If we divide that number by the size of each stack
     * entry, we get the number of parameters.  This same trick is used in
     * the debugger's stackdump.
     */

    addr = returnpc + 1;
    switch( instruction )
      {
	case ADDQL4 :
	    return( 1 );
	case ADDQL8 :  
	    return( 2 );
	case LEA    :
	case ADDW   :  
	    count = (int) *addr;
	    break;
	case ADDL   :  
	    count = (int) *((unsigned *)addr);
	    break;
        default     :  
	    return( 0 );
      }
    nparams = count / 4;
    if( nparams > MAX_NUM_ARGS )
      {
	return( 0 );
      }
    return( nparams );
  }

#include "Vquerykernel.h"

extern MachineConfigurationReply MachineConfig;
extern short FramebufferType;
extern FramebufferEnable();

Kabort( str ) register char *str;

  /* Kernel abort: print message on console and stop with
   * only refresh interrupt enabled.
   */
  {
    disable;	 /* Ensure that Kabort can't be recursively
		    called from an interrupt routine. */

    StopMouse(); /* Needed for Cadlinc */
    if (FramebufferType != PRF_NONE) FramebufferEnable(1);

    K_puts( "Kernel abort: " ); 
    K_puts( str ); 
    K_puts( "\r\n" );
    printx("%s\n", version);
    StackDump();
    asm("	trap #14");
  }
