/* On-board driver for Communication Machinery Corporation's (CMC)
 *   Ethernet Node Processor (ENP).  Paul Roy, Neguine Navab 
 */

#include "machine.h"
#include "ethercmc.h"

/* Allocate space to hold pointers to K1 Kernel routines. */
int  (*K1Status)();
int  (*K1Receive)();
int  (*K1Transmit)();
int  (*K1Timer)();

/* Forward declarations */
int  RcvInterrupt();
int  XmitInterrupt();
int  BusInterrupt();

/* Allocate space for the receive and transmit buffers. */
Buffer XmitBuffers[NUM_XMIT_BUF];
Buffer RcvBuffers[NUM_RCV_BUF];

/* Since, supposedly the K1 kernel doesn't like the control blocks to be 
 *   local to the invoking routine, we allocate them all globally.
 */
XmitBufCtrl  XmitCtrl[NUM_XMIT_BUF]; /* will be linked to 'XmitBuffers' */
RcvBufCtrl   RcvCtrl[NUM_RCV_BUF];   /* will be linked to 'RcvBuffers' */
StatusCtrl   StatCtrl;
InitControl  InitCtrl;

/* Used to interrupt host via status command. */
StatusCtrl   InterruptReq;


/* Stack of pointers to free transmit buffer control blocks. */
XmitBufCtrl *EmptyBuf[NUM_XMIT_BUF];

/* One less than the number of buffers available.  Acts as an index into
 *   EmptyBuf[].  Accessing and changing this variable must be indivisible.
 */
int    XmitBufAvailable;

XmitRecord     *XmitRec;   /* Record used during transmit for communication
                            *   between host and enp. */

RcvRecord      *RcvRec;    /* Record used during receive for communication
                            *   between host and enp. */

QueryRecord    *QueryRec;  /* Record used during query for communication
                            *   between host and enp. */

uschar         *interrupt_loc_ptr;  /* the type of interrupt is written here */
unsigned short   XmitFlag;     /* flags that reveal state of the records.
                                *   set by the host, cleared by the enp
                                */
unsigned short   RcvFlag;
unsigned short   QueryFlag;

unsigned short	 EnetReceiver; /* == 1 if RcvRec is being accessed */

/* We need a receive queue because we may get receive interrupts faster than
 *   we can process them.  The size of the queue is greater than the
 *   maximum number of possible valid entries  => don't worry about it filling.
 */
RcvBufCtrl       *ReceiveQueue[NUM_RCV_BUF];
unsigned short   front, back;  /* queue indices */
                              

main()

  {
     /* set up the stack pointer at the top of RAM */
#ifdef ENP10
     asm ("	lea /F1FDFC, sp	"); 
#endif
#ifdef ENP30
     asm ("	lea /F1DFFC, sp	"); 
#endif


    /* wait for host to issue initialize request */
    while( *(uschar *) ENP_STARTUP_MBOX != NET_INIT )
      ;
   
    Net_Initialize(); /* initialize Lance chip, buffers, etc. */

    while( 1 )
      {
        /* wait while no work to do */
	while ( !XmitFlag && !RcvFlag && !QueryFlag )
          ;

        if( XmitFlag )
          Net_Transmit();

        else if( RcvFlag )
          Net_Receive();

        else if( QueryFlag )
          {
            if( QueryRec->type == NET_STATS )
              Net_Stats();
            else
              Net_Modify();
          }

      } /* end while */

  }  /* end main() */




Net_Initialize()

  {
    HostInitRequest *HostInitReq;
    EnpInitResponse *EnpInitResp;
    InitResponse    *InitResp;
    int             (*func)(), i;

    XmitFlag = RcvFlag = QueryFlag = 0;
    EnetReceiver = 0;  /* no current receiver yet */
    front = back = 0;  /* initialize receive queue pointers */
    *(char *)INTERRUPT_REG = 0;  /* no ints. currently being serviced by host */
    Enable();                    /* enable interrupts */
#ifdef ENPDEBUG
    *(unsigned *)TEST_LOC1 = 0;
    *(unsigned *)TEST_LOC2 = 0;
    *(unsigned *)TEST_LOC3 = 0;
    *(unsigned *)TEST_LOC4 = 0;
    *(unsigned *)TEST_LOC5 = 0;
#endif

    HostInitReq = (HostInitRequest *) ENP_STARTUP_INIT_BLOCK;
    EnpInitResp = (EnpInitResponse *) ENP_STARTUP_INIT_BLOCK;

    /* get the info passed from the host */   
    XmitRec = (XmitRecord *) HostInitReq->XmitRec;
    RcvRec = (RcvRecord *) HostInitReq->RcvRec;
    QueryRec = (QueryRecord *) HostInitReq->QueryRec;
    interrupt_loc_ptr = (uschar *) HostInitReq->interrupt_loc_ptr;
    
    /* Set up initialize block */
    InitCtrl.mode = DRX; /*disable receiver */ /* 0;  /* normal operation */ 
    for (i = 0; i < SHORTS_PER_ADDR_FILTER; i++)
      InitCtrl.LogAddrFilter[i] = HostInitReq->LogAddrFilter[i];

    InitCtrl.RcvBufNum = NUM_RCV_BUF;
    InitCtrl.XmitBufNum = NUM_XMIT_BUF;
    InitCtrl.RcvInterrupt = RcvInterrupt;
    InitCtrl.XmitInterrupt = XmitInterrupt;
    InitCtrl.BusInterrupt = BusInterrupt;

    /* Call kernel Initialize routine */
    func = *( (int (**)())(K1_STARTUP_MBOX));
    InitResp = (InitResponse *)((*func)( &InitCtrl ));

    /* store pointers to K1 routines */
    K1Status = InitResp->StatusRtnAddr;
    K1Receive = InitResp->RcvRtnAddr;
    K1Transmit = InitResp->XmitRtnAddr;
    K1Timer = InitResp->TimerRtnAddr;

    for( i = 0; i < SHORTS_PER_ENET_ADDR; i++ )
      EnpInitResp->EthernetAddr[i] = InitResp->EthernetAddr[i];

    /* Start up the lance via the Status command */
    StatCtrl.FunctionCode = STRT;
    func = (int(*)())K1Status;
    (*func)( &StatCtrl );

    /* set the addresses of the record flags */
    EnpInitResp->XmitFlagPtr = (unsigned short *) ENPtoSysAddr( &XmitFlag );
    EnpInitResp->RcvFlagPtr = (unsigned short *) ENPtoSysAddr( &RcvFlag );
    EnpInitResp->QueryFlagPtr = (unsigned short *) ENPtoSysAddr( &QueryFlag );

    /* clear mbox so that host will stop busy-waiting */
    *(uschar *) ENP_STARTUP_MBOX = 0;

    /* Now connect control blocks to the buffers and hand them to k1 kernel. */
    func = (int (*)())K1Receive;
    for( i = 0; i < NUM_RCV_BUF; i++ )
      {
        RcvCtrl[i].BufAddr = &RcvBuffers[i];
        RcvCtrl[i].BufLength = BUFFER_LENGTH;  /* this never changes */
        RcvCtrl[i].status = ENP | STP; 
        RcvCtrl[i].LinkAddr = NULL;
  
        (*func)( &RcvCtrl[i] ); /* Call K1 kernel */
      }

    for( i = 0; i < NUM_XMIT_BUF; i++ )
      XmitCtrl[i].BufAddr = &XmitBuffers[i];

    /* set up pointers to empty xmit control blocks */
    for( i = 0; i < NUM_XMIT_BUF; i++ )
      EmptyBuf[i] = &XmitCtrl[i];
    XmitBufAvailable = NUM_XMIT_BUF - 1;

  }



Net_Transmit()

  {

    XmitBufCtrl              *XmitCtrl;  /* issued to k1 kernel */
    Interrupt_mask           mask;
    register    XmitPiece    *piece;
    register    char         *PktPtr, *data, *BufPtr;
    register    int          i, packet_length = 0, numpieces, (*func)();

/*
#ifdef ENPDEBUG
    printx("ENP: Transmitting\n");
#endif
 */
#ifdef ENPDEBUG
    Incr_Test_Loc3();
#endif
    while( XmitBufAvailable < 0 )
      ;  /* wait for a buffer to be given back in interrupt routine */
  
    mask = Mask_interrupts();  /* don't want a transmit interrupt */
    XmitCtrl = EmptyBuf[XmitBufAvailable--];
    Restore_interrupt_mask(mask);
    PktPtr = BufPtr = (char *) XmitCtrl->BufAddr; 

    /* If there is only one piece then the header is built into that piece. */
    if( (numpieces = XmitRec->numpieces) > 1 )
      {
        /* copy the header into the packet buffer - structure assignment */
        *( (Enet10Header *) PktPtr ) = XmitRec->EnetHeader;
        PktPtr += sizeof( Enet10Header );
        packet_length += sizeof( Enet10Header );
        numpieces--;
      }

    for( i = 0; i < numpieces; i++ )
      {
	piece = &(XmitRec->piece[i]);
	data = (char *) piece->address;
	packet_length += piece->length;
    
	/* Note that we are assuming that the packet length will never
	 *   be greater than the max. packet length which is BUFFER_LENGTH.
	 */
     
	/* Copy from the host memory to the xmit buffer.
	 *   We use the fast copy routine coded in 68000 assembler.
	 */
	Copy( PktPtr, data, piece->length );
	PktPtr += piece->length;
      }
  
    XmitCtrl->LinkAddr = NULL;
    XmitCtrl->BufLength = packet_length;
    XmitCtrl->status = ENP | STP;

#ifdef ENPDEBUG
    Incr_Test_Loc4();
#endif
    /* start the transmit before notifying host */
    func = (int (*)())K1Transmit;
    (*func)( XmitCtrl );  /* Call K1 kernel */

    XmitFlag = 0;  /* done processing the transmit */
#ifdef ENPDEBUG
    Incr_Test_Loc5();
#endif
    Interrupt_Host( NET_XMIT ); 

  }  



Net_Receive()

  {
    /* host is done processing the last receive */

    RcvBufCtrl     *RcvBlock;
    int            (*func)();

    /* Is there another receive that needs processing? */

    /* We don't want an interrupt after the test and before 'EnetReceiver'
     *   is cleared in the else clause. */
    Disable();  
    if( front != back )
      {
        Enable();
        /* yes, so process it */
     
        RcvFlag = 0;       /* enp done processing the last receive.
                            *   must clear this before interrupt host. */

        RcvBlock = ReceiveQueue[back];

        /* copy the data to the host kernel */
        CopyToHost( RcvBlock );
        Interrupt_Host( NET_RCV );         /* set the host off and running */
              
        /* give the receive buffer back to the LANCE */
        RcvBlock->status = ENP | STP;        /* probably not needed */
        RcvBlock->LinkAddr = NULL;
  
        func = (int (*)())K1Receive;
        (*func)( RcvBlock );                 /* Call K1 kernel */

        back = (back + 1) % NUM_RCV_BUF;

        /* note that we leave 'EnetReceiver' set to 1 */
      }
    else
      {
        EnetReceiver = 0;  /* the rcv record is no longer in use */
        Enable();
        RcvFlag = 0;       /* enp done processing the last receive */
      }



  }



Net_Stats()

  {
    int         (*func)();
 

    StatCtrl.FunctionCode = SREQ;  /* says to copy stats into stats block */
    StatCtrl.StatsBlockAddr = (StatsBlock *) QueryRec->StatsArray;

    func = (int(*)())K1Status;
    (*func)( &StatCtrl );  /* Call K1 kernel */

    QueryFlag = 0;  /* done processing the query */
    Interrupt_Host( NET_STATS ); 

  }




Net_Modify()

  {
    int         (*func)();

    /* reset statistics */
    StatCtrl.FunctionCode = SRESET;

    func = (int (*)())K1Status;
    (*func)( &StatCtrl );

    QueryFlag = 0;  /* done processing the query */
    Interrupt_Host( NET_MODIFY );

  }



int XmitInterrupt( XmitCtrl )
  register XmitBufCtrl *XmitCtrl;

  {
#ifdef ENPDEBUG
    Incr_Test_Loc1();
#endif 
    /* put the free buffer back on stack */
    EmptyBuf[++XmitBufAvailable] = XmitCtrl;
 
  }




int RcvInterrupt( RcvBlock )
  register RcvBufCtrl *RcvBlock;

  {
    EnetBlock  *EnetPacket;
    int        (*func)();

#ifdef ENPDEBUG
    Incr_Test_Loc2();
#endif 
    if( !EnetReceiver )
      {
        /* process the interrupt immediately */

        /* copy the data to the host kernel */
        CopyToHost( RcvBlock );
        Interrupt_Host( NET_RCV );         /* set the host off and running */
              
        /* give back the receive buffer immediately */
        RcvBlock->status = ENP | STP; /* probably not needed */
        RcvBlock->LinkAddr = NULL;
  
        func = (int (*)())K1Receive;
        (*func)( RcvBlock );          /* Call K1 kernel */

        EnetReceiver = 1;
      }
    else
      {
        /* queue for later processing - note that we have no need to check
         *   if the queue is full because it can never get full */
        ReceiveQueue[front] = RcvBlock;
        front = (front + 1) % NUM_RCV_BUF;
      }
  }



int BusInterrupt()
  {
    ;
  }




/* Code stolen from kernel functions.  To work properly, the host's
 *   EPROM space must be mapped into DVMA space so that K_putchar will work.
 */
#ifdef ENPDEBUG

/* printx.c: simple debugging-output functions */

/* These routines will print on anything, provided you define putchar
 * appropriately.  This version is for the kernel, and uses a separately
 * compiled putchar routine.  Note: putchar should convert \n to \r\n
 * for nice results.
 */


#include "process.h"

#undef putchar
#define putchar PutChar		/* This macro definition is needed to avoid
				   conflict with the defn. in Vio.h - which
				   gets included in various kernel source
				   files.  PutChar() is a machine-dependent
				   kernel-level version of putchar. */


print(s)
  register char *s;
  {
    while (*s)
	putchar(*s++);
  }


printx(string)
  char *string;
  {
    register int *argp = (int *) &string;
    register char *s = string;
    register int c;

    argp++;
    while(*s)
      {
	if (*s == '%')
	  {
	    s++;
	    switch(*s++)
	      {
	      case 's': print(*argp++);  break;
	      case 'x': hexprint(*argp++);  break;
	      case 'd': decprint(*argp++);  break;
	      case 'o': octprint(*argp++);  break;
	      case 'c': 
		c = *argp++;
		putchar(c);
		break;
	      default:
		putchar(*(s-2));
		putchar(*(s-1));
	      }
	  }
	else putchar(*s++);
      }
  }


/* unsigned hexadecimal print */
hexprint(n)
  register unsigned n;
  {
    register unsigned i, digit, pflag = 0;

    for (i=0; i<8; i++)
      {
	digit = (n >> 28);
	if (digit != 0 || i == 7) pflag = 1;
	if (pflag) 
	  {
	    if (digit > 9) digit += 'a'-10;
	    else digit += '0';
	    putchar(digit);
	  }
	n <<= 4;
      }
  }


/* Unsigned octal print.  We build the character string backwards. */
octprint(n)
  register unsigned n;
  {
    register unsigned digit;
    static char octbuf[14];
    register char *p = octbuf+13;

    *p-- = 0;
    if (n==0)  { putchar('0'); return; }
    while (n != 0)
      {
	digit = n & 07;
	n >>= 3;
	*p-- = digit + '0';
      }
    p++;
    print(p);
  }


/* Signed decimal print.  We build the character string backwards. */
decprint(n)
  register int n;
  {
    register unsigned digit, nflag=0;
    static char decbuf[12];
    register char *p = decbuf+11;

    *p-- = 0;
    if (n<0)   { n = -n; nflag++; }
    if (n==0)  { putchar('0'); return; }
    while (n != 0)
      {
	digit = n % 10;
	n = n / 10;
	*p-- = digit + '0';
      }
    if (nflag) *p-- = '-';
    p++;
    print(p);
  }


/*
 * putchar:
 * Kernel version of putchar.
 * NOTE: the standard macro definition of putchar must be undefined.
 */

PutChar(c)
    char c;
  {
    K_putchar(c);
  }



/*
 * fakeemt.c
 *
 * Fake Emulator Trap package for SMI Suns.
 */

#include "memory.h"
#include "sunromvec.h"


/*
 * int K_putchar(c)	-- prints character c on Console
 */
int K_putchar(c)
char c;
{
  register int i;
  /* give the character to the host */

  *(char *)PUTCHAR_LOC = c;
  Interrupt_Host( HOST_PUTCHAR ); 

  /* wait until the host has printed the character */
  while( *(char *)PUTCHAR_LOC == c )
    for( i = 0; i < 50; i++ )
      ;  /* delay loop */

}


#endif ENPDEBUG

