/*
 * Distributed V Kernel 
 * Copyright (c) 1982 by Stanford University.
 *
 * Kernel Ethernet driver for Communication Machinery Corporation's (CMC)
 *   Ethernet Node Processor (ENP).  Paul Roy 
 */

#include "Vethernet.h"
#include "Vikc.h"
#include "Vquerykernel.h"
#include "interrupt.h"
#include "dm.h"
#include "ikc.h"
#include "memory.h"
#include "process.h"
#include "ethercmc.h"


typedef Process_id (*PFPID)();

/* Imports */
extern SystemCode NotSupported();
extern DeviceInstance *GetDevice();
extern unsigned char *LastPeripheral;
extern InitializeAlien();
extern Process *AllocatePd();
extern Process Idle_process;
extern void MapDvma();

/* Exports */

extern SystemCode EnetCreate();
extern SystemCode EnetRead();
extern SystemCode EnetWrite();
extern SystemCode EnetModify();
extern SystemCode EnetQuery();
extern SystemCode EnetRelease();
extern SystemCode EnetCheckRequest();
extern SystemCode EnetPowerup();

extern               EnetReadPacket(), ReleaseSegBuffer();
extern HostSegBuffer *GetSegBuffer();

/* In general, we want variables shared between the host and enp to be in
 *   host memory to avoid possible timeouts while the host is accessing enp
 *   memory.  The exceptions are the flags because the enp busy-waits on these.
 */
XmitRecord      _XmitRec;    /* Record used during transmit for communication
                              *   between host and enp.  To access, 'EnetWriter'
                              *   must be locked.  
                              */
RcvRecord       _RcvRec;     /* Record used during receive for communication
                              *   between host and enp.
                              */
QueryRecord     _QueryRec;   /* Record used during query and modify for
                              *   communication between host and enp.
                              */
/* buffers to hold received segments */
HostSegBuffer   SegmentBuffers[NUM_SEG_BUFFERS];

/* Global array used to contain statistics. */
StatsBlock      StatsArray;
uschar          interrupt_loc;            /* contains type of interrupt */

/* variables used for mapping */
char            *xmit_pd_virt_addr, *xmit_seg_virt_addr; 
char            *xmit_packet_virt_addr, *rcv_pd_virt_addr;
char            *xmit_pd_phys_addr, *xmit_seg_phys_addr; 
char            *xmit_packet_phys_addr, *rcv_pd_phys_addr;
char            *XmitRec_pa, *RcvRec_pa, *QueryRec_pa, *host_segbufs_pa;
char		*stats_array_pa, *interrupt_loc_pa;

XmitRecord      *XmitRec = &_XmitRec;
RcvRecord       *RcvRec = &_RcvRec;
QueryRecord     *QueryRec = &_QueryRec;

HostSegBuffer   *SegmentList;  /* pointer to head of free segment buffer list */
unsigned char   SegBuffer_lock;  /* must lock this before accessing the list 
                                  * 'SegBuffer_lock must only be acc. w/ TSET*/ 

/* These flag pointers point to enp memory. */
unsigned short  *XmitFlagPtr;  /* ptrs to flag that reveal state of the records.
                                *   set by the host, cleared by the enp
                                */
unsigned short  *RcvFlagPtr;
unsigned short  *QueryFlagPtr;

unsigned char	EnetWriter;  /* == 1 if XmitRec is being accessed.
                              *   'EnetWriter' must only be accessed w/ TSET. 
                              */

unsigned char	EnetQueryer; /* == 1 if QueryRec is being accessed.
                              *   'EnetQueryer' must only be accessed w/ TSET. 
                              */

NetAddress	EnetHostNumber;           /* physical ethernet address */
DeviceInstance	*EthernetInstance = NULL; /* ptr to DeviceInstance */
				
int		NetAlarm = 0;

NetAddress	MulticastDefault = V_MULTICAST_ADDRESS;

/* Queue for out-going IKP packets. */
SyncQueue IkpOutq;  


/* Macro expansion to interrupt-invoked C call to Ethernetinterrupt */
Call_inthandler(EnetInterrupt)


SystemCode EnetCreate( pd, inst )
    Process *pd;
    register DeviceInstance *inst;
  /* Create an instance for the Ethernet interface.
   */
 {
    register CreateInstanceRequest *req = (CreateInstanceRequest *) &pd->msg;
    if( req->filemode != FCREATE ) return( MODE_NOT_SUPPORTED );

    if( EthernetInstance != NULL && 
        MapPid( EthernetInstance->owner) )
      {
	return( BUSY );
      }

    /*
     * Initialize the device instance, can assume all fields
     * are zero except for owner and id.
     */

    inst->readfunc = EnetRead;
    inst->writefunc = EnetWrite;
    inst->modifyfunc = EnetModify;
    inst->queryfunc = EnetQuery;
    inst->releasefunc = EnetRelease;
    inst->reader = 0;
    inst->writer = 0;

    EthernetInstance = inst; /* Record the instance for interrupt routine */

    inst->type = (READABLE+WRITEABLE+VARIABLE_BLOCK+STREAM);
    inst->blocksize = ENET10_MAX_PACKET;

    return( OK );
  }

 
SystemCode EnetPowerup()

  /* Powerup and initialize the Ethernet Interface Board.
   *  Done just once, from kernel Powerup() routine. 
   */
  {
    extern int           Asm_EnetInterrupt();
    extern HostSegBuffer *GetSegBuffer();
    register int         (**intvec)() = (int(**)()) ENET_INT_LEVEL;
    register int         i;
    HostInitRequest      *HostInitReq;
    EnpInitResponse      *EnpInitResp;

    printx("CMC powerup\n");    
    /* Indicate we have a CMC board */
    *LastPeripheral++ = PRF_ENET_CMC;  
    IkpOutq.tail = (Process *) &(IkpOutq.head);
    IkpOutq.type = IKP_OUT_QUEUE;
    
    /* Plug interrupt location */
    *intvec = Asm_EnetInterrupt;

    EnetWriter = EnetQueryer = 0;

    /* Get the enp to start executing by giving it the address of the base
     *   of the on-board code, and subsequently giving it a "jump" command
     *   in the location that it continuously monitors upon reset.
     */
    *((long *) ENPtoSysAddr(K1_STARTUP_MBOX)) = ENP_CODE_START;
    *((long *) ENPtoSysAddr(JUMP_COMMAND_LOC)) = JUMP_COMMAND;

    /* Allocate Dvma space and map all the static data structures 
     *   into DVMA space so that the enp can access them. */
    printx("Setting up DVMA\n");
    SetupDvma();
    printx("Back from DVMA set up\n");

    /* initialize the receive record */
    if( (RcvRec->pd = AllocatePd()) == NULL )
      Kabort("Couldn't get pd in Powerup()");
    printx("got a pd\n");
    RcvRec->pd->packetType &= ALIEN_EMBEDDED;  /* this pd will contain alien */
    printx("Mapping Dvma...");
    /* map the receive pd into DVMA space */
    MapDvma( rcv_pd_virt_addr, RcvRec->pd, sizeof(Process) );
    printx("done\n");

    /* Compute the Multibus address of the receive pd's image in
     *   DVMA space */
    RcvRec->pd_pa = (Process *) ((char *)rcv_pd_phys_addr + 
                      ((char *)RcvRec->pd - (char *)downtopage(RcvRec->pd)));

    /* link all the free buffers together */
    SegmentList = &SegmentBuffers[0];
    for( i = 0; i < NUM_SEG_BUFFERS-1; i++ )
      *((unsigned *) &SegmentBuffers[i]) = (unsigned) &SegmentBuffers[i+1];
    *((unsigned *) &SegmentBuffers[NUM_SEG_BUFFERS-1]) = NULL;
    
    if( (RcvRec->data = (char *) GetSegBuffer( &RcvRec->data_pa)) == NULL )
      Kabort("Couldn't get segment buffer in Powerup()");

    /* build the init request in a fixed block in Enp space */
    HostInitReq = (HostInitRequest *) ENPtoSysAddr( ENP_STARTUP_INIT_BLOCK );

    for (i = 0; i < SHORTS_PER_ADDR_FILTER; i++)
      HostInitReq->LogAddrFilter[i] = 0xFFFF;  /* accept all multicasts */

    HostInitReq->XmitRec = (XmitRecord *) XmitRec_pa;
    HostInitReq->RcvRec = (RcvRecord *) RcvRec_pa;
    HostInitReq->QueryRec = (QueryRecord *) QueryRec_pa;
    HostInitReq->interrupt_loc_ptr = (uschar *) interrupt_loc_pa;
    /* cause our code running on-board  to initialize */

    printx("puttng NET_INIT into startup mbox\n");
    (*(uschar *) ENPtoSysAddr(ENP_STARTUP_MBOX)) = NET_INIT; 

#ifdef ENPDEBUG
    printx("waiting for enp to initialize...");
#endif
    /* wait for enp to complete initialization */
    while( (*(uschar *) ENPtoSysAddr(ENP_STARTUP_MBOX)) == NET_INIT )
      for( i = 0; i < 1000; i++ )
        ;  /* delay loop */
#ifdef ENPDEBUG
    printx("done\n");
#endif

    /* response from enp comes back in same block */
    EnpInitResp = (EnpInitResponse *) ENPtoSysAddr( ENP_STARTUP_INIT_BLOCK );

    /* get pointers to flags that reveal the state of the records */
    XmitFlagPtr = EnpInitResp->XmitFlagPtr;
    RcvFlagPtr = EnpInitResp->RcvFlagPtr;
    QueryFlagPtr = EnpInitResp->QueryFlagPtr;

    /* get station address */
    EnetHostNumber.addrhigh = EnpInitResp->EthernetAddr[0];
    EnetHostNumber.addrmid = EnpInitResp->EthernetAddr[1];
    EnetHostNumber.addrlow = EnpInitResp->EthernetAddr[2];

#ifdef ENPDEBUG
    printx("Enet Addr = %x %x %x\n",EnetHostNumber.addrhigh,
         EnetHostNumber.addrmid,EnetHostNumber.addrlow);
#endif
  }



SetupDvma()

  {
    /* Allocate Dvma space and set up page maps that won't be changing. 
     *   Note that we assume AllocateDvmaSpace() allocates on page boundaries.
     */
    
    char     *pa, *va; /* dummy variables */

    /* Since the bottom 4k of multibus memory space is inaccessible by the 
     *   Enp, we allocate 4k and let it go to waste.  Assumes that 
     *   AllocateDvmaSpace() allocates memory starting from the bottom.
     */
    AllocateDvmaSpace( 4096, &pa, &va );

    AllocateDvmaSpace( sizeof(Process), &xmit_pd_phys_addr,
                                               &xmit_pd_virt_addr );

    AllocateDvmaSpace( sizeof(HostSegBuffer), &xmit_seg_phys_addr,
                                               &xmit_seg_virt_addr );

    AllocateDvmaSpace( ENET10_MAX_PACKET, &xmit_packet_phys_addr,
                                               &xmit_packet_virt_addr );

    AllocateDvmaSpace( sizeof(Process), &rcv_pd_phys_addr,
                                               &rcv_pd_virt_addr );

    /* this stuff won't change so, in addition to allocating, we map now */

    AllocateDvmaSpace( sizeof(XmitRecord), &pa, &va );
    MapDvma( va, &_XmitRec, sizeof(XmitRecord) );
    XmitRec_pa = (char *) pa + ((char *)&_XmitRec -
                                               (char *)downtopage(&_XmitRec));
    AllocateDvmaSpace( sizeof(RcvRecord), &pa, &va );
    MapDvma( va, &_RcvRec, sizeof(RcvRecord) );
    RcvRec_pa = (char *) pa + ((char *)&_RcvRec -
                                               (char *)downtopage(&_RcvRec));
    AllocateDvmaSpace( sizeof(QueryRecord), &pa, &va );
    MapDvma( va, &_QueryRec, sizeof(QueryRecord) );
    QueryRec_pa = (char *) pa + ((char *)&_QueryRec -
                                               (char *)downtopage(&_QueryRec));
    AllocateDvmaSpace( sizeof(HostSegBuffer)*NUM_SEG_BUFFERS, 
                                                                   &pa, &va );
    MapDvma( va, SegmentBuffers, sizeof(HostSegBuffer)*NUM_SEG_BUFFERS );
    host_segbufs_pa = (char *) pa + ((char *)SegmentBuffers -
                                          (char *)downtopage(SegmentBuffers));
    AllocateDvmaSpace( sizeof(StatsBlock), &pa, &va );
    MapDvma( va, &StatsArray, sizeof(StatsBlock) );
    stats_array_pa = (char *) pa + ((char *)&StatsArray -
                                               (char *)downtopage(&StatsArray));
    AllocateDvmaSpace( sizeof(uschar), &pa, &va );
    MapDvma( va, &interrupt_loc, sizeof(uschar) );
    interrupt_loc_pa = (char *) pa + ((char *)&interrupt_loc -
                                           (char *)downtopage(&interrupt_loc));

  }
                                               


SystemCode EnetRelease( pd, inst )
    Process *pd;
    DeviceInstance *inst;
   /*
    * Release the ethernet instance.
    */
  {
    IoRequest *req = (IoRequest *) &(pd->msg);

    inst->owner = 0; /* Free the instance */
    /* It is assumed that the reader will unblock on next packet
     * if currently blocked.
     */
    return( OK );
  }


SystemCode EnetRead( pd, inst ) 
    register Process *pd;
    register DeviceInstance *inst;
 /*
  * Handle a read instance request for the Ethernet interface.
  */
  {
    IoRequest *req = (IoRequest *) &(pd->msg);
    SystemCode	r;

    /*
     * Reply for RETRY if already a waiting reader process.
     * This effectively provides busy-wait queuing for reading.
     */
    if( MapPid(inst->reader) ) return( RETRY );

    if( (r=EnetCheckRequest(pd, req)) != OK ) return( r );

    /* Reader waits for a packet to arrive */
    inst->reader = pd->pid;
    pd->state = AWAITING_INT;

    /* Note that this is different from the 3Com driver which saves packets
     *   around if a Reader has not yet arrived.
     */

    return( NO_REPLY );
  }


SystemCode EnetWrite( pd, inst )
    register Process *pd;
    register DeviceInstance *inst;

   /* Handle a write instance request to the Ethernet interface.
    */
  {
    register IoRequest *req = (IoRequest *) &(pd->msg);
    SystemCode         r;

    /* Reply for RETRY if currently writing.
     * This effectively provides busy-wait queuing for writing.
     */
    if( TestAndSet( &EnetWriter ) != 0 ) 
      return( RETRY );

    if( (r = EnetCheckRequest(pd, req)) != OK ) 
      return( r );

    XmitRec->current_pd = pd;
    XmitRec->numpieces = 1;
    /* Compute the Multibus address of the receive pd's image in
     *   DVMA space */
    XmitRec->piece[0].address = ( (char *)xmit_packet_phys_addr + 
            ((char *)req->bufferptr - (char *)downtopage(req->bufferptr)) );
    XmitRec->piece[0].length = req->bytecount;

    /* map the already-built packet into DVMA space */
    MapDvma( xmit_packet_virt_addr, req->bufferptr, req->bytecount );

    XmitRec->xmit_again = 0;  /* don't have to transmit again for this req. */

    *XmitFlagPtr = 1;  /* cause the enp to process the record */

    /* the process will be added to the ready list in the interrupt routine */

    return( NO_REPLY );
  }



SystemCode EnetModify( pd, inst, dirIndex )
    register Process *pd;
    DeviceInstance   *inst;
    unsigned short   dirIndex;
  {
    /* reset statistics */

    if( TestAndSet( &EnetQueryer ) != 0 ) 
      return( RETRY );

    QueryRec->type = NET_MODIFY;
    *QueryFlagPtr = 1;  /* cause the enp to process the record */

    return( OK );
  }    



SystemCode EnetQuery( pd, inst, dirIndex ) 
    Process *pd;
    DeviceInstance *inst;
    unsigned short dirIndex;
  {
    register QueryEnetReply *reply = (QueryEnetReply *) &(pd->msg);

    if( TestAndSet( &EnetQueryer ) != 0 ) 
      return( RETRY );

    QueryRec->type = NET_STATS;
    QueryRec->current_pd = pd;
    QueryRec->StatsArray = (StatsBlock *) stats_array_pa;
    *QueryFlagPtr = 1;  /* cause the enp to process the record */

    pd->state = AWAITING_INT;  /* must wait for response */
  
    return( NO_REPLY );
  } 



SystemCode EnetCheckRequest( pd, req ) 
    register Process *pd;
    register IoRequest *req;

  /* Check that the read or write request has a legitimate buffer, etc.
   */
  {
    register unsigned count;
    register SystemCode r;

    /*  Check length */
    count = req->bytecount;

    req->bytecount = 0; /* To be left zero if a check fails */
    if( count > ENET10_MAX_PACKET || (count <= IO_MSG_BUFFER) )
	r = BAD_BYTE_COUNT;
    else  /* Make sure data pointer is valid.
	   *  Check that on a word boundary (WHY? --TPM)
	   *  and not in the kernel area.
	   */
    if( (BadUserPtr(req->bufferptr)) ||
    (pd->team->team_space.size < (req->bufferptr + count)) ||
    ((int) req->bufferptr) & 1 )
	r = BAD_BUFFER;
    else
      {
	req->bytecount = count;
	r = OK;
      }
    return( r );
  }



EnetInterrupt()

   /* Handle an interrupt from the ethernet interface.
    */
  {
    register Process *pd, *last_pd;
    uschar           code;
    extern int       K_putchar();

    code = (uschar) interrupt_loc;
    /* clear the interrupt */
    *((char *)SYS_INTERRUPT_REG) = 0;
 
    switch( (uschar) code )
      {  
        case NET_RCV: 
#ifdef ENPDEBUG
            printx("Interrupt...Receive\n");
#endif 
	    EnetReadPacket();
  	    return;

        case NET_XMIT:

#ifdef ENPDEBUG
            printx("Interrupt...Transmit\n");
#endif 
          /* deal with the last transmitter */
          if( XmitRec->xmit_again )
            last_pd = XmitRec->current_pd;  /* last guy has more to transmit */
          else
            {
              last_pd = 0;
              /* Finished with the process that just transmitted;
               *   route to the next queue. */
              if( XmitRec->numpieces == 1 ||
                              (XmitRec->current_pd->state == READY) )
                {
                  /* if numpieces == 1 then xmit initiated from EnetWrite() */ 
                  Addready( XmitRec->current_pd );
                }
              else
                DelayProcess( XmitRec->current_pd );
            }
             
	  /* If IkpOutq is non-empty, transmit next interkernel packet. */
	  Lockq( &IkpOutq );
	  if( (pd=IkpOutq.head) != NULL )
	    {
              /* the queue is not empty */
	      pd->queuePtr = NULL;
	      if( (IkpOutq.head = pd->link) == NULL )
		  IkpOutq.tail = (Process *) &(IkpOutq.head);
              if( last_pd )
                {
                  /* requeue the last guy on the end of the Ikp queue */
                  last_pd->link = NULL;
                  last_pd->queuePtr = &IkpOutq;
	          IkpOutq.tail = (IkpOutq.tail->link = last_pd);
                }
	      Unlockq( &IkpOutq );
	      WriteNextKPacket( pd );
	    }
	  else
	    {
	      Unlockq( &IkpOutq );
              if( last_pd )
                {
                  /* transmit again for the last guy */  
	          WriteNextKPacket( last_pd );
                }
              else
                {
                  /* No transmission in progress now */
	          EnetWriter = 0; 
	        }
            }
          break;  /* end of NET_XMIT */


        case NET_MODIFY:

          /* nothing to do except free up the query record */
          EnetQueryer = 0;
          break;


        case NET_STATS:
          {
            QueryEnetReply *reply = (QueryEnetReply *) 
                                               &(QueryRec->current_pd->msg);

            reply->NetworkType = ENET_TYPE_10MBIT;
            reply->NumCollisions = StatsArray.CollisionErrors;
            reply->NumOverflows = StatsArray.MissedPackets;
            reply->NumCRCErrors = StatsArray.CRCErrors;
            reply->NumSyncErrors = StatsArray.FramingErrors;
            reply->NumTimeOuts = StatsArray.RetryFailures;
            reply->ReceiveMask = 0;
            reply->NumValidPackets = StatsArray.EnetReceived;
            reply->HostAddress.e10 = EnetHostNumber;
          
            reply->replycode = OK;
            Addready( QueryRec->current_pd );

            EnetQueryer = 0;   /* make the query record available again */
            break;
          }
#ifdef ENPDEBUG
        case HOST_PUTCHAR:
          K_putchar( *(char *)SYS_PUTCHAR_LOC );
          /* cause the Enp to stop busy-waiting */
          *(char *)SYS_PUTCHAR_LOC = 0xFF;  /* invalid ascii character */
          break;
#endif ENPDEBUG
      }
  }


EnetReadPacket()

   /*   Only invoked by the interrupt routine.  The received packet is 
    * described by 'RcvRec'.  The packet has been scattered into 
    * RcvRec->EnetHeader, RcvRec->last_pd->(kernel packet part), RcvRec->data.
    */
  {
    extern PFPID         RemoteTrapTable[];
    register kPacket   *kp;
    register IoRequest *req;
    register Process   *last_pd;
    register unsigned  bytes;
    register short     *p;
    register unsigned  short packetType;
    Enet10Header       EnetHeader;
    Process            *pd, *newpd;
    DeviceInstance     *inst;
    Team               *oldteam;

    /* We want to extract the necessary information out of the receive record 
     *   and fill in new pointers as quickly as possible.
     */   
    EnetHeader = RcvRec->EnetHeader;  /* structure assignment */
    last_pd = RcvRec->pd;
    if( (bytes = RcvRec->segment_length) )
      {
        /* kludge: we store the pointer to the segment in an unused field
         *   of the process descriptor
         */
        last_pd->timeout_count = (unsigned) RcvRec->data;
        
        /* get a new buffer pointer for the next received segment */
        if( (RcvRec->data = (char *) GetSegBuffer( &RcvRec->data_pa )) == NULL )
          Kabort("Couldn't get a buffer in EnetReadPacket");
 
        /* the segment was already mapped into DVMA when it was allocated */
      }
    /* else: no segment present so 'RcvRec->data' is still valid */

    if( (RcvRec->pd = AllocatePd()) == NULL )
      Kabort("Couldn't get pd in EnetReadPacket");
    RcvRec->pd->packetType &= ALIEN_EMBEDDED;/* this pd will contain an alien */
    /* Compute the Multibus address of the receive pd's image in
     *   DVMA space */
    RcvRec->pd_pa = (Process *) ((char *)rcv_pd_phys_addr + 
                      ((char *)RcvRec->pd - (char *)downtopage(RcvRec->pd)));

    /* map the receive pd into DVMA space */
    MapDvma( rcv_pd_virt_addr, RcvRec->pd, sizeof(Process) );

    *RcvFlagPtr = 1;  /* allow the enp to process the next receive */


    /* Now go ahead and process the receive that caused the interrupt */

    /* Look at the first few words of the packet to determine
       if it is a kernel packet and not addressing an alias process */

    kp = (kPacket *) &last_pd->packetType;
    if( EnetHeader.EtherType == KERNEL_PACKET && DifferentIKCByteOrder(kp) )
      {
	SwapIKPacket(kp);
	kp->packetType &= ~IKC_LITTLE_ENDIAN; /* Just to be tidy */
      }
    if( EnetHeader.EtherType != KERNEL_PACKET ||
	(kp->dstPid & REMOTE_ALIAS_PROCESS) )
      {
	/* First case: it is not a kernel packet */
	/* and not to a remote alias process */

	if( (inst = EthernetInstance) == NULL )
	  {
	    /* Ethernet device not open */
	    goto below;  /* discard the packet */
	  }
	if( !MAP_TO_RPD(pd, inst->reader) )
	  {
	    /* No reader process */
	    if( !MAP_TO_RPD(pd, inst->owner) )
	      {
		/* Ethernet device not open */
		inst->owner = 0; /* Free the instance */
		EthernetInstance = NULL;
	      }
	    /* If no reader then we bite the bullet and discard
	     *   the packet (even if the reader may show up soon). 
	     */
	    goto below;
	  }
  
	req = (IoRequest *) &(pd->msg);
	/* First, fill in the bytecount in the reply */
	if ( (bytes+ENET_PLUS_KERNEL_HEADER_SIZE) < req->bytecount )
	  req->bytecount = bytes + ENET_PLUS_KERNEL_HEADER_SIZE;
	/* Now decide where to put the data */
	if( req->bytecount <= IO_MSG_BUFFER ) 
	  p = (short *) req->shortbuffer;
	else
	  p = (short *) req->bufferptr;
  
	/* Copy data to correct location */
	/* Note that we take a bath here because now we have to
	 *   "gather" the packet back together.
	 */
	oldteam = GetAddressableTeam();
	SetAddressableTeam(pd->team);

	/* need to gather the enet header, kernel packet, and data */
	*((Enet10Header *) p) = EnetHeader;  
	Copy( ((char *) p)+ENET10_HEADER_SIZE, &last_pd->packetType,
	  Min( sizeof(kPacket), req->bytecount - ENET10_HEADER_SIZE ) );
	if( req->bytecount > ENET_PLUS_KERNEL_HEADER_SIZE )
	  {
	    Copy( ((char *) p)+ENET_PLUS_KERNEL_HEADER_SIZE, 
		   (char *) last_pd->timeout_count,
		    req->bytecount - ENET_PLUS_KERNEL_HEADER_SIZE );
	  }
	SetAddressableTeam(oldteam);
  
	inst->reader = 0;
	req->requestcode = OK;
	Addready( pd );  /* send the reader process on its way */

      }
    else
      {
	/* Second case : a kernel packet */
	if (kp->packetType == remoteForward)
	  {
	    HostCacheEnter(kp->forwarder>>16, 
		EnetHeader.SrcHost);
	  }
	else
	  {
	    HostCacheEnter(kp->srcPid>>16, EnetHeader.SrcHost);
	  }
  
	/* Ensure in correct packetType range. */
	packetType = kp->packetType & 0xF;
	(*RemoteTrapTable[packetType])(kp);

      }

    /* release the pd and segment buffer (if necessary) */

   below:
    /* if the alien pd is not used by IKP level then discard it 
     *   ALIEN_EMBEDDED bit gets toggled in Alienate() */
    if( kp->packetType & ALIEN_EMBEDDED ) 
      FreePd( last_pd, 0 );

    /* if a segment buffer was used, release it */
    if( bytes )
      ReleaseSegBuffer( (HostSegBuffer *) last_pd->timeout_count );
  
  }



HostSegBuffer *GetSegBuffer( buf_phys_addr )
  char  **buf_phys_addr;  

  {
    HostSegBuffer  *free_buffer = NULL;

    /* lock the free buffer list first */
    while( TestAndSet( &SegBuffer_lock ) != 0 ) 
      ;

    if( (free_buffer = SegmentList) != NULL )
      {
        /* Compute the Multibus address of the segment buffer's image in
         *   DVMA space */
        *buf_phys_addr = ( (char *) host_segbufs_pa + 
                            ((char *)free_buffer - (char *)SegmentBuffers) );

        SegmentList = (HostSegBuffer *) *((unsigned *) SegmentList);
      }

    SegBuffer_lock = 0;
    return( free_buffer );
  }



ReleaseSegBuffer( free_buffer )
  HostSegBuffer *free_buffer;

  {
    /* lock the free buffer list first */

    while( TestAndSet( &SegBuffer_lock ) != 0 ) 
      ;

    *((unsigned *) free_buffer) = (unsigned) SegmentList;
    SegmentList = free_buffer;

    SegBuffer_lock = 0;
  }




#define SwapPids( pd )				\
  {						\
    register Process_id tmp;			\
    tmp = pd->pid;				\
    pd->pid = pd->blocked_on;			\
    pd->blocked_on =tmp;			\
  }



WriteKPacket( pd )
    register Process *pd;

   /*
    * This routine has to be called with all network fields filled in.
    * A corresponding packet will be written out, with a segment
    * appended starting at currentSegmentPtr and length.
    * On return currentSegmentPtr is incremented by length.
    * Routing rule: as a WF processor, this routes the pd to
    * the delay queue if the pd->state != READY, else adds
    * it into the ready queue.
    */
  {
    register SystemCode error;

#ifdef ENPDEBUG
    printx("WriteKPacket\n");
#endif 
    pd->currentSegmentPtr = pd->segmentPtr;

#ifdef VM
    /* Be sure requested segment is in physical memory before starting */
    /* %%% not quite right -- must lock exactly once so have to resume
     *  BELOW this code after unblocking */
    error = LockSegment(pd, pd->segmentPtr, pd->segmentSize, PA_IKCOUT);
    if (error == NO_REPLY) return;	/* will come back later */
    if (error != OK)
      {
	/* This should be an exception, probably */
	printx("WriteKPacket: LockSegment failed, pd %x, error %x\n");
	return;	/* process hangs */
      }
#endif VM

    /* Check if xmit record is being used. */
    if( TestAndSet( &EnetWriter ) == 0 ) 
      {
	/* no, so go ahead and use it */
	WriteNextKPacket( pd );
      }
    else	
      {
#ifdef ENPDEBUG
        printx("...queueing\n");
#endif 
   	/* Queue pd for later transmission. */
	switch (pd->packetType)
	  {
	  case breathOfLife:
	  case remoteMoveToReply:
	    /* Add to head of queue */
	    Lockq( &IkpOutq );
	    pd->queuePtr = &IkpOutq;
	    pd->link = IkpOutq.head;
	    if (IkpOutq.head == NULL) IkpOutq.tail = pd;
	    IkpOutq.head = pd;
	    Unlockq( &IkpOutq );
	    break;

	  default:
	    /* Add to end of queue */
	    pd->link = NULL;
	    Lockq( &IkpOutq );
	    pd->queuePtr = &IkpOutq;
	    IkpOutq.tail = (IkpOutq.tail->link = pd);
	    Unlockq( &IkpOutq );
	    break;
	  }
      }
  }


#define SwapPids( pd )				\
  {						\
    register Process_id tmp;			\
    tmp = pd->pid;				\
    pd->pid = pd->blocked_on;			\
    pd->blocked_on =tmp;			\
  }


WriteNextKPacket( pd )
    register Process *pd;

  {
    extern NetAddress HostCacheLookup();
    register unsigned bytes;
    register unsigned len, no_length_hack = 1;

#ifdef ENPDEBUG
    printx("WriteNextKPacket\n");
#endif
    bytes = 0;
    if( (pd->packetType == remoteMoveFromReq) ||
	(pd->packetType == remoteMoveToReply) )
      {
	no_length_hack = 0; /* Kludge for two special cases. */	
      }
    else if( pd->segmentSize )  /* Non-zero segment being sent. */
      {
	len = (unsigned)pd->segmentPtr + pd->segmentSize -
	    		(unsigned)pd->currentSegmentPtr;
	pd->length = min(len, MAX_APPENDED_SEGMENT);
	pd->remoteaddress = (Unspec *)
		    ( (unsigned)pd->remoteSegmentPtr
		      + ( (unsigned)pd->currentSegmentPtr
			  - (unsigned)pd->segmentPtr ) );
	bytes = (pd->length + 1) & ~1;
     }
    else pd->length = len = 0;


    if( AlienProcess(pd) ) SwapPids( pd );

    if( (pd->blocked_on&GROUP_ID_BIT) )
      {
	if( (pd->blocked_on&LOCAL_GROUP_BIT) )
	  {
	    ProcessId id;
	    /* Turn off group bit so cache lookup might work. */
	    id = pd->blocked_on & ~GROUP_ID_BIT;
	    XmitRec->EnetHeader.DestHost = HostCacheLookup(id >> 16);
	  }
	else /* Use a multicast address. */
	  {
	    XmitRec->EnetHeader.DestHost = MulticastDefault;
	    /* Now plug in the logical host group identifier from groupId. */
	    XmitRec->EnetHeader.DestHost.addrlow = (pd->blocked_on>>17);
	  }
      }
    else
      {
	XmitRec->EnetHeader.DestHost = HostCacheLookup(pd->blocked_on >> 16);
      }
    XmitRec->EnetHeader.SrcHost = EnetHostNumber;
    XmitRec->EnetHeader.EtherType = KERNEL_PACKET;

    /* kernel packet header */
    /* Compute the Multibus address of the packet header's image in
     *   DVMA space */
    XmitRec->piece[0].address = ( (char *)xmit_pd_phys_addr + 
       ((char *)(&pd->packetType) - (char *)downtopage(&pd->packetType)) );
    XmitRec->piece[0].length = sizeof( kPacket );

    /* map the pd into DVMA space */
    MapDvma( xmit_pd_virt_addr, &(pd->packetType), sizeof(Process) );

    if( AlienProcess(pd) ) SwapPids( pd );
    pd->packetType &= ~IKC_LITTLE_ENDIAN; /* Tell the world we're big-endian */

    if( (pd->length != 0) && no_length_hack )
      {
	/* segment */
        /* Compute the Multibus address of the segment's image in
         *   DVMA space */
        XmitRec->piece[0].address = ( (char *)xmit_seg_phys_addr + 
  ((char *)pd->currentSegmentPtr - (char *)downtopage(pd->currentSegmentPtr)) );
        XmitRec->piece[1].length = bytes;

        /* map the segment into DVMA space */
        MapDvma( xmit_seg_virt_addr, pd->currentSegmentPtr, bytes );

        XmitRec->numpieces = 3;  /* including header */
	pd->currentSegmentPtr = (Unspec *)
		    ( (unsigned)pd->currentSegmentPtr + pd->length );
      }
    else
      XmitRec->numpieces = 2;

    /* Check if we will have finished transmission for this pd. */
    if( no_length_hack && (len > MAX_APPENDED_SEGMENT) )
      {
        /* No: set a flag for interrupt routine. */
        XmitRec->xmit_again = 1;
      }
    else
      XmitRec->xmit_again = 0;

    XmitRec->current_pd = pd; /* record pd for interrupt routine */
    *XmitFlagPtr = 1;       /* cause the enp to process the record */

  }


int ReadDataPacket( copier, kp )
    register Process *copier;
    register kPacketWithSegment *kp;

  /* Copy kp->length bytes from segment portion of packet
   * associated with kp (kernel packet) into address space of copier,
   * starting at kp->remoteaddress.  The segment portion is pointed to by
   * 'timeout_count' of the alien pd that the kp is embedded in.
   */
  {
    register Process *alien;
    Team             *oldteam;

    /* Calculate address of alien, assuming &(alien->packetType) == kp */
    alien = (Process *) ( (char *) kp -
		 ((char *)&Idle_process.packetType - (char *)&Idle_process ));

    oldteam = GetAddressableTeam();
    SetAddressableTeam(copier->team);
    Copy( kp->remoteaddress, (char *) alien->timeout_count, kp->length );
    SetAddressableTeam(oldteam);
    return( OK );
  }



int NetCheck()
  {
    /* Invoked by timer interrupt routine.  Nothing to do because non-kernel
     *   packets are not saved around if there is no reader process.
     */
    return;
  }	  



/* TestAndSet is called as a function with a single argument being the
 *   address of the (unsigned char) semaphore.
 *   Returns 0 if the semaphore is 0 else returns > 0.
 */
asm("	.globl  TestAndSet	");
asm("TestAndSet:		");
asm("	clrl  d0		");
asm("	movl sp@(4), a0 	");
asm("	tas  a0@		");
asm("	sne  d0			");
asm("	rts			");	



