/*
 * Distributed V Kernel 
 * Copyright (c) 1982 by Stanford University.
 *
 * Kernel Ethernet driver
 * 3Com 10 Mbit version ("ec")
 */

#include "Vethernet.h"
#include "Vikc.h"
#include "Vquerykernel.h"
#include "interrupt.h"
#include "enetxln.h"
#include "dm.h"
#include "ikc.h"

typedef Process_id (*PFPID)();

/* Imports */
extern unsigned char *LastPeripheral;

/* Exports */

extern SystemCode EnetCreate();
extern SystemCode EnetRead();
extern SystemCode EnetWrite();
extern SystemCode EnetModify();
extern SystemCode EnetQuery();
extern SystemCode EnetRelease();
extern SystemCode EnetCheckRequest();
extern SystemCode EnetReadPacket();
extern SystemCode EnetPowerup();
extern int DiscardDataPacket();
extern int ReadDataPacket();

/* Forward */
struct MsgBuf *GetMsgBuf();

NetAddress	EnetHostNumber = 0;
				/* physical ethernet address */
DeviceInstance	*EthernetInstance = NULL;
				/* ptr to DeviceInstance for Ethernet */
ProcessId	EnetWriter = NULL;
				/* Non-zero if a writer using Ethernet */
int		EnetReceiveMask = ENET_DEFAULT_MASK;
				/* addresses to listen for */
int		EnetBadCounts = 0;	/* Number of bad count values */
unsigned short	EnetCollisions = 0;	/* Number of collision errors */
unsigned short	EnetOverflows = 0;	/* Queue overflow errors */
unsigned short	EnetCRCerrors = 0;	/* Packets with bad CRC's */
unsigned short	EnetSyncErrors = 0;	/* Receiver out of sync */
unsigned short	EnetTimeouts = 0;	/* Transmitter timeouts */
int		EnetValidPackets = 0;
unsigned short  EnetPromiscuous = 0;	/* Are we in promiscuous mode. */
unsigned short	EnetIntPending = 0;	/* Which interrupts have been disabled
					 *  to save packets for a reader to
					 *  pick up later */
EnetBlock 	*EnetPacket;	/* Points to current incoming ethernet pkt. */

/*
 * Significant addresses referring to Multibus memory.
 */

unsigned	MultibusMemAddr;	/* Base address of my area of */
					/* Multibus memory. */
char		*VMultibusMemAddr;	/* Virtual address of above. */

char		*ReadDataBuf;		/* Buffer for incoming packets. */
char		*WriteDataBuf;		/* Buffer for outgoing packets. */
char		*StatsBuffer;		/* Buffer for statistics replies. */

struct MsgQs	*MsgQPtr;		/* Start of request/reply queues. */
unsigned	MsgQVirAddr;		/* Virtual address of same. */
unsigned	MsgQPhysAddr;		/* Physical address of same. */
unsigned	MsgQVirSegAddr;		/* Start of segment of same--Exos's */
					/*  16 bit offsets are calculated */
					/*  from here.  */
int		NetAlarm = 0;		/* Set to nonzero when an unclaimed
					   user packet is received.  Dec-
					   remented on every click.  When
					   it reaches 0, packet is tossed. */
int		BytesInBuffer = 0;	/* Zero if there is no unclaimed
					   user packet in ReadDataBuf.
					   Else, the length of the packet. */

	/* NOTE:  In the current version of this driver, only one
	 *  buffer is reserved in Multibus memory for incoming packets.
	 *  This buffer can hold either a packet for a client or for
	 *  the kernel.  If a packet comes in for a client which hasn't
	 *  asked for it, it must be saved for ENET_READER_TIMEOUT clicks.
	 */

NetAddress	MulticastDefault = V_MULTICAST_ADDRESS;

SyncQueue IkpOutq;  /* Queue for out-going IKP packets. */

/* Macro expansion to interrupt-invoked C call to Ethernetinterrupt */
Call_inthandler(EnetInterrupt)


SystemCode EnetCreate( pd, inst )
    Process *pd;
    register DeviceInstance *inst;
  /* Create an instance for the Ethernet interface.
   */
 {
    register CreateInstanceRequest *req = (CreateInstanceRequest *) &(pd->msg);
    if( req->filemode != FCREATE ) return( MODE_NOT_SUPPORTED );

    if( EthernetInstance != NULL && 
        MapPid( EthernetInstance->owner) )
      {
	return( BUSY );
      }

    /*
     * Initialize the device instance, can assume all fields
     * are zero except for owner and id.
     */

    inst->readfunc = EnetRead;
    inst->writefunc = EnetWrite;
    inst->modifyfunc = EnetModify;
    inst->queryfunc = EnetQuery;
    inst->releasefunc = EnetRelease;
    inst->reader = 0;
    inst->writer = 0;

    EthernetInstance = inst; /* Record the instance for interrupt routine */

    inst->type = (READABLE+WRITEABLE+VARIABLE_BLOCK+STREAM);
    inst->blocksize = ENET10_MAX_PACKET;

    return( OK );
  }

 
SystemCode EnetPowerup()

  /* Powerup and initialize the Ethernet Interface Board.
   *  Done just once, from kernel Powerup() routine. 
   */
  {
    extern int Asm_EnetInterrupt();
    register int (**intvec)() = (int(**)()) ENET_INT_LEVEL;
    
    /* Indicate we have a 3Com board */
    *LastPeripheral++ = PRF_ENET_EXCELAN;
    IkpOutq.tail = (Process *) &(IkpOutq.head);
    IkpOutq.type = IKP_OUT_QUEUE;
    EnetReceiveMask = (ENET_SELF+ENET_BROADCAST);
    
    /* Plug interrupt location */
    *intvec = Asm_EnetInterrupt;

    EnetReset( ENET_DEFAULT_MASK );
  }

/* NOTE: This version of the driver ignores the recieve mask */
EnetReset( receiveMask )
    int receiveMask;
  {
    int dummy, i, bytes;
    long addr;
    struct ReadMsg *readmsg;
    struct SystemBuffers *systembuf;
    struct MsgBuf *bufptr, *nextbufptr;
    struct Init *configmsg;
    struct testp *testpat;

    /* Reset board. */

    dummy = PORTA;			/* Reading PORTA does reset. */
    i = 0;
    while( ( PORTB & X_ERROR ) == 0 )	/* Test status--wait for response. */
      if( i++ > 400000 )
	{
	  K_puts( "Excelan Driver:  Interface didn't reset." );
	  return( BAD_STATE );
	}

    /* Set up Multibus Memory--the shared memory between board and processor. */

    bytes = sizeof( struct SystemBuffers ) + sizeof( struct MsgQs );
    AllocateMultibusMemory( bytes, &MultibusMemAddr, &VMultibusMemAddr );
    systembuf = (struct SystemBuffers *) VMultibusMemAddr;
    ReadDataBuf = systembuf->readbuffer;
    WriteDataBuf = systembuf->writebuffer;
    StatsBuffer = systembuf->statsbuffer;
    MsgQVirAddr = (long)( VMultibusMemAddr + sizeof( struct SystemBuffers ) );
    MsgQPhysAddr = VirToPhys( MsgQVirAddr );
    MsgQVirSegAddr = (unsigned) VMultibusMemAddr;
	  /* EXOS requires this to be on a 16-byte boundary.  We know it
	   *  is since AllocateMultibusMemory aligns to page boundaries */
    clear( MsgQVirAddr, sizeof( struct MsgQs ) );

    /* Initialize message queues. */

    MsgQPtr = (struct MsgQs *) MsgQVirAddr;
    MsgQPtr->xreqhdr = HostToExAddr( MsgQPtr->requestbufs );
    MsgQPtr->xreptail = HostToExAddr( MsgQPtr->replybufs );
    MsgQPtr->hreqtail = MsgQPtr->requestbufs;	/* Not used--see GetMsgBuf. */
    MsgQPtr->hrephdr = MsgQPtr->replybufs;

    /* Initialize request buffers--Host to Exos buffers. */

    bufptr = &( MsgQPtr->requestbufs[ NREQBUFS - 1 ] );
    for( i = 0; i < NREQBUFS; i++ )
      {
	nextbufptr = &( MsgQPtr->requestbufs[i] );
	bufptr->next = nextbufptr;
	bufptr->msg.link = HostToExAddr( nextbufptr );
	bufptr->msg.length = MSGLEN;
	bufptr->msg.status = HOST_OWNER | EXOS_DONE;
	bufptr = nextbufptr;
      }

    /* Initialize reply buffers--Exos to Host buffers. */

    bufptr = &( MsgQPtr->replybufs[ NREPBUFS - 1 ] );
    for( i = 0; i < NREPBUFS; i++ )
      {
	nextbufptr = &( MsgQPtr->replybufs[i] );
	bufptr->next = nextbufptr;
	bufptr->msg.link = HostToExAddr( nextbufptr );
	bufptr->msg.length = MSGLEN;
	bufptr->msg.status = EXOS_OWNER | HOST_DONE;
	bufptr = nextbufptr;
      }

    /* Construct configuration message. */

    configmsg = (struct Init *) ReadDataBuf;	/* Beginning of buffer space. */
    clear( configmsg, sizeof *configmsg );	/* Defaults all values to 0. */

    configmsg->res0 = 1;			/* Excelan doc: pgs. 24-5. */
    configmsg->code = 0xff;
    configmsg->mode = 0;			/* Link level cntrlr mode. */
    configmsg->format[0] = 1;			/* Look at test pattern. */
    configmsg->format[1] = 1;
    testpat = (struct testp *) configmsg->mmap;
    testpat->b0 = 1;
    testpat->b1 = 3;
    testpat->b2 = 7;
    testpat->b3 = 0xf;
    testpat->w0 = 0x103;
    testpat->w1 = 0x70f;
    testpat->l0 = 0x103070f;
    configmsg->amode = 3;			/* Absolute address mode. */
    configmsg->mvblk = -1L;			/* 0xffffffff  */
    configmsg->nproc = 0xff;
    configmsg->nmbox = 0xff;
    configmsg->naddr = 0xff;
    configmsg->nhost = 1;
    configmsg->hxitype = 3;			/* Interrupt on some level. */
    configmsg->xhitype = 3;
    configmsg->hxseg = VirToPhys( MsgQVirSegAddr );
				/* Request Q offsets calculated from here. */
    configmsg->xhseg = VirToPhys( MsgQVirSegAddr );
				/* Reply Q offsets calculated from here. */
    configmsg->hxhead = ( (int) &MsgQPtr->xreqhdr - MsgQVirSegAddr );
    configmsg->xhhead = ( (int) &MsgQPtr->xreptail - MsgQVirSegAddr );

    /* Send FF FF 00 00 for synchronization. */
    addr = 0x0000ffff;
    for( i = 0; i < 4; i++ )
      {
	while( PORTB & X_NOT_READY );		/* Wait for device ready. */
	PORTB = addr & 0xff;
	addr >>= 8;
      }

    /* Now send physical address of configmsg, LSB first. */
    addr = VirToPhys( configmsg );
    for( i = 0; i < 4; i++ )
      {
	while( PORTB & X_NOT_READY );		/* Wait for device ready. */
	PORTB = addr & 0xff;
	addr >>= 8;
      }

    /* Wait for Exos to change code. */

    i = 0;
    while( configmsg->code == 0xff )
      if( i++ > 1000000 )
	{
	  Kabort( "Excelan driver: Timeout waiting for board to configure." );
          break;
	}
    if( configmsg->code != 0 ) Kabort( "Excelan driver: Error on initialization." );

    SetMode( CONNECT_TO_NET );

    /* Construct and send a read request--allowing reception of any packets
     * which might come in.
     */

    readmsg = (struct ReadMsg *) GetMsgBuf();
    if( !readmsg ) Kabort( "Excelan driver: No read msg buffer." );

    readmsg->request = NET_READ;
    readmsg->nblocks = 1;
    readmsg->block[0].len = ENET10_MAX_PACKET;
    readmsg->block[0].addr = VirToPhys( ReadDataBuf );
    readmsg->id = NET_READ;			/* Not used currently. */
    readmsg->status = EXOS_OWNER | HOST_DONE;
    PORTB = 0;					/* Interrupt board to notify
						   it of new read request. */

    /* Get ethernet address of this host. */
    GetSlot(PHYS_ADDRS_SLOT, &EnetHostNumber);
    return( OK );
  }

SystemCode EnetRelease( pd, inst )
    Process *pd;
    DeviceInstance *inst;
   /*
    * Release the ethernet instance.
    */
  {
    IoRequest *req = (IoRequest *) &(pd->msg);

    inst->owner = 0; /* Free the instance */
    /* It is assumed that the reader will unblock on next packet
     * if currently blocked.
     */
    return( OK );
  }


SystemCode EnetRead( pd, inst ) 
    register Process *pd;
    register DeviceInstance *inst;
 /*
  * Handle a read instance request for the Ethernet interface.
  */
  {
    IoRequest *req = (IoRequest *) &(pd->msg);
    SystemCode	r;

    /*
     * Reply for RETRY if already a waiting reader process.
     * This effectively provides busy-wait queuing for reading.
     */
    if( MapPid(inst->reader) ) return( RETRY );

    if( (r=EnetCheckRequest(pd, req)) != OK ) return( r );

    /* Check if packet waiting in network interface.
     */
    /* Wait for a packet to arrive */
    inst->reader = pd->pid;
    pd->state = AWAITING_INT;

    if( EnetReadPacket( 1 ) == OK ) return( OK );

    return( NO_REPLY );
  }


SystemCode EnetWrite( pd, inst )
    register Process *pd;
    register DeviceInstance *inst;

   /* Handle a write instance request to the Ethernet interface.
    */
  {
    extern ProcessId EnetWriter;
    register IoRequest *req = (IoRequest *) &(pd->msg);
    SystemCode r;
    register short *enetptr;
    register unsigned bytes;
    struct XmitMsg *xmitmsg;

    if( (r = EnetCheckRequest(pd, req)) != OK ) return( r );

    /*
     * Reply for RETRY if currently writing.
     * This effectively provides busy-wait queuing for writing.
     */
    if( EnetWriter ) return( RETRY );

    bytes = req->bytecount;

    EnetWriter = pd->pid;
  
    /* Transfer a packet to the Ethernet Interface Board. */

    xmitmsg = (struct XmitMsg *) GetMsgBuf();
    if( !xmitmsg ) Kabort( "Excelan Driver: EnetWrite can't get msg buf." );

    Copy( WriteDataBuf, req->bufferptr, bytes );
    				/* Copy user's packet into system buffer. */

    xmitmsg->request = NET_XMIT;
    xmitmsg->nblocks = 1;
    xmitmsg->block[0].len = bytes;
    xmitmsg->block[0].addr = (long) VirToPhys( WriteDataBuf );
    xmitmsg->id = NET_XMIT;	/* Right now, only one transmit out at a time.*/
    xmitmsg->status = EXOS_OWNER|HOST_DONE;
    PORTB = 0;			/* Interrupt the board. */

    /* Don't wait for the interrupt. No point */

    return( OK );
  }

/* NOTE: EnetModify not implemented in this driver */

SystemCode EnetModify( pd, inst, dirIndex )
    register Process *pd;
    DeviceInstance *inst;
    unsigned short dirIndex;
  {

   return( OK );
  }    

SystemCode EnetQuery( pd, inst, dirIndex ) 
    Process *pd;
    DeviceInstance *inst;
    unsigned short dirIndex;
  {
    register QueryEnetReply *reply = (QueryEnetReply *) &(pd->msg);

    /* Send a NET_STATS request to the board, and return its answers to the
     * querier.  Note that we are normally configuring the interface to
     * ignore packets with CRC or alignment errors, so these figures may
     * be 0 all the time.  Also note that the board does not count the
     * number of collisions, so an estimate of these is maintained in
     * EnetInterrupt.
     */

    struct StatsMsg *statsmsg;
    struct StatsArray *statsarray;

    statsmsg = (struct StatsMsg *) GetMsgBuf();
    if( !statsmsg )
      Kabort( "Excelan Driver: EnetQuery couldn't get a msg buffer." );

    statsmsg->request = NET_STATS;
    statsmsg->reqmask = READREQ;
    statsmsg->res1 = 0;
    statsmsg->nobj = NUM_STATS_OBJS;
    statsmsg->index = 0;		/* Start of stats array. */
    statsmsg->bufaddr = (long) VirToPhys( StatsBuffer );
    statsmsg->id = NET_STATS;
    statsmsg->status = EXOS_OWNER | HOST_DONE;
    PORTB = 0;

    WaitForCompletion( NET_STATS );	/* Stats calls don't take long-- */
					/* we can wait? */

    statsarray = (struct StatsArray *) StatsBuffer;

    reply->NetworkType = ENET_TYPE_10MBIT;
    reply->NumCollisions = EnetCollisions;
    reply->NumOverflows = statsarray->frLostNoBuffers;
    reply->NumCRCErrors = statsarray->frRecvdCRCError;
    reply->NumSyncErrors = statsarray->frRecvdAlignError;
    reply->NumTimeOuts = statsarray->frAbortExcessCollisions;
    reply->ReceiveMask = ENET_DEFAULT_MASK;	/* not really implemented */
    reply->NumValidPackets = statsarray->frRecvdNoError;
    reply->HostAddress.e10 = EnetHostNumber;

    return( OK );
  } 


SystemCode EnetCheckRequest( pd, req ) 
    register Process *pd;
    register IoRequest *req;

  /* Check that the read or write request has a legitimate buffer, etc.
   */
  {
    register unsigned count;
    register SystemCode r;

    /*  Check length */
    count = req->bytecount;

    req->bytecount = 0; /* To be left zero if a check fails */
    if( count > ENET10_MAX_PACKET || (count <= IO_MSG_BUFFER) )
	r = BAD_BYTE_COUNT;
    else  /* Make sure data pointer is valid.
	   *  Check that on a word boundary (WHY? --TPM)
	   *  and not in the kernel area.
	   */
    if( (BadUserPtr(req->bufferptr)) ||
    (pd->team->team_space.size < (req->bufferptr + count)) ||
    ((int) req->bufferptr) & 1 )
	r = BAD_BUFFER;
    else
      {
	req->bytecount = count;
	r = OK;
      }
    return( r );
  }


EnetInterrupt()

   /* Handle an interrupt from the ethernet interface.
    */
  {
    extern ProcessId EnetWriter;
    register Process *pd;
    register unsigned short mecsr;
    register struct MsgBuf *replybuf;
    register struct ReadMsg *readmsg;
 
   /*
    * Handle an interrupt from the ethernet interface.  These occur whenever
    * Exos changes one of the queues--either reads a request or writes a
    * reply.  We are, at this point, only interested in replies, so we ignore
    * interrupts which haven't changed the reply queue.
    */
    PORTA = 1; /* clear the interrupt */
    replybuf = MsgQPtr -> hrephdr;
    while( replybuf->msg.status == (HOST_OWNER|EXOS_DONE) )
      {
	/* A reply is ready! */
        if ( replybuf->msg.request == NET_XMIT)
          {

          /* Unless something catastrophic happens, the reply field of the */
          /* message buffer will have 0 if there were no collisions, 1 if */
          /* there was 1 collision, 2 if there were more than 1 collision. */
            if( replybuf->msg.reply < 3 )
	        EnetCollisions += replybuf->msg.reply;

	   /* If IkpOutq is non-empty, transmit next interkernel packet. */
	    if( (pd=IkpOutq.head) != NULL )
	      {
	        Lockq( &IkpOutq );
	        if( (IkpOutq.head = pd->link) == NULL )
		    IkpOutq.tail = (Process *) &(IkpOutq.head);
	        pd->queuePtr = NULL;
	        Unlockq( &IkpOutq );
	        WriteNextKPacket( pd );
	      }
	    else EnetWriter = 0; /* No transmission in progress now */
          }
       else if ( replybuf->msg.request == NET_READ)
          {
	    BytesInBuffer = replybuf->msg.block[0].len;
	    EnetReadPacket( 0 );
            /* Request a new packet from the board.
	     */
	    readmsg = (struct ReadMsg *) GetMsgBuf();
	    if( !readmsg )
	         Kabort( "Excelan Driver: Couldn't get a msg buffer." );

	    readmsg->request = NET_READ;
	    readmsg->nblocks = 1;
	    readmsg->block[0].len = ENET10_MAX_PACKET;
	    readmsg->block[0].addr = VirToPhys( ReadDataBuf );
	    readmsg->id = NET_READ;
	    readmsg->status = EXOS_OWNER | HOST_DONE;
	    PORTB = 0;
	  }

	/* else  NET Command.  We busy-waited for this to be done so */
	/*   nothing has to be done here.  See SetMode for more info. */

	/* Return this buffer to Exos's control and look at the next to */
	/* see if there was more than one reason for this interrupt. */

	replybuf->msg.length = MSGLEN;
	replybuf->msg.status = EXOS_OWNER|HOST_DONE;
	replybuf = replybuf->next;
      }

    /* Look at this reply buffer next time we get an interrupt. */
    MsgQPtr->hrephdr = replybuf;
  }


SystemCode EnetReadPacket( invoker )
    unsigned invoker;
   /*
    *  Physically transfer a packet from the Ethernet
    *  Interface Board into the reading process's buffer.
    * Takes slightly different actions, depending on invoker,
    * which can be interrupt routine - 0, client - 1, or
    * NetCheck - 2.
    */
  {
    extern PFPID RemoteTrapTable[];
    Process *pd;
    register DeviceInstance *inst;
    register kPacket *kp;
    register IoRequest *req;
    Team *oldteam;
    register short *p;
    SystemCode ret;
    /* SendGroupMembers() may find a remote alias process group member. */
    extern int DeliverToEnetReader;

    /* If we are entering from NetCheck then it is possible
     * that the packet buffer has been overwritten (and used)
     * while the timer has been counting down.
     */
    if( BytesInBuffer == 0 ) return(END_OF_FILE);
    
    /* This is a rudimentary packet check */
    if(BytesInBuffer < MAGIC_MIN_PACKET_SIZE )
      {
	/* Ignore this poor excuse for a packet and ask for others. */
 	/* that might be coming in. */
	ret = END_OF_FILE;
	goto finishpacket;
      }

    /* Look at the first few words of the packet to determine
     * if it is a kernel packet and not addressing an alias process
     */
    DeliverToEnetReader = 0;
    EnetPacket = (EnetBlock *) ReadDataBuf;
    kp = (kPacket *) EnetPacket->data;
    if( EnetPacket->EtherType == KERNEL_PACKET && DifferentIKCByteOrder(kp) )
      {
	SwapIKPacket(kp);
	kp->packetType &= ~IKC_LITTLE_ENDIAN; /* Just to be tidy */
      }
    if( EnetPacket->EtherType != KERNEL_PACKET ||
	(kp->dstPid & REMOTE_ALIAS_PROCESS) )
      {

nonKernelPacket:
	/* First case: it is not a kernel packet
	 * and not to a remote alias process
	 */
	ret = NO_REPLY;
	if( (inst = EthernetInstance) == NULL )
	    /* Ethernet device not open */
	    return( ret );

	if( !MAP_TO_RPD(pd, inst->reader) )
	  /* No reader process */
	  {
	    if( !MAP_TO_RPD(pd, inst->owner) )
	      {
		/* Ethernet device not open */
		inst->owner = 0; /* Free the instance */
		EthernetInstance = NULL;
		inst = NULL;
		EnetIntPending = 0;
		EnetReset( ENET_DEFAULT_MASK );
	        goto finishpacket;
	      }

#ifdef undef
            /* This is a chunk of code from the 3com driver
	     * that really isn't applicable here. The 3com
	     * driver disables interrupts on a full buffer.
	     * Here we just let it get overwritten with the
	     * next packet that comes along. This driver
	     * should probably be modified to have a single
	     * buffer for storing the last unread user packet.
	     * (Besides - this driver doesn't support
	     * promiscuous mode yet.)
	     */
	     
	    /* If in promiscuous mode, dont save packet if
	     * it is a kernel packet - too risky.*/
	    if( EnetPromiscuous &&
		(EnetPacket->EtherType == KERNEL_PACKET &&
		(kp->dstPid & REMOTE_ALIAS_PROCESS)) )
		goto finishpacket;
#endif
	    /* There is an instance but no reader.
	     * Let the packet sit in the read buffer,
	     * checking to see if a reader comes along
	     * every ENET_READER_TIMEOUT clicks. The 
	     * next packet to come in will overwrite 
	     * the buffer.
	     */
	    NetAlarm = ENET_READER_TIMEOUT;
	    return( ret );
	  }

	/* There is a reader waiting for this packet. Transfer the
	 * packet from the read buffer to the waiting process buffer.
	 */
	req = (IoRequest *) &(pd->msg);
	/* First, fill in the bytecount in the reply */
	if ( BytesInBuffer < req->bytecount )
	    req->bytecount = BytesInBuffer;
	/* Now decide where to put the data */
	if( req->bytecount <= IO_MSG_BUFFER ) 
	    p = (short *) req->shortbuffer;
	else
	    p = (short *) req->bufferptr;

	/* Copy data to correct location */
	oldteam = GetAddressableTeam();
	SetAddressableTeam(pd->team);
	Copy( p, EnetPacket, req->bytecount );
	SetAddressableTeam(oldteam);
	ret = OK;
	inst->reader = 0;

	req->requestcode = OK;
	Addready( pd );
      }
    else
      {
	/* Second case : a kernel packet */
	if (kp->packetType == remoteForward)
	  {
/*	    HostCacheEnter(kp->forwarder>>16, EnetPacket->SrcHost); */
	  }
	else
	  {
	    HostCacheEnter(kp->srcPid>>16, EnetPacket->SrcHost);
	  }
	/* Ensure in correct packetType range. */
	kp->packetType = kp->packetType & 0xF;
	(*RemoteTrapTable[kp->packetType])(kp);
	ret = NO_REPLY;
	/* If in promiscuous mode, let the enet reader get too. */
	if( EnetPromiscuous || DeliverToEnetReader ) goto nonKernelPacket;
      }
finishpacket:
    BytesInBuffer = 0;
    ++EnetValidPackets;
    return( ret );
  }


WriteKPacket( pd )
    register Process *pd;

    /*
	This routine has to be called with all network fields filled in.
	A corresponding packet will be written out, with a segment
	appended starting at currentSegmentPtr and length.
	On return currentSegmentPtr is incremented by length.
    * Routing rule: as a WF processor, this routes the pd to
    * the delay queue if the pd->state != READY, else adds
    * it into the ready queue.
    */
  {
    extern Process_id EnetWriter;

/*%%%*/if (pd->queuePtr != NULL)
	{
	  printx("WriteKPacket: process %x already on queue %x\n", 
		pd, pd->queuePtr);
	  StackDump();
	  ;asm("trap #14");;
        }


    pd->currentSegmentPtr = pd->segmentPtr;
    /* Check if device is busy. */
    if( !EnetWriter ) /* Test and set required here. */
      {
	EnetWriter = 1;
	WriteNextKPacket( pd );
      }
    else	
      {   /* Queue pd for later transmission. */
	pd->link = NULL;
	Lockq( &IkpOutq );
	pd->queuePtr = &IkpOutq;
	IkpOutq.tail = (IkpOutq.tail->link = pd);
	Unlockq( &IkpOutq );
      }
  }

#define SwapPids( pd )				\
  {						\
    register Process_id tmp;			\
    tmp = pd->pid;				\
    pd->pid = pd->blocked_on;			\
    pd->blocked_on =tmp;			\
  }

WriteNextKPacket( pd )
    register Process *pd;

  {
    extern NetAddress HostCacheLookup();
    register unsigned bytes, i;
    register unsigned *ptr, *pktPtr;
    register EnetBlock *enetPacket;
    register unsigned len, no_length_hack = 1;
    struct XmitMsg *xmitmsg;

    bytes = 0;
    if( (pd->packetType == remoteMoveFromReq) ||
	(pd->packetType == remoteMoveToReply) )
      {
	no_length_hack = 0; /* Kludge for two special cases. */	
      }
    else if( pd->segmentSize )  /* Non-zero segment being sent. */
      {
	len = (unsigned)pd->segmentPtr + pd->segmentSize -
	    		(unsigned)pd->currentSegmentPtr;
	pd->length = min(len, MAX_APPENDED_SEGMENT);
	pd->remoteaddress = (Unspec *)
		    ( (unsigned)pd->remoteSegmentPtr
		      + ( (unsigned)pd->currentSegmentPtr
			  - (unsigned)pd->segmentPtr ) );
	bytes = (pd->length + 1) & ~1;
     }
    else pd->length = len = 0;

    if( AlienProcess(pd) ) SwapPids( pd );

    /* Fill in ethernet packet header */
    enetPacket = (EnetBlock *) WriteDataBuf;
		/* Save this ptr for later use. */
    if( (pd->blocked_on&GROUP_ID_BIT) )
      {
	if( (pd->blocked_on&LOCAL_GROUP_BIT) )
	  {
	    ProcessId id;
	    /* Turn off group bit so cache lookup might work. */
	    id = pd->blocked_on & ~GROUP_ID_BIT;
	    enetPacket->DestHost = HostCacheLookup(id >> 16);
	  }
	else /* Use a multicast address. */
	  {
	    enetPacket->DestHost = MulticastDefault;
	    /* Now plug in the logical host group identifier from groupId. */
	    enetPacket->DestHost.addrlow = (pd->blocked_on>>17);
	  }
      }
    else
      {
	enetPacket->DestHost = HostCacheLookup(pd->blocked_on >> 16);
      }
    enetPacket->SrcHost = EnetHostNumber;
    enetPacket->EtherType = KERNEL_PACKET;

    pd->packetType &= ~IKC_LITTLE_ENDIAN; /* Tell the world we're big-endian */

    /* Send kernel packet header */
    ptr = (unsigned *) &(pd->packetType);
    pktPtr = (unsigned *) enetPacket->data;
    i = (WORDS_PER_PACKET >> 1);
				/* Assume an even value. */
    while( i-- ) 
      {
        *pktPtr++ = *ptr++;
        *pktPtr++ = *ptr++;
      }
    if( AlienProcess(pd) ) SwapPids( pd );

    if( (pd->length != 0) && no_length_hack )
      {
	Team *oldteam;

	oldteam = GetAddressableTeam();
	SetAddressableTeam(pd->team);
	/* Send segment if any */
	Copy(pktPtr, pd->currentSegmentPtr, bytes );
	SetAddressableTeam(oldteam);
	pd->currentSegmentPtr = (Unspec *)
		    ( (unsigned)pd->currentSegmentPtr + pd->length );
      }

    /* Hand the packet off to the ethernet board. */
    xmitmsg = (struct XmitMsg *) GetMsgBuf();
    if (!xmitmsg)
        Kabort("Excelan driver:WriteNextKPacket couldn't get a msg buffer.");
    xmitmsg->request = NET_XMIT;
    xmitmsg->nblocks = 1;
    /* Length includes Ethernet header, packet itself, and */
    /* appended data bytes. */
    xmitmsg->block[0].len = sizeof( Enet10Header ) +
			     sizeof( kPacket ) + bytes;
    xmitmsg->block[0].addr = VirToPhys( WriteDataBuf );
    xmitmsg->id = NET_XMIT;
    xmitmsg->status = EXOS_OWNER | HOST_DONE;
    PORTB = 0;

    /* Check if we have finished transmission for this pd. */
    if( no_length_hack && (len > MAX_APPENDED_SEGMENT) )
      { /* No: requeue on Ikp queue - at the end.*/
	pd->link = NULL;
	Lockq( &IkpOutq );
	pd->queuePtr = &IkpOutq;
	IkpOutq.tail = (IkpOutq.tail->link = pd);
	Unlockq( &IkpOutq );
      }
    else
      { /* Finished with this process; route to next queue. */
	if( pd->state == READY ) Addready( pd );
	else DelayProcess( pd );
      }
  }


int ReadDataPacket( copier, kp )
    register Process *copier;
    register kPacketWithSegment *kp;
  /* Copy kp->length bytes from segment portion of packet
   * associated with kp (kernel packet) into address space of copier,
   * starting at kp->remoteaddress.
   */
  {
    register Team *oldteam;

    oldteam = GetAddressableTeam();
    SetAddressableTeam(copier->team);
    /* Copy starting right AFTER the msg array. */
    Copy( kp->remoteaddress, kp->data, kp->length );
    SetAddressableTeam(oldteam);

    return( OK );
  }


int DiscardDataPacket( kp ) 
    register kPacket *kp;
  {
    return( OK );
  }

int NetCheck()
  {
    register Process *pd;

    if( EthernetInstance != NULL && MAP_TO_RPD( pd, EthernetInstance->reader ) )
      {
	/* Reader is present: return immediately */
	/* Note: this allows the reader to read this packet even if
	 *  the owner has gone away -- we don't detect that case here */
	return;
      }
    /* Reader not present: read out of interface and re-enable the interrupt */
    EnetReadPacket( 2 );
  }	  

struct MsgBuf *IGetMsgBuf()
  {
    /* Find an empty request buffer.  Note that we start looking for an
     * empty one where Exos is about to look for a request--not at the
     * end of the request queue, which would be MsgQPtr->hreqtail.  It
     * doesn't matter in our case, since we never send more than one
     * message at a time, but when I'm sure I know what's going on, I'm
     * going to go back and fix this.  Besides, it's the only use of
     * the macro ExToHostAddr.  --EJB
     */
    register int i;
    register struct MsgBuf *bufptr;

    bufptr = (struct MsgBuf *) ExToHostAddr( MsgQPtr->xreqhdr );
    i = 0;
    while( i++ < NREQBUFS  &&  bufptr->msg.status != (HOST_OWNER|EXOS_DONE) )
      bufptr = bufptr->next;
    if( bufptr->msg.status == (HOST_OWNER|EXOS_DONE) )
      {
	bufptr->msg.status = HOST_OWNER|HOST_DONE;
	bufptr->msg.res = 0;
	clear( &bufptr->msg.pad, MSGLEN );
	return( bufptr );
      }
    else
	return( NULL );
  }



struct MsgBuf *GetMsgBuf()
  {
    /*
     * Try 50 times to get a message buffer.
     */
    register timer;
    register struct MsgBuf *bufptr;

    timer = 50;
    while( timer-- != 0 )
      if( (bufptr = IGetMsgBuf()) != NULL ) return( bufptr );
    return( (struct MsgBuf *) NULL );
  }


SetMode( newmode )
	unsigned newmode;
  {
    /*
     * Set the mode of the interface.
     */
    register struct ModeMsg *modemsg;
    register timer;

    modemsg = (struct ModeMsg *) GetMsgBuf();
    if( !modemsg ) Kabort( "Excelan Driver: Mode set couldn't get msg buffer.");

    modemsg->request = NET_MODE;
    modemsg->reqmask = WRITEREQ;
    modemsg->mode = newmode;
    modemsg->id = NET_MODE;	/* No more than one mode cmd at a time. */
    modemsg->status = EXOS_OWNER|HOST_DONE;
    PORTB = 0;			/* Interrupt interface. */

    /* Wait for a response from the interface.  NOTE: We are being extremely
     * cautious and not reenabling interrupts at this point (they're turned
     * off in the kernel).  Therefore, the only way for Exos to tell us it
     * is done with this command is to add a new reply to the tail of the
     * reply queue.  This scheme will work ONLY because there are no other
     * requests outstanding at the time this routine is called.
     */

    timer = X_TIMEOUT;
    while( timer-- != 0 )
      if( (MsgQPtr->hrephdr)->msg.status == HOST_OWNER|EXOS_DONE )
	{
	  /* Since I am effectively processing an interrupt before */
	  /* it arrives, I must do all the things EnetInterrupt would. */
	  /* EnetInterrupt will then ignore the interrupt when it is sent, */
	  /* because it won't see any buffers which belong to the Host. */
	  (MsgQPtr->hrephdr)->msg.length = MSGLEN;
	  (MsgQPtr->hrephdr)->msg.status = EXOS_OWNER | HOST_DONE;
	  (MsgQPtr->hrephdr) = (MsgQPtr->hrephdr)->next;
	  return;
	}
    K_puts( "Excelan Driver: SetMode timed out." );
  }



GetSlot( slotno, dst )
	int slotno;
	char *dst;
  {
    /*
     * Get the address stored in the address slot slotno.  When slotno =
     * PHYS_ADRS_SLOT, returns the physical address of the board.
     */
    register struct AddrsMsg *addrmsg;
    register int timer;
    int i;
    char *src;
    struct AddrsMsg *msgptr;

    addrmsg = (struct AddrsMsg *) GetMsgBuf();
    if( !addrmsg ) Kabort( "Excelan Driver: GetSlot can't get a msg buffer." );

    addrmsg->request = NET_ADDRS;
    addrmsg->reqmask = READREQ;
    addrmsg->slot = slotno;
    addrmsg->id = NET_ADDRS;	/* No more than one outstanding at a time. */
    addrmsg->status = EXOS_OWNER|HOST_DONE;
    PORTB = 0;			/* Interrupt the board. */

    /* For the reasoning behind this section, see SetMode. */

    timer = X_TIMEOUT;
    while( timer-- != 0 )
      if( (MsgQPtr->hrephdr)->msg.status == HOST_OWNER|EXOS_DONE )
	{
	  msgptr = (struct AddrsMsg *) &( (MsgQPtr->hrephdr)->msg );
	  src = (char *) &( msgptr->addr );
	  for( i = 0; i < sizeof( NetAddress ); i++ )
	    *dst++ = *src++;

	  /* Since I am effectively processing an interrupt before */
	  /* it arrives, I must do all the things EnetInterrupt would. */
	  /* EnetInterrupt will then ignore the interrupt when it is sent, */
	  /* because it won't see any buffers which belong to the Host. */
	  (MsgQPtr->hrephdr)->msg.length = MSGLEN;
	  (MsgQPtr->hrephdr)->msg.status = EXOS_OWNER | HOST_DONE;
	  (MsgQPtr->hrephdr) = (MsgQPtr->hrephdr)->next;
	  return;
	}
    K_puts( "Excelan Driver: GetSlot timed out." );
  }



WaitForCompletion( id )
	long id;
  {
    /* Wait for the completion of the command whose id is id. */
    /* The strategy is to examine all the reply buffers in the reply */
    /* message queue until one appears with the correct id, or timeout */
    /* occurs. */

    int timer, i;
    int found = FALSE;
    struct MsgBuf *bufptr;

    timer = X_TIMEOUT;
    bufptr = MsgQPtr->hrephdr;
    while( timer-- > 0  &&  !found )
      {
	for( i = 0; i < NREPBUFS; i++ )
	  {
	    if( bufptr->msg.status == (HOST_OWNER | EXOS_DONE) &&
	      bufptr->msg.id == id )
	      found = TRUE;
	    bufptr = bufptr->next;
	  }
      }
    if( !found ) Kabort( "Excelan Driver: WaitForCompletion timed out." );
  }

SystemCode AddLogicalHostGroup( lhg ) GroupId lhg; { return( OK ); }
SystemCode DeleteLogicalHostGroup( lhg ) GroupId lhg; { return( OK ); }
