/*
 * Distributed V Kernel 
 * Copyright (c) 1982 by Stanford University.
 *
 * Kernel Ethernet driver
 * 3Com 10 Mbit version ("ec")
 */

#include "Venviron.h"
#include "Vethernet.h"
#include "Vikc.h"
#include "Vquerykernel.h"
#include "interrupt.h"
#include "enet3com.h"
#include "dm.h"
#include "ikc.h"

typedef Process_id (*PFPID)();

typedef struct { char *ptr; unsigned long bytes; } BufferList;

/* Imports */
extern unsigned char *LastPeripheral;

/* Variables imported from enet.c */
extern DeviceInstance	*EthernetInstance;
extern ProcessId	EnetWriter;	/* Pid of writer; NULL if free */
extern NetAddress	MulticastDefault;
extern char		UserPacketBuffer[];
extern int		UserPacketLength;
extern SyncQueue	IkpOutq;

/* Exports */
extern SystemCode EnetModify();
extern SystemCode EnetQuery();
extern SystemCode EnetPowerup();
extern NetworkWrite();

/* Used internally */
SystemCode EnetReadPacket();

NetAddress	EnetHostNumber;	/* physical ethernet address */
int		EnetReceiveMask = ENET_DEFAULT_MASK;
int		EnetBadCounts = 0;	/* Number of bad count values */
unsigned short	EnetCollisions = 0;	/* Number of collision errors */
unsigned short  EnetPacketCollisions = 0; /* Collisions for current packet. */
unsigned short	EnetOverflows = 0;	/* Queue overflow errors */
unsigned short	EnetCRCerrors = 0;	/* Packets with bad CRC's */
unsigned short	EnetSyncErrors = 0;	/* Receiver out of sync */
unsigned short	EnetTimeouts = 0;	/* Transmitter timeouts */
int		EnetValidPackets = 0;
int		NetAlarm = 0;		/* Not used in this driver */
unsigned short  EnetRxmode;		/* Mode for MECSR register. */
unsigned short  EnetPromiscuous = 0;	/* Are we in promiscuous mode. */
EnetBlock 	*EnetPacket;	/* Points to current incoming ethernet pkt. */

/* Macro expansion to interrupt-invoked C call to Ethernetinterrupt */
Call_inthandler(EnetInterrupt)


SystemCode EnetPowerup()
  /* Powerup and initialize the Ethernet Interface Board.
   *  Done just once, from kernel Powerup() routine. 
   */
  {
    extern int Asm_EnetInterrupt();
    register int (**intvec)() = (int(**)()) ENET_INT_LEVEL;
    
    /* Indicate we have a 3Com board */
    *LastPeripheral++ = PRF_ENET_3COM;
    IkpOutq.tail = (Process *) &(IkpOutq.head);
    IkpOutq.type = IKP_OUT_QUEUE;
    EnetReceiveMask = (ENET_SELF+ENET_BROADCAST);
    srand(GenerateRandomNumber());	/* for collision backoff */
    
    /* Plug interrupt location */
    *intvec = Asm_EnetInterrupt;

    EnetReset( ENET_DEFAULT_MASK );
  }


EnetReset( receiveMask )
    int receiveMask;
  {
    short rxmode;

    /* Change the receive mask. */
    EnetReceiveMask = receiveMask;
    EnetPromiscuous = 0;
    switch( receiveMask )
      {
	case ENET_BROADCAST: rxmode = RXmulti;
		break;
	case ENET_PROMISCUOUS: rxmode = RXall; EnetPromiscuous = 1;
		break;
	case ENET_SELF: rxmode = RXbroad;
		break;
 	case ENET_DEFAULT_MASK: rxmode = RXmulti;
		break;
	default: rxmode = RXmulti; EnetReceiveMask = ENET_DEFAULT_MASK;
     }
    EnetRxmode = rxmode;
    /* Clear all interrupts, flush all buffers */
    *(unsigned short *)MECSR = RESET;

    /*
     * Read Ethernet address of this host from the On-board PROM
     * and put it in the Address RAM.  Flick the AMSW bit in
     * MECSR to give the controller its address
     */

    /* Get ethernet address of this host. */
    EnetHostNumber = *(NetAddress *)MEADRAM = *(NetAddress *)MEADROM;

    SetCSR(MECSR, AMSW);	/* Give the ME its address */

    /*
     * Establish the ME receive criteria, and hand it both
     * receiver buffers
     */
    SetCSR(MECSR, rxmode | ABSW | BBSW);

    /* Enable receive interrupts */
    SetCSR(MECSR, AINTEN | BINTEN);
  }


EnetInterrupt()
   /* Handle an interrupt from the ethernet interface.
    */
  {
    extern ProcessId EnetWriter;
    register Process *pd;
    register unsigned short mecsr;
 
    /*
     * Determine type of interrupt.
     */
    mecsr = *(unsigned short *)MECSR;
    if( (mecsr & (JAM+JINTEN)) == (JAM+JINTEN) )
      {
	/* Collision */
	TryEnetTransmit(); 
	return;
      }
    
    /* Which buffers have packets ? */
    if ( ((mecsr & (ABSW | BBSW)) == 0) && ( mecsr & RBBA ) )
      {
	/* Both buffers are full, and B filled before A */
	EnetReadPacket( MEBHDR );
	mecsr |= BBSW;			/* Don't read from B later ! */
	SetCSR(MECSR, BBSW | BINTEN );	/* Enable buffer B */
      }

    if ( ( mecsr & ABSW ) == 0 )
      {
	EnetReadPacket( MEAHDR );
	SetCSR(MECSR, ABSW | AINTEN );
      }

    if ( ( mecsr & BBSW ) == 0 )
      {
	EnetReadPacket( MEBHDR );
	SetCSR(MECSR, BBSW | BINTEN );
      }

    if( (mecsr & (TBSW+TINTEN)) == (0+TINTEN) )
      {
	/* Transmitter interrupt */
	ResetCSR(MECSR, TINTEN);

	/* If IkpOutq is non-empty, transmit next interkernel packet. */
	Lockq( &IkpOutq );
	if( (pd=IkpOutq.head) != NULL )
	  {
	    if( (IkpOutq.head = pd->link) == NULL )
		IkpOutq.tail = (Process *) &(IkpOutq.head);
	    pd->queuePtr = NULL;
	    Unlockq( &IkpOutq );
	    WriteNextKPacket( pd );
	  }
	else
	  {
	    Unlockq( &IkpOutq );
	    EnetWriter = 0; /* No transmission in progress now */
	  }
      }
  }


SystemCode EnetReadPacket( mehdr )
short *mehdr;
   /*
    * Physically transfer a packet from the board's buffer pointed to
    * by mehdr (what does that stand for ? - Lance) and process it.
    */
  {
    extern PFPID RemoteTrapTable[];
    Process *pd;
    register DeviceInstance *inst;
    register kPacket *kp;
    register IoRequest *req;
    Team *oldteam;
    register unsigned	bytes;
    register short	*p;
    register unsigned short	mecsr;
    /* SendGroupMembers() may find a remote alias process group member. */
    extern int DeliverToEnetReader;

    bytes = (*mehdr & FREEmask) - 2;  /* Length of packet is free pointer
				       * less size of header (2 bytes)
				       */

    if( (*mehdr & (FCSerr|FRAMerr|RANGerr)) != 0 ) 
      {
	if( *mehdr & (FCSerr|FRAMerr) ) ++EnetCRCerrors;
	if( (bytes > ENET10_MAX_PACKET) || (bytes<60) || *mehdr & RANGerr )
	    ++EnetBadCounts;
	return;
      }

    EnetValidPackets++;
    DeliverToEnetReader = 0;
    /* Look at the first few words of the packet to determine
     * if it is a kernel packet and not addressing an alias process.
     */
    EnetPacket = (EnetBlock *) (mehdr + 1);
    kp = (kPacket *) EnetPacket->data;
    if((EnetPacket->EtherType==KERNEL_PACKET) && DifferentIKCByteOrder(kp))
      {
	SwapIKPacket(kp);
	kp->packetType &= ~IKC_LITTLE_ENDIAN; /* Just to be tidy */
      }
    if( EnetPacket->EtherType != KERNEL_PACKET ||
	(kp->dstPid & REMOTE_ALIAS_PROCESS) )
      {
nonKernelPacket:
	/* This is a user packet */
	if( (inst = EthernetInstance) == NULL ) /* Ethernet device not open */
	    return;
	if( !MAP_TO_RPD(pd, inst->reader) )
	  {
            /* No reader process */
	    if( !MAP_TO_RPD(pd, inst->owner) )
	      {
		inst->owner = 0; /* Free the instance */
		EthernetInstance = NULL;
		inst = NULL;
		EnetReset( ENET_DEFAULT_MASK );
		return;
	      }
	    /* If in promiscuous mode, dont save packet if
	     * it is a kernel packet - too risky.*/
	    if( EnetPromiscuous &&
		(EnetPacket->EtherType == KERNEL_PACKET &&
		(kp->dstPid & REMOTE_ALIAS_PROCESS)) )
		return;
	    /* The instance owner hasn't issued a read request yet.
	     * Save this packet in a kernel buffer.
	     */
	    UserPacketLength = bytes;
	    Copy( UserPacketBuffer, EnetPacket, bytes );
	    return;
	  }
	/* There is a valid owner with an outstanding read request.
	 * Give him the pcaket.
	 */
	req = (IoRequest *) &(pd->msg);
	/* First, fill in the bytecount in the reply */
	if ( bytes < req->bytecount )
	    req->bytecount = bytes;
	/* Now decide where to put the data */
	if( req->bytecount <= IO_MSG_BUFFER ) 
	    p = (short *) req->shortbuffer;
	else
	    p = (short *) req->bufferptr;

	/* Copy data to correct location */
	oldteam = GetAddressableTeam();
	SetAddressableTeam(pd->team);
	Copy( p, EnetPacket, req->bytecount );
	SetAddressableTeam(oldteam);
	inst->reader = 0;
	req->requestcode = OK;
	Addready( pd );
      }
    else
      {
	/* Second case : a kernel packet */
	if (kp->packetType != remoteForward)
	    HostCacheEnter(kp->srcPid>>16, EnetPacket->SrcHost);

	/* Ensure in correct packetType range. */
	kp->packetType = kp->packetType & 0xF;
	(*RemoteTrapTable[kp->packetType])(kp);
	/* If in promiscuous mode, let the enet reader get too. */
	if( EnetPromiscuous || DeliverToEnetReader ) goto nonKernelPacket;
      }
  }


NetworkWrite( bufferptr )
register BufferList *bufferptr;
  {
    register short *enetptr;
    register unsigned bytes;
    register BufferList *bl;
    register char *bp;

    if (bufferptr->bytes < sizeof(Enet10Header)) Kabort("Bad network header");
  
    ((Enet10Header *) bufferptr->ptr)->SrcHost = EnetHostNumber;

    for (bytes = 0, bl = bufferptr; bl->ptr != 0; bl++) bytes += bl->bytes;

    /* Transfer a packet to the Ethernet Interface Board. */
    enetptr = (short *)(SENDEND - bytes - (bytes&1)); /* make it even */
    bp = (char *) enetptr;
    while (bufferptr->ptr)
      {
	if (bufferptr->bytes)
	  {
	    Copy(bp, bufferptr->ptr, bufferptr->bytes);
	    bp += bufferptr->bytes;
	  }
	bufferptr++;
      }

    *(unsigned short *)METXHDR = (unsigned)enetptr - (unsigned)METXHDR;
    EnetPacketCollisions = 0;    
    SetCSR(MECSR, EnetRxmode | TBSW | TINTEN | JINTEN);
  }


TryEnetTransmit()
  {
    if( *(unsigned short *)(MECSR) & JAM )
      {
	++EnetCollisions;

        if (++EnetPacketCollisions > 10)
          {
	    /*
	     * Quietly give up trying to send this packet, although invoker
	     * assumes it was sent successfully.  No way to report the error
	     * at this point -- but Ethernets are assumed unreliable anyway.
	     * We enable transmitter interrupts so the next process on
	     * the IkpOutq (if any) will get serviced.  If the net is sick
	     * (e.g., unterminated) timeouts will probably continue to happen,
	     * but at least we won't wedge, and will recover when the net
	     * is reconnected.
	     */
	    EnetReset(EnetReceiveMask);
	    ++EnetTimeouts;
            SetCSR(MECSR, TINTEN);
          }
        else
          {
	    unsigned short delay;

	    /*
	     * Determine number of slot times to delay using binary
	     * exponential backoff.  The manual claims that delay time 
	     * can be zero, but passing a zero makes the board delay 65536 
	     * slot times, so we make sure the delay time is at least 1.
	     */
	    delay = (rand() & (0x0000ffff >> (16-EnetPacketCollisions))) + 1;
            *(unsigned short *)(MEBACK) = (-delay);	/* two's complement */
            SetCSR(MECSR, JAM | TINTEN | JINTEN);
          }
      }
  }


SystemCode EnetModify( pd, inst, dirIndex )
    register Process *pd;
    DeviceInstance *inst;
    unsigned short dirIndex;
  {
    register QueryEnetReply *reply = (QueryEnetReply *) &(pd->msg);

    if( reply->NumCollisions != -1 )
	EnetCollisions = reply->NumCollisions;
    if( reply->NumOverflows != -1 )
	EnetOverflows = reply->NumOverflows;
    if( reply->NumCRCErrors != -1 )
	EnetCRCerrors = reply->NumCRCErrors;
    if( reply->NumSyncErrors != -1 )
	EnetBadCounts = reply->NumSyncErrors;	/* kludge */
    if( reply->NumTimeOuts != -1 )
	EnetTimeouts = reply->NumTimeOuts;
    if( reply->NumValidPackets != -1 )
	EnetValidPackets = reply->NumValidPackets;
    if( reply->ReceiveMask != EnetReceiveMask &&
        reply->ReceiveMask != -1 ) EnetReset( reply->ReceiveMask );

   return( OK );
 }    

SystemCode EnetQuery( pd, inst, dirIndex ) 
    Process *pd;
    DeviceInstance *inst;
    unsigned short dirIndex;
  {
    register QueryEnetReply *reply = (QueryEnetReply *) &(pd->msg);

    reply->NetworkType = ENET_TYPE_10MBIT;
    reply->NumCollisions = EnetCollisions;
    reply->NumOverflows = EnetOverflows;
    reply->NumCRCErrors = EnetCRCerrors;
    reply->NumSyncErrors = EnetBadCounts;	/* kludge */
    reply->NumTimeOuts = EnetTimeouts;
    reply->ReceiveMask = EnetReceiveMask;
    reply->NumValidPackets = EnetValidPackets;
    reply->HostAddress.e10 = EnetHostNumber;

    return( OK );
  } 

int NetCheck() {}
SystemCode AddLogicalHostGroup(lhg) GroupId lhg; { return( OK ); }
SystemCode DeleteLogicalHostGroup(lhg) GroupId lhg; { return( OK ); }
