/*
 * V Kernel - Copyright (c) 1982 by Stanford University
 *
 * Kernel Console support routines for the device server
 *
 * The "console" provides access to the default keyboard and display
 *  for standalone programs.  The association between this logical
 *  device and physical device(s) depends on the workstation's
 *  type and configuration.
 */

#include "Venviron.h"
#include "Vteams.h"
#include "Vnaming.h"
#include "Vquerykernel.h"
#include "Vkeyboard.h"
#include "dm.h"
#include "process.h"
#include "machine.h"
#include "cbuff.h"
#ifdef SUN2
#include "sunromvec.h"
#endif

#define	BELL '\007'

/* Imports */
extern SystemCode NotSupported();
extern SystemCode SerialCreate();
extern SystemCode SerialRead();
extern SystemCode SerialWrite();
extern SystemCode SerialRelease();
extern short K_mayget();
extern SystemCode SccBellKludge();

extern short ConsoleInputDevice, ConsoleOutputDevice;
#ifdef SUN2
extern struct sunromvec *RomVecPtr;
#endif

/* Exports */
extern SystemCode ConsoleCreate();
extern SMIKeyboardPoll();  /* called by timer routine */

/* Private */
static SystemCode FbWrite();
static SystemCode FbRelease();
static SystemCode SMIKeyboardRead();
static SystemCode SMIKeyboardRelease();
static SystemCode SMIKeyboardModify();
static SystemCode SMIKeyboardQuery();
static SMIKeyboardInit();
DeviceInstance *ConsoleInInstance, *ConsoleOutInstance;
	  /* used only if different from serial line 0 */
CircularBuffer CbuffSMI;
static int SMIKeyboardTranslation;

/* Routines */
SystemCode ConsoleCreate( pd, inst )  
    Process *pd;
    DeviceInstance *inst;
  {
  /*
   * Create instances for the console.  The instance returned is
   *  used for console output.  Instance id + 1 is used for console
   *  input.  Its parameters can be obtained by doing a QueryInstance.
   */
    CreateInstanceRequest *req = (CreateInstanceRequest *) &pd->msg;
    register DeviceInstance *oldInst;
    SystemCode code;

    /* Do we need to use serial line 0 for the console? */
    if ( ConsoleInputDevice == PRF_ASCII_TERMINAL ||
         ConsoleInputDevice == PRF_CADLINC_KEYBOARD ||
	 ConsoleOutputDevice == PRF_ASCII_TERMINAL )
      {
        /* Open serial line 0 for keyboard and/or display */
        inst->unit = 0;  /* Use line 0 */
#ifdef SUN2
        if ( (code = SccCreate(pd, inst)) != OK ) return (code);
#else
        if ( (code = SerialCreate(pd, inst)) != OK ) return (code);
#endif

	ConsoleInInstance = inst+1;
        ConsoleOutInstance = inst;

	/* Can we use the framebuffer for console output? */
	if ( ConsoleOutputDevice == PRF_FRAMEBUFFER_SUN1 ||
	     ConsoleOutputDevice == PRF_FRAMEBUFFER_SUN2 ||
	     ConsoleOutputDevice == PRF_FRAMEBUFFER_SUN3 )
	  {
	    /* Yes, call the PROM monitor to do it */
	    inst->writefunc = FbWrite;
	    inst->type = (WRITEABLE+STREAM);
	    inst->blocksize = sizeof(char);  /* character by character */
	    inst->lastblock = 0;
	    inst->lastbytes = inst->blocksize;
	  }

	/* Should we poll for console input? */
	if ( ConsoleInputDevice == PRF_SMI100_KEYBOARD ||
	     ConsoleInputDevice == PRF_SMI120_KEYBOARD )
	  {
	    /* Yes, call the PROM monitor to do it */
	    inst++;
	    inst->readfunc = SMIKeyboardRead;
	    inst->type = (READABLE+STREAM);
	    inst->blocksize = sizeof(char);  /* character by character */
	    inst->lastblock = MAXUNSIGNED;   /* may grow indefinitely */
	    inst->lastbytes = MAXUNSIGNED;
	    SMIKeyboardInit();
	    CbuffFlush(&CbuffSMI);	/* Flush any buffered characters */
	  }

	return (OK);
      }

    /* The console is unrelated to serial device 0 */
    /* Call the PROM monitor to do both input and output */
    if (req->filemode != FCREATE) return (MODE_NOT_SUPPORTED);

    if( (oldInst = ConsoleInInstance) != NULL )
      {
	if( MapPid( oldInst->owner ) )  return (BUSY);
	ConsoleInInstance = 0;
	oldInst->owner = 0;
      }

    if( (oldInst = ConsoleOutInstance) != NULL )
      {
	if( MapPid( oldInst->owner ) )  return (BUSY);
	ConsoleOutInstance = 0;
	oldInst->owner = 0;
      }

    /*
     * Initialize the device instance descriptors, can assume all fields
     * are zero except for id and first instance's owner.
     */

    /* Output descriptor */
    inst->readfunc = NotSupported;
    inst->writefunc = FbWrite;
    inst->modifyfunc = NotSupported;
    inst->queryfunc = NotSupported;
    inst->releasefunc = FbRelease;

    inst->type = (WRITEABLE+STREAM);
    inst->blocksize = sizeof(char);  /* character by character for now */
    inst->lastblock = 0;
    inst->lastbytes = inst->blocksize;
    ConsoleOutInstance = inst;

    /* Input descriptor */
    inst++;
    inst->owner = pd->pid;
    inst->readfunc = SMIKeyboardRead;
    inst->writefunc = NotSupported;
    inst->modifyfunc = SMIKeyboardModify;
    inst->queryfunc = SMIKeyboardQuery;
    inst->releasefunc = SMIKeyboardRelease;

    inst->type = (READABLE+STREAM);
    inst->blocksize = sizeof(char);  /* assume character by character for now */
    inst->lastblock = MAXUNSIGNED;   /* may grow indefinitely */
    inst->lastbytes = MAXUNSIGNED;
    ConsoleInInstance = inst; /* Record the instance for interrupt routine */

    SMIKeyboardInit();
    CbuffFlush(&CbuffSMI);	/* Flush any buffered characters */

    return( OK );

  }


static SystemCode FbWrite( pd, inst )
    register Process *pd;
    register DeviceInstance *inst;
    /*
     * Handle a write instance request for the frame buffer
     */

  {
    register IoRequest *req = (IoRequest *) &pd->msg;
    register IoReply *reply = (IoReply *) &pd->msg;
    register char c;
    register unsigned bytecount;

    bytecount = req->bytecount;
    reply->bytecount = 0;
    if( bytecount > inst->blocksize ) return( BAD_BYTE_COUNT );
    if( bytecount == 0 ) return( OK );
/* Disable block number checking for now */
/*    if( req->blocknumber != inst->lastblock+1 ) return( BAD_BLOCK_NO );*/

    c = req->shortbuffer[0] & 0x7F;
    inst->lastblock++;
    reply->bytecount = 1;

    if (c == BELL)
      {
        /* Warning: kludge follows */
#ifdef SUN2
	if (ConsoleInputDevice == PRF_CONSOLE_SMI120)
	    return ( SccBellKludge( pd ) );
#else
	if (ConsoleInputDevice == PRF_CADLINC_KEYBOARD) 
	    lineput( 0, '\004' );  /* For CadLinc keyboards */
	else if (ConsoleInputDevice == PRF_ASCII_TERMINAL)
	    lineput( 0, BELL );    /* For terminals */
#endif SUN2	    
	/* end kludge */
      }
    else
	K_putchar( c );

    return( OK );
  }



static SystemCode FbRelease(pd, inst)
    Process *pd;
    DeviceInstance *inst;
  {

    inst->owner = 0;
    return (OK);
  }



static SMIKeyboardInit()
  {
    SMIKeyboardTranslation = TR_ASCII;
#ifdef SUN2
    (*RomVecPtr->v_InitGetkey)();
#endif
  }

SMIKeyboardPoll()
  {
    /* Called by timer interrupt if workstation type == SMI */
    register short c;
    register Process *pd;
    register DeviceInstance *inst;

    if ( (inst = ConsoleInInstance) != NULL &&
		(c = K_mayget()) != -1 )
      {
	if ( MAP_TO_RPD(pd, inst->reader) )
	  {
	    /* Reader waiting -- give it the character */
	    SMIKeyboardReturnChar(&pd->msg, inst, c);
	    Addready( pd );
	  }
	else /* Reader not waiting -- queue the character */
	    CbuffPut(&CbuffSMI, c);
      }

  }

static SystemCode SMIKeyboardRead( pd, inst )
    Process *pd;
    DeviceInstance 	*inst;
/*
 * Handle a read instance request for the SMI keyboard
 */
  {
    IoRequest *req = (IoRequest *) &pd->msg;
    register char c;
    register IoReply *reply = (IoReply *) &pd->msg;
    register unsigned bytecount;
    Process *reader;

    bytecount = req->bytecount;
    reply->bytecount = 0;

    /* Reply RETRY if already a waiting reader */
    if ( inst->reader && MAP_TO_RPD(reader, inst->reader) ) 
	return (RETRY);

    if ( bytecount > inst->blocksize ) return (BAD_BYTE_COUNT);
    if ( bytecount == 0 ) return (OK);
    if ( req->blocknumber != inst->nextblock ) return (BAD_BLOCK_NO);

    disable;
    /* Try to get a character from the buffer */
    if ( (c = CbuffGet(&CbuffSMI)) != CBUFF_EMPTY )
      {
	SMIKeyboardReturnChar(reply, inst, c);
	return (OK);
      }

    /* Wait for character to come in */
    inst->reader = pd->pid;
    pd->state = AWAITING_INT;

    return (NO_REPLY);
  }

static SMIKeyboardReturnChar(reply, inst, c)
    IoReply		*reply;
    DeviceInstance	*inst;
    char		c;
  {
    reply->replycode = OK;
    reply->shortbuffer[0] = c;
    reply->bytecount = 1;
    inst->nextblock++;
    inst->reader = 0;
  }



static SystemCode SMIKeyboardRelease( pd, inst )
    Process *pd;
    DeviceInstance *inst;
  /*
   * Release SMI keyboard (console) instance
   */
  {
    IoRequest  *req = (IoRequest *) &pd->msg;
    IoReply *reply;
    register Process *rwpd;

    if( rwpd = MapPid( inst->reader) ) 
				/* Unblock reader process */
      {
	inst->reader = 0;
	reply = (IoReply *) &rwpd->msg;
	reply->replycode = END_OF_FILE;
	reply->bytecount = 0;
	Addready( rwpd );
      }
    inst->owner = 0;
    return( OK );
  }


static SystemCode SMIKeyboardModify( pd, inst )
    Process *pd;
    DeviceInstance *inst;
/*
 * Modify translation of SMI keyboard
 */
  {
    ModifyKeyboardRequest *req = (ModifyKeyboardRequest *) &pd->msg;
    
#ifdef SUN2
    SMIKeyboardTranslation = req->translation;
    (*RomVecPtr->v_InitGetkey)();
    *(RomVecPtr->v_translation) = SMIKeyboardTranslation;
    
    return OK;
#else
    return REQUEST_NOT_SUPPORTED;
#endif
  }

static SystemCode SMIKeyboardQuery( pd, inst )
    Process *pd;
    DeviceInstance *inst;
/*
 * Query translation of SMI keyboard
 */
  {
    QueryKeyboardReply *reply = (QueryKeyboardReply *) &pd->msg;
    
    reply->translation = SMIKeyboardTranslation;
    
    return OK;
  }


CreateStdIo( pd )
    Process *pd;
  /* Create a console device instance for the root team's standard i/o */
  {
    Message msg;
    register CreateInstanceRequest *req = (CreateInstanceRequest *) &pd->msg;
    register CreateInstanceReply *reply = (CreateInstanceReply *) &pd->msg;
    register RootMessage *rmsg = (RootMessage *) msg;
    extern Process_id KernelServerPid;
    extern SystemCode CreateDeviceInstance();

    req->requestcode = CREATE_INSTANCE;
    req->filemode = FCREATE;
    req->filename = "console";
    req->filenamelen = 7;
    req->filenameindex = 0;
    req->contextid = DEFAULT_CONTEXT;

    if (CreateDeviceInstance( pd ) != OK) 
	Kabort("Error creating std i/o");

    ConsoleInInstance->owner = pd->pid;
    ConsoleOutInstance->owner = pd->pid;

    rmsg->replycode = OK;
    rmsg->rootflags = STDOUT_APPEND | STDERR_APPEND;
    rmsg->stdinserver = rmsg->stdoutserver = rmsg->stderrserver = 
    	reply->fileserver;
    rmsg->stdinfile = reply->fileid + 1;
    rmsg->stdoutfile = rmsg->stderrfile = reply->fileid;
    rmsg->envblock = NULL;

    Copy_msg(&pd->msg, rmsg);
  }

