@Part( process, root "manual" )
@Chapter(Processes and Interprocess Communication)@Label(process)
@index(Process)
@index(InterProcess Communication)

The V kernel supports processes as abstractions, with operations for process
management and interprocess communication.
Several processes may share an address space on one host. Processes
sharing an address space are collectively referred to as a @i[team].

The V kernel also 
supports the concept of a process group. Any process may create
a new group and processes may join or leave a group. Most functions that 
operate on a process also operate on a process group. This is achieved 
by specifying a group identifier in place of the process identifier. 
Thus, for example,
to destroy all the processes in the group specified by @t[groupid], 
the function
@t[DestroyProcess(groupid)] can be invoked. (A single process
can be viewed as a special process group: a group with just one member,
with the process identifier serving as the group identifier.)

Similarly, messages may be sent to a group of processes, simply by
addressing them to a group identifier. Typical usages of group communication
are @i[notification] (e.g. to notify other processes of some event) and
@i[query] (e.g. to locate a specific server). With local area networks
providing broadcast and multicast facilities in hardware, group
communication can be significantly more efficient than using repeated
one-to-one communication.

The process and interprocess communication-related 
functions in the V C library provide services and/or
interfaces between processes and the V kernel.
They have no direct analog in
the standard Unix C library.
These functions provide a convenient interface to kernel-provided
services.  Some of the functions execute kernel trap instructions,
while others send messages to the kernel-server inside the kernel.

A kernel operation executes as a single indivisible function call
as far as the C programmer is concerned.
Each kernel operation takes zero or more arguments and
returns a single value.

In the descriptions below,
the @i[active process] or @i[invoking process]
always refers to the process that executed the kernel operation.

Some operations such as SetTeamPriority and SetTime are intended to be
used only by ``operating system'' or management processes and should
not be used by application programs.

This chapter is divided into four sections: 1) process-related kernel
operations, 2) other process-related functions, 3) process group related
kernel functions and 4) interprocess communication-related functions.



@section(Process-Related Kernel Operations)

@function{SystemCode ClearModifedPages(pid)
    ProcessId pid;}@index(ClearModifiedPages)
Clears the dirty bits for all pages in the address space in which the process 
specified by @t[pid] resides.

@function{ProcessId CreateProcess(priority, initialpc, initialsp)
    short priority;  char *initialpc, *initialsp;}@index(CreateProcess)
Create a new process with the specified priority,
initial program counter and initial stack pointer
and return its unique process identifier.

The priority must be between 0 and 255 inclusive, with 0 the highest
priority. (See the discussion on priorities with the description of
@t[QueryProcessPriority()].)
@t[initialpc] is the address of the first instruction of the process to be
executed outside of the kernel.
Generally, @t[initialsp]
specifies the initialization of the stack and general registers
and is processor-specific.
In the case of the Motorola 68000,
@t[initialsp] is a simple long word value that is assigned to the 
user stack pointer.

The process is created awaiting reply from the invoking process
and in the same team space.
The segment access is set up to provide read and write access
to the entire team space of the newly created process.
The creator must reply to the newly created process before it
can execute.
If there are no resources to create the process or the priority
is illegal,
a pid of 0 is returned.

Usually programmers will prefer the @t[Create()] call described later
in this chapter.

@function{ProcessId CreateTeam(priority, initialpc, initialsp, lhost)
    short priority;  char *initialpc, *initialsp; ProcessId lhost}
		@index(CreateTeam)
Create a new team with initial or root process having the specified
priority, initial program counter, and initial stack pointer.  @t[lhost] 
specifies which (existing) logical host the new team should be placed into.  
A value of 0 specifies the logical host of the invoker.

@t[CreateTeam()] is similar to @t[CreateProcess()]
except the new process is created
on a new team.
The new team initially has a null team space.
It is intended that the creator of the team will initialize the team
address space and root process state using @t[SetTeamSize()],
@t[MoveTo()], and @t[WriteProcessState()]. @t[priority] must be a value
between 0 and 255.

CreateTeam returns 0 if there are no resources to create the team
or the root process, or the priority is illegal.

Warning: @t[CreateTeam()] will be restricted to the first team in the near
future.

@function{ProcessId Creator(pid)
    ProcessId pid;}@index(Creator)
Return the process id of the process that created @t[pid].
If @t[pid] is zero, return the creator of the invoking process.
If @t[pid]
does not exist or is the root process of the initial team, return 0.

@function{SystemCode DestroyProcess(pid)
    ProcessId pid;}@index(DestroyProcess)
Destroy the specified process and all processes that it created.
When a process is destroyed, it stops executing,
its pid becomes invalid, and all processes blocked on it
become unblocked (eventually).
@Foot(Processes blocked on a nonexistent processes are detected
and unblocked by the clock interrupt routine checking periodically.)
@t[DestroyProcess()] may also be used to destroy a process group by
specifying a group identifier with @t[pid].
@t[DestroyProcess()] returns OK if @t[pid] if successful,
else a reply code indicating the reason for failure.
@t[DestroyProcess(0)] is suicide. If @t[pid] specifies a process group,
then all processes in that group (and their descendants) are destroyed.

Usually programmers will prefer the @t[Destroy()] call described later
in this chapter.

@function{ProcessId GetObjectOwner(objectPid)
    ProcessId objectPid;}@index(GetObjectOwner)
Return the process-id of the owner of the specifed object.  Currently the 
only type of object supported is a team.   Thus @t[objectPid] must specify a 
process on the team whose owner is desired.

@function{ProcessId GetPid(logicalid, scope)
    int logicalid, scope;}@index(GetPid)
Return the pid of the process registered using @t[SetPid()]
with the specified @t[logicalid] and @t[scope], or 0 if not set.

The scope is one of:
@begin(description)
LOCAL@us()PID@\Return a locally registered process only.

ANY@us()PID@\Return a local or remote process.
@end(description)
If @t[logicalid] is ACTIVE@us()PROCESS,
the pid of the invoking process is returned.
If the scope is
@i(any), the kernel first looks for a locally registered process; if
one is not found, the kernel broadcasts a request
for a process identifier registered as this logical id to other
workstations running the V kernel on the network.
In this way, a kernel can discover the process identifiers of the
standard server processes from other kernels,
or at least from the kernel that is running the server process of interest.

@b[Note]: @t[GetPid()] and @t[SetPid] are being phased out.
New programs should use the group communication facility instead.
The REMOTE@us()PID scope available in previous releases is no longer
supported.

@function{ProcessId GetTeamRoot(pid)
    ProcessId pid;}@index(GetTeamRoot)
Return the process identifier of the root process of the team
containing @t[pid], or zero if @t[pid] is not a valid process identifier.
A @t[pid] of zero specifies the invoking process.

@function{char *GetTeamSize(pid)
    ProcessId pid;}@index(GetTeamSize)
Return the first unused location in the team space associated
with @t[pid], as set by @t[SetTeamSize()].  
If @t[pid] is zero, the size of the invoking
process's team is returned.
If @t[pid] does not exist, 0 is returned.


@function{QueryKernel(pid, groupSelect, reply)
    ProcessId pid;  int groupSelect;  Message reply;}@index(QueryKernel)
@label[QueryKernel]
Query the kernel on the host where process @t[pid] is resident.  A @t[pid]
of zero specifies the invoking process's kernel.

The @t[groupSelect] field specifies what information is to be returned
in the @t[reply] message.  The available group selection codes are
MACHINE@US()CONFIG, to return information about the processor configuration,
PERIPHERAL@US()CONFIG, to return a list of peripherals available on the
machine, KERNEL@US()CONFIG, to return the kernel's configuration parameters,
MEMORY@US()STATS, to return memory usage statistics, and KERNEL@US()STATS, to
return other kernel statistics.  These codes, and the corresponding 
structures that may be returned, are defined in the standard header file
<Vquerykernel.h>.

@function{SystemCode QueryProcessorUsage(pid, usage, tusage)
    ProcessId pid; unsigned *usage, *tusage;}@index(QueryProcessorUsage)
Return the time allocated so far by the processor to 
process  @t[pid]
in @t[*usage] and return total for the entire team in @t[*tusage] if
@t[tusage] is non-zero. Time is returned in "clicks". If @t[pid] is zero,
the time for the invoking process is returned. If @t[pid] is equal
to the logical host name (LHN) shifted left by 16 bits, the time allocated
to the idle process is returned.

The function itself returns a replycode indicating success
or otherwise.


@function{unsigned short QueryProcessPriority(pid)
    ProcessId pid;}@index(QueryProcessPriority)
Returns the composite priority of the process with @t[pid]. A @t[pid] of zero
specifies the invoking process. If @t[pid] does not exist, 0 is returned.

The 16 bit composite priority field of a process effectively consists of 
two concatenated 8 bit fields. 
The higher-order field contains the team priority;
the lower-order field the process priority within the team.
These are initialized when the processes are created (see
@t[CreateProcess()] and @t[CreateTeam()]) and may be manipulated with
@t[SetProcessPriority()] and @t[SetTeamPriority()].
The ready process with the lowest number in its composite priority 
field runs. 



@function{ProcessId QueryProcessState(pid, pb)
    ProcessId pid;  ProcessBlock *pb;}@index(QueryProcessState)
Copy the state of the process into the structure pointed to by @t[pb].
The various fields in the structure are defined in <Vprocess.h>.
Their meanings should be self-explanatory.

The message buffer is only available if @t[pid] is
the invoking process or is awaiting reply from the invoking process.
If not, the appropriate fields in the
structure are zeroed.

If @t[pid] is zero, the process state of the invoking process is
returned.  If @t[pid] does not exist, 0 is returned; otherwise,
@t[pid] is returned.

@function{ProcessId ReadProcessState(pid, state)
    ProcessId pid;  Processor@uf()state *state;}@index(ReadProcessState)
Copy the machine-specific processor state into the structure
pointed to by @t[state].  The information returned is a subset of
that returned by @t[QueryProcessState()].

If @t[pid] is zero, the processor state of the invoking process is
returned.  If @t[pid] does not exist, 0 is returned; otherwise,
@t[pid] is returned.

@function{SystemCode ReturnModifiedPages(pid, buffer, bufferlen)
    ProcessId pid; unsigned buffer[256]; unsigned *bufferlen}
		@index(ReturnModifiedPages)
Clears the @i[modified] bit of each  page table entry for the team
specified by @t[pid].  Returns the starting address of each page of memory
in the team whose modified bit was on before being cleared.  The size
of the buffer may not be sufficient to contain the
starting address of all dirty pages in the team's address
space, although it should be sufficient for most.
The operation does not clear the modified bit
of any page whose starting address it cannot place in the invoker's buffer.
Thus, the invoker need simply reinvoke @t[ReturnModifiedPages]
to continue with the the operation if the return code indicates that more
dirty pages might be outstanding.  This is indicated with a return code of
RETRY instead of OK.  The last valid page address returned is
delimited by a NULL word in the case where the buffer is not completely
filled.  @t[bufferlen] returns the number of modified page addresses returned 
in @t[buffer].

@function{int SameTeam(pid1, pid2)
    ProcessId pid1, pid2;}@index(SameTeam)
Return true (nonzero) if the processes specified both exist and
are on the same team; otherwise return false.  If either pid is
zero, the invoking process is assumed.

@function{SetObjectOwner(objectPid, newOwner)
    ProcessId objectPid, newOwner;}@index(SetObjectOwner)
Set the kernel-maintained owner of the object specified by @t[objectPid].  
Currently the only type of object supported is a team.  Thus @t[objectPid] 
must be the pid of a process on a particular team.  Ownership of a team 
implies that unrestricted access to the team's address space using @t[MoveTo] 
and @t[MoveFrom] operations without having a process in the team first send a 
message to the owner.

@function{SetPid(logicalid, pid, scope)
    int logicalid, scope;  ProcessId pid;}@index(SetPid)
Associate @t[pid] with the specified logical id within the specified scope.
Subsequent calls to @t[GetPid()] with this @t[logicalid] and @t[scope] return
this pid.
This provides an efficient, low-level naming service.

The scope is one of:
@begin(description)
LOCAL@us()PID@\Register the process locally.

ANY@us()PID@\Register the process globally.
@end(description)
The @i(local) scope is intended for servers
serving only the local workstation.
The @i(any) scope permits both local and
remote access.

@b[Note]: @t[GetPid()] and @t[SetPid] are being phased out.
New programs should use the group communication facility instead.
The REMOTE@us()PID scope available in previous releases is no longer
supported.

@function{SystemCode SetProcessPriority(pid, priority, decay)
    ProcessId pid;  unsigned short priority,  unsigned decay;}
@index(SetProcessPriority)
Set the priority of process @t[pid] to @t[priority].
If @t[pid] is zero the priority of the invoking process is set to
@t[priority].
@t[priority] may be any integer between 0 and 255.
(See @t[QueryProcessPriority()] for a discussion of process priorities.)
If @t[decay] is nonzero, the priority is
incremented every @t[decay] "clicks" until it reaches 255.

@function{SystemCode SetTeamPriority(pid, priority)
    ProcessId pid;  unsigned short priority;}@index(SetTeamPriority)
Set the team priority of the team associated with @t[pid] to @t[priority].
If @t[pid] is zero the priority of the invoking process's team is set to
@t[priority].
@t[priority] must be an integer between 0 and 255.
(See @t[QueryProcessPriority()] for a discussion of team priorities.)
Teams with @t[priority] 254 or 255 do not run.

@t[SetTeamPriority()] changes the absolute
scheduling priority of each process on the team
by modifying the team priority field of the composite priority for each
process.
This operation is intended for implementing macro-level scheduling
and is restricted in use to the first team.
Other teams should use @t[ChangeTeamPriority()] to request special
scheduling service.

@function{char *SetTeamSize(pid, addr)
    ProcessId pid;  char *addr;}@index(SetTeamSize)
Sets the first unused address for the team containing @t[pid] to @t[addr].
The new team size may be either greater or smaller than the
previous size.
The new team size is returned; this will normally be equal to @t[addr].
If there was not enough memory available to grant the request,
the return value will be less than @t[addr]; if @t[addr] was below the starting
address for team spaces on the host machine, the team space will be set
to null and its starting address will be returned.
Thus @t[SetTeamSize(pid, 0)] is a machine-independent way
of setting a team space to null.

A pid of 0 specifies the invoking process.
Only the creator of the team or members of the team may change
the team size and (consequently) the specified process must be local.
@comment[We don't check for this -- should we?  --TPM]


@function{int ValidPid(pid)
    ProcessId pid;}@index(ValidPid)
Return true (nonzero) if @t[pid] is a valid process or group identifier;
otherwise return false.

@function{ProcessId WriteProcessState(pid, state)
    ProcessId pid;  Processor@uf()state *state;}@index(WriteProcessState)
Copy the specified process state record into the
kernel state of the process specified by @t[pid] and return @t[pid].

The specified process must be the invoking process, or
awaiting reply from the invoking process.
@t[WriteProcessState()] returns 0 if the process does not exist,
is not awaiting reply or there is a problem with the
state record.
The kernel checks that the new state cannot compromise
the integrity or security of the kernel.

A @t[pid] of 0 specifies the invoking process.  A process that
writes its own processor state affects only the machine-independent
per-process area information kept as part of the state record
(see section @ref[PerProcess]).



@section(Logical Host-Related Functions)

@function{ProcessId CreateHost(priority, initialpc, initialsp)
    short priority;  char *initialpc, *initialsp; ProcessId lhost;}
		@index(CreateHost)
Create a new team in a separate logical host
with initial or root process having the specified
priority, initial program counter, and initial stack pointer.
This routine is the same as @t[CreateTeam] except that a new 
logical host is created for the new team.

@function{SystemCode DestroyHost(pid)
    ProcessId pid;}@index(DestroyHost)
Destroy the logical host in which the process specified by @t[pid] resides.  
When a process that is frozen is destroyed, any queue local IPC operations on 
it will be re-executed rather than returned with a failure status code.  
Also, queued reply messages will be forwarded rather than simply destroyed. 

@function{SystemCode ExtractHost(pid, optype, buffer, length)
    ProcessId pid; int optype; char *buffer; unsigned *length;}
		@index(ExtractHost)
Extract the kernel descriptor information for the logical host in which the 
process @t[pid] resides and place it in the buffer pointed to by @t[buffer].  
@t[length] specifies the size of buffer provided on invocation and returns 
the size of the descriptor information returned.  
This operation can only be invoked on local logical hosts.  
@t[optype] is either the 
manifest constant @t[QUERY_HOST_CASE] or @t[EXTRACT_HOST_CASE] (specifed in 
@t[Vmigrate.h]).  The former indicates that only summary information should 
be returned on the number of processes, teams and memory used by the logical 
host.  The is returned in a @t[HostResourcesRec] structure, as defined 
in @t[Vmigrate.h].  The latter optype indicates that the full kernel 
descriptor information for the logical host should be returned.
The full descriptor 
information returned is @i[not] intended to be interpreted outside the kernel 
and should only be used by the @t[TransferHost] operation to be reinstalled 
into another machine's kernel.

@function{SystemCode FreezeHost(pid)
    ProcessId pid;}@index(FreezeHost)
Freeze the logical host in which @t[pid] resides so that its address spaces 
and the kernel state associated with its teams and processes are not 
modified.  This will cause all kernel operations on the logical host to be 
deferred and the priority of the teams in the logical host to be set to 
non-runnable.

@function{SystemCode TransferHost(pid, buffer, length)
    ProcessId pid; char *buffer; unsigned length}@index(TransferHost)
Takes the kernel descriptor information generated by @t[ExtractHost] and 
installs it into an existing logical host that must have an equivalent number 
of teams already in it.  Furthermore, the teams must have only a root process 
in them.  The existing logical host is renamed by this operation to be the 
logical host specified in the descriptor information, as are all the teams in 
it.  The result is a logical host that is equivalent to the one described in 
the descriptor information and the effective deletion of the "blank" logical 
host that was used as an installation base.  The owner of the "new" logical 
host is the invoker of the operation.

@function{SystemCode UnfreezeHost(pid)
    ProcessId pid;}@index(UnfreezeHost)
Unfreeze the logical host in which @t[pid] resides.  All deferred kernel 
server and IPC operations are executed and the priority of the teams in the 
logical host are set to their previous runnable values.



@section(Other Process-Related Functions)

@function{ProcessId Create(priority, function, stacksize)
    short priority;  char *function;  unsigned stacksize}@index(Create)
Create a new process executing
the specified function with the specified priority and stack size.  
The new
process is blocked, waiting for a reply from the creator.  The function
@t[Ready()] should be used to start the process running.
The new process is on the same team as its creator, and
inherits the creator's standard input, output, and
error files, and the creator's current context (current working
directory).

@t[Create] returns the pid of the new process,
or zero if a process could not be created.
This function is usually preferable to calling the kernel operation
@t[CreateProcess()] directly.

@function{ProcessId Ready(pid, nargs, a1, ..., an)
    ProcessId pid;  unsigned nargs;  Unspec a1, ..., an;}@index(Ready)
Set up the stack of the specified
process and reply to it, thus placing it on the ready queue.
The values @t[a1, ..., an] appear as arguments to the root function
of the new process, while @t[nargs] is the number of arguments passed.
Zero is returned if there is a problem, else @t[pid] is returned.

@function{Destroy(pid)
    ProcessId pid;}@index(Destroy)
Destroy the specified process.  If the destroyed process was on the same team
as the invoking process, the memory allocated to its stack by @t[Create()]
is freed.  @b[Warning]: Do not invoke @t[Destroy()] on a process that
was not created by @t[Create()]; use @t[DestroyProcess()] in that case.

@function{Suicide()}@index(Suicide)
Destroy the invoking process and free its stack.  @t[Suicide()] is
identical to @t[Destroy(0)], and the same warning applies.

@function{exit(status)
    int status;}@index(exit)
Terminate the execution of the team (i.e., program),
after closing all open files.
The @t[status] is sent to the creator of the team requesting termination.
Thus, using the V executive@index(Executive), control is returned to the
command interpreter.
In bare kernel mode,@index(bare kernel mode)
control is returned to the @c[PROM] monitor@index(PROM monitor).

@function{abort()}@index(abort)
Abort execution of the team by causing an exception in the calling process.
This routine can also be called with parameters.  If it is, the standard
exception handler will interpret the first parameter as a pointer to a
@t[printf]-style format string.  The other parameters will be interpreted
as values to be printed using that string.  In an effort to keep the
standard exception handler simple and robust, the number of @t[%s]'s in the
format string must not exceed 8, nor may any of the strings (either the
format string or strings to be printed) exceed 128 characters in length.

The format specifier @t[%z] is included in addition to the usual specifiers.
@t[%z] will interpret its argument as a @t[SystemCode]@index(SystemCode),
and print the result of running @t[ErrorString]@index(ErrorString) with
that code as its parameter.

@function{ChangeTeamPriority(pid, priority)
    ProcessId pid;  short priority;}
Set the priority of the team in which process @t[pid] resides 
to @t[priority].  If @t[pid] is 0 then the invoking process is 
implied.  @t[priority] must be one of the manifest constants: REAL_TIME1 
through REAL_TIME4 (of which REAL_TIME1 is the most privileged priority), 
FOREGROUND, BACKGROUND, GUEST, and STOP_TEAM_PRIORITY.  The first team runs 
at priority REAL_TIME3, locally invoked foreground programs run at 
FOREGROUND, locally invoked concurrent (&) programs run at BACKGROUND, and 
remotely invoked programs run at GUEST priority.  STOP_TEAM_PRIORITY makes 
the processes of a team nonrunnable.  FOREGROUND, BACKGROUND, and GUEST 
programs are time-sliced in a round-robin scheme, with lower priority teams 
only getting the time slice if no higher priority teams exist.
Management of team priorities is done by the team server, which uses 
the privileged @t[SetTeamPriority] kernel operation to 
actually change team priorities.

@Section(Process Group Operations)
@Index(Process Group Operations)

@function{GroupId CreateGroup(initial@uf()member, type)
    ProcessId initial@uf()member; unsigned type;}@index(CreateGroup)
Create a new group of the specified @t[type]
(UNRESTRICTED@us()GROUP@US()BIT|LOCAL@US()GROUP@US()BIT) and make 
@t[initial@uf()member] the first member.
The invoking process is made the first member of the process group if
@t[initial@uf()member] is 0.
If the UNRESTRICTED@US()GROUP@US()BIT is set in @t[type], then any process may join
the group, otherwise only processes of the same user may join.
One can specify that only processes on the same host as
@t[initial@uf()member] may join the group with the
LOCAL@US()GROUP@US()BIT bit in @t[type], thus allowing certain optimizations.
@t[initial@uf()member] may also specify a process group, in which case
every member of 
@t[initial@uf()member] becomes a member of the newly created group.

Returns the group id of the newly created group if successful, 0 otherwise.


@function{SystemCode JoinGroup(groupId, pid)
    GroupId groupId; ProcessId pid;}@index(JoinGroup)
Add the process or process group specified by @t[pid] to the 
process group @t[groupId]. Group @t[groupid] must
exist. Well known groups are defined in the include file
<Vgroupids.h>. If @t[pid] is 0, the
invoking process is added to the group. 
If @t[pid] specifies a process group, every process of that
group joins the group specified by @t[groupid]. Returns OK if successful, else
a reply code indicating the reason for failure.

@function{SystemCode LeaveGroup(groupId, pid)
    GroupId groupId; ProcessId pid;}@index(LeaveGroup)
@t[pid] leaves the process group with group id @t[groupid]. If
@t[pid] is 0, the invoking process leaves the group.
Again, @t[pid] may either specify a process or a process group.

@function{SystemCode QueryGroup(groupId, pid)
    GroupId groupId; ProcessId pid;}@index(QueryGroup)
Query the kernel to see if @t[pid] (the invoking process if @t[pid] is 0)
would be allowed to
join the group with the specified @t[groupId].
Returns OK if so, otherwise NO@US()PERMISSION if not
allowed, DUPLICATE@US()NAME if already in, and NOT@US()FOUND
if group does not exist (not at least one member located).



@section(Interprocess Communication)
@index(Interprocess Communication)


@function{int AwaitingReply(frompid, awaitingpid)
    ProcessId frompid, waitingpid;}@index(AwaitingReply)
Return true (nonzero) if @t[awaitingpid] is awaiting reply from
@t[frompid]; otherwise return false.


@function{SystemCode ForceException(pid)
    ProcessId pid;}
Causes process @t[pid] to send an exception message to the exception server.  
The exception type is FORCEEXCEPTION.


@function{SystemCode ForceSend(msg, fromPid, toPid)
    Message msg; ProcessId fromPid, toPid;}@index(ForceSend)
Force process @t[fromPid] to send @t[msg] to 
process @t[toPid].  @t[ForceSend]cannot be reinvoked on a process until the 
first invocation is terminated by replying to the process.  I.e. there can 
only be at most a single @t[ForceSend] in effect for any given process.


@function{ProcessId Forward(msg, frompid, topid)
    Message msg;  ProcessId frompid, topid;}@index(Forward)
Forward the message pointed to by @t[msg] to the process specified
by @t[topid] as though it had been sent by the process @t[frompid].

The process specified by
@t[frompid] must be awaiting reply from the invoking process.
The effect of this operation is the same as if
@t[frompid] had sent
directly to @t[topid], except that the invoking process is
noted as the forwarder of the message.
Note that @t[Forward()] does not block.

@t[Forward()] returns @t[topid] if it was successful, 0 if unsuccessful.
If @t[topid] is invalid, @t[frompid] is unblocked with an indication
that its @t[Send()] failed.
(Namely, the @t[Send()] returns zero, and the replycode field of 
the reply message is set to BAD@us()FORWARD.)

@function{ProcessId Forwarder(pid)
    ProcessId pid;}@index(Forwarder)
Return the process id that forwarded the last message received from
@t[pid], providing @t[pid] is still awaiting reply from the invoking
process.  If the message was not forwarded, @t[pid] is returned.
If @t[pid] does not exist or is not awaiting reply from the invoking
process, 0 is returned.
If the last message received was sent to a process group, @t[Forwarder()]
returns the group identifier the message was sent to.


@function{ProcessId GetReply(msg, timeout) 
    Message msg;  int timeout;}@index(GetReply)
Returns the next reply message from a group @t[Send()] in @t[msg] and returns
the process identifier of the replying process. If no messages are available
within the timeout period, @t[GetReply()] returns 0. A typical message
transaction thus consists of a @t[Send()] (which returns the first reply)
followed by any number of @t[GetReply()]. However,
all replies for a message transaction are discarded when
the process sends again, initiating a
new message transaction. (Note: Many library routines, such as @t[printf()]
are implemented with message passing primitives, thus ending the last
message transaction when they are called.)
The timeout is given in clicks.

@function{SystemCode MoveFrom(srcpid, dest, src, count)
    ProcessId srcpid;  char *dest, *src;  unsigned count;}@index(MoveFrom)
Copy @t[count] bytes from the memory segment starting at @t[src] in the
team space of @t[srcpid] to the segment starting at @t[dest] in the
invoking process's space, and return the standard system reply code OK.

Unless the invoker is the owner of the team in which @t[srcpid] resides,
the @t[srcpid] process must be awaiting reply from the invoking process
and must have provided read access to the segment of memory in its
space using the message format conventions described for @t[Send()].
@t[MoveFrom()] returns a standard system reply
code indicating the reason for failure
if any of these conditions are violated.

@function{SystemCode MoveTo(destpid, dest, src, count)
    ProcessId destpid;  char *dest, *src;  unsigned count;}@index(MoveTo)
Copy @t[count] bytes from the segment starting at @t[src] in the invoking
process's team space to the segment starting at @t[dest]
in the team space of the @t[destpid] process,
and return the standard system reply code OK.

Unless the invoker is the owner of the team in which @t[srcpid] resides,
the @t[destpid] process must be awaiting reply from the invoking process
and must have provided write access to the segment of memory in its
space using the message format conventions described under @t[Send()].
@t[MoveTo()] returns a standard system reply
code indicating the reason for failure
if any of these conditions are violated.


@function{ProcessId Receive(msg)
    Message msg;}
Suspend the invoking process until a message is available from
a sending process, returning the pid of this process, and placing
the message in the array pointed to by @t[msg]. To determine if the message
was sent to a process group see @t[Forwarder()].

@function{ProcessId ReceiveWithSegment(msg, segbuf, segsize)
    Message msg;  char *segbuf;  unsigned *segsize;}@index(ReceiveWithSegment)
Suspend the invoking process until a message is available from a sending
process, returning the pid of this process, and placing
the message in the array pointed to by @t[msg]
and at most the first @t[*segsize]
bytes of the segment included with the message
in the buffer starting at @t[segbuf].
The actual number of bytes in the portion of the segment 
received is returned in @t[*segsize].
(Note: This may be zero even if a segment is specified in the message.)
Additional parts of the segment specified in the message may be transferred
with @t[MoveFrom()].

@function{ProcessId ReceiveSpecific(msg, pid)
    Message msg;  ProcessId pid;}@index(ReceiveSpecific)
Suspend the invoking process until a message is available from
the process @t[pid] or from a process in the process group specified by
@t[pid], returning the pid of this process, and placing
the message in the array pointed to by @t[msg].

If @t[pid] is not a valid process or group identifier,
@t[ReceiveSpecific()] returns 0.

@function{ProcessId Reply(msg, pid)
    Message msg;  ProcessId pid;}@index(Reply)
Send the specified reply message to the process specified
by @t[pid] and return @t[pid].

The specified process must be awaiting reply from the invoking process.
Zero is returned if the process does not exist or is not awaiting
reply.
Note: Messages that have been received but not replied to consume kernel
resources until the receiver exits.
Therefore, each process should invoke @t[Reply()] on every
message it receives.
If no reply is required,
then @t[Reply()] should be invoked with a message whose replycode is set to
DISCARD@us()REPLY.
Such a reply message is not delivered to the sender, but releases
kernel resources and allows
the sender to (eventually) unblock (with a KERNEL@US()TIMEOUT error reply
code if no replies were received at all).


@function{ReplyWithSegment(msg, pid, src, dest, bytes)
    Message msg;  ProcessId pid;  char *src, *dest;  unsigned bytes;}
@index(ReplyWithSegment)
Send
the specified reply message and segment
to the process specified by @t[pid] and return @t[pid].

The specified process must be awaiting reply from the invoking process.
Zero is returned if the process does not exist or is not awaiting
reply.
The segment size is currently limited to 1024 bytes.
A @t[ReplyWithSegment()] with a nonzero segment size
may only be used to reply to an idempotent request (see @t[Send()]).


@function{ProcessId Send(msg, pid)
    Message msg;  ProcessId pid;}@index(Send)
If pid specifies a single process group, send 
the message in @t[msg] to the specified process,
blocking the invoking process until the message is both
received and replied to.
The array specified by @t[msg] is assumed to be 8 long words.
The reply message overwrites the
original message in the array.

If @t[Send()] completes successfully,
it returns the pid of the process that replied to the message.
The pid returned will differ from that specified in the call
if the message is forwarded by the receiver to another process
that in turn replies to it.
If the send fails (for instance, because the intended receiver
does not exist), @t[Send()] returns the pid of the process the
message was last forwarded to (the pid it was sent to, if
it was never forwarded).  The kernel indicates the reason for the
failure by overwriting the first 16 bits of the message with
a standard system reply code.  (This places it in the @i[replycode]
field for reply messages that follow the standard system format.)

If @t[pid] is a process group identifier, the message is sent to all
processes in the group on a @i(best effort basis) and @t[Send()]
blocks until a first process replies. The first reply message
overwrites the original message. Further replies of the current message
transaction may be received
with @t[GetReply()]. Send initiates a new
message transaction, effectively flushing all messages of the last transaction.
 

All messages must follow the kernel
message format conventions as follows.
The first 16 bits of the message are considered to be a request code
or reply code.  Some of high-order 8 bits within request and reply codes
are assigned special meanings, and currently-unused bits within this
subfield are reserved for future use.
The bit names given below are defined in the standard header file
<Venviron.h>.
@begin(Description)
REPLY@us()BIT@\is reset if a request message is being sent;
set if a reply message.

SYSTEM@us()CODE@\is set if the request code or reply code
is considered a standard system code.
Applications can use special request codes and reply codes internal to their
programs but use standard ones for interfacing to other programs and the
system.
@end(description)

Several other bits are interpreted with the following special meanings
if the message is a request.
@begin(description)
READ@us()BIT@\is set
if read access is provided to a memory segment.@index(segment)
If this bit is set, the kernel interprets the last 2 words of the message
as specifying a pointer to the start of the segment and the size
in bytes of the segment, respectively.
The kernel then makes the segment available to the receiving process
using @t[MoveTo] and @t[MoveFrom].

WRITE@us()BIT@\is set if write access is provided to a memory segment.
The segment is specified as described above.
Both read and write access can be provided by setting both bits 4 and 5.

DATAGRAM@us()SEND@us()BIT@\Experimentally,
the V kernel currently supports the concept of
real-time communication. In this mode, messages are communicated to a single
process or a group of processes on a best effort basis. A process will only
receive the message if it is receive-blocked waiting for it. The send
operation does not block. Thus, one cannot reply to a real-time send. This
type of communication is intended for situations, where, for example, a
process continuously, in regular intervals, sends update information to a
group. This mode of communication is specified by setting the
DATAGRAM@US()SEND@US()BIT of the requestcode of the message.
@end(Description)

It is intended and assumed that most requests can be assigned
a request code that is stored in the first 16 bits of the request message,
so that the bits are set correctly for the request by the value
of the request code.

The following bits have special meaning in reply codes:
@begin(description)
ANONYMOUS@us()REPLY@us()BIT@\Reply as the forwarder of the message.
This feature allows processes to join groups and reply to messages
anonymously.

REPLY@us()SEGMENT@us()BIT@\Reply segment has been specified.
If this bit is set in a call to @t[ReplyWithSeg()],
the kernel interprets the last 2 words of the message
as specifying a pointer to the start of the reply segment and the size
in bytes of the segment, respectively.

IMMEDIATE@us()REPLY@us()BIT@\Don't delay this reply, even if
it is to a group send.  If this bit is not set,
replies to group sends are delayed slightly within the replying kernel to 
avoid swamping the sending kernel with back-to-back packets.
@end(description)
