@Part( mem, root "manual" )
@Chapter(Memory Management)@Label(mem)

Blocks within a managed pool of memory can be dynamically allocated and
freed within the address space of a team using the functions described below.
These routines provide essentially the same
functionality as the standard C library.  The memory allocation routines are
provided on a per-team basis.

Note that there is one pool of free storage for all processes in the team:
when using the standard library versions,
programmers must be careful to synchronize the processes allocating and
freeing this storage.
A set of memory management routines with internal locking
for mutual exclusion is also
available (see @t[lockedmalloc], below).
These routines run more slowly than the standard versions.

@function{char *malloc(size)
    unsigned size;}@index(malloc)
Returns a pointer to a memory block that is @t[size] bytes long.
@c[NULL] is returned if there is not enough memory available.

@function{free(ptr)
    char *ptr;}@index(free)
The memory pointed to is returned to the free storage pool.  @t[ptr]
must point to a block allocated by one of the routines listed here.

@function{char *realloc(ptr, size)
    char *ptr;  unsigned size;}@index(realloc)
Changes the size of the block pointed to by @t[ptr]
to be @t[size] bytes.  Returns a possibly moved pointer.

@function{char *calloc(elements, size)
    unsigned elements, size;}@index(calloc)
Equivalent to @t[malloc(elements*size)],
except the area is cleared to zero.
Provided for allocating arrays.

@function{cfree(ptr, elements, size)
    char *ptr;  unsigned elements, size;}@index(cfree)
Frees storage allocated by @t[calloc()].  Actually, this
function is identical to @t[free(ptr)], which may be used instead.
@t[elements] and @t[size] are ignored.

@function{unsigned Copy(destination, source, count)
    char *destination, *source;  unsigned count;}@index(Copy)
A fast block transfer function.  
Transfers
@t[count] bytes from @t[source] to @t[destination].
Returns @t[count].
Restriction: the source and destination blocks must not overlap.

@function{unsigned blt(destination, source, count)
    char *destination, *source;  unsigned count;}@index(blt)
Identical to @t[Copy()].

@function{char *Zero(ptr, n)
    char *ptr;  unsigned n;}@index(Zero)
Zero memory.  Writes @t[n] bytes of zeros starting at @t[ptr],
and returns @t[ptr].

@function{clear(ptr, n)
    char *ptr;  unsigned n;}@index(clear)
Clear memory.  Writes @t[n] bytes of zeros starting at @t[ptr]. 

@function{swab(pfrom, pto, n)
    char *pfrom, *pto;  unsigned n;}@index(swab)
Swap the bytes in @t[n]
16-bit words starting at the location @t[pfrom] into a block starting at
the location @t[pto].



The following functions are of interest only to those managing memory (using
the kernel primitives) in addition to that provided by the above routines.
The current implementation of @t[malloc()] prevents these routines from adding
space below the current top of the pool.

@function{GiveToMalloc(start,length)
    char *start;  int length;}@index{GiveToMalloc}
Add the @t[length] bytes of memory at @t[start] to the pool used by the
allocators described above, returning the number of bytes actually installed
after alignment and error-checking is done.

@function{char * GetMoreMallocSpace(min,actual)
    int min, *actual;}@index{GetMoreMallocSpace}
@t[Malloc()] calls this function to
acquire more space for its pool;  a default
version is supplied, which is replaced if the programmer supplies a routine
of this name.  @t[GetMoreMallocSpace()] should return a pointer to at least
@t[min] bytes of space and set @t[*actual] to the
number of bytes made available; @t[NULL] may be returned if no more space is
to be added to the pool.

In the default version, free memory is determined and extended
based on the memory map and memory usage of the team (using the
V kernel operations @t[GetTeamSize()] and @t[SetTeamSize()]).

@section(Use in multi-process teams)
The standard library versions of the allocation and deallocation
routines do not enforce exclusion
among processes within a team; so disastrous things may happen if two or
more processes access them simultaneously.  A multi-process team may use the
routines safely by enforcing its own exclusion (@i[e.g.], by having all
allocation/deallocation occur in a single process), or by explicitly linking
in a provided version of these routines that does provide locking.
The routines affected are malloc, realloc, free, calloc, cfree, 
and GiveToMalloc.  (Note, however, that calls to these routines may be
hidden in other standard library routines as well.)
The locking version may be accessed using the compiler flag @t{-llockedmalloc};
to include it use, for example:
@programexample{cc68 -V -r @i{other flags} yourfile.o -llockedmalloc @i{other libraries}}
This provides full exclusion for all of the routines mentioned, but at a
execution-time penalty of up to about 25%.
