@part( locking, root "manual" )
@chapter(Intra-Team Locking)@label[locking]@index[locking]

The V kernel provides message-passing as a means of synchronizing processes, 
and mutual exclusion may be enforced by the use of a server process that 
executes the critical section for clients.  Such an arrangement is not always 
suitable, however:  for processes communicating via a shared data structure 
the overhead of a message exchange may exceed by an order of magnitude the
cost of performing the critical section.

The V library includes support for cheap mutual exclusion among processes in 
a single team.  @i[Spin locks] ensure mutual exclusion in the presence of
contention, but in its absence they introduce very little overhead.
Spin locks also maintain a count indicating the level of contention so that 
the programmer can continue to assess their suitability after they are in use.
@comment{Incorporate overview of other locking primitives (if any) here.}

@comment{@section{Spin Locks}
@index{spin locks}@label{spinlock}
Spin locks are essentially binary semaphores without queueing:
when a process fails in an attempt to acquire a lock, it simply delays 
(instead of busy-waiting) before trying again ("spinning" the lock).

@b[Advantages:]  In the absence of contention, spin locks are @i[fast].
The optimized macro forms require only from one to six machine instructions 
each; the procedure forms add only the cost of a single-argument procedure 
call.

@b[Disadvantages:]
A process that fails in an attempt to acquire a lock 
delays one tick before trying again; the locking overhead in the presence of 
contention is therefore higher than it would be for a message exchange with a 
server process.  Also, spin locks are not fair:  the order in which processes
acquire the lock is not determined by the order in which they begin their
attempts.

Spin locks are best suited for cases in which contention is expected to occur 
only rarely.  The repeated attempts at the lock render them less suitable 
when the lock is held for long periods of time (several clicks), and the 
delay period (one click) may be too long for some applications with real-time 
constraints.

Each spin lock maintains a @i[contention count], incremented each time 
that a process is forced to delay in an attempt to acquire the lock.  The 
counter is incremented without mutual exclusion; its value is therefore not 
guaranteed precise, but should still provide a rough indication of the level 
of contention.

The following subroutines are provided in the library, with needed
definitions in Vspinlock.h.@index{Vspinlock.h}

@function{AcquireSpinLock(lock)
    SpinLockType *lock;}@index[AcquireSpinLock]
Wait until the named lock is acquired before returning.  The delay on failure 
is one click.

@function{ReleaseSpinLock(lock)
    SpinLockType *lock;}@index[ReleaseSpinLock]
Release the named lock.

Additionally, the macro @t[SpinLockCount(lock)] provides access to the 
contention count; it is of type @t[short integer] and may be assigned to as 
well as read.

Locks must be initialized to either @t[SpinLockLocked] 
or @t[SpinLockUnlocked], which also set the contention count to zero.


More efficient macro forms of the locking operations are provided for
the common cases of the lock being a global variable or an argument to the
procedure invoking the operation.  The costs of these forms of the operations  
are in the range of one to six machine instructions.
The compiler and lint, however, cannot properly check theses forms, which 
may result in either spurious error messages or failure to detect real errors.

@function{AcquireGlobalSpinLock(lockName)
    SpinLockType lockName;}@index[AcquireGlobalSpinLock]
Equivalent to @t[AcquireSpinLock(&lockName)], where @t[lockName] is the name 
of a global (extern) variable.  (This will also work if the global 
variable @t[lockName] is a struct and the lock is its first component.)

@function{ReleaseGlobalSpinLock(lockName)
    SpinLockType lockName;}@index[ReleaseGlobalSpinLock]
Equivalent to @t[ReleaseSpinLock(&lockName)], where @t[lockName] is the name 
of a global (extern) variable.

@function{AcquireArgumentSpinLock()}@index[AcquireArgumentSpinLock]
Equivalent to @t[AcquireSpinLock(p)], where @t[p] is the @i[first] argument
of the containing procedure.  (This will also work if the lock is 
the first component in a struct @t[*p].)

@function{ReleaseArgumentSpinLock()}@index[ReleaseArgumentSpinLock]
Equivalent to @t[ReleaseSpinLock(p)], where @t[p] is the first argument of
the containing procedure.
