@chapter(Fields: Using an AVT as a Menu)
@label(Fields)
@index(fields)
@index(Menu)
These routines allow you to set up a table of @i[fields] in an AVT.
They can be selected with the mouse, so that you can have a menu.
The advantages over the standard pop-up menu are that
you can have more choices, you can display more information with each choice,
and the menu is always there.

With each field, you can associate a value, which can be displayed and edited.

The menu is an array of @t[Field]s. These are defined in @t{<fields.h>}.
Each @t[Field] consists of:
@begin(programexample)
typedef struct
  {
    short row;		/* @r{field's row number in AVT} */
    short col;		/* @r{leftmost character of field} */
    short width;	/* @r{width of field} */
    long *value;
    int (*proc)();
    char *format;	/* @r{format in which to display} *value */
  } Field;
@end(programexample)
@t[row] and @t[col] indicate where in the AVT the field begins.
(row=1 and col=1 is the top left corner of the AVT.)
@t[width] is the length of the field in characters.
Only one-line fields are supported.
@t[proc] is not used by the package itself.
The intended usage is:
@begin(programexample)
field = GetField(...);
if (field) (*field->proc))(field->value);
@r{or perhaps:}
if (field) (*field->proc))(field);
@end(programexample)
@t(format) is discussed below.

@section{Formats}

@t[format] is a format like those used by @t[printf] and @t[scanf].
Together with the @t{value}, it determines the string to be displayed in
the field. This string must be a least @t{width} characters long.
It is a zero-terminated C (asciz) string.
Formats are of the form:
@begin(programexample)
@i{prefix} @i{[conversion]} @i{suffix}
@end(programexample)
Here prefix and suffix is constant text which is displayed.
If a @t{%} is to be displayed, it must be written as @t{%%}.
The following utility routine will do a string copy analogous to @t{strncpy},
except that @t{%}s are automatically copied:
@function[char * StrToFormat(f, s, n)
    char *f;  /* @r{destination string buffer where '@t(%)'s are to be doubled} */
    char *s;  /* @r{source string} */
    int n;    /* @r{count - buffer size} */
]

The optional @i{conversion} describes how @t{value} is to be
displayed/read. Its form is:
@begin(programexample)
@i(@t{%}[[@t{-}][@t{0}]fieldwidth][@t{.}precision][@g{l}]c)
@end(programexample)
Here the @t{%} indicates the beginning of the conversion specification.
The conversion type letter @i[c] marks the end of the conversion specification.
The format is exactly as used by @t[printf], except that there may be a
data length specification @g{l}.
If @t{value} is a @t{short *} rather than a @t{int *}, @g{l} must be
given as @t{h}.
If the @t{value} is a @t{double *} rather than a @t{float *},
@g{l} must be @t{l}, or the conversion type letter @i[c] must be capitalized.

When fields are displayed, @t[sprintf] is used to do the conversion.
The length specification @g[l] is only used to dereference @t[value]
(except for fields where the conversion type letter is @t[s]); it is
stripped from the format before being passed to @t[sprintf].

On input to fields, only the length specification @g[l] and the type code
@i[c] are passed to @t[sscanf].
If the type code is @t[e] or @t[g], it is changed to @t[f].
A type code of @t[S] (or @t[ls]) means that the whole input
line (including spaces) should be accepted.

@Section{The Field Table as a Menu: Selecting an Action}
@function{Field * GetField(menu, menuLength, buttons, avt)
    Field *menu;
    int menuLength;
    short buttons;
    File *avt;         /* output AVT */
}
If @t[button != 0], it is assumed that the mouse is down on procedure entry.
@t[GetField] returns when the button state changes; if it changes
to non-zero, @t[GetField] fails by returning zero.
If @t[button == 0], @t[GetField] will first wait for an event.
(It will fail unless it is a mouse button being pressed down.)

As long as the user keeps the mouse button down, display the selected
field (if any) in inverse video.
When the user releases the button, return the last selected @t[Field], or
if none, return 0.

The menu is terminated by the first negative @t[row] field, or when the
@t[menuLength] count is exhausted.

@Section(Displaying Fields)
@function{PutField(buffer, field)
    char *buffer;  /* @r{destination string buffer} */
    Field *field;  /* @r{source format and value} */
}
More or less like @t{sprintf(buffer, field->format, *field->value)}.

@function{DisplayFields(menu, menuLength, avt)
    Field *menu;
    int menuLength;    /* see GetField function */
    File *avt;         /* output AVT where fields are to be written */
}
Display in the @t[avt] all the @t[string] fields,
at the positions given by the @t[row] and @t[col] components.

The @t[width] components are ignored. This allows convenient display
of material which the user cannot select ("write-protected" fields)
either by using fields with @t[width <= 0] or by having a @t[string]
longer than the @t[width].

@section{User Input to Fields}
@function{EditField(field, stuff, out, in)
    Field *field; /* @r{field whose @t{*value} is to be edited} */ 
    int stuff; /* @r{0: old text should be cleared: 1: stuff into editor} */
    File *out, *in; /* @r{input and output sides of AVT to use} */
}
Move the cursor to the conversion part of the @t{field}.
If @t{stuff} is 0, the old value is cleared from the screen;
if it is 1, the old value is placed in the line editing buffer.
Enter line-edit mode, and wait for the user to type in a line.
If the user types @t{^G}, abort, redisplay old value and return -1.
Else parse the line using @t{field->format}.
If this succeeds, update @t{*field->value}, returning 1, else 0.
In any case, redisplay things correctly.

@function{EditStdFld(field)}
Equivalent to @t{EditField(field, 1, stdout, stdin)}

@function{ReadStdFld(field)}
Equivalent to @t{EditField(field, 0, stdout, stdin)}

@Section{An Example}
@begin{BigProgramExample}
/* This is a program which adds up integers, optionally scaled */
#include <stdio.h>
#include <fields.h>
double Scale = 1.0, Total = 0.0;
int Value = 0;

Quit() { ... cleanup actions ...; exit(-1);}
@hinge
NewValue(f)
    Field *f;
  {
    if (ReadStdFld(f) == 1)
        Total += Value * Scale;
  }
@hinge
Fields Menu[] =
  {
/* VAL (defined in fields.h) coerces pointers and values to (int *) */
    {1, 41, 10, VAL &Scale, EditStdFld, "Scale: %G      "},
    {1,  1, 15, VAL &Value, NewValue,   "New value: %-8d"},
    {2,  1,  0, VAL &Total, 0,          "Total: %G.        "},
    {5,  1,  8, 0,          Quit,       "--Quit--"},
    LASTFIELD /* defined in fields.h */
  };
@hinge
main()
  { Field *field;
    while (1)
      {
	putc('L' & 31, stdout); /* write FormFeed to clear screen */
	DisplayFields(Menu, 999, stdout);
	field = GetField(Menu, 999, 0, stdout);
	if (field) (*(field->proc)) (field);
      }
  }
@End{BigProgramExample}

Since the screen is updated every time here, we do not have to worry about
garbage being left behind when the field becomes shorter.
However, one of two fixes shown above can be used when this is not desired:
In the @t{Value} field, we make @i{sure} the field doesn't become shorter,
by left justification if needed. This loses if we want to output punctuation
after the value, as in the @t{Total} field. In this case, we can make sure
that we output enough trailing spaces to erase the garbage.

@section{Limitations}
No facilities yet for arrays.
