@part(xlisp, root "manual")
@chapter(xlisp: An Experimental Object Oriented Language)

This chapter is adapted from the document @i[XLISP: An Experimental 
Object Oriented Language], Version 1.4, January 1, 1985, by David
Betz, 114 Davenport Ave., Manchester, NH  03103.

@section(Introduction)

XLISP is an experimental programming language combining some
of  the  features  of LISP with an object oriented extension
capability.  It was  implemented  to  allow  experimentation
with  object oriented programming on small computers.  There
are currently implementations running on  the  PDP-11  under
UNIX  V7, on the VAX-11 under VAX/VMS and Berkeley VAX/UNIX,
and on the 8088/8086 under CP/M-86 or MS-DOS.  A version  is
currently  being  developed for the 68000 under CP/M-68K and
for the Apple Macintosh.  It is completely  written  in  the
programming  language  'C'  and is easily extended with user
written built-in functions and classes.  It is available  in
source   form   free  of  charge  to  non-commercial  users.
Prospective commercial users should contact the  author  for
permission to use XLISP.

Many traditional LISP functions are built  into  XLISP.   In
addition,  XLISP defines the objects 'Object' and 'Class' as
primitives.   'Object'  is  the  only  class  that  has   no
superclass  and  hence  is  the  root of the class heirarchy
tree.  'Class'  is  the  class  of  which  all  classes  are
instances  (it  is  the  only  object that is an instance of
itself).

This document is intended  to  be  a  brief  description  of
XLISP.    It   assumes  some  knowledge  of  LISP  and  some
understanding   of   the   concepts   of   object   oriented
programming.

Version 1.2 of XLISP differs from  version  1.1  in  several
ways.   It  supports  many  more Lisp functions.  Also, many
version 1.1  functions  have  been  renamed  and/or  changed
slightly  to follow traditional Lisp usage.  One of the most
frequently reported problems in version  1.1  resulted  from
many  functions being named after their equivilent functions
in the C language.  This turned  out  to  be  confusing  for
people who were trying to learn XLISP using traditional LISP
texts as references.  Version 1.2 renames these functions to
be  compatible  with  more  traditional  dialects  of  LISP.
Version 1.3 introduces many new  LISP  functions  and  moves
closer  to the goal of being compatible with the Common Lisp
standard.  Version 1.4 introduces user  error  handling  and
breakpoint  support  as  well as more Common Lisp compatible
functions.

A recommended text for learning LISP programming is the book
"LISP"  by Winston and Horn and published by Addison Wesley.
The first edition of this book is based on MacLisp  and  the
second  edition is based on Common Lisp.  Future versions of
XLISP will continue to migrate  towards  compatibility  with
Common Lisp.

@section(A Note From the Author)

If you have any problems with XLISP, feel free to contact me
for  help  or  advice.   Please remember that since XLISP is
available in source form in  a  high  level  language,  many
users  have  been  making versions available on a variety of
machines.  If you call to report a problem with  a  specific
version,  I may not be able to help you if that version runs
on a machine to which I don't have access.  Please have  the
version  number  of the version that you are running readily
accessible before calling me.

If you find a bug  in  XLISP,  first  try  to  fix  the  bug
yourself  using  the  source  code  provided.   If  you  are
successful in fixing the bug, send the bug report along with
the  fix to me.  If you don't have access to a C compiler or
are unable to fix a bug, please send the bug  report  to  me
and I'll try to fix it.

Any suggestions for improvements  will  be  welcomed.   Feel
free  to  extend  the  language  in  whatever way suits your
needs.  However, PLEASE DO  NOT  RELEASE  ENHANCED  VERSIONS
WITHOUT  CHECKING  WITH  ME  FIRST!!  I would like to be the
clearing house for new features added to XLISP.  If you want
to  add  features for your own personal use, go ahead.  But,
if you want to distribute your enhanced version, contact  me
first.  Please remember that the goal of XLISP is to provide
a language to learn and  experiment  with  LISP  and  object
oriented programming on small computers.

@section(XLISP Command Loop)

When XLISP is started, it first  tries  to  load  "init.lsp"
from  the  default directory.  It then loads any files named
as parameters on the command line (after appending ".lsp" to
their names).  It then issues the following prompt:

@begin(programexample)
>
@end(programexample)

This indicates that XLISP is waiting for an expression to be
typed.   When  an  incomplete expression has been typed (one
where the left and right parens don't match)  XLISP  changes
its prompt to:

@begin(programexample)
n>
@end(programexample)

where n is an integer indicating how  many  levels  of  left
parens remain unclosed.

When a complete expression has been entered, XLISP  attempts
to  evaluate  that  expression.  If the expression evaluates
successfully, XLISP prints the result of the evaluation  and
then  returns  to  the  initial  prompt  waiting for another
expression to be typed.

Input can be aborted at any time by typing the CONTROL-G key
(it may be necessary to follow CONTROL-G by RETURN).

@section(Break Command Loop)

When  XLISP  encounters  an  error   while   evaluating   an
expression, it attempts to handle the error in the following
way:

If  the  symbol  '*breakenable*'  is   true,   the   message
corresponding  to  the  error  is  printed.  If the error is
correctable, the correction  message  is  printed.   If  the
symbol '*tracenable*' is true, a trace back is printed.  The
number of entries printed depends on the value of the symbol
'*tracelimit*'.   If  this  symbol is set to something other
than a number, the  entire  trace  back  stack  is  printed.
XLISP  then  enters a read/eval/print loop to allow the user
to examine the state of the interpreter in  the  context  of
the  error.   This  loop  differs  from the normal top-leval
read/eval/print loop in that if the user  types  the  symbol
'continue' XLISP will continue from a correctable error.  If
the user types the symbol 'quit' XLISP will abort the  break
loop  and return to the top level or the next lower numbered
break loop.  When in a break loop, XLISP prefixes the  break
level to the normal prompt.

If the symbol '*breakenable*' is  nil,  XLISP  looks  for  a
surrounding   errset  function.   If  one  is  found,  XLISP
examines the value of the print flag.  If this flag is true,
the error message is printed.  In any case, XLISP causes the
errset function call to return nil.

If there is no surrounding errset function, XLISP prints the
error message and returns to the top level.


@section(Data Types)

There are several different data types  available  to  XLISP
programmers.
@begin(itemize)
lists

symbols

strings

integers

objects

file pointers

subrs/fsubrs (built-in functions)
@end(itemize)

Another data type is the stream.  A stream is  a  list  node
whose car points to the head of a list of integers and whose
cdr points to the last list node  of  the  list.   An  empty
stream  is  a  list node whose car and cdr are nil.  Each of
the integers in the  list  represents  a  character  in  the
stream.   When  a character is read from a stream, the first
integer from the head of the list is removed  and  returned.
When  a  character  is  written  to  a  stream,  the integer
representing the character code of the character is appended
to  the  end of the list.  When a function indicates that it
takes an input source as a  parameter,  this  parameter  can
either  be  an  input  file pointer or a stream.  Similarly,
when a function indicates that it takes an output sink as  a
parameter,  this  parameter  can  either  be  an output file
pointer or a stream.


@section(The Evaluator)

The process of evaluation in XLISP:

@begin(itemize)
Integers,  strings,  objects,  file  pointers,  and
subrs evaluate to themselves

Symbols evaluate to the value associated with their
current binding

@begin(multiple)
Lists are evaluated by evaluating the first element
of the list

@begin(itemize)
If it evaluates to a subr, the  remaining  list
elements  are  evaluated and the subr is called
with these evaluated expressions as arguments.

If it evaluates  to  an  fsubr,  the  fsubr  is
called  using  the  remaining  list elements as
arguments  (they  are  evaluated  by  the  subr
itself if necessary)

If it evaluates to a list and the  car  of  the
list  is  'lambda', the remaining list elements
are evaluated and the resulting expressions are
bound  to  the  formal  arguments of the lambda
expression.   The  body  of  the  function   is
executed within this new binding environment.

If it evaluates to a list and the  car  of  the
list  is  'macro',  the remaining list elements
are bound to the formal arguments of the  macro
expression.    The  body  of  the  function  is
executed within this new  binding  environment.
The result of this evaluation is considered the
macro expansion.  This result is then evaluated
in place of the original expression.

If it evaluates to an object, the  second  list
element  is  evaluated  and  used  as a message
selector.  The message formed by combining  the
selector  with the values of the remaining list
elements is sent to the object.
@end(itemize)
@end(multiple)
@end(itemize)

@section(Lexical Conventions)

The following conventions are followed when  entering  XLISP
programs:

Comments in XLISP code begin with a semi-colon character and
continue to the end of the line.

Symbol names  in  XLISP  can  consist  of  any  sequence  of
non-blank printable characters except the following:

@begin(verbatim)
              ( ) ' ` , " ;
@end(verbatim)

Upper and lower case characters are distinct.   The  symbols
'CAR' and 'car' are not the same.  The names of all built-in
functions are in lower case.   The  names  of  all  built-in
objects  are  lower  case  with  an initial capital.  Symbol
names must not begin with a digit.

Integer literals consist of a sequence of digits  optionally
beginning with a '+' or '-'.  The range of values an integer
can represent is limited by the size of a  C  'int'  on  the
machine that XLISP is running on.

Literal strings are sequences of  characters  surrounded  by
double  quotes.   Within quoted strings the '\' character is
used to allow non-printable characters to be included.   The
codes recognized are:

@begin(description, spread=0)
\\@\means the character '\'

\n@\means newline

\t@\means tab

\r@\means return

\e@\means escape

\nnn@\means the character whose octal code is nnn
@end(description)

XLISP defines several useful read macros:

@begin(description, spread = 0)
'<expr>@\== (quote <expr>)

#'<expr>@\== (function <expr>)

`<expr>@\== (backquote <expr>)

,<expr>@\== (comma <expr>)

,@@<expr>@\== (comma-at <expr>)
@end(description)


@section(Objects)

Definitions:

@begin(description)
selector@\a symbol used to select  an  appropriate
method

message@\a selector and a list of actual arguments

method@\the code that implements a message
@end(description)

Since XLISP was  created  to  provide  a  simple  basis  for
experimenting  with  object oriented programming, one of the
primitive data types included was 'object'.   In  XLISP,  an
object  consists of a data structure containing a pointer to
the object's class as well as a list containing  the  values
of the object's instance variables.

Officially, there is no way to see inside an object (look at
the  values  of  its  instance  variables).  The only way to
communicate with an object is by sending it a message.  When
the  XLISP  evaluator  evaluates  a  list the value of whose
first element is an object, it interprets the value  of  the
second  element  of the list (which must be a symbol) as the
message selector.  The evaluator determines the class of the
receiving object and attempts to find a method corresponding
to the message selector in the set of messages  defined  for
that  class.   If  the  message is not found in the object's
class and the class has a super-class, the search  continues
by  looking  at  the  messages  defined for the super-class.
This process continues from  one  super-class  to  the  next
until  a  method  for the message is found.  If no method is
found, an error occurs.

When a method is found, the evaluator  binds  the  receiving
object  to  the  symbol 'self', binds the class in which the
method was found to the symbol 'msgclass', and evaluates the
method  using the remaining elements of the original list as
arguments  to  the  method.   These  arguments  are   always
evaluated prior to being bound to their corresponding formal
arguments.  The result of evaluating the method becomes  the
result of the expression.

Classes:
@begin(programexample)

      Object  THE TOP OF THE CLASS HEIRARCHY

          Messages:

              show    SHOW AN OBJECT'S INSTANCE VARIABLES
                  returns     the object
@hinge
              class   RETURN THE CLASS OF AN OBJECT
                  returns     the class of the object
@hinge
              isnew   THE DEFAULT OBJECT INITIALIZATION ROUTINE
                  returns     the object
@hinge
              sendsuper <sel> [<args>...] SEND SUPERCLASS A MESSAGE
                  <sel>       the message selector
                  <args>      the message arguments
                  returns     the result of sending the message

@hinge
      Class   THE CLASS OF ALL OBJECT CLASSES (including itself)

          Messages:

              new     CREATE A NEW INSTANCE OF A CLASS
                  returns     the new class object
@hinge
              isnew [<scls>]  INITIALIZE A NEW CLASS
                  <scls>      the superclass
                  returns     the new class object
@hinge
              answer <msg> <fargs> <code>     ADD A MESSAGE TO A CLASS
                  <msg>       the message symbol
                  <fargs>     the formal argument list
                                this list is of the form:
                                  (<farg>...
                                   [&optional <oarg>...]
                                   [&rest <rarg>]
                                   [&aux <aux>...])
                                where
                                  <farg>   a formal argument
                                  <oarg>   an optional argument
						(default is nil)
                                  <rarg>   bound to the rest of the 
						arguments
                                  <aux>    a auxiliary variable
				  		(set to nil)
                  <code>      a list of executable expressions
                  returns     the object
@hinge
              ivars <vars>    DEFINE THE LIST OF INSTANCE VARIABLES
                  <vars>      the list of instance variable symbols
                  returns     the object
@hinge
              cvars <vars>    DEFINE THE LIST OF CLASS VARIABLES
                  <vars>      the list of class variable symbols
                  returns     the object
@end(programexample)


When a new instance of a class is  created  by  sending  the
message  'new'  to  an  existing  class, the message 'isnew'
followed by whatever parameters were  passed  to  the  'new'
message is sent to the newly created object.

When a new class is created by sending the 'new' message  to
the  object  'Class', an optional parameter may be specified
indicating  the  superclass  of  the  new  class.   If  this
parameter  is  omitted,  the new class will be a subclass of
'Object'.  A class inherits all  instance  variables,  class
variables, and methods from its super-class.


@section(Symbols)


@begin(description)
self@\the  current  object  (within  a   message
context)

msgclass@\the class in which  the  current  method
was found

*oblist*@\the object list

*keylist*@\the keyword list

*standard-input*@\the standard input file

*standard-output*@\the standard output file

*breakenable*@\flag controlling entering the break
loop on errors

*tracenable*@\flag controlling trace back printout
on errors and breaks

*tracelimit*@\maximum number of  levels  of  trace
back information printed on errors and breaks

*evalhook*@\user  substitute  for  the  evaluator
function

*applyhook*@\(not yet implemented)

*unbound*@\indicator for unbound symbols
@end(description)

@section(Evaluation Functions)

@begin(programexample)
      (eval <expr>)  EVALUATE AN XLISP EXPRESSION
          <expr>      the expression to be evaluated
          returns     the result of evaluating the expression

@hinge
      (apply <fun> <args>)  APPLY A FUNCTION TO A LIST OF ARGUMENTS
          <fun>       the function to apply (or function symbol)
          <args>      the argument list
          returns     the result of applying the function 
			to the argument list

@hinge
      (funcall <fun> <arg>...)  CALL A FUNCTION WITH ARGUMENTS
          <fun>       the function to call (or function symbol)
          <arg>       arguments to pass to the function
          returns     the result of calling the function 
	  		with the arguments

@hinge
      (quote <expr>)  RETURN AN EXPRESSION UNEVALUATED
          <expr>      the expression to be quoted (quoted)
          returns     <expr> unevaluated

@hinge
      (function <expr>)  QUOTE A FUNCTION (THIS IS THE SAME AS QUOTE)
          <expr>      the function to be quoted (quoted)
          returns     <expr> unevaluated

@hinge
      (backquote <expr>)  FILL IN A TEMPLATE
          <expr>      the template
          returns     a copy of the template with comma and comma-at
		        expressions expanded (see the Common Lisp
		        reference manual)
@end(programexample)

@section(Symbol Functions)

@begin(programexample)
      (set <sym> <expr>)  SET THE VALUE OF A SYMBOL
          <sym>       the symbol being set
          <expr>      the new value
          returns     the new value

@hinge
      (setq [<sym> <expr>]...)  SET THE VALUE OF A SYMBOL
          <sym>       the symbol being set (quoted)
          <expr>      the new value
          returns     the new value

@hinge
      (setf [<place> <expr>]...)  SET THE VALUE OF A FIELD
          <place>     specifies the field to set (quoted):
                          <sym>                the value of a symbol
                          (car <expr>)         the car of a list node
                          (cdr <expr>)         the cdr of a list node
                          (get <sym> <prop>)   the value of a property
                          (symbol-value <sym>) the value of a symbol
                          (symbol-plist <sym>) the property list 
						 of a symbol
          <value>     the new value
          returns     the new value

@hinge
      (defun <sym> <fargs> <expr>...)  DEFINE A FUNCTION

@hinge
      (defmacro <sym> <fargs> <expr>...)  DEFINE A MACRO
          <sym>       symbol being defined (quoted)
          <fargs>     list of formal arguments (quoted)
                        this list is of the form:
                          (<farg>...
                           [&optional <oarg>...]
                           [&rest <rarg>]
                           [&aux <aux>...])
                        where
                          <farg>  is a formal argument
                          <oarg>  is an optional argument (default nil)
                          <rarg>  bound to the rest of the arguments
                          <aux>   is an auxiliary variable (set to nil)
          <expr>      expressions constituting the body of the
                      function (quoted)
          returns     the function symbol

@hinge
      (gensym [<tag>])  GENERATE A SYMBOL
          <tag>       string or number
          returns     the new symbol

@hinge
      (intern <pname>)  MAKE AN INTERNED SYMBOL
          <pname>     the symbol's print name string
          returns     the new symbol

@hinge
      (make-symbol <pname>)  MAKE AN UNINTERNED SYMBOL
          <pname>     the symbol's print name string
          returns     the new symbol

@hinge
      (symbol-name <sym>)  GET THE PRINT NAME OF A SYMBOL
          <sym>       the symbol
          returns     the symbol's print name

@hinge
      (symbol-value <sym>)  GET THE VALUE OF A SYMBOL
          <sym>       the symbol
          returns     the symbol's value

@hinge
      (symbol-plist <sym>)  GET THE PROPERTY LIST OF A SYMBOL
          <sym>       the symbol
          returns     the symbol's property list
@end(programexample)


@section(Property List Functions)
@begin(programexample)

      (get <sym> <prop>)  GET THE VALUE OF A PROPERTY
          <sym>       the symbol
          <prop>      the property symbol
          returns     the property value or nil

@hinge
      (remprop <prop> <sym>)  REMOVE A PROPERTY
          <sym>       the symbol
          <prop>      the property symbol
          returns     nil
@end(programexample)

@section(List Functions)

@begin(programexample)
      (car <expr>)  RETURN THE CAR OF A LIST NODE
          <expr>      the list node
          returns     the car of the list node

@hinge
      (cdr <expr>)  RETURN THE CDR OF A LIST NODE
          <expr>      the list node
          returns     the cdr of the list node

@hinge
      (caar <expr>) == (car (car <expr>))
      (cadr <expr>) == (car (cdr <expr>))
      (cdar <expr>) == (cdr (car <expr>))
      (cddr <expr>) == (cdr (cdr <expr>))

@hinge
      (cons <expr1> <expr2>)  CONSTRUCT A NEW LIST NODE
          <expr1>     the car of the new list node
          <expr2>     the cdr of the new list node
          returns     the new list node

@hinge
      (list <expr>...)  CREATE A LIST OF VALUES
          <expr>      expressions to be combined into a list
          returns     the new list

@hinge
      (append <expr>...)  APPEND LISTS
          <expr>      lists whose elements are to be appended
          returns     the new list

@hinge
      (reverse <expr>)  REVERSE A LIST
          <expr>      the list to reverse
          returns     a new list in the reverse order

@hinge
      (last <list>)  RETURN THE LAST LIST NODE OF A LIST
          <list>      the list
          returns     the last list node in the list

@hinge
      (member <expr> <list> [<key> <test>])  FIND AN EXPRESSION 
					       IN A LIST
          <expr>      the expression to find
          <list>      the list to search
          <key>       the keyword :test or :test-not
          <test>      the test function (defaults to eql)
          returns     the remainder of the list starting
	  		with the expression

@hinge
      (assoc <expr> <alist> [<key> <test>])  FIND AN EXPRESSION
      					       IN AN A-LIST
          <expr>      the expression to find
          <alist>     the association list
          <key>       the keyword :test or :test-not
          <test>      the test function (defaults to eql)
          returns     the alist entry or nil

@hinge
      (remove <expr> <list> [<key> <test>])  REMOVE AN EXPRESSION
      					       FROM A LIST
          <expr>      the expression to delete
          <list>      the list
          <key>       the keyword :test or :test-not
          <test>      the test function (defaults to eql)
          returns     the list with the matching expressions deleted

@hinge
      (length <expr>)  FIND THE LENGTH OF A LIST
          <expr>      the list
          returns     the length of the list

@hinge
      (nth <n> <list>)  RETURN THE NTH ELEMENT OF A LIST
          <n>         the number of the element to return (zero origin)
          <list>      the list
          returns     the nth element 
			or nil if the list isn't that long

@hinge
      (nthcdr <n> <list>)  RETURN THE NTH CDR OF A LIST
          <n>         the number of the element to return (zero origin)
          <list>      the list
          returns     the nth cdr
			or nil if the list isn't that long

@hinge
      (mapc <fcn> <list1>...<listn>)  APPLY FUNCTION
					TO SUCCESSIVE CARS
          <fcn>       the function or function name
          <list1..n>  a list for each argument of the function
          returns     the first list of arguments

@hinge
      (mapcar <fcn> <list1>...<listn>)  APPLY FUNCTION
      					  TO SUCCESSIVE CARS
          <fcn>       the function or function name
          <list1..n>  a list for each argument of the function
          returns     the list of values returned
			by each function invocation

@hinge
      (mapl <fcn> <list1>...<listn>)  APPLY FUNCTION TO SUCCESSIVE CDRS
          <fcn>       the function or function name
          <list1..n>  a list for each argument of the function
          returns     the first list of arguments

@hinge
      (maplist <fcn> <list1>...<listn>)  APPLY FUNCTION
					   TO SUCCESSIVE CDRS
          <fcn>       the function or function name
          <list1..n>  a list for each argument of the function
          returns     the list of values returned
	  		by each function invocation

@hinge
      (subst <to> <from> <expr> [<key> <test>])  SUBSTITUTE EXPRESSIONS
          <to>        the new expression
          <from>      the old expression
          <expr>      the expression in which to do the substitutions
          <key>       the keyword :test or :test-not
          <test>      the test function (defaults to eql)
          returns     the expression with substitutions

@hinge
      (sublis <alist> <expr> [<key> <test>])  SUBSTITUTE
      					        USING AN A-LIST
          <alist>     the association list
          <expr>      the expression in which to substitute
          <key>       the keyword :test or :test-not
          <test>      the test function (defaults to eql)
          returns     the expression with substitutions
@end(programexample)

@section(Destructive List Functions)
@begin(programexample)
      (rplaca <list> <expr>)  REPLACE THE CAR OF A LIST NODE
          <list>      the list node
          <expr>      the new value for the car of the list node
          returns     the list node after updating the car

@hinge
      (rplacd <list> <expr>)  REPLACE THE CDR OF A LIST NODE
          <list>      the list node
          <expr>      the new value for the cdr of the list node
          returns     the list node after updating the cdr

@hinge
      (nconc <list>...)  DESTRUCTIVELY CONCATENATE LISTS
          <list>      lists to concatenate
          returns     the result of concatenating the lists

@hinge
      (delete <expr> <list> [<key> <test>])  DELETE AN EXPRESSION
      					       FROM A LIST
          <expr>      the expression to delete
          <list>      the list
          <key>       the keyword :test or :test-not
          <test>      the test function (defaults to eql)
          returns     the list with the matching expressions deleted
@end(programexample)

@section(Predicate Functions)
@begin(ProgramExample)

      (atom <expr>)  IS THIS AN ATOM?
          <expr>      the expression to check
          returns     t if the value is an atom, nil otherwise

@hinge
      (symbolp <expr>)  IS THIS A SYMBOL?
          <expr>      the expression to check
          returns     t if the expression is a symbol, nil otherwise

@hinge
      (numberp <expr>)  IS THIS A NUMBER?
          <expr>      the expression to check
          returns     t if the expression is a symbol, nil otherwise

@hinge
      (null <expr>)  IS THIS AN EMPTY LIST?
          <expr>      the list to check
          returns     t if the list is empty, nil otherwise

@hinge
      (not <expr>)  IS THIS FALSE?
          <expr>      the expression to check
          return      t if the expression is nil, nil otherwise

@hinge
      (listp <expr>)  IS THIS A LIST?
          <expr>      the expression to check
          returns     t if the value is a list node or nil,
	  	      nil otherwise

@hinge
      (consp <expr>)  IS THIS A NON-EMPTY LIST?
          <expr>      the expression to check
          returns     t if the value is a list node, nil otherwise

@hinge
      (boundp <sym>)  IS THIS A BOUND SYMBOL?
          <sym>       the symbol
          returns     t if a value is bound to the symbol,
	  	      nil otherwise

@hinge
      (minusp <expr>)  IS THIS NUMBER NEGATIVE?
          <expr>      the number to test
          returns     t if the number is negative, nil otherwise

@hinge
      (zerop <expr>)  IS THIS NUMBER ZERO?
          <expr>      the number to test
          returns     t if the number is zero, nil otherwise

@hinge
      (plusp <expr>)  IS THIS NUMBER POSITIVE?
          <expr>      the number to test
          returns     t if the number is positive, nil otherwise

@hinge
      (evenp <expr>)  IS THIS NUMBER EVEN?
          <expr>      the number to test
          returns     t if the number is even, nil otherwise

@hinge
      (oddp <expr>)  IS THIS NUMBER ODD?
          <expr>      the number to test
          returns     t if the number is odd, nil otherwise

@hinge
      (eq <expr1> <expr2>)  ARE THE EXPRESSIONS IDENTICAL?
          <expr1>     the first expression
          <expr2>     the second expression
          returns     t if they are equal, nil otherwise

@hinge
      (eql <expr1> <expr2>)  ARE THE EXPRESSIONS IDENTICAL?
                              (WORKS WITH NUMBERS AND STRINGS)
          <expr1>     the first expression
          <expr2>     the second expression
          returns     t if they are equal, nil otherwise

@hinge
      (equal <expr1> <expr2>)  ARE THE EXPRESSIONS EQUAL?
          <expr1>     the first expression
          <expr2>     the second expression
          returns     t if they are equal, nil otherwise
@end(ProgramExample)

@section(Control Functions)
@begin(ProgramExample)

      (cond <pair>...)  EVALUATE CONDITIONALLY
          <pair>      pair consisting of:
                          (<pred> <expr>...)
                        where
                          <pred>      is a predicate expression
                          <expr>      evaluated if the predicate
                                      is not nil
          returns     the value of the first expression whose predicate
                      is not nil

@hinge
      (and <expr>...)  THE LOGICAL AND OF A LIST OF EXPRESSIONS
          <expr>...   the expressions to be ANDed
          returns     nil if any expression evaluates to nil,
                      otherwise the value of the last expression
                      (evaluation of expressions stops after the first
                       expression that evaluates to nil)

@hinge
      (or <expr>...)  THE LOGICAL OR OF A LIST OF EXPRESSIONS
          <expr>...   the expressions to be ORed
          returns     nil if all expressions evaluate to nil,
                      else the value of the first non-nil expression
                      (evaluation of expressions stops after the first
                       expression that does not evaluate to nil)

@hinge
      (if <texpr> <expr1> [<expr2>])  EXECUTE EXPRESSIONS CONDITIONALLY
          <texpr>     the test expression
          <expr1>     the expression to be evaluated
			if texpr is non-nil
          <expr2>     the expression to be evaluated if texpr is nil
          returns     the value of the selected expression

@hinge
      (let (<binding>...) <expr>...)  BIND SYMBOLS AND
      					EVALUATE EXPRESSIONS

      (let* (<binding>...) <expr>...)  LET WITH SEQUENTIAL BINDING
          <binding>   the variable bindings each of which is either:
                      1)  a symbol (which is initialized to nil)
                      2)  a list whose car is a symbol and whose cadr
                              is an initialization expression
          <expr>...   the expressions to be evaluated
          returns     the value of the last expression

@hinge
      (catch <sym> [<expr>]...)  EVALUATE EXPRESSIONS AND CATCH THROWS
          <sym>       the catch tag
          <expr>...   expressions to evaluate
          returns     the value of the last expression or
	  		the throw expression

@hinge
      (throw <sym> [<expr>])  THROW TO A CATCH
          <sym>       the catch tag
          <expr>      the value for the catch to return (default nil)
          returns     never returns
@end(ProgramExample)


@section(Looping Functions)

@begin(ProgramExample)
      (do ([<binding>]...) (<texpr> [<rexpr>]...) [<expr>]...)
      (do* ([<binding>]...) (<texpr> [<rexpr>]...) [<expr>]...)
          <binding>   the variable bindings each of which is either:
                      1)  a symbol (which is initialized to nil)
                      2)  a list of the form: (<sym> <init> [<step>])
                          where:
                              <sym>  is the symbol to bind
                              <init> is the initial value of the symbol
                              <step> is a step expression
          <texpr>     the termination test expression
          <rexpr>...  result expressions (the default is nil)
          <expr>...   the body of the loop (treated like a prog)
          returns     the value of the last result expression

@hinge
      (dolist (<sym> <expr> [<rexpr>]) [<expr>]...)  LOOP THRU A LIST
          <sym>       the symbol to bind to each list element
          <expr>      the list expression
          <rexpr>     the result expression (the default is nil)
          <expr>...   the body of the loop (treated like a prog)

@hinge
      (dotimes (<sym> <expr> [<rexpr>]) [<expr>]...)  LOOP FROM ZERO
      						        TO N-1
          <sym>       the symbol to bind to each value from 0 to n-1
          <expr>      the number of times to loop
          <rexpr>     the result expression (the default is nil)
          <expr>...   the body of the loop (treated like a prog)

@end(ProgramExample)

@section(The Program Feature)
@begin(ProgramExample)

      (prog (<binding>...) [<expr>]...)  THE PROGRAM FEATURE
      (prog* (<binding>...) [<expr>]...)  PROG WITH SEQUENTIAL BINDING
          <binding>   the variable bindings each of which is either:
                      1)  a symbol (which is initialized to nil)
                      2)  a list whose car is a symbol and whose cadr
                              is an initialization expression
          <expr>      expressions to evaluate or tags (symbols)
          returns     nil or the argument passed to the return function

@hinge
      (go <sym>)  GO TO A TAG WITHIN A PROG CONSTRUCT
          <sym>       the tag (quoted)
          returns     never returns

@hinge
      (return [<expr>])  CAUSE A PROG CONSTRUCT TO RETURN A VALUE
          <expr>      the value (defaults to nil)
          returns     never returns

@hinge
      (prog1 <expr1> [<expr>]...)  EXECUTE EXPRESSIONS SEQUENTIALLY
          <expr1>     the first expression to evaluate
          <expr>...   the remaining expressions to evaluate
          returns     the value of the first expression

@hinge
      (prog2 <expr1> <expr2> [<expr>]...)  EXECUTE EXPRESSIONS
      					     SEQUENTIALLY
          <expr1>     the first expression to evaluate
          <expr2>     the second expression to evaluate
          <expr>...   the remaining expressions to evaluate
          returns     the value of the second expression

@hinge
      (progn [<expr>]...)  EXECUTE EXPRESSIONS SEQUENTIALLY
          <expr>...   the expressions to evaluate
          returns     the value of the last expression (or nil)

@end(ProgramExample)

@section(Debugging and Error Handling)
@begin(ProgramExample)
      (error <emsg> [<arg>])  SIGNAL A NON-CORRECTABLE ERROR
          <emsg>      the error message string
          <arg>       the argument expression
	  		(printed after the message)
          returns     never returns

@hinge
      (cerror <cmsg> <emsg> [<arg>])  SIGNAL A CORRECTABLE ERROR
          <cmsg>      the continue message string
          <emsg>      the error message string
          <arg>       the argument expression
	  		(printed after the message)
          returns     nil when continued from the break loop

@hinge
      (break [<bmsg> [<arg>]])  ENTER A BREAK LOOP
          <bmsg>      the break message string
	  		(defaults to "**BREAK**")
          <arg>       the argument expression
	  		(printed after the message)
          returns     nil when continued from the break loop

@hinge
      (errset <expr> [<pflag>])  TRAP ERRORS
          <expr>      the expression to execute
          <pflag>     flag to control printing of the error message
          returns     the value of the last expression consed with nil
                      or nil on error

@hinge
      (baktrace [<n>])  PRINT N LEVELS OF TRACE BACK INFORMATION
          <n>         the number of levels (defaults to all levels)
          returns     nil

@hinge
      (evalhook <expr> <ehook> <ahook>)  EVALUATE AN EXPRESSION
      					   WITH HOOKS
          <expr>      the expression to evaluate
          <ehook>     the value for *evalhook*
          <ahook>     the value for *applyhook*
          returns     the result of evaluating the expression
@end(ProgramExample)

@section(Arithmetic Functions)
@begin(ProgramExample)
      (+ <expr>...)  ADD A LIST OF NUMBERS
          <expr>...   the numbers
          returns     the result of the addition

@hinge
      (- <expr>...)  SUBTRACT A LIST OF NUMBERS
      		       OR NEGATE A SINGLE NUMBER
          <expr>...   the numbers
          returns     the result of the subtraction

@hinge
      (* <expr>...)  MULTIPLY A LIST OF NUMBERS
          <expr>...   the numbers
          returns     the result of the multiplication

@hinge
      (/ <expr>...)  DIVIDE A LIST OF NUMBERS
          <expr>...   the numbers
          returns     the result of the division

@hinge
      (1+ <expr>)  ADD ONE TO A NUMBER
          <expr>      the number
          returns     the number plus one

@hinge
      (1- <expr>)  SUBTRACT ONE FROM A NUMBER
          <expr>      the number
          returns     the number minus one

@hinge
      (rem <expr>...)  REMAINDER OF A LIST OF NUMBERS
          <expr>...   the numbers
          returns     the result of the remainder operation

@hinge
      (min <expr>...)  THE SMALLEST OF A LIST OF NUMBERS
          <expr>...   the expressions to be checked
          returns     the smallest number in the list

@hinge
      (max <expr>...)  THE LARGEST OF A LIST OF NUMBERS
          <expr>...   the expressions to be checked
          returns     the largest number in the list

@hinge
      (abs <expr>)  THE ABSOLUTE VALUE OF A NUMBER
          <expr>      the number
          returns     the absolute value of the number
@end(ProgramExample)

@section(Bitwise Logical Functions)
@begin(ProgramExample)

      (bit-and <expr>...)  THE BITWISE AND OF A LIST OF NUMBERS
          <expr>      the numbers
          returns     the result of the and operation

@hinge
      (bit-ior <expr...)  THE BITWISE INCLUSIVE OR OF A LIST OF NUMBERS
          <expr>      the numbers
          returns     the result of the inclusive or operation

@hinge
      (bit-xor <expr...)  THE BITWISE EXCLUSIVE OR OF A LIST OF NUMBERS
          <expr>      the numbers
          returns     the result of the exclusive or operation

@hinge
      (bit-not <expr>)  THE BITWISE NOT OF A NUMBER
          <expr>      the number
          returns     the bitwise inversion of number
@end(ProgramExample)

@section(Relational Functions)
The relational functions can be used to compare integers  or
strings.   The  functions  '='  and '/=' can also be used to
compare other types.  The result  of  these  comparisons  is
computed the same way as for 'eq'.

@begin(ProgramExample)
      (< <e1> <e2>)  TEST FOR LESS THAN
          <e1>        the left operand of the comparison
          <e2>        the right operand of the comparison
          returns     the result of comparing <e1> with <e2>

@hinge
      (<= <e1> <e2>)  TEST FOR LESS THAN OR EQUAL TO
          <e1>        the left operand of the comparison
          <e2>        the right operand of the comparison
          returns     the result of comparing <e1> with <e2>

@hinge
      (= <e1> <e2>)  TEST FOR EQUAL TO
          <e1>        the left operand of the comparison
          <e2>        the right operand of the comparison
          returns     the result of comparing <e1> with <e2>

@hinge
      (/= <e1> <e2>)  TEST FOR NOT EQUAL TO
          <e1>        the left operand of the comparison
          <e2>        the right operand of the comparison
          returns     the result of comparing <e1> with <e2>

@hinge
      (>= <e1> <e2>)  TEST FOR GREATER THAN OR EQUAL TO
          <e1>        the left operand of the comparison
          <e2>        the right operand of the comparison
          returns     the result of comparing <e1> with <e2>

@hinge
      (> <e1> <e2>)  TEST FOR GREATER THAN
          <e1>        the left operand of the comparison
          <e2>        the right operand of the comparison
          returns     the result of comparing <e1> with <e2>
@end(ProgramExample)


@section(String Functions)
@begin(ProgramExample)

      (strcat <expr>...)  CONCATENATE STRINGS
          <expr>...   the strings to concatenate
          returns     the result of concatenating the strings

@hinge
      (strlen <expr>)  COMPUTE THE LENGTH OF A STRING
          <expr>      the string
          returns     the length of the string

@hinge
      (substr <expr> <sexpr> [<lexpr>]) EXTRACT A SUBSTRING
          <expr>      the string
          <sexpr>     the starting position
          <lexpr>     the length (default is rest of string)
          returns     substring starting at <sexpr> for <lexpr>

@hinge
      (ascii <expr>)  NUMERIC VALUE OF CHARACTER
          <expr>      the string
          returns     the ascii code of the first character

@hinge
      (chr <expr>)  CHARACTER EQUIVALENT OF ASCII VALUE
          <expr>      the numeric expression
          returns     a one character string
	  		whose first character is <expr>

@hinge
      (atoi <expr>)  CONVERT AN ASCII STRING TO AN INTEGER
          <expr>      the string
          returns     the integer value of the string expression

@hinge
      (itoa <expr>)  CONVERT AN INTEGER TO AN ASCII STRING
          <expr>      the integer
          returns     the string representation of the integer value
@end(ProgramExample)

@section(Input/Output Functions)
@begin(ProgramExample)
      (read [<source> [<eof>]])  READ AN XLISP EXPRESSION
          <source>    the input source (default is standard input)
          <eof>       the value to return on end of file (default nil)
          returns     the expression read

@hinge
      (print <expr> [<sink>])  PRINT A LIST OF VALUES ON A NEW LINE
          <expr>      the expressions to be printed
          <sink>      the output sink (default is standard output)
          returns     nil

@hinge
      (prin1 <expr> [<sink>])  PRINT A LIST OF VALUES
          <expr>      the expressions to be printed
          <sink>      the output sink (default is standard output)
          returns     nil

@hinge
      (princ <expr> [<sink>])  PRINT A LIST OF VALUES WITHOUT QUOTING
          <expr>      the expressions to be printed
          <sink>      the output sink (default is standard output)
          returns     nil

@hinge
      (terpri [<sink>])  TERMINATE THE CURRENT PRINT LINE
          <sink>      the output sink (default is standard output)
          returns     nil

@hinge
      (flatsize <expr>)  LENGTH OF PRINTED REPRESENTATION USING PRIN1
          <expr>      the expression
          returns     the length

@hinge
      (flatc <expr>)  LENGTH OF PRINTED REPRESENTATION USING PRINC
          <expr>      the expression
          returns     the length

@hinge
      (explode <expr>)  CHARACTERS IN PRINTED REPRESENTATION
			  USING PRIN1
          <expr>      the expression
          returns     the list of characters

@hinge
      (explodec <expr>)  CHARACTERS IN PRINTED REPRESENTATION
			   USING PRINC
          <expr>      the expression
          returns     the list of characters

@hinge
      (maknam <list>)  BUILD AN UNINTERNED SYMBOL FROM
			 A LIST OF CHARACTERS
          <list>      list of characters in symbol name
          returns     the symbol

@hinge
      (implode <list>)  BUILD AN INTERNED SYMBOL FROM
			  A LIST OF CHARACTERS
          <list>      list of characters in symbol name
          returns     the symbol
@end(ProgramExample)

@section(File I/O Functions)
@begin(ProgramExample)

      (openi <fname>)  OPEN AN INPUT FILE
          <fname>     the file name string
          returns     a file pointer

@hinge
      (openo <fname>)  OPEN AN OUTPUT FILE
          <fname>     the file name string
          returns     a file pointer

@hinge
      (close <fp>)  CLOSE A FILE
          <fp>        the file pointer
          returns     nil

@hinge
      (read-char [<source>])  READ A CHARACTER FROM A FILE OR STREAM
          <source>    the input source (default is standard input)
          returns     the character (integer)

@hinge
      (peek-char [<flag> [<source>]])  PEEK AT THE NEXT CHARACTER
          <flag>      flag for skipping white space (default is nil)
          <source>    the input source (default is standard input)
          returns     the character (integer)

@hinge
      (write-char <ch> [<sink>])  WRITE A CHARACTER TO A FILE OR STREAM
          <ch>        the character to put (integer)
          <sink>      the output sink (default is standard output)
          returns     the character (integer)

@hinge
      (readline [<source>])  READ A LINE FROM A FILE OR STREAM
          <source>    the input source (default is standard input)
          returns     the input string
@end(ProgramExample)

@section(System Functions)

@begin(ProgramExample)
      (load <fname> [<vflag> [<pflag>]])  LOAD AN XLISP SOURCE FILE
          <fname>     the filename string (".lsp" is appended)
          <vflag>     the verbose flag (default is t)
          <pflag>     the print flag (default is nil)
          returns     the filename

@hinge
      (gc)  FORCE GARBAGE COLLECTION
          returns     nil

@hinge
      (expand <num>)  EXPAND MEMORY BY ADDING SEGMENTS
          <num>       the number of segments to add
          returns     the number of segments added

@hinge
      (alloc <num>)  CHANGE NUMBER OF NODES TO ALLOCATE IN EACH SEGMENT
          <num>       the number of nodes to allocate
          returns     the old number of nodes to allocate

@hinge
      (mem)  SHOW MEMORY ALLOCATION STATISTICS
          returns     nil

@hinge
      (type <expr>)  RETURNS THE TYPE OF THE EXPRESSION
          <expr>      the expression to return the type of
          returns     nil if the value is nil, else one of the symbols:
                          SYM   for symbols
                          OBJ   for objects
                          LIST  for list nodes
                          SUBR  for subroutine nodes
			  	  with evaluated arguments
                          FSUBR for subroutine nodes with
			  	  unevaluated arguments
                          STR   for string nodes
                          INT   for integer nodes
                          FPTR  for file pointer nodes

@hinge
      (exit)  EXIT XLISP
          returns     never returns

@end(ProgramExample)
