/********************************************************/
/*							*/
/*		YALE layout editor			*/
/*							*/
/*		(C) COPYRIGHT 1982			*/
/*		BOARD OF TRUSTEES			*/
/*	LELAND STANFORD JUNIOR UNIVERSITY		*/
/*	  STANFORD, CA. 94305, U. S. A.			*/
/*		THOMAS R. DAVIS				*/
/*							*/
/********************************************************/

/* FILE: status.c */

/* The routines in this file are meant to support a status window for
 * YALE.  It should present such information as the current cell being
 * edited, the current file open for i/o, the currently-selected layer,
 * the current default widths of all the layers, the world coordinates of
 * the most recently moused point, and dy and dx from the last such point.
 * The user will be able to mouse the stipple patterns to set the current
 * layer.
 */

#include "aledefs.h"


#define StatusXmin 550		/* initial screen coordinates for Status window */
#define StatusXmax 1020
#define StatusYmin 7
#define StatusYmax 120


#define dx 40
#define width 32
#define yMin 32
#define yMax 48
#define xBase 152

extern short DefaultLayerWidth,
	     DefaultMetalWidth,
	     DefaultPolyWidth,
	     DefaultDiffWidth,
	     DefaultImplantWidth,
	     DefaultBuriedWidth,
	     DefaultGlassWidth,
	     DefaultContactWidth;

extern PROCESS_LAYER DefaultLayer;

short StatusVGT;			/* VGT index for staus window */
static short StatusSymbol;		/* Top symbol in status window */
short AltSDF;				/* SDF number for non-graphics */


static short MetalNum, PolyNum, DiffNum, 
		ImplantNum, BuriedNum, 
		GlassNum, ContactNum,
		WidthNum, DisplayXNum, DisplayYNum, DXNum, DYNum,
		FileNameNum, CurrentCellNum, RepeatCmdNum, FreeNum;

static short Outline;
static char WidthLine[35];
static char DisplayX[] = "x: 0      ";
static char DisplayY[] = "y: 0      ";
static char DisplayDX[] = "dx: 0      ";
static char DisplayDY[] = "dy: 0      ";
static char FileName[] = "File: (none)       ";
static char CurrentCell[] = "Cell: (none)       ";
static char RepeatCmd[] = "Rpt Cmd: NONE        ";
/*static char FreeBytes[] = "Free List:        "; */
static char *LayerNames = "Metl Poly Diff Impl Burd Glas Cont";

InitStatusVGT()
  {
    WriteWidthLine ();
    StatusSymbol = UNIQUE;
    StatusVGT = CreateVGT(AltSDF, GRAPHICS, StatusSymbol,"Yale Status");
    BuildStippleMenu();
    CreateView( StatusVGT, 
		StatusXmin, StatusYmin,
		StatusXmax, StatusYmax, 0, -16, 0, FALSE);
  }

/* StatusLayerSelect: takes a pair of coordinates which are guaranteed
 * to lie within the status window, and finds out if they lie within one
 * of the layer "menu" rectangles.  If so, the layer default is changed to
 * be the appropriate thing, and the selection is indicated in the status
 * window.
 */

StatusLayerSelect(x, y)
  short x, y;
{
    short layer;
    LISTTYPE FindSelectedObject (), list;

    list = FindSelectedObject(AltSDF, x, y, StatusVGT, JustRects);

    if (list.NumOfElements == 0)
        return;

    EditSymbol(AltSDF, StatusSymbol);
    layer = list.Header->item;
    if (layer == MetalNum)
     {
        ChangeItem(AltSDF,Outline, xBase, xBase+width, yMin, yMax, AllEdges, SDF_OUTLINE);
        DefaultLayer = NM;
        DefaultLayerWidth = DefaultMetalWidth;
      } else
    if (layer == PolyNum)
      {
        ChangeItem(AltSDF,Outline,xBase+dx, xBase+width+dx, yMin, yMax, AllEdges, SDF_OUTLINE);
        DefaultLayer = NP;
        DefaultLayerWidth = DefaultPolyWidth;
      } else
    if (layer == DiffNum)
      {
        ChangeItem(AltSDF,Outline,xBase+2*dx,xBase+width+2*dx, yMin,yMax,AllEdges,SDF_OUTLINE);
        DefaultLayer = ND;
        DefaultLayerWidth = DefaultDiffWidth;
      } else
    if (layer == ImplantNum)
      {
        ChangeItem(AltSDF,Outline,xBase+3*dx,xBase+width+3*dx,yMin,yMax,AllEdges,SDF_OUTLINE);
        DefaultLayer = NI;
        DefaultLayerWidth = DefaultImplantWidth;
      } else
    if (layer == BuriedNum)
      {
        ChangeItem(AltSDF,Outline,xBase+4*dx,xBase+width+4*dx,yMin,yMax,AllEdges,SDF_OUTLINE);
        DefaultLayer = NB;
        DefaultLayerWidth = DefaultBuriedWidth;
      } else
    if (layer == GlassNum)
      {
        ChangeItem(AltSDF,Outline,xBase+5*dx,xBase+width+5*dx,yMin,yMax,AllEdges,SDF_OUTLINE);
        DefaultLayer = NG;
        DefaultLayerWidth = DefaultGlassWidth;
      } else
    if (layer == ContactNum)
      {
        ChangeItem(AltSDF,Outline,xBase+6*dx,xBase+width+6*dx,yMin,yMax,AllEdges,SDF_OUTLINE);
        DefaultLayer = NC;
        DefaultLayerWidth = DefaultContactWidth;
      }
    
    EndSymbol(AltSDF, StatusSymbol, StatusVGT); 
}


BuildStippleMenu()
  {
    short dxMultiplier, response;

	/*
	 * reset the stipple menu to the defaults
	 */
    DeleteSymbol(AltSDF, StatusSymbol);
    response = DefineSymbol(AltSDF, StatusSymbol, "Status VGT");
    if (response!=StatusSymbol && Debug)
      {
        printf("Status Symbol %d define returned %d\n", StatusSymbol, response);
      }
    AddItem(AltSDF, (MetalNum = UNIQUE), xBase, xBase+width, yMin, yMax, NM,
	SDF_FILLED_RECTANGLE, NULL);
    AddItem(AltSDF, (PolyNum = UNIQUE), xBase + dx, xBase+width + dx, yMin, yMax, NP,
	SDF_FILLED_RECTANGLE, NULL);
    AddItem(AltSDF, (DiffNum = UNIQUE), xBase+2*dx, xBase+width+2*dx, yMin, yMax, ND,
	SDF_FILLED_RECTANGLE, NULL);
    AddItem(AltSDF, (ImplantNum = UNIQUE), xBase+3*dx, xBase+width+3*dx, yMin, yMax, NI,
	SDF_FILLED_RECTANGLE, NULL);
    AddItem(AltSDF, (BuriedNum = UNIQUE), xBase+4*dx, xBase+width+4*dx, yMin, yMax, NB,
	SDF_FILLED_RECTANGLE, NULL);
    AddItem(AltSDF, (GlassNum = UNIQUE), xBase+5*dx, xBase+width+5*dx, yMin, yMax, NG,
	SDF_FILLED_RECTANGLE, NULL);
    AddItem(AltSDF, (ContactNum = UNIQUE), xBase+6*dx, xBase+width+6*dx, yMin, yMax, NC,
	SDF_FILLED_RECTANGLE, NULL);

    switch (DefaultLayer)
     {
      case NM:
    	dxMultiplier = 0;
    	break;
      case NB:
    	dxMultiplier = 4;
    	break;
      case NI:
    	dxMultiplier = 3;
    	break;
      case NC:
    	dxMultiplier = 6;
    	break;
      case NP:
    	dxMultiplier = 1;
    	break;
      case ND:
    	dxMultiplier = 2;
    	break;
      case NG:
    	dxMultiplier = 5;
    	break;
     }
    AddItem(AltSDF, Outline=UNIQUE, xBase + dxMultiplier*dx,
    	xBase+width + dxMultiplier*dx, yMin, yMax, AllEdges, SDF_OUTLINE, NULL);
    
    AddItem(AltSDF, WidthNum=UNIQUE, xBase, xBase+272, yMax-32, yMax-16, 0,
    		 SDF_SIMPLE_TEXT, WidthLine);
    AddItem(AltSDF, UNIQUE, xBase, xBase+272, yMax+2, yMax+18, 0, SDF_SIMPLE_TEXT, LayerNames);
    AddItem(AltSDF, DisplayXNum=UNIQUE, 16, 100, yMax+2, yMax+18, 0, SDF_SIMPLE_TEXT, DisplayX);
    AddItem(AltSDF, DisplayYNum=UNIQUE, 16, 100, yMax-14, yMax+2, 0, SDF_SIMPLE_TEXT, DisplayY);
    AddItem(AltSDF, DXNum=UNIQUE, 16, 100, yMax-30, yMax-14, 0, SDF_SIMPLE_TEXT, DisplayDX);
    AddItem(AltSDF, DYNum=UNIQUE, 16, 100, yMax-46, yMax-30, 0, SDF_SIMPLE_TEXT, DisplayDY);
    AddItem(AltSDF, RepeatCmdNum=UNIQUE, 16, 180, yMax-62, yMax-46, 0,SDF_SIMPLE_TEXT,RepeatCmd);
/*    AddItem(AltSDF, FreeNum=UNIQUE, 196, 366, yMax-62, yMax-46, 0, SDF_SIMPLE_TEXT, FreeBytes);
 */
    AddItem(AltSDF, FileNameNum=UNIQUE, 106, 200, yMax-46, yMax-30, 0, SDF_SIMPLE_TEXT, FileName);
    AddItem(AltSDF, CurrentCellNum=UNIQUE, 270, 1000, yMax-46, yMax-30, 0, SDF_SIMPLE_TEXT, CurrentCell);

    EndSymbol(AltSDF, StatusSymbol, 0);
    DisplayItem(AltSDF, StatusSymbol, StatusVGT);
}

WriteWidthLine()
  {
    EditSymbol(AltSDF, StatusSymbol);
    sprintf(WidthLine, "%4d %4d %4d %4d %4d %4d %4d",
	DefaultMetalWidth,
	DefaultPolyWidth,
	DefaultDiffWidth,
	DefaultImplantWidth,
	DefaultBuriedWidth,
	DefaultGlassWidth,
	DefaultContactWidth);
    ChangeItem(AltSDF, WidthNum, xBase, xBase+272, yMax-32, yMax-16, 0,
    		 SDF_SIMPLE_TEXT, WidthLine);
    EndSymbol(AltSDF, StatusSymbol, StatusVGT);
  }

ResetXYStatus(newX, newY)
  short newX, newY;
 {
    static short oldX = 0;
    static short oldY = 0;
    short dX, dY;
    
    dX = newX - oldX;
    dY = newY - oldY;
    oldX = newX;
    oldY = newY;

    sprintf(DisplayX, "x: %s", MakeHalfNumber(newX));
    sprintf(DisplayY, "y: %s", MakeHalfNumber(newY));
    sprintf(DisplayDX, "dx: %s", MakeHalfNumber(dX));
    sprintf(DisplayDY, "dy: %s", MakeHalfNumber(dY));
/*    sprintf(FreeBytes, "Free List: %d", FreeStorageCount());
 */
    EditSymbol(AltSDF, StatusSymbol);
    ChangeItem(AltSDF, DisplayXNum, 16, 100, yMax+2, yMax+18, 0, SDF_SIMPLE_TEXT, DisplayX);
    ChangeItem(AltSDF, DisplayYNum, 16, 100, yMax-14, yMax+2, 0, SDF_SIMPLE_TEXT, DisplayY);
    ChangeItem(AltSDF, DXNum, 16, 100, yMax-30, yMax-14, 0, SDF_SIMPLE_TEXT, DisplayDX);
    ChangeItem(AltSDF, DYNum, 16, 100, yMax-46, yMax-30, 0, SDF_SIMPLE_TEXT, DisplayDY);
/*    ChangeItem(AltSDF, FreeNum, 196, 366, yMax-62, yMax-46, 0, SDF_SIMPLE_TEXT, FreeBytes);
 */

    EndSymbol(AltSDF, StatusSymbol, StatusVGT);
}

ResetRepeatCmd(str)
  char *str;
{
    sprintf(RepeatCmd, "Rpt Cmd: %.12s", str);
    EditSymbol(AltSDF, StatusSymbol);
    ChangeItem(AltSDF,RepeatCmdNum, 16, 180, yMax-62, yMax-46, 0, SDF_SIMPLE_TEXT, RepeatCmd);
    EndSymbol(AltSDF, StatusSymbol, StatusVGT);
}

ResetCellName(str)
  char *str;
{
    sprintf(CurrentCell, "Cell: %.13s", str);
    EditSymbol(AltSDF, StatusSymbol);
    ChangeItem(AltSDF, CurrentCellNum, 270, 1000, yMax-46, yMax-30, 0, SDF_SIMPLE_TEXT, CurrentCell);

    EndSymbol(AltSDF, StatusSymbol, StatusVGT);
}


ResetFileName(str)
char *str;
{
    short nameLength;
    char *ptr, *lastSlash;
    BOOLEAN slashFlag;

    nameLength = 0;
    slashFlag = FALSE;
    ptr = str;
    while (*ptr)
      {
        if (*ptr == '/')
    	  {
    	    slashFlag = TRUE;
    	    lastSlash = ++ptr;
    	    ptr--;
    	  }
        nameLength++;
        ptr++;
      }
    if (nameLength < 14)
        sprintf(FileName, "File: %.13s", str);
    else
        sprintf(FileName, "File: %.13s", lastSlash);
    EditSymbol(AltSDF, StatusSymbol);
    ChangeItem(AltSDF, FileNameNum, 106, 106, yMax-46, yMax-30, 0, SDF_SIMPLE_TEXT, FileName);
    EndSymbol(AltSDF, StatusSymbol,StatusVGT);
}
    
