/********************************************************/
/*							*/
/*		YALE layout editor			*/
/*							*/
/*		(C) COPYRIGHT 1982			*/
/*		BOARD OF TRUSTEES			*/
/*	LELAND STANFORD JUNIOR UNIVERSITY		*/
/*	  STANFORD, CA. 94305, U. S. A.			*/
/*		THOMAS R. DAVIS				*/
/*							*/
/********************************************************/

/* FILE: evalexp.c */

/* The main routine contained herein is an expression evaluator.
 * Given the current state of the symbol table and an expression
 * pointer, it returns the integer value of the expression.  The
 * routine may have side-effects -- assignment is an expression.  It
 * is probably important that this routine be optimized later.
 * For now, if it works, that's good enough.
 *
 * The routine does no error checking, so the expression parser had
 * better do a good job!
 *
 * The whole calculation is based upon the frame-pointer frame_base,
 * which I think can be handled as a global.  For debugging, it is
 * set arbitrarily to 10.
 */

#include "aledefs.h"

#define GET1 (*e++) /* twiddles the expression ptr! */
#define GET2 ((*e+256)%256)*256 + (*(e+1)+256)%256

#define StdAssign 0
#define XAssign 1
#define YAssign 2
#define PntAssign 3

	/************************************************/
	/*		evalexp:			*/
	/************************************************/

int evalexp(exp, frameBase)
EXPRESSION exp;
short frameBase;
{
int stack[30];
short stackPtr = -1;
short expToken, GetConnPoint();
int temp, lb1, lb2, ub1, ub2, width;
short assignmentType;
register unsigned char *e;

/*if (exp.postfix == 0 && exp.string == 0)*/
if (exp == NIL_EXPRESSION)
    {
    TtyBlinkError("NIL expression");
    return(0);
    }

e = (unsigned char *) exp;
assignmentType = StdAssign;
while (Eend != (expToken=GET1))
    switch(expToken)
	{
	case Eplus:
	    stack[stackPtr-1] += stack[stackPtr];
	    stackPtr--;
	    break;
	case Eminus:
	    stack[stackPtr-1] -= stack[stackPtr];
	    stackPtr--;
	    break;
	case Etimes:
	    stack[stackPtr-1] *= stack[stackPtr];
	    stackPtr--;
	    stack[stackPtr] >>= 1; /* divide by two */
	    break;
	case Ediv:
	    stack[stackPtr-1] /= stack[stackPtr];
	    stackPtr--;
	    stack[stackPtr] <<= 1; /* multiply by 2 */
	    break;
	case Emod:
	    stack[stackPtr-1] %= stack[stackPtr];
	    stackPtr--;
	    break;
	case Euminus:
	    stack[stackPtr] = -stack[stackPtr];
	    break;
	case Edotx:
	    if (FrameStack[stack[stackPtr]+
				frameBase] == NIL)
		{
		ErrorPrint(33);
		return(0);
		}
	    stack[stackPtr] =
		(ConnPointArray[FrameStack[stack[stackPtr]+
						  frameBase]]).x;
	    break;
	case Edoty:
	    if (FrameStack[stack[stackPtr]+
				frameBase] == NIL)
		{
		ErrorPrint(33);
		return(0);
		}
	    stack[stackPtr] =
		(ConnPointArray[FrameStack[stack[stackPtr]+
						  frameBase]]).y;
	    break;
	case EQdotx:
	    assignmentType = XAssign;
	    break;
	case EQdoty:
	    assignmentType = YAssign;
	    break;
	case Eand:
	    stack[stackPtr-1] &= stack[stackPtr];
	    stackPtr--;
	    break;
	case Eor:
	    stack[stackPtr-1] |= stack[stackPtr];
	    stackPtr--;
	    break;
	case Enot:
	    stack[stackPtr] = ~stack[stackPtr];
	    break;
	case EgreaterEq:
	    stack[stackPtr-1] = (stack[stackPtr-1]>=stack[stackPtr]);
	    stackPtr--;
	    break;
	case ElessEq:
	    stack[stackPtr-1] = (stack[stackPtr-1]<=stack[stackPtr]);
	    stackPtr--;
	    break;
	case EnotEq:
	    stack[stackPtr-1] = (stack[stackPtr-1]!=stack[stackPtr]);
	    stackPtr--;
	    break;
	case Eequal:
	    stack[stackPtr-1] = (stack[stackPtr-1]==stack[stackPtr]);
	    stackPtr--;
	    break;
	case E8int:
	    stack[++stackPtr] = GET1&255; /* chars are signed */
	    break;
	case E16int:
	    stack[++stackPtr] = GET2;
	    e += 2;
	    break;
	case Esymbol:
	    temp = GET2;
	    if (temp < 0)
		stack[++stackPtr] = FrameStack[-temp];
	    else
		stack[++stackPtr] = FrameStack[frameBase+temp];
	    e += 2;
	    break;
	case EQsymbol:
	    stack[++stackPtr] = GET2;
	    e += 2;
	    break;
	case Ead0:
	    /* essentially a no-op Ead0 is mostly for symmetry. */
	    break;
	case Ead1: /* the 2's below are because of the half- */
		   /* integer storage convention. */
	    lb1 = 2*GET1;
	    ub1 = 2*GET1;
	    width = GET1;
	    if (lb1 > stack[stackPtr] || ub1 < stack[stackPtr])
		{
		ErrorPrint(36);
		return(0);
		}
	    stack[stackPtr-1] += width*(stack[stackPtr]-lb1)/2;
	    stackPtr--;
	    break;
	case Ead2:
	    lb1 = 2*GET1;  /* get the array dimensions off of the */
	    ub1 = 2*GET1;  /* expression stack */
	    lb2 = 2*GET1;
	    ub2 = 2*GET1;
	    width = GET1;
	    if (lb1 > stack[stackPtr-1] || ub1 < stack[stackPtr-1] ||
		lb2 > stack[stackPtr] || ub2 < stack[stackPtr])
		{
		ErrorPrint(36);
		return(0);
		}
	    stack[stackPtr-2] +=
		  width*( (stack[stackPtr]-lb2)/2 +
		  (ub2-lb2+2)*(stack[stackPtr-1]-lb1)/4 );
	    stackPtr -= 2;
	    break;
	case Esigvec:
	    lb1 = 2*GET1;
	    ub1 = 2*GET1;
	    if (lb1 > stack[stackPtr] || ub1 < stack[stackPtr])
		{
		ErrorPrint(35);
		return(0);
		}
	    stack[stackPtr-1]
		=frameBase+(stack[stackPtr]>>1)-lb1+stack[stackPtr-1];
	    stackPtr--;
	    break;
	case Elsigvec:
	    lb1 = 2*GET1;
	    ub1 = 2*GET1;
	    if (lb1 > stack[stackPtr] || ub1 < stack[stackPtr])
		{
		ErrorPrint(35);
		return(0);
		}
	    stack[stackPtr-2]
		= frameBase + (stack[stackPtr] >> 1) - lb1 +
		  stack[stackPtr-1] + stack[stackPtr-2];
	    stackPtr -= 2;
	    break;
	case Elocal:
	    stack[stackPtr-1]
		= FrameStack[frameBase +
				  stack[stackPtr] +
				  stack[stackPtr-1]];
	    stackPtr--;
	    break;
	case Eassign:
	    switch (assignmentType)
		{
		case StdAssign:
		    FrameStack[frameBase +
				    stack[stackPtr-1]]=stack[stackPtr];
		    stack[stackPtr-1]=stack[stackPtr]; /* return */
		    stackPtr--;
		    break;
		case XAssign:
		    if (FrameStack[stack[stackPtr-1]+frameBase]==NIL)
			FrameStack[stack[stackPtr-1]+frameBase]
							=GetConnPoint();
		    ConnPointArray[FrameStack[stack[stackPtr-1]+
					     frameBase]].x=stack[stackPtr];
		    stack[stackPtr-1] = stack[stackPtr];
		    stackPtr--;
		    break;
		case YAssign:
		    if (FrameStack[stack[stackPtr-1]+frameBase]==NIL)
			FrameStack[stack[stackPtr-1]+frameBase]
							=GetConnPoint();
		    ConnPointArray[FrameStack[stack[stackPtr-1]+
					     frameBase]].y=stack[stackPtr];
		    stack[stackPtr-1] = stack[stackPtr];
		    stackPtr--;
		    break;
		}
	    break;
	default:
	    {
	    ErrorPrint(32);
	    return(0);
	    }
	}
return(stack[0]);
}
