
/*
 * Routines for implementing key tables.
 */

#include "Vio.h"
#include "key.h"
#include "chardef.h"


#define VedInitFileName "[home].Ved_pro"
#define MAX_INITFILE_LINE_LENGTH 80

extern char *fgets();

extern FunctionListRec FunctionList[];
extern KeyTableRec KeyTable[];
extern int ProvidePrefixArgumentKey;
extern int ProvidePrefixArgument();


#define EndChar(str) ((*(str) >= ' ') && (*(str) < '\177') && \
    		    ((*((str)+1) == ' ') || (*((str)+1) == '\t')))

typedef struct ParseTableType
  {
    char *str;
    int value;
  }
    ParseTableRec;

ParseTableRec ParseTable[] =
  {
    {"uarrow", UP_ARROW},
    {"darrow", DOWN_ARROW},
    {"rarrow", RIGHT_ARROW},
    {"larrow", LEFT_ARROW},
    {"del", DEL},
    {"\\t", '\t'},
    {"\\n", '\n'},
    {"\\r", '\r'},
    {"pf1", PF1},
    {"pf2", PF2},
    {"pf3", PF3},
    {"pf4", PF4},
    {"smi-pf1", SMI_PF1},
    {"smi-pf2", SMI_PF2},
    {"smi-pf3", SMI_PF3},
    {"smi-pf4", SMI_PF4},
    {"set-up", SET_UP},
    {"no_scrl", NO_SCRL},
    {"break", BREAK},
    {"k.", KPERIOD},
    {"k,", KCOMMA},
    {"k-", KMINUS},
    {"enter", ENTER},
    {"k0", K0},
    {"k1", K1},
    {"k2", K2},
    {"k3", K3},
    {"k4", K4},
    {"k5", K5},
    {"k6", K6},
    {"k7", K7},
    {"k8", K8},
    {"k9", K9},
    {NULL, 0}
  };


/* Function declarations */
int 	SelfInsert(),
	InsertReturn(),
	InsertTab(),
	ProvidePrefixArgument(),
	ExitEditor(),
	ForwardCharacter(),
	BackwardCharacter(),
	BeginningOfLine(),
	EndOfLine(),
	NextLine(),
	PreviousLine(),
	ScrollOneLineUp(),
	ScrollOneLineDown(),
	ForwardWord(),
	BackwardWord(),
	CaseWordUpper(),
	CaseWordLower(),
	CaseWordCapitalize(),
	NextHalfPage(),
	PreviousHalfPage(),
	NextPage(),
	PreviousPage(),
	RedrawDisplay(),
	BeginningOfWindow(),
	EndOfWindow(),
	LineToTopOfWindow(),
	BeginningOfFile(),
	EndOfFile(),
	GotoRequestedLine(),
	RequestTagSearch(),
	RequestStringSearch(),
	RepeatStringSearch(),
	RequestReverseStringSearch(),
	RepeatReverseStringSearch(),
	QueryReplace(),
	BackwardHackingTabs(),
	DeleteNextCharacter(),
	DeletePreviousCharacter(),
	DeleteNextWord(),
	DeletePreviousWord(),
	TransposeCharacters(),
	NewlineAndIndent(),
	IndentLikePreviousLine(),
	NewlineAndBackup(),
	KillToEndOfLine(),
	YankKillBufferAfterCursor(),
	YankKillBufferBeforeCursor(),
	VisitFile(),
	SaveCurrentBuffer(),
	WriteNamedFile(),
	WriteModifiedFiles(),
	MarkUnmodified(),
	ToggleBackup(),
	ToggleVerifyWrite(),
	ToggleAutoLineFeed(),
	ChangeContext(),
	InsertFile(),
	OpenBrace(),
	CloseBrace(),
	OpenBrace1(),
	CloseBrace1(),
	IndentFour(),
	OutdentFour(),
	SetMark(),
	ExchangeDotAndMark(),
	IndentRegion(),
	WriteRegion(),
	DeleteToKillBuffer(),
	VisitFileNewBuffer(),
	DeleteWindow(),
	YankToNewWindow(),
	RegionToNewWindow(),
	MergeWindows(),
	GoToBuffer(),
	OrderBuffers(),
	OrderBuffersBackwards(),
	Beep();

FunctionListRec FunctionList[] =
  {
    {"SelfInsert", SelfInsert},
    {"InsertReturn", InsertReturn},
    {"InsertTab", InsertTab},
    {"ProvidePrefixArgument", ProvidePrefixArgument},
    {"ExitEditor", ExitEditor},
    {"ForwardCharacter", ForwardCharacter},
    {"BackwardCharacter", BackwardCharacter},
    {"BeginningOfLine", BeginningOfLine},
    {"EndOfLine", EndOfLine},
    {"NextLine", NextLine},
    {"PreviousLine", PreviousLine},
    {"ScrollOneLineUp", ScrollOneLineUp},
    {"ScrollOneLineDown", ScrollOneLineDown},
    {"ForwardWord", ForwardWord},
    {"BackwardWord", BackwardWord},
    {"CaseWordUpper", CaseWordUpper},
    {"CaseWordLower", CaseWordLower},
    {"CaseWordCapitalize", CaseWordCapitalize},
    {"NextHalfPage", NextHalfPage},
    {"PreviousHalfPage", PreviousHalfPage},
    {"NextPage", NextPage},
    {"PreviousPage", PreviousPage},
    {"RedrawDisplay", RedrawDisplay},
    {"BeginningOfWindow", BeginningOfWindow},
    {"EndOfWindow", EndOfWindow},
    {"LineToTopOfWindow", LineToTopOfWindow},
    {"BeginningOfFile", BeginningOfFile},
    {"EndOfFile", EndOfFile},
    {"GotoRequestedLine", GotoRequestedLine},
    {"RequestTagSearch", RequestTagSearch},
    {"RequestStringSearch", RequestStringSearch},
    {"RepeatStringSearch", RepeatStringSearch},
    {"RequestReverseStringSearch", RequestReverseStringSearch},
    {"RepeatReverseStringSearch", RepeatReverseStringSearch},
    {"QueryReplace", QueryReplace},
    {"BackwardHackingTabs", BackwardHackingTabs},
    {"DeleteNextCharacter", DeleteNextCharacter},
    {"DeletePreviousCharacter", DeletePreviousCharacter},
    {"DeleteNextWord", DeleteNextWord},
    {"DeletePreviousWord", DeletePreviousWord},
    {"TransposeCharacters", TransposeCharacters},
    {"NewlineAndIndent", NewlineAndIndent},
    {"IndentLikePreviousLine", IndentLikePreviousLine},
    {"NewlineAndBackup", NewlineAndBackup},
    {"KillToEndOfLine", KillToEndOfLine},
    {"YankKillBufferAfterCursor", YankKillBufferAfterCursor},
    {"YankKillBufferBeforeCursor", YankKillBufferBeforeCursor},
    {"VisitFile", VisitFile},
    {"SaveCurrentBuffer", SaveCurrentBuffer},
    {"WriteNamedFile", WriteNamedFile},
    {"WriteModifiedFiles", WriteModifiedFiles},
    {"MarkUnmodified", MarkUnmodified},
    {"ToggleBackup", ToggleBackup},
    {"ToggleVerifyWrite", ToggleVerifyWrite},
    {"ToggleAutoLineFeed", ToggleAutoLineFeed},
    {"ChangeContext", ChangeContext},
    {"InsertFile", InsertFile},
    {"OpenBrace", OpenBrace},
    {"CloseBrace", CloseBrace},
    {"OpenBrace1", OpenBrace1},
    {"CloseBrace1", CloseBrace1},
    {"IndentFour", IndentFour},
    {"OutdentFour", OutdentFour},
    {"SetMark", SetMark},
    {"ExchangeDotAndMark", ExchangeDotAndMark},
    {"IndentRegion", IndentRegion},
    {"WriteRegion", WriteRegion},
    {"DeleteToKillBuffer", DeleteToKillBuffer},
    {"VisitFileNewBuffer", VisitFileNewBuffer},
    {"DeleteWindow", DeleteWindow},
    {"YankToNewWindow", YankToNewWindow},
    {"RegionToNewWindow", RegionToNewWindow},
    {"MergeWindows", MergeWindows},
    {"GoToBuffer", GoToBuffer},
    {"OrderBuffers", OrderBuffers},
    {"OrderBuffersBackwards", OrderBuffersBackwards},
    {"Beep", Beep},
    {NULL, NULL}
  };

char *DefaultKeyBindings[] =
  {
    "\\r InsertReturn",
    "\\n NewlineAndIndent",
    "\\t InsertTab",
    "Esc-\\t IndentLikePreviousLine",
    "^u ProvidePrefixArgument",
    "^x-^z ExitEditor",
    "^c ExitEditor",
    "^f ForwardCharacter",
    "Ansi-c ForwardCharacter",
    "rarrow ForwardCharacter",
    "^b BackwardCharacter",
    "Ansi-d BackwardCharacter",
    "larrow BackwardCharacter",
    "^a BeginningOfLine",
    "^e EndOfLine",
    "^n NextLine",
    "Ansi-b NextLine",
    "darrow NextLine",
    "^p PreviousLine",
    "Ansi-a PreviousLine",
    "uarrow PreviousLine",
    "^z ScrollOneLineUp",
    "pf1 ScrollOneLineUp",
    "smi-pf1 ScrollOneLineUp",
    "Esc-z ScrollOneLineDown",
    "pf2 ScrollOneLineDown",
    "smi-pf2 ScrollOneLineDown",
    "Esc-f ForwardWord",
    "Esc-b BackwardWord",
    "Esc-u CaseWordUpper",
    "Esc-l CaseWordLower",
    "Esc-c CaseWordCapitalize",
    "^v NextPage",
    "Esc-v PreviousPage",
    "Esc-darrow NextHalfPage",
    "Esc-Ansi-b NextHalfPage",
    "Esc-pf1 NextHalfPage",
    "Esc-smi-pf1 NextHalfPage",
    "Esc-uarrow PreviousHalfPage",
    "Esc-Ansi-a PreviousHalfPage",
    "Esc-pf2 PreviousHalfPage",
    "Esc-smi-pf2 PreviousHalfPage",
    "^l RedrawDisplay",
    "Esc-, BeginningOfWindow",
    "Ansi-h BeginningOfWindow",
    "Esc-. EndOfWindow",
    "Esc-! LineToTopOfWindow",
    "Esc-< BeginningOfFile",
    "Esc-> EndOfFile",
    "Esc-g GotoRequestedLine",
    "^x-t RequestTagSearch",
    "^s RequestStringSearch",
    "Esc-s RepeatStringSearch",
    "pf3 RepeatStringSearch",
    "smi-pf3 RepeatStringSearch",
    "^r RequestReverseStringSearch",
    "Esc-r RepeatReverseStringSearch",
    "pf4 RepeatReverseStringSearch",
    "smi-pf4 RepeatReverseStringSearch",
    "Esc-q QueryReplace",
    "^h DeletePreviousCharacter",
    "del DeletePreviousCharacter",
    "^d DeleteNextCharacter",
    "Esc-d DeleteNextWord",
    "Esc-h DeletePreviousWord",
    "^t TransposeCharacters",
    "^o NewlineAndBackup",
    "^k KillToEndOfLine",
    "^y YankKillBufferAfterCursor",
    "Esc-y YankKillBufferBeforeCursor",
    "^x-^v VisitFile",
    "^x-^s SaveCurrentBuffer",
    "^x-^w WriteNamedFile",
    "Esc-^m WriteModifiedFiles",
    "Esc-~ MarkUnmodified",
    "^x-b ToggleBackup",
    "^x-v ToggleVerifyWrite",
    "^x-l ToggleAutoLineFeed",
    "^x-c ChangeContext",
    "^x-^i InsertFile",
    "Esc-{ OpenBrace1",
    "Esc-} CloseBrace1",
    "Esc-  IndentFour",
    "Esc-rarrow IndentFour",
    "Esc-Ansi-c  IndentFour",
    "Esc-^h OutdentFour",
    "Esc-larrow OutdentFour",
    "Esc-Ansi-d OutdentFour",
    "^@ SetMark",
    "^x-^m SetMark",
    "set-up SetMark",
    "^x-^x ExchangeDotAndMark",
    "Esc-i IndentRegion",
    "^x-^r WriteRegion",
    "^x-^k DeleteToKillBuffer",
    "^w DeleteToKillBuffer",
    "^x-g VisitFileNewBuffer",
    "^x-G VisitFileNewBuffer",
    "^x-d DeleteWindow",
    "^x-y YankToNewWindow",
    "^x-a RegionToNewWindow",
    "^x-m MergeWindows",
    "^x-1 GoToBuffer",
    "^x-2 GoToBuffer",
    "^x-3 GoToBuffer",
    "^x-4 GoToBuffer",
    "^x-5 GoToBuffer",
    "^x-6 GoToBuffer",
    "^x-7 GoToBuffer",
    "^x-8 GoToBuffer",
    "^x-9 GoToBuffer",
    "^x-o OrderBuffers",
    "^g Beep",
    "Esc-^g Beep",
    "Ansi-^g Beep",
    "^x-^g Beep",
    NULL
  };

KeyTableRec KeyTable[KEY_TABLE_SIZE];

int InitErrorFlag = 0;


/*
 * SkipWhiteSpace:
 */

char *SkipWhiteSpace(str)
    char *str;
  {
    while ((*str == ' ') || (*str == '\t'))
      {
        str++;
      }
    return(str);
  }


/*
 * UpperCase:
 */

char UpperCase(c)
    register char c;
  {
    if ((c >= 'a') && (c <= 'z'))
      {
	c = c - 'a' + 'A';
      }
    return(c);
  }


/*
 * InitKeyTable:
 */

InitKeyTable()
  {
    int i;

    /* Set default key binding to be the error-bell. */
    for (i = 0; i < KEY_TABLE_SIZE; i++)
      {
	KeyTable[i].fcn = Beep;
      }
    /* Set up the self-inserting characters. */
    for (i = ' '; i < '\177'; i++)
      {
	KeyTable[i].fcn = SelfInsert;
      }
    /* Set up the default edit-key bindings. */
    for (i = 0; DefaultKeyBindings[i] != NULL; i++)
      {
	SetKeyDefn(DefaultKeyBindings[i]);
      }
    /* Process init file. */
    ProcessInitFile();
  }


/*
 * ProcessInitFile:
 */

ProcessInitFile()
  {
    File *initFile;
    SystemCode error;
    char lineBuf[MAX_INITFILE_LINE_LENGTH], *p;

    initFile = Open(VedInitFileName, FREAD+FRELEASE_ON_CLOSE, &error);
    if (initFile == NULL)
      {
        if (error != NOT_FOUND)
	  {
	    printf("ERROR while trying to open %s: %s\n", VedInitFileName,
		ErrorString(error));
	  }
	return;
      }
    p = fgets(lineBuf, MAX_INITFILE_LINE_LENGTH, initFile);
    while (!Eof(initFile))
      {
        *(p + strlen(p) - 1) = '\0';
				/* Cut out the trailing '\n' */
	ProcessInitLine(p);
	p = fgets(lineBuf, MAX_INITFILE_LINE_LENGTH, initFile);
      }
    Close(initFile);
    if (InitErrorFlag)
      {
	exit(1);
      }
  }


/*
 * ProcessInitLine:
 */

ProcessInitLine(str)
    char *str;
  {
    str = SkipWhiteSpace(str);
    if (strlen(str) == 0)
      {
	return;
      }
    if (*str == '(')
      {
	ProcessInitCmd(str);
      }
    else
      {
	SetKeyDefn(str);
      }
  }


/*
 * ProcessInitCmd:
 */

ProcessInitCmd(string)
    char *string;
  {
    extern int CheckPointInterval;
    extern short NrowsDefault;
    extern int AutoLineFeedFlag;
    extern int backupoption;
    extern int VerifyWriteOption;
    char *str;
    int i;
    int flag = 0;

    str = string + 1;
    if (strncmp(Lower(str), "checkpoint", 10) == 0)
      {
	str += 10;
	str = SkipWhiteSpace(str);
	i = atoi(str);
	if (i <= 0) 
	  {
	    flag = 1;
	  }
	else
	  {
	    CheckPointInterval = i;
	  }
      }
    else if (strncmp(Lower(str), "defaultrows", 11) == 0)
      {
	str += 11;
	str = SkipWhiteSpace(str);
	i = atoi(str);
	if (i <= 0) 
	  {
	    flag = 1;
	  }
	else
	  {
	    NrowsDefault = i;
	  }
      }
    else if (strncmp(Lower(str), "autolinefeed", 12) == 0)
      {
	str += 12;
	str = SkipWhiteSpace(str);
	if (strncmp(Lower(str), "on", 2) == 0)
	  {
	    AutoLineFeedFlag = 1;
	  }
	else if (strncmp(Lower(str), "off", 3) == 0)
	  {
	    AutoLineFeedFlag = 0;
	  }
	else
	  {
	    flag = 1;
	  }
      }
    else if (strncmp(Lower(str), "backup", 6) == 0)
      {
	str += 6;
	str = SkipWhiteSpace(str);
	if (strncmp(Lower(str), "on", 2) == 0)
	  {
	    backupoption = 1;
	  }
	else if (strncmp(Lower(str), "off", 3) == 0)
	  {
	    backupoption = 0;
	  }
	else
	  {
	    flag = 1;
	  }
      }
    else if (strncmp(Lower(str), "verifywrite", 11) == 0)
      {
	str += 11;
	str = SkipWhiteSpace(str);
	if (strncmp(Lower(str), "on", 2) == 0)
	  {
	    VerifyWriteOption = 1;
	  }
	else if (strncmp(Lower(str), "off", 3) == 0)
	  {
	    VerifyWriteOption = 0;
	  }
	else
	  {
	    flag = 1;
	  }
      }
    else
      {
        flag = 1;
      }
    if (flag)
      {
	printf("ERROR: Invalid line encountered in %s: \n  %s\n",
		VedInitFileName, string);
	InitErrorFlag = 1;
      }
  }


/*
 * GetKey:
 * Returns the key action specified in str.
 * Prints an error msg and returns 0 if an invalid key comb.
 * is specified.
 */

int GetKey(str)
    char *str;
  {
    int i;
    int value;
    char *strKeep = str;

    if (strncmp(str, "^x-", 3) == 0)
				/* CTRL-x cases */
      {
	str += 3;
	value = GetKey(str);
	if (value != -1)
	  {
	    value = CONTROLX(value);
	  }
	return(value);
      }
    else if (*str == '^')	/* CTRL cases */
      {
	str++;
	if (EndChar(str))
	  {
	    value = CONTROL(UpperCase(*str));
	  }
	else
	  {
	    value = GetKeyErrorMsg(strKeep);
	  }
	return(value);
      }
    else if (strncmp(str, "Esc-", 4) == 0)
				/* ESC case */
      {
	str += 4;
	value = GetKey(str);
	if (value != -1)
	  {
	    value = ESCAPE(value);
	  }
	return(value);
      }
    else if (strncmp(str, "Ansi-", 5) == 0)
				/* ANSI case */
      {
	str += 5;
	value = GetKey(str);
	if (value != -1)
	  {
	    if ((value >= 'a') && (value <= 'z'))
	      {
		value = UpperCase(value);
	      }
	    value = ANSI(value);
	  }
	return(value);
      }

    for (i = 0; ParseTable[i].str != NULL; i++)
      {
	if (strncmp(str, ParseTable[i].str, strlen(ParseTable[i].str)) == 0)
	  {
	    return(ParseTable[i].value);
	  }
      }

    if (EndChar(str))	/* Simple char. case */
      {
	return(*str);
      }
    else
      {
        return(GetKeyErrorMsg(strKeep));
      }
  }


/*
 * GetFcn:
 * Returns the edit function specified in str.
 * Returns 0 if none found.
 */

int GetFcn(str)
    char *str;
  {
    char *strKeep = str;
    FunctionListRec *p;

    /* Skip over key specified. */
    while ((*str != ' ') && (*str != '\t'))
      {
	str++;
      }
    /* Skip over white space. */
    str = SkipWhiteSpace(str);
    /* str now points to the function name.  Search the function list for
       the corresponding function ptr. */
    for (p = FunctionList; p->str != NULL; p++)
      {
	if (strcmp(p->str, str) == 0)
	  {
	    return((int) p->fcn);
	  }
      }
    NewMsg("ERROR - invalid function name specified in: ");
    Msg(strKeep);
    return(0);
  }


/*
 * GetKeyErrorMsg:
 */

int GetKeyErrorMsg(str)
    char *str;
  {
    NewMsg("ERROR - invalid key combination specified in: ");
    Msg(str);
    return(-1);
  }


/*
 * SetKeyDefn:
 */

SetKeyDefn(str)
    char *str;
  {
    int key, fcn;

    key = GetKey(str);
    if (key == -1)
      {
	return;
      }
    if (MapKey(key) == (KEY_TABLE_SIZE-1))
      {
	key = GetKeyErrorMsg(str);
	return;
      }
    fcn = GetFcn(str);
    if (fcn == 0)
      {
	return;
      }
    if (fcn == (int) ProvidePrefixArgument)
      {
	ProvidePrefixArgumentKey = key;
      }
    KeyTable[MapKey(key)].fcn = (PFI) fcn;
  }


/*
 * MapKey:
 * Maps key values into the KeyTable.
 * Currently a dummy routine.
 */

int MapKey(ch)
    register int ch;
  {
    register c;

    /* Is it something that maps directly into the table? */
    if ((ch >= 0) && (ch < 0x800))
      {
	return(ch);
      }
    /* Is it an SMI-PF? */
    c = ch - 0x800 - 'P';
    if ((c >= 0) && (c <= 4))
      {
	return(c + 0x800);
      }
    /* Is it an ESCAPE(SMI-PF)? */
    c = c - 0x100;
    if ((c >= 0) && (c <= 4))
      {
	return(c + 0x800 + 4);
      }
    /* We don't know what this is. */
    return(KEY_TABLE_SIZE - 1);
  }
