/*
 * Functions which are like emacs mlisp functions.  Part 2.
 */


#include "vedit.h"
#include "chardef.h"

extern struct ckpdesc *CkpNames;	/* head of ckp file name list */
extern int QuitEditor;


/*
 * NextHalfPage:
 * Move view 1/2 screen down in the text.
 */

NextHalfPage(ch)
    int ch;
  {
    int n;
    Mark m;

    n = Nrows/2;
    if (!rows[n].exists)
      {
	Beep();
	return;
      }
    while(rows[n].continues)
      {
	n--;
      }
    m = Rowmark(n);
    Display(m);
    if (currow < Nrows/2) 
      {
	Setcursor(Rowmark(0), zeropos);
	wishcol = curcol;
      }
    else 
      {
	MarkSetcursor(curmark);
      }
  }


/*
 * PreviousHalfPage:
 * Move view 1/2 screen up in the text.
 */

PreviousHalfPage(ch)
    int ch;
  {
    Mark m;

    m = Backrows(Rowmark(0),Nrows/2);
    Display(m);
    if (currow >= Nrows/2) 
      {
	Setcursor(Rowmark(0), zeropos);
	wishcol = curcol;
      }
    else 
      {
        MarkSetcursor(curmark);
      }
  }


/*
 * GotoRequestedLine:
 * Go to the line specified by input requested from the user.
 */

GotoRequestedLine(ch)
    int ch;
  {
    int i, n;

    NewMsg("Go to line: ");  
    n = Getline(linebuf, 72);
    BackToPad();
    i = atoi(linebuf);
    if (i<1) 
      {
	NewMsg("Invalid line number.");
	return;
      }
    n = GotoLine(i);
    if (n < i) 
      {
	printf("\nfile ends at line %d",n);
	Flush(stdout);
      }
  }


/*
 * RequestTagSearch:
 * Query user for a name and then do a TagSearch.
 */

RequestTagSearch(ch)
    int ch;
  {
    int i, n;

    NewMsg("Tag:"); 
    n = Getline(linebuf,80);
    if (n > 0) 
      { /* Transfer tag to search string */
	for( i = 0; i < n; i++)
	  {
	    searchstring[i] = linebuf[i];
	  }
	searchlen = n;
	searchstring[n] = NULL;	/* not the ved convention */
	TagSearch( searchstring );
      }
  }


/*
 * RequestStringSearch:
 * Query user for a string to search for and then find it.
 */

RequestStringSearch(ch)
    int ch;
  {
    int i, n;

    NewMsg("Search:");  
    n = Getline(linebuf,80);
    if (n < 0) 
      {
	return;
      }
    if (n > 0) 
      {  /* new search string, not old one */
	for (i=0; i<n; i++)  
	  {
	    searchstring[i] = linebuf[i];
	  }
	searchlen = n;
      }
    SelFind(Search(searchstring, searchlen));
  }


/*
 * RepeatStringSearch:
 * Repeat string search with same arguments.
 */

RepeatStringSearch(ch)
    int ch;
  {
    SelFind(Search(searchstring, searchlen));
  }


/*
 * RequestReverseStringSearch:
 * Query user for a string to search for and then find it.
 */

RequestReverseStringSearch(ch)
    int ch;
  {
    int i, n;

    NewMsg("Reverse search:");  
    n = Getline(linebuf,80);
    if (n < 0) 
      {
	return;
      }
    if (n > 0) 
      {  /* new search string, not old one */
	for (i=0; i<n; i++)  
	  {
	    searchstring[i] = linebuf[i];
	  }
	searchlen = n;
      }
    SelFind(ReverseSearch(searchstring, searchlen));
  }


/*
 * RepeatReverseStringSearch:
 * Repeat string search with same arguments.
 */

RepeatReverseStringSearch(ch)
    int ch;
  {
    SelFind(ReverseSearch(searchstring, searchlen));
  }


/*
 * QueryReplace:
 */

QueryReplace(ch)
    int ch;
  {
    int i, n;

    NewMsg("Replace: ");  
    n = Getline(linebuf, 80);
    if (n < 0) 
      {
        return;
      }
    if (n > 0) 
      {
	for (i=0; i<n; i++) 
	  {
	    searchstring[i] = linebuf[i];
	  }
	searchlen = n;
      }
    Msg("  with: ");  
    replen = Getline(repstring, 80);
    if (replen < 0) 
      {
	replen = 0;
	return;
      }
    SelFind(Search(searchstring, searchlen));
  }


/*
 * NewlineAndIndent:
 */

NewlineAndIndent(ch)
    int ch;
  {
    int i;

    i = CountIndent(curmark);
    SelfInsert('\n');
    InsertIndent(i);
  }


/*
 * NewlineAndBackup:
 */

NewlineAndBackup(ch)
    int ch;
  {
    PreAdjust(curmark,curpos,&eolmark,&eolpos);
    TextInsertChar('\n');
    if (MarkEQ(curmark,eolmark)) {
	eolmark.cp++;
	Advance(&eolmark);
	}
    AdjustDisplay(curmark,eolmark,&curpos,&eolpos);
    Setcursor(curmark,curpos);
  }


/*
 * ExitEditor:
 * Exit preamble routine.
 * Checks to see whether any files have been modified.
 * If so, then queries the user for further action.
 * flag specifies whether the user is queried to confirm exiting.
 * quitFlag signals whether to actually exit the editor.
 */

ExitEditor(ch)
    int ch;
  {
    int flag;
    int n;

    if (ch == CONTROL('C'))
      {
	flag = 1;
      }
    else
      {
	flag = 0;
      }
    QuitEditor = 1;		/* Quit unless something comes up. */
    if (AnyBufsModified())
      {
	NewMsg("There are modified buffers.  Save them? (y)");
	n = Yesno();
	switch (n)
	  {
	    case -1:	/* ^G abort */
		QuitEditor = 0;
		break;
	    case 0:		/* No */
		break;
	    case 1:		/* Yes */
	    case 2:		/* Return */
		if (!WriteModifiedFiles())
		  {
		    QuitEditor = 0;
			    /* Don't exit - couldn't save buffers! */
		  }
		break;
	  }
      }
    else if (flag)
      {
	NewMsg("Confirm to quit? (y)");
	if (Yesno() <= 0)
	  {
	    QuitEditor = 0;	/* no or ^G abort */
	  }
      }
    if (QuitEditor)
      {
	RemoveAllCkpFiles();	/* removes all the CKP files opened here */
	SelectPad(stdout);
	QuitEditor = -1;	/* This is what is actually used for term. */
      }
  }

/* RemoveAllCkpFiles:
 * Upon normal exit, want to remove all files in the CkpNames list.
 * Normal exit is ^x-^z, and also ^x-d of last window.
 */

RemoveAllCkpFiles()
  {
    SystemCode	 	code;
    struct ckpdesc	*ctemp;

    for( ctemp = CkpNames; ctemp; ctemp = ctemp->nextckpdesc )
      {
	printf( "\nRemoving %s ... ", ctemp->ckpfilename );
        code = RemoveFile( ctemp->ckpfilename );
        if( code != OK )
	  {
	    Beep();
	    printf( "\n  Error in removing %s: ", ctemp->ckpfilename );
	    printf( "%s", ErrorString( code ) );
	  }
	else
	    printf( "done." );
       }
  }

/*
 * IndentLikePreviousLine:
 * Indents the current line like the previous line.
 */

IndentLikePreviousLine(ch)
    int ch;
  {
    Mark m;
    int i;

    m = Backrows(curmark, 0);
    MarkSetcursor(m);
    Retract(&m);
    i = CountIndent(m);
    InsertIndent(i);
  }


/*
 * VisitFile:
 */

VisitFile(ch)
    int ch;
  {
    int n;
    Chunk nchunk;

    if (SaveIfModified() == 0)
      {
	return;
      }
    NewMsg("visit file: ");
    if ( (n = Getline(linebuf,72)) < 0 )
      {
	return;
      }
    
    FreeText(headmark.chunk);
    BackToPad();
    if ( n > 0 )
      { 
	nchunk = ReadFile( OpenName( filename, linebuf ) );
      }
    else 
      {
	nchunk = NULL;
	filename[ 0 ] = NULL;
      }
  
    if (nchunk == NULL) 
      {
        ClearBuffer();
      }
    else 
      {
        InitBuffer(nchunk,readfile_end);
      }
    Display(headmark);
    Setcursor(headmark, zeropos);  
    wishcol = curcol;
    modified = 0;
    ckpmodified = 0;
  }


/*
 * SaveCurrentBuffer:
 * Save current buffer in file.
 */

SaveCurrentBuffer(ch)
    int ch;
  {
    WriteWithAsk();
    Banner();
  }


/*
 * WriteNamedFile:
 */

WriteNamedFile(ch)
    int ch;
  {
    int n;

    NewMsg("write file: ");  
    n = Getline(linebuf,72);
    if (n>0) 
      {
	/* try to write the file, then open for reading so we can
	   get the name.  Yes, this is a hack */
	if ( !(ckpmodified = modified = 
			!WriteFile( linebuf, headmark, endmark )) )
	  {
	    Close( OpenName( filename, linebuf ) );
	  }
	Banner();
      }
  }


/*
 * MarkUnmodified:
 */

MarkUnmodified(ch)
    int ch;
  {
    modified = 0;
    ckpmodified = 0;
  }


/*
 * ToggleBackup:
 */

ToggleBackup(ch)
    int ch;
  {
    backupoption ^= 1;
    if (backupoption) 
      {
        NewMsg("Backup option on");
      }
    else 
      {
        NewMsg("Backup option off");
      }
  }


/*
 * InsertFile:
 */

InsertFile(ch)
    int ch;
  {
    int n;
    Chunk nchunk;

    NewMsg("insert file: ");  
    n = Getline(linebuf,79);
    if (n <= 0) 
      {
	return;
      }
    BackToPad();
    nchunk = ReadFile( OpenName(filename2, linebuf) );
    if (!nchunk)		/* file not found */
      {
	return;
      }
    InsertIntoBuffer(nchunk);
  }


/*
 * MergeWindows:
 */

MergeWindows(ch)
    int ch;
  {
    int n, i;
    Chunk nchunk;

    NewMsg("Select window to merge ");
    n = PickWindow(0);
    if (n<0) 
      {
        NewMsg("Aborted");  
	return;
      }
    if (n == thisbuffer) 
      {
	NewMsg("Can't merge a window into itself");
	return;
      }
    /* now we delete the chosen buffer but keep its text */
    i = thisbuffer;
    PickBuffer(n);
    DestroyProcess(keyproc);
    free(rows);
    free(buffers[thisbuffer].ptr->fileRec);
    free(buffers[thisbuffer].ptr);
    buffers[thisbuffer].ptr = NULL;
    buffers[thisbuffer].pad = 0;
    Close(pad);
    nchunk = headmark.chunk;
    SelectBuffer(i);
    InsertIntoBuffer(nchunk);
  }


/*
 * ChangeContext:
 */

ChangeContext(ch)
    int ch;
  {
    int n;
    SystemCode error;

    NewMsg("New context name: " );
    n = Getline(linebuf,79);
    BackToPad();
    if (n <= 0)
      {
	return;
      }
    if ( (error = ChangeDirectory( linebuf )) != OK )
      {
	NewMsg( "ChangeDirectory: " );
	Msg( ErrorString( error ) );
	return;
      }
    SetContextBanner();
  }


/*
 * SetMark:
 */

SetMark(ch)
    int ch;
  {
    regionmark = curmark;  
    wishcol = curcol;
    NewMsg("Mark set");  
  }


/*
 * WriteRegion:
 */

WriteRegion(ch)
    int ch;
  {
    int n;

    NewMsg("Write region to file: ");
    n = Getline(linebuf,79);
    if (n <= 0)
      {
	return;
      }
    if (MarkEQ(regionmark,curmark)) 
      { 
	NewMsg("empty region not written"); 
      }
    else
      {
	if( MarkGT(regionmark,curmark) == 1 )
	  {
	    WriteFile( linebuf, curmark, regionmark);
	  }
	else if( MarkGT(regionmark,curmark) == 0 )
	  {
	    WriteFile( linebuf, regionmark, curmark);
	  }
	else
	  printf( "\nMarkGT returned -1: cannot write region" );
	wishcol = curcol;  
      }
  }


/*
 * DeleteWindow:
 *
 * CRZ 5/20/85 Changed so that upon deleting the last window (which returns -1
 *		from DeleteBuffer), it calls ExitEditor with some other than
 *		CNTL-c, so that it will not ask for confirmation, and then will
 *		delete all of the CKP opened in this session.
 */

DeleteWindow(ch)
    int ch;
  {
    int n;

    if (SaveIfModified() == 0)
      {
	return;
      }
    n = DeleteBuffer();
    if (n < 0) 
      {
	ExitEditor( 'n' );	/* CONTROL('C') means that it will ask */
				/* confirmation, we don't want that.   */
				/* Call ExitEditor to remove CKP files. */
      }
  }


/*
 * YankToNewWindow:
 */

YankToNewWindow(ch)
    int ch;
  {
    int n;
    Chunk nchunk;

    if (killbuffer == NULL || killbuffer->length == 0) 
      {
	NewMsg("Nothing to yank");
	return;
      }
    nchunk = CopyText(killbuffer);
    Flush(pad);  
    RedrawPad(pad);
    StoreBuffer(thisbuffer);
    filename[0] = '\0';	/* no file name */
    n = NewBuffer(nchunk, NrowsDefault);
    if (n) 
      {
        modified = 1;
	ckpmodified = 1;
      }
    else 
      {
        FreeText(nchunk);
      }
    wishcol = 0;
  }


/*
 * RegionToNewWindow:
 */

RegionToNewWindow(ch)
    int ch;
  {
    int n;
    Chunk nchunk;

    if (MarkEQ(regionmark,curmark)) 
      {
	NewMsg("Region is empty");
	return;
      }
    if (MarkGT(regionmark,curmark))
      {
	nchunk = BlockDelete(curmark,regionmark);
      }
    else
      {
        nchunk = BlockDelete(regionmark,curmark);
      }
    Display(Rowmark(0));
    MarkSetcursor(curmark);
    RedrawPad(pad);
    /* now we make the new buffer and window */
    StoreBuffer(thisbuffer);
    filename[0] = '\0';	/* no file name */
    n = NewBuffer(nchunk, NrowsDefault);
    if (n)
      {
        modified = 1;
        ckpmodified = 1;
      }
    else
      {
	if (killbuffer) 
	  {
	    FreeText(killbuffer);
	  }
	killbuffer = nchunk;
      }
    wishcol = 0;
  }


/*
 * IndentFour:
 */

IndentFour(ch)
    int ch;
  {
    ShoveRight(4);
  }


/*
 * OutdentFour:
 */

OutdentFour(ch)
    int ch;
  {
    ShoveLeft(4);
  }


/*
 * GoToBuffer:
 * Selects the buffer whose number is 1 less than that specified by the
 * low-order 8 bits of ch.
 * NOTE: GoToBuffer assumes that the low-order 8 bits represent an Ascii
 * digit.  Among other things, this implies that this function cannot be
 * bound to ^x-1 through ^x-9.
 */

GoToBuffer(ch)
    int ch;
  {
    ch = (ch & 0xff) - '0';
    ch = ch - 1;
    if ((ch < 0) || (ch > 8))
      {
	Beep(ch);
	return;
      }
    PickBuffer(ch);
  }


/*
 * InsertReturn:
 */

InsertReturn(ch)
    int ch;
  {
    SelfInsert('\n');
  }


/*
 * InsertTab:
 */

InsertTab(ch)
    int ch;
  {
    SelfInsert('\t');
  }


/*
 * OrderBuffers:
 * Orders the buffers displayed.
 */

OrderBuffers(ch)
    int ch;
  {
    register int i;
    int lowBuf;

    for (i = Maxbufs-1; i >= 0; i--)
      {
	if (buffers[i].pad != 0)
	  {
	    lowBuf = i;
	    SelectPad(buffers[i].pad);
	  }
      }
    PickBuffer(lowBuf);
  }


/*
 * OrderBuffersBackwards:
 * Orders the buffers displayed.
 */

OrderBuffersBackwards(ch)
    int ch;
  {
    register int i;
    int highBuf;

    for (i = 0; i < Maxbufs; i++)
      {
	if (buffers[i].pad != 0)
	  {
	    highBuf = i;
	    SelectPad(buffers[i].pad);
	  }
      }
    PickBuffer(highBuf);
  }
