/*
 * Talk - interworkstation communications
 * Andrew B. Hastings
 *
 * handle multiple "views" within a pad
 */

#include <Vgts.h>
#include <Vio.h>
#include <Vtermagent.h>

/* Exports */
File *viewFile1;	/* Files to record lines from each view */
File *viewFile2;	/* viewFile2 is flush()ed after each line */

/* ANSI escape sequences - recognized by VGTS pad */
#define CLEARSCREN "\033[2J"
#define CURSORPOSN "\033[%d;%df"
#define DELETELINE "\033[M"
#define ERASETOEOL "\033[K"
#define INSERTLINE "\033[L"

#define MAXVIEWS ((PadHeight-PadTop)/4)
#define PadTop	1		/* size of view at top of pad */

struct
  {
    short	x;		/* cursor position within view */
    short	y;
    char	buffer[PadWidth+1];
    char	header[PadWidth];
  } viewCur[MAXVIEWS+1];
short viewCount = 0;		/* number of views active */
short viewLines = PadHeight;	/* lines per view */

short cursorX = 0;		/* actual cursor position within pad */
short cursorY = 0;

File *pad;			/* pad we're writing to */


viewInit(outputpad)
    File *outputpad;
  /*
   * initialize view package using outputpad.
   * returns maximum number of views
   */
  {
    short i;

    pad = outputpad;
    ModifyPad(pad, CR_Input);
    fputs(CLEARSCREN, pad);
    for (i=0; i<=MAXVIEWS; i++)
      viewCur[i].x = 1;
    return(MAXVIEWS);
  }


viewBanner(s)
    char *s;
  /*
   * stores banner on pad
   * banner is also written verbatim to record files
   */
  {
    SetVgtBanner(pad, s);
    if (viewFile1)
      fprintf(viewFile1, "%s\n", s);
    if (viewFile2)
      {
	fprintf(viewFile2, "%s\n", s);
	fflush(viewFile2);
      }
  }


viewAdd(s)
    char *s;
  /*
   * creates another view in pad, shrinking existing views.
   * 's' is header which stays on second line of view.
   * (first line of view is divider)
   * returns ordinal of view created, -1 if error.
   */
  {
    short y, i;
    short oldLines, oldCount;

    if (viewCount == MAXVIEWS)
      return(-1);
    oldCount = viewCount;
    oldLines = viewLines;
    viewCount++;
    viewLines = (PadHeight-PadTop) / viewCount;
    for (i=oldCount-1; i>=0; i--)
      {
	y = oldLines;
	gotoXY(1, i*oldLines+viewCur[i].y+1+PadTop);
	while ((viewCur[i].y < y) && (y > viewLines))
	  {
	    fputs(DELETELINE, pad);
	    y--;
	  }
	gotoXY(1, i*oldLines+3+PadTop);
	while (y > viewLines)
	  {
	    fputs(DELETELINE, pad);
	    y--;
	    viewCur[i].y--;
	  }
      }
    strcpy(viewCur[oldCount].header, s);
    viewCur[oldCount].x = 1;
    viewCur[oldCount].y = 1;
    viewPuts(oldCount, "- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -\n");
    viewPuts(oldCount, s);
    viewPutc(oldCount, '\n');
    return(oldCount);
  }


viewDel(view)
    short view;
  /*
   * deletes indicated view.
   * ordinals of all views greater than deleted view will decrease
   * by one.
   */
  {
    short i, y;
    short oldCount, oldLines;

    if ((view < 0) || (view >= viewCount))
      return(-1);
    oldCount = viewCount;
    oldLines = viewLines;
    viewCount--;
    if (viewCount == 0)
      viewLines = PadHeight;
    else
      viewLines = (PadHeight-PadTop) / viewCount;
    gotoXY(1, view*oldLines + 1 + PadTop);
    for (i=0; i<oldLines; i++)
      fputs(DELETELINE, pad);
    for (i=view; i<viewCount; i++)
      viewCur[i] = viewCur[i+1];
    for (i=0; i<viewCount; i++)
      {
	gotoXY(1, i*viewLines+oldLines + 1 + PadTop);
	for (y=oldLines; y<viewLines; y++)
	  fputs(INSERTLINE, pad);
      }
    fflush(pad);
    return(viewCount);
  }



viewPutc(view, c)
    short view;
    char c;
  /*
   * write a character to indicated view.
   * view will scroll independently of other views if necessary.
   * if view is maximum view (as returned by viewInit), write
   * character to 1-line view at top of pad above other views.
   * character is buffered and written to record files when end-of-line
   * is encountered.  (except characters written to 1-line view at
   * top of pad are never written to record files)
   */
  {
    short y;

    if (view == MAXVIEWS)
      viewCur[view].y = -view*viewLines;
    else if ((view < 0) || (view >= viewCount))
      return(-1);
    y = viewCur[view].y + view*viewLines + PadTop;
    switch (c)
      {
	case '\b':
	case '\177':
	  /* ^H or ^? - erase character */
	  if (viewCur[view].x > 1)
	    {
	      gotoXY(--viewCur[view].x, y);
	      putc(' ', pad);
	      putc('\b', pad);
	    }
	  break;

	case '\025':
	  /* ^U - erase line */
	  gotoXY(1, y);
	  fputs(ERASETOEOL, pad);
	  viewCur[view].x = 1;
	  break;

	case '\t':
	  /* ^I - skip to next tab stop */
	  do
	    viewPutc(view, ' ');
	  while (viewCur[view].x % 8);
	  break;

	case '\n':
	  /* newline - move to next line, scroll if needed */
newline:  if (view == MAXVIEWS)
	    {
	      gotoXY(1, y);
	      fputs(ERASETOEOL, pad);
	      viewCur[view].x = 1;
	      break;
	    }
	  viewCur[view].buffer[viewCur[view].x-1] = '\0';
	  if (viewCur[view].y > 2)
	    {
	      if (viewFile1)
		fprintf(viewFile1, "%s\t%s\n", viewCur[view].header,
							viewCur[view].buffer);
	      if (viewFile2)
		{
		  fprintf(viewFile2, "%s\t%s\n", viewCur[view].header,
							viewCur[view].buffer);
		  fflush(viewFile2);
		}
	    }
	  if (viewCur[view].y == viewLines)
	    {
	      gotoXY(1, view*viewLines + 3 + PadTop);
	      fputs(DELETELINE, pad);
	      gotoXY(1, y);
	      fputs(INSERTLINE, pad);
	    }
	  else
	    {
	      viewCur[view].y++;
	      gotoXY(1, y+1);
	    }
	  viewCur[view].x = 1;
	  break;

	case '\007':
	  /* ^G - bell */
	  putc('\007', pad);
	  break;

	default:
	  /* otherwise, put it on pad */
	  if (c >= ' ')
	    {
	      gotoXY(viewCur[view].x, y);
	      viewCur[view].buffer[viewCur[view].x-1] = c;
	      putc(c, pad);
	      viewCur[view].x++;
	      if (++cursorX > PadWidth)
		goto newline;
	    }
	  break;

      }
  }


viewPuts(view, s)
    short view;
    char *s;
  /*
   * call viewPutc repeatedly to put a string on indicated view
   */
  {
    if ((view < 0) || (view >= viewCount && view != MAXVIEWS))
      return(-1);
    while (*s)
      viewPutc(view, *s++);
    fflush(pad);
  }


viewFlush()
  /*
   * call fflush on pad.  for efficiency, pad is never flushed after
   * viewPutc, always flushed after viewPuts.
   */
  {
    fflush(pad);
  }


gotoXY(x, y)
    short x, y;
  /*
   * internal routine used only by view package
   */
  {
    if ((x < 1) || (x > PadWidth) || (y < 1) || (y > PadHeight))
      return(-1);
    if ((x != cursorX) || (y != cursorY))
      {
	fprintf(pad, CURSORPOSN, y, x);
	cursorY = y;
	cursorX = x;
      }
  }
