/*
 * silpress.c - convert sil files to .Press and spool to dover
 *
 * Bill Nowicki November 1983
 *
 */
 
# define DPR "dpr"	/* program to spool a press file */
# define McsPerInch 2540
# define PageLength McsPerInch*(8*11-3)/8/* 10.625 inches */
# define PageWidth  McsPerInch*17/2.

# define LeftMargin	1000		/* left margin (in Micas) */
# define BottomMargin  	10		/* bottom margin (in micas) */
					/* 2600 is too large */
# define ConversionFactor 35.180	/* Micas per SIL pixel */
					/* 35.175 is too small */
					/* 35.25 is too Large */

# define SilXtoPress(x) (int)((x)*ConversionFactor+LeftMargin)
# define SilYtoPress(y) (int)((y)*ConversionFactor+BottomMargin)

# define TemplateFudge	64		/* Offset for Template 64 (SilPixels) */
# define TextFudge 7			/* Offset for text (Micas) */

# define MaxFonts 64

# define TRUE  1
# define FALSE 0

# include <stdio.h>
# include <pwd.h>
# include <sys/types.h>
# include <sys/stat.h>
# include "press.h"
# include "sil.h"
# include "dfont.h"
 
	/* 
	 * face encodings: 
	 * Note that the design size is encoded here for TEX fonts
	 *
	 */
#define ITALIC	1
#define BOLD	2


struct stat S;
char   *ctime ();
char   *getenv ();
long    time ();
struct passwd  *getpwuid ();


short 	Debug = 0;		/* True means extra debug printfs */
char sysname[31];		/* System name */

struct font fonts[MaxFonts];	/* possible fonts at one time */
int     NumberFonts = 1;	/* number of fonts known about */
char    TempName[50] = "/tmp/Dover XXXXXX";
int     Copies = 1;		/* number of copies to make */
int	NotifyFlag =0;		/* true if notification on spooling wanted */
char   *Banner = 0;		/* label to appear on the cover page */
char    Names[54] = "";		/* list of file names */
int     PressOnly = 0;		/* true if press file only wanted */

int     SeenFile = 0;		/* true if a file has been processed */

char   *FileName;		/* name of file currently being pressed
				*/
char   *FileDate;		/* last mod date of file being pressed
				*/
struct font  *CurrentFont;		/* current Font */
int	HaveOpenEntity = FALSE; 	/* if have an entity open */
int	CurrentFontSet = -1;		/* current font Set for this entity */
 
char   *UsersName = 0;		/* user specified name for filing output */
int     Page = 1;		/* current page number */
int     TotalPages = 0;		/* total number of pages printed */
int     TruncChars = 0;		/* number of characters truncated */
int     UndefChars = 0;		/* number of characters skipped because
				   they weren't defined in some font */
 
int HaveOpenPressFile = FALSE;	/* output press file opened */

struct Object Table[MaxObjects];	/* table of all objects */

char FontMap[] =  
  {
  	/*
	 * Map SIL internal font numbers into Press numbers.
	 * Note I only use the first 8, and add 8 if italic.
	 */
    Helvetica10,
    Helvetica10B,
    Helvetica7,
    Helvetica7B,
    Template64,
    Template64,
    Template64,
    Template64,
    Helvetica10I,
    Helvetica10BI,
    Helvetica7I,
    Helvetica7BI,
    Template64,
    Template64,
    Template64,
    Template64
  };

crash (s, a, b)
register char  *s; 
{
	/*
	 * Send a message to the terminal, then die horribly.
	 * something is very wrong..... (JAG's idea of humor)
	 */
    fprintf (stderr, s, a, b);
    fprintf (stderr, "\n");
    exit (1);
}

Quit()
  {
    exit(1);
  }

char   *fontname (p)
  struct font * p; 
 {
	/*
	 * return the name of a font,
	 * given a pointer to the font structure
	 */
    static char res[50];
 
    if (p->face<0)
      {
       int designSize = (- p->face/2) - 1;
       if ( p->size == designSize )
             sprintf (res, "%s%d", p -> name, p->size );
        else sprintf (res, "%s%d at %d", p -> name, designSize, p->size );
       return( res );
      }
    sprintf (res, "%s%d%s", p -> name, p -> size,
	    p -> face == (BOLD | ITALIC) ? "bi" :
	    p -> face & BOLD ? "b" :
	    p -> face & ITALIC ? "i" :
	    "");
    return (res);
}



OpenPage () 
 {
	/*
	 */
    int i;
    long counter;

	/* 
	 * put out a page heading to the press file,
	 * initialize the data & entity parts.
	 */
 
    BeginPage();
    HaveOpenEntity = FALSE;
    CurrentFont = NULL;
    TotalPages++;
    fprintf(stderr, "%d",  Page++ );
    if (Debug) fprintf(stderr, "Begining of page %d\n", Page);
  }


ClosePage () 
 {
	/*
	 * terminate a page.  Put out the entity list and entity trailer,
	 * pad out the block, and add an entry to the parts list.
	 */		    
   if (Debug) fprintf( stderr, "End of page\n" );
 
    if (HaveOpenEntity)
     {
	EndEntity();
	HaveOpenEntity = FALSE;
      }
    EndPage();
    fprintf(stderr, "." );
 }


 
AlreadyPress () 
 {
/*
 * check to see if the current input file is already in press format,
 * if it is just copy it & update the DocumentDir & ship it.
 * returns true if in press format
 */
    register    n;
 
    fseek (stdin, -512, 2);
    n = ( getchar() & 255)*256 + (getchar() & 255);
    rewind (stdin);
    if (n != 27183)
	return (0);
    /*
     * *** Normally we would call "cz" at this point to do the real work.
     */
}
 

CopyFile () 
 {
	/*
	 * Copy the standard input file to the press file, assumes that the last
	 * entity (page) has been flushed, 
	 */
     register struct Object *ob;
     short objs = 0;

    if ((S.st_mode & S_IFMT) == S_IFREG && 
        (S.st_size & 0777) == 0 && S.st_size > 0777
	    && AlreadyPress ())
	return;
    rewind(stdin);
    if (!HaveOpenPressFile) 
     {
	PressStart(TempName,1);
	HaveOpenPressFile = TRUE;
     }
    ParseFile(stdin,NULL,NULL,NULL,NULL);
    OpenPage();
    for (ob = Table+FirstItem;ob<Table+MaxObjects;ob++)
     if (ob->state==Used)
      switch (ob->kind)
       {
	 case Line:
            DoRule( ob->xmin, ob->xmax, ob->ymin, ob->ymax);
	    objs++;
	    break;

        case Text:
	  {
	    short fontIndex, base;

	    fontIndex = FontMap[ob->fontNumber];
	    if (fontIndex==Template64) 
		  base = ob->ymax - TemplateFudge;
	      else
		  base = BaseOf(ob->string,fontIndex,ob->ymin,ob->ymax);
            DoText( ob->xmin, base, fontIndex, ob->string);
	    objs++;
	  }
       }
    ClosePage();
 }


DoRule( xmin, xmax, ymin, ymax )
  { 
    int pw,ph,x,y;
    
    x = SilXtoPress(xmin);
    y = SilYtoPress(ymin);
    
    if (xmax<xmin) xmax = xmin;
    if (ymax<ymin) ymax = ymin;
    
    pw =  (xmax-xmin+1)*ConversionFactor;
    ph =  (ymax-ymin+1)*ConversionFactor;

    if (!HaveOpenEntity)
    {
	BeginEntity(0,CurrentFont->set,
	    0,0,0,0,
	    MicasPageWidth,MicasPageHeight);
	HaveOpenEntity = TRUE;
	CurrentFontSet = CurrentFont->set;
    }
    if (Debug) 
        fprintf(stderr, "PressShowRectangle (%d,%d)@(%d,%d)\n",
		pw,ph,x,y);
     PressShowRectangle( x, y, pw, ph );
  }


DoText(left,bottom,fontIndex,string)
    char *string;
  {
    int x, y;

    if (Debug) 
        fprintf(stderr, "DoText left=%d, bottom=%d, fontIndex=%d, string=%s\n",
		left,bottom,fontIndex,string);

    x =  SilXtoPress(left);
    y =  SilYtoPress(bottom)+TextFudge;

   ChangeFont(fontIndex);
   PutStringontoPage(x,y,string);
  }


ChangeFont(n)
	long n;
 {
 	/*
	 * Change to font number n 
	 */
    register struct font *f = fonts + n;

    if (CurrentFont == f) return;

    CurrentFont = f;
    if(n<0 || n>=NumberFonts)
	crash("Change to invalid font: %d",n);
    if (Debug) fprintf( stderr, "Font changed to Press %d\n", n);
    if (HaveOpenEntity && CurrentFontSet != f->set)
    {
	EndEntity();
	HaveOpenEntity = FALSE;
    }
    if (!HaveOpenEntity)
    {
	BeginEntity(0,CurrentFont->set,
	    0,0,0,0,
	    MicasPageWidth,MicasPageHeight);
	HaveOpenEntity = TRUE;
	CurrentFontSet = CurrentFont->set;
    }
    f -> bc = 0;
    f -> ec = 127;
    PressFont( n );
 }


DumpFonts ()
 {
	/*
	 * dump the font directory to the press file 
	 */
    register    struct font * f;
    for (f = &fonts[NumberFonts-1]; f>= &fonts[0]; f--)
      {
	if (f->bc <= f->ec)
  	    EnterFontinDirectory(
		f->set,         /* font set */
		f->number,      /* font number within set */
		f->bc,      	/* first character to define */
		f->ec,      	/* last character to define */
		f->name,        /* font family name */
		f->face,        /* encoded face information */
		f->bc,      	/* source character for first defn */
		f->size,   	/* font size in points */
		f->rotation     /* rotation */
		);
     }
 }
 
 

addstr (s)
char * s; 
 {
       /*
        * add a string to the name list 
	*/
    if (s && strlen (Names) + strlen (s) <= 52)
	strcat (Names, s);
 }
 


 
ProcessArg (p)
register    char * p; 
 {
 	/*
	 * Process one argument, given our state
	 */
    static  enum State 
      {
	normal, pressname,  grabname,
	grabheader, getcopies, getbanner
      }
            state = normal;
 
    switch (state) 
    {
case pressname:
	strcpy (TempName, p);
	state = normal;
	break;

case grabname:
	UsersName = p;
	state = normal;
	break;

case getcopies:
	Copies = atoi (p);
	if (Copies < 1) crash ("bad number of copies");
	state = normal;
	break;

case getbanner:
	Banner = p;
	state = normal;
	break;

default:
	if (*p == '-')
	    while (*++p)
		switch (*p) 
		{
	case 'b':
		state = getbanner;
		break;
	case 'c':
		state = getcopies;
		break;
	case 'd':
		Debug = 1;
		break;
	case 'h':
		state = grabheader;
		break;
	case 'n':
		state = grabname;
		break;
	case 'p':
		PressOnly++;
		state = pressname;
		break;
	case 'N':
		NotifyFlag++;
		break;
	default:
		printf ("Unknown option: %c\n", *p);
		SeenFile++;
		break;
	    }
	else 
	 {
	    FileName = p;
	    if (freopen (FileName, "r", stdin) == NULL) 
	    {
	       strcat(FileName, ".sil");
	       if (freopen( FileName, "r", stdin) == NULL) 
  	        {
		  printf ("Can't open %s\n", FileName);
		  exit (1);
		}
	      }
	    ReadAltFonts(FileName);
	    DoFonts();
	    fstat (fileno (stdin), &S);
	    FileDate = ctime (&S.st_mtime);
	    if (*Names)
		addstr (", ");
	    addstr (FileName);
	    CopyFile ();
	    fclose (stdin);
	    SeenFile = 1;
	}
    }
}


DeclareFont(n,name,size,face)
    int n;		/* font index number */
    char *name;		/* string name of family */
    int size;		/* in points */
    int face;		/* Parc face code */
  {
    register struct font *f;
    
    f = fonts + n;
    f->number = n;
    strcpy(f->name, name);
    f->size = size;
    f->face = face;
    
    f->set = 0;
    f->bc = 127;
    f->ec = 0;
    f->rotation = 0;
    
    if (n>NumberFonts) NumberFonts = n;
  }


extern char SmallFontName[];
extern char BigFontName[];

GetSize(name,fam)
    char *name, *fam;
  {
    register char *p;

    for (p = name;*p && (*p < '0' || *p > '9');p++) 
	  *fam++ = *p;

    *fam++ = 0;
    return(atoi(p));
  }
      

DoFonts()
  {
    char small[100], big[100];
    int smallSize, bigSize;

    smallSize = GetSize(SmallFontName,small);
    bigSize = GetSize(BigFontName,big);

    printf("Using fonts %s %d and %s %d\n",small,smallSize,big,bigSize);
    fflush(stdout);

    DeclareFont(Helvetica10, big,bigSize,0);
    DeclareFont(Helvetica10B, big,bigSize,BOLD);
    DeclareFont(Helvetica7, small,smallSize,0);
    DeclareFont(Helvetica7B, small,smallSize,BOLD);
    DeclareFont(Helvetica7I, small,smallSize,ITALIC);
    DeclareFont(Helvetica7BI, small,smallSize,BOLD+ITALIC);
    DeclareFont(Template64,"Template",64,0);
  }


main (argc, argv)
 char ** argv; 
 {
    register    char * p,
               *arg;

    gethostname(sysname, sizeof sysname); 
    mktemp (TempName);

    if ( (p = getenv ("SIL")) || (p = getenv("CZ")) )
	while (1) 
	 {
	    register char quote = ' ';
	    while (*p == ' ')
		p++;
	    if(*p=='"' || *p=='\'') quote = *p++;
	    arg = p;
	    while (*p != quote && *p != '\0')
		p++;
	    if (*p == '\0') 
	     {
		if (*arg)
		    ProcessArg (arg);
		break;
	     }
	    *p++ = '\0';
	    ProcessArg (arg);
	}
    while (argc > 1) 
     {
	argc--;
	ProcessArg (*++argv);
     }
    if (!SeenFile) 
     {
	addstr ("It came from out of the blue");
	FileName = 0;
	FileDate = "";
	fstat (fileno (stdin), &S);
	if ((S.st_mode & S_IFMT) == S_IFREG)
	    FileDate = ctime (&S.st_mtime);
	CopyFile ();
     }
    if (HaveOpenPressFile)
    {
	DumpFonts();

	addstr(" [");
	addstr(sysname);
	addstr("]");

	if (Banner != 0)
	    PressSetBanner(Banner);
	else
	    PressSetBanner(Names);

	PressSetCopies(Copies);

	if (UsersName != 0)
	    PressSetCreator(UsersName);

	PressFinish();
     }
    fprintf(stderr,"\n");

    if (TruncChars)
	printf ("%d characters omitted because of long lines.\n",
		TruncChars);
    if (UndefChars)
	printf ("%d characters omitted because of incomplete fonts.\n",
		UndefChars);
    if (PressOnly)
	    printf ("press file left on %s\n", TempName);
	else 
	 {
	   if (NotifyFlag)
	       execlp ( DPR, DPR, "-n", TempName, 0);
	   else
	       execlp( DPR, DPR, TempName, 0);
	    printf ("Couldn't execute %s.  Press file in %s\n", DPR, TempName);
	}
}


