/*
 * siledit.c - edit .sil files on the VGTS
 * Bill Nowicki Januray 1983
 *
 * Wishlist:
 *	- Alternate font selection
 *	- Undelete
 *	- Handle Macros. Does anybody really use these?
 *	- Newfile
 */

# include "sil.h"

# include <Vgts.h>
# include <text.h>

# ifdef Vsystem
# include <Vio.h>
# else Vsystem
# define File FILE
# include <stdio.h>
# endif Vsystem

/*
 * definitions for .sil files
 */

char FontMap[] =  
  {
  	/*
	 * Map SIL internal font numbers into VGTS numbers.
	 * Note I only use the first 8, and add 8 if italic.
	 * Note that some of the entries can be changed at run-time
	 * if alternate fonts are selected.
	 */
    Helvetica10,	/* 0 */
    Helvetica10B,	/* 1 */
    Helvetica7,		/* 2 */
    Helvetica7B,	/* 3 */
    Template64,		/* 4 */
    Template64,		/* 5 */
    Template64,		/* 6 */
    Template64,		/* 7 */
    Helvetica10I,	/* 8 */
    Helvetica10BI,	/* 9 */
    Helvetica7I,	/* 10 */
    Helvetica7BI,	/* 11 */
    Template64,		/* 12 */
    Template64,		/* 13 */
    Template64,		/* 14 (Line font) */
    Template64		/* 15 (Erroneous font) */
  };

extern char SmallFontName[];
extern char BigFontName[];


unsigned char
	SmallFont,
	SmallFontB,
	SmallFontI,
	SmallFontBI,
	BigFont,
	BigFontB,
	BigFontI,
	BigFontBI;

struct Object Table[MaxObjects];	/* table of all objects */
struct Object *SelectList = NULL;	/* list of selected objects */

short CurrentObject = 0;	/* where to start looking for empty objects */
short Debug = 0;		/* extra debugging information */
short Changed = 0;		/* If we made changes since last save */
short MarkX = 0, MarkY = 0;	/* coordinates of the mark */
short Width = 1;		/* width of lines */
short CurrentFont = 0;		/* internal font number for new text */
short sdf, vgt;			/* the sdf and vgt we use */

MaybeQuit()
  {
  	/*
	 * Luser has selected Quit from the menu, see if changed
	 * since last save.
	 */
    if (Changed)
      {
        char answer;

        printf("Picture changed since last Write; confirm with 'y':");
	answer = getchar();
	printf("%c\r\n",answer);
	if (answer != 'y')
          return;
      }
    Quit();
  }

Quit()
  {
    /*
     * gracefully exit, deleting sdf and vgt
     */
     DeleteVGT(vgt,1);
     if (Debug) printf("Deleted vgt\r\n");
     DeleteSDF(sdf);
     if (Debug) printf("Deleted sdf\r\n");
     ResetTTY();
     exit();
  }


PutByte(c,f)
    File *f;
    char c;
  {
    /*
     * write a byte to a file
     */
    putc( (c&255),f);
  }

PutShort(n,f)
    File *f;
    short n;
  {
    /*
     * write a short to a file, in big endian order
     */
    putc( (n>>8)&255, f); 
    putc( (n & 255), f);
  }

main(argc, argv)
 char **argv;
 {

    short xminb, xmaxb, yminb, ymaxb;
    int i,vecs;
    short item, width, height;
    File *f;
    char fileName[256];
    char vgtName[256];

    if (argc>1 && argv[1][0]=='-')
      {
        char *p = argv[1]+1;
	argc--; argv++;
	while (*p)
	  {
	    switch (*p)
	      {
	        case 'd': Debug=1; break;
		default: 
		  printf("Unknown option: %c\n", *p);
	      }
	    p++;
	  }
      }
      
     if ( argc > 1 ) f=fopen( argv[1], "r" );
     else 
      {
        printf("usage: siledit [-d] file\r\n");
	exit();
      }
     strcpy(fileName,argv[1]);
     if (f==NULL)
         {
           printf( "Cannot open %s\r\n", argv[1] );
         }
     strcpy(vgtName,"Sil Edit ");
     strcat(vgtName,fileName);

# ifdef Vsystem
     printf("SIL format editor for VGTS\r\n");
# else Vsystem
     printf("Remote SIL format editor for VGTS\r\n");
# endif Vsystem

     GetTTY();
     sdf = CreateSDF();
     DefineSymbol(sdf,MarkSymbol,"mark");
     AddItem(sdf,0,-5,5,5,-5,0,SDF_GENERAL_LINE,NULL);
     AddItem(sdf,0,5,-5,5,-5,0,SDF_GENERAL_LINE,NULL);
     EndSymbol(sdf,MarkSymbol,0);
     DefineSymbol( sdf, TopSymbol, fileName );
     AddItem(sdf,0,-Border,Border+XLimit,-Border,0,
     	NP,SDF_FILLED_RECTANGLE,NULL);
     AddItem(sdf,0,-Border,Border+XLimit,YOffset,YOffset+Border,
     	NP,SDF_FILLED_RECTANGLE,NULL);
     AddItem(sdf,0,-Border,0,-Border,YOffset+Border,
     	NP,SDF_FILLED_RECTANGLE,NULL);
     AddItem(sdf,0,XLimit,Border+XLimit,-Border,YOffset+Border,
     	NP,SDF_FILLED_RECTANGLE,NULL);
     AddCall(sdf,MarkItem,0,0,MarkSymbol);
     vgt = CreateVGT(sdf, GRAPHICS+ZOOMABLE, TopSymbol, vgtName );
     if (Debug) printf("sdf=%d, vgt=%d\r\n", sdf, vgt);
     EndSymbol( sdf, TopSymbol, vgt );

     ParseFile(f,&xminb,&xmaxb,&yminb,&ymaxb);
     	/*
	 * Create the right sized view depending on the size of the picture.
	 */
     width = 32 + xmaxb - xminb;
     height = 48 + ymaxb - yminb;
     DefaultView(vgt, width, height, xminb-16, yminb-16, 0, 0, 0, 0);
     ReadAltFonts(fileName);
     DefineSilFonts();
     DisplaySil(width,height);
     fclose(f);
     printf("Use mouse buttons: Mark, Select, Menu\r\n");
     while (1) MainLoop(fileName);
  }

static PopUpEntry fontMenu[] = 
      {
        NULL, 	0,
	NULL,	1,
	NULL,	2,
	NULL,	3,
	NULL,	10,
	"Template 64",		4,
	0, 0
      };


char *FontName(num)
  {
    /*
     * returns the string name of the given font number by looking in
     * the above font menu.
     */
     register PopUpEntry *p;

     for (p = fontMenu;p->string;p++)
 	if (p->menuNumber==num) return(p->string);
     return("Unknown");
  }

DefineSilFonts()
  {
  	/*
	 * Tell the VGTS which fonts we want to use
	 */
    printf("Defining fonts %s and %s:", SmallFontName, BigFontName);

    FontMap[0] = BigFont = OneFont(BigFontName,"", fontMenu+0,255);
    FontMap[1] = BigFontB = OneFont(BigFontName,"B", fontMenu+1,BigFont);
    FontMap[2] = SmallFont = OneFont(SmallFontName,"", fontMenu+2,255);
    FontMap[3] = SmallFontB = OneFont(SmallFontName,"B", fontMenu+3,SmallFont);

    FontMap[8] = BigFontI = OneFont(BigFontName,"I", NULL,BigFont);
    FontMap[9] = BigFontBI = OneFont(BigFontName,"BI", NULL,BigFontB);
    FontMap[10] = SmallFontI = OneFont(SmallFontName,"I", fontMenu+4,SmallFont);
    FontMap[11] = SmallFontBI = OneFont(SmallFontName,"BI", NULL, SmallFontB);

    DefineFont("Template64",NULL); 
    printf(" Done loading fonts\r\n");
  }


OneFont(generic,face,entry,alt)
  char *generic, *face;		/* family and face code strings */
  PopUpEntry *entry;		/* Points to a menu entry */
  unsigned char alt;		/* Alternate font number if not found */
  {
	/*
	 * Handle the definition of one font
	 */
    char name[128];
    unsigned char value;
    int length;

    strcpy(name,generic);
    strcat(name,face);
    putchar('.'); fflush(stdout);
    length = strlen(name);
    if (entry)
      {
        entry->string = (char *) malloc(length+1);
        strcpy(entry->string,name);
      }
    value = DefineFont(name,NULL);
    if (value==255)
      {
        if (alt==255)
            printf("Unable to locate font %s\r\n",name);
	else value = alt;
      }
    return(value);
  }


DisplaySil(width,height)
    short width,height;
  {
	/*
	 * Draw the initial screen, and some misc. initialization.
	 * calculate time for display and size of bitmap.
	 */
     long start, end, objs=0;
     register struct Object *ob;
     double rate;
  
    SelectList = NULL;
    start = GetMilliseconds();
    EditSymbol(sdf,TopSymbol);
    for (ob = Table+FirstItem;ob<Table+MaxObjects;ob++)
     if (ob->state==Used)
      switch (ob->kind)
       {
	 case Line:
            AddItem( sdf, ob->number, ob->xmin, ob->xmax, 
	    	ob->ymin, ob->ymax, BLACK,
	    	SDF_FILLED_RECTANGLE, NULL);
	    objs++;
	    break;

        case Text:
	  {
	    short base, font;

	    font = FontMap[ob->fontNumber];
	    base = BaseOf(ob->string,font,ob->ymin,ob->ymax);
            AddItem( sdf, ob->number, ob->xmin, ob->xmax, 
	    	base, ob->ymax, font, 
	    	SDF_TEXT, ob->string);
	    objs++;
	  }
       }
     EndSymbol( sdf, TopSymbol, vgt );
     end = GetMilliseconds();
     if (start==end) end++;
     rate = (objs*1000.0)/(double)(end-start);
     printf("%d objects in %d milliseconds, or %6.2f objects/second\r\n",
     	objs, end-start, rate);
     printf("Bitmap is %d x %d = %d bits, or %d bytes\r\n",
     	width, height, width*height, (width*height+7)/8 );
     printf("SDF contains %d bytes\r\n", objs*20);
     CurrentObject = objs + 1;
  }


PutSilFile(f)
  {
  	/*
	 * write the data structure to the indicated SIL file
	 */
    register struct Object *ob = Table+FirstItem;
    short silymin, silymax, length;

    PutShort(Password,f);
    for (;ob<Table+MaxObjects;ob++)
     if (ob->state==Used)
      {
        silymin = (YOffset - ob->ymax) & CoordMask;
        silymax = (YOffset - ob->ymin + 1) & CoordMask;
	if (Debug) 
	  printf("%s object %d xmin=%d, ymin=%d\r\n",
	    ob->kind==Text ? "Text" : "Line", ob->number, 
	    ob->xmin, ob->ymin );
	if (ob->kind==Text && 
	  (ob->string==NULL || strlen(ob->string)==0)) 
	    {
	      if (Debug) printf("%s string\r\n",
	        ob->string ? "Empty" : "NULL" );
	      continue;
	    }
        PutShort(-1,f);
	PutShort(ob->xmin,f);
	PutShort(silymin,f);
	PutShort(ob->xmax+1,f);
	PutShort( (ob->fontNumber<<12) | silymax, f);
	if (ob->kind==Text)
	  {
	    register char *p = ob->string;
	    
	    if (Debug) printf("Put string %s\r\n", p);
	    length = strlen(p);
	    PutByte(length,f);
	    for (;*p;p++) PutByte(*p,f);
	    if ( (length&1)==0) PutByte(0,f);
	  }
      }
  }

static PopUpEntry mainPopup[] = 
  {
    "Quit", 		CommandQuit,
    "Line Width",	CommandWidth,
    "Delete",		CommandDelete,
    "Unselect",		CommandUnselect,
    "Draw Line",	CommandDraw,
    "Add Text",		CommandAddText,
    "Modify Text",	CommandText,
    "Write", 		CommandWrite,
# ifdef Vsystem
    "Stretch Line",	CommandStretch,
    "Move",		CommandMove,
    "Copy",		CommandCopy,
    "Box",		CommandBox,
    "Select Area",	CommandSelectArea,
# else Vsystem
    "Debug", 		CommandDebug,
# endif Vsystem
    0, 0
  };


MainLoop(name)
  char *name;
  {
    	/*
	 * main command loop 
	 */
    short x, y, buttons;
    
    GetMouseClick(&x,&y,&buttons);
    switch (buttons)
      {
        case LeftButton:		DoMark(x,y);	break;
	case MiddleButton:		DoSelect(x,y,1);break;
	case LeftButton+MiddleButton:	DoLine(x,y);	break;
	case MiddleButton+RightButton:	DoSelect(x,y,0);break;
	case RightButton:
		switch (popup(mainPopup))
		  {
		    case CommandQuit:	MaybeQuit();	break;
		    case CommandDelete:	DoDelete();	break;
		    case CommandUnselect:DoUnselect();	break;
		    case CommandWrite:	DoWrite(name);	break;
		    case CommandDraw:	DoDraw();	break;
		    case CommandAddText:DoAddText();	break;
		    case CommandText:	DoText();	break;
		    case CommandWidth:	DoWidth();	break;
# ifdef Vsystem
		    case CommandMove:	DoMove(0);	break;
		    case CommandCopy:	DoMove(1);	break;
		    case CommandBox:	DoBox();	break;
		    case CommandStretch:DoStretch();	break;
		    case CommandSelectArea:DoArea();	break;
# endif Vsystem
		    case CommandDebug:
		    		if (Debug) 
				  {
				    printf("Debug mode OFF\r\n");
				    Debug = 0;
				  }
				else
				  {
				    printf("Debug mode ON\r\n");
				    Debug = 1;
				  }
				break;
		  }
		break;
	case LeftButton+MiddleButton+RightButton:	break;
	default:	printf("Strange button combination: %d\r\n", buttons);
      }
  }

struct Object *GetFreeObject()
  {
  	/*
	 * return a free object desriptor.
	 * Search the object table from the last point we found one,
	 * and wrap around the end.
	 */
    register struct Object *ob = Table + CurrentObject;
    short num = CurrentObject;
    
    while (1)
      {
        if (ob->state==Empty) 
	  {
	      ob->number=num;
	      CurrentObject=num+1;
	      ob->state=Used;
    	      return(ob);
	  }
	if (++num >= MaxObjects)
	  {
	    num = FirstItem;
	    ob = Table + num;
	  }
	 else ob++;
	if (CurrentObject==num) return(NULL);
      }
  }

Contain(xp,yp)
    short *xp, *yp;
  {
    /*
     * make sure that the point is within range
     */
    if (*xp < 0) *xp = 0;
    if (*xp > XLimit) *xp = XLimit;
    if (*yp < 0) *yp = 0;
    if (*yp > YOffset) *yp = YOffset;
  }


static PopUpEntry textPopup[] = 
  {
    "Edit Text",	CommandEditText,
    "Default Font",	CommandFont,
    "Change Font",	CommandChange,
    0, 0
  };

DoText()
  {
  	/*
	 * Modify some text
	 */
    register struct Object *ob;

    printf("Modify text M: ");
    switch (popup(textPopup))
      {
	    case CommandEditText:	DoEditText();	break;
	    case CommandFont:		DoFont();	break;
	    case CommandChange:		DoChange();	break;
	    defaut:			printf("\r\n");
      }
  }


DoAddText()
  {
  	/*
	 * Add some text
	 */
    register struct Object *ob;
    char string[256];
    int length;
    
    printf("Enter some text: ");
    ResetTTY();
    gets(string);
    GetTTY();
    length = strlen(string);
    if (length==0)
      {
        printf("You did not type anything.\r\n");
	return;
      }

    ob = GetFreeObject();
    if (ob==NULL)
      {
        printf("Too many objects already\r\n");
	return;
      }
    Changed++;
    ob->kind = Text;
    ob->string = (char *)malloc(length+1);
    strcpy(ob->string,string);
    ob->fontNumber=CurrentFont;
    ob->xmin = MarkX;
    ob->xmax = MarkX+length*8;
    ob->ymin = MarkY;
    ob->ymax = MarkY+16;
    EditSymbol(sdf,TopSymbol);
    if (FontMap[ ob->fontNumber ]==Template64) 
      {
        ob->ymax = MarkY+63;
        AddItem( sdf, ob->number, ob->xmin, ob->xmax, ob->ymax, 0,
	    FontMap[ ob->fontNumber ], SDF_TEXT, string);
      }
    else
      {
        AddItem( sdf, ob->number, ob->xmin, ob->xmax, ob->ymin, ob->ymax,
	    FontMap[ ob->fontNumber ], SDF_TEXT, string);
	if (Descenders(ob->string)) ob->ymin -= DescentFudge;
      }
   InquireItem(sdf,ob->number, &(ob->xmin), &(ob->xmax), 
	    	    NULL, &(ob->ymax), NULL, NULL, NULL);
    EndSymbol(sdf,TopSymbol,vgt);
  }



DoFont()
  {
  	/*
	 * Select a new default font for new text.
	 * some day, of course, we want to allow this to be
	 * user-defined.
	 */
    short newFont;
    
    printf("New default font M: ");
    newFont = popup(fontMenu);
    if (newFont<0)
      {
        printf("not changed\r\n");
	return;
      }
    printf("new font is %s\r\n", FontName(newFont));
    CurrentFont = newFont;
  }

char *Sel(s)
  char *s;
  {
  	/*
	 * Returns a pointer to a string with the Inverse video
	 * bit on the front of it.
	 */
    static char buf[256];
    buf[0] = BegInverse;
    strcpy(buf+1,s);
    return(buf);
  }

DoEditText()
  {
  	/*
	 * Change the content of text already displayed
	 */
    register struct Object *ob = SelectList;
    char string[256];
    int length;

   for (;ob;ob = ob->next)
	if (ob->kind == Text) break;

   if (ob==NULL)
      {
        printf("You have to select some text first\r\n");
	return;
      }
    printf("Enter new text: ");
    ResetTTY();
# ifdef Vsystem
    length = strlen(ob->string);
    EditLine(stdout,ob->string,length);
# endif Vsystem
    gets(string);
    GetTTY();
    length = strlen(string);
    if (length==0)
      {
        printf("You did not type anything.\r\n");
	return;
      }
    Changed++;
    EditSymbol(sdf,TopSymbol);
    for (;ob;ob = ob->next)
      {
	switch (ob->kind)
	  {
	    case Text:
	        ob->string = (char *)malloc(length+1);
    		strcpy(ob->string,string);
		ChangeItem( sdf, ob->number, ob->xmin, ob->xmax, 
		  BaseOf(ob->string,FontMap[ ob->fontNumber ],
		  		ob->ymin,ob->ymax) , 0,
#ifdef INVERT
		    FontMap[ ob->fontNumber ], SDF_TEXT, Sel(ob->string));
#else INVERT
		    FontMap[ ob->fontNumber ], SDF_TEXT, ob->string);
	        InquireItem(sdf,ob->number, &(ob->xmin), &(ob->xmax), 
	    	    NULL, &(ob->ymax), NULL, NULL, NULL);
	        ChangeItem(sdf,ob->number+SelectFlag,ob->xmin, ob->xmax, 
			ob->ymin - 1, ob->ymax + 1, 
			HILIGHT, SDF_FILLED_RECTANGLE, NULL);
#endif INVERT
		break;
	  }
      }
    EndSymbol(sdf,TopSymbol,vgt);
  }


DoChange()
  {
  	/*
	 * Change the font of text already displayed
	 */
    short newFont;
    register struct Object *ob = SelectList;

    if (ob==NULL)
      {
        printf("You have to select some text first\r\n");
	return;
      }
    printf("Select a font to impose M: ");
    newFont = popup(fontMenu);
    if (newFont<0)
      {
        printf("not changed\r\n");
	return;
      }
   Changed++;
   EditSymbol(sdf,TopSymbol);
    for (;ob;ob = ob->next)
      {
	switch (ob->kind)
	  {
	    case Text:
	        ob->fontNumber = newFont;
		ChangeItem( sdf, ob->number, ob->xmin, ob->xmax, 
		  BaseOf(ob->string,FontMap[ ob->fontNumber ],
		  		ob->ymin,ob->ymax) , 0,
#ifdef INVERT
		    FontMap[ ob->fontNumber ], SDF_TEXT, Sel(ob->string));
#else INVERT
		    FontMap[ ob->fontNumber ], SDF_TEXT, ob->string);
	        InquireItem(sdf,ob->number, &(ob->xmin), &(ob->xmax), 
	    	    NULL, &(ob->ymax), NULL, NULL, NULL);
	        ChangeItem(sdf,ob->number+SelectFlag,ob->xmin, ob->xmax, 
			ob->ymin - 1, ob->ymax + 1, 
			HILIGHT, SDF_FILLED_RECTANGLE, NULL);
#endif INVERT
		break;
	  }
      }
    EndSymbol(sdf,TopSymbol,vgt);

    printf("font imposed is %s\r\n", FontName(newFont));
  }


static PopUpEntry widthMenu[] = 
      {
        "1 (thin)",	1,
	"2",		2,
	"3",		3,
	"4",		4,
	"5",		5,
	"6",		6,
	"7",		7,
	"8 (thick)",	8,
	0, 0
      };

DoWidth()
  {
  	/*
	 * Select a new default line width
	 */
    short newWidth;
    
    printf("New line width (currently %d) M: ", Width);
    newWidth = popup(widthMenu);
    if (newWidth<0)
      {
        printf("not changed\r\n");
	return;
      }
    printf("new width is %d\r\n", newWidth);
    Width = newWidth;
  }


DoDraw()
  {
  	/*
	 * Draw a line
	 */
    short x, y, buttons;
    
    printf("Draw to point B: ");
    GetMouseClick(&x,&y,&buttons);
    DoLine(x,y);
    printf("\r\n");
  }

DoLine(x,y)
  short x,y;
  {
    register struct Object *ob;

    ob = GetFreeObject();
    if (ob==NULL)
      {
        printf("Too many objects already");
	return;
      }
    Changed++;
    Contain(&x,&y);
    ob->kind=Line;
    ob->fontNumber=LineFont;
    if ( abs(MarkX-x) < abs(MarkY-y) )
      {
        /*
	 * Vertical "line"
	 */
	ob->ymin = min(MarkY,y);
	ob->ymax = max(MarkY,y);
	ob->xmin = MarkX;
	ob->xmax = MarkX+Width-1;
	x = MarkX;
      }
    else
      {
        /*
	 * Horizontal "line"
	 */
	ob->xmin = min(MarkX,x);
	ob->xmax = max(MarkX,x);
	ob->ymin = MarkY;
	ob->ymax = MarkY+Width-1;
	y = MarkY;
      }
    EditSymbol(sdf,TopSymbol);
    AddItem( sdf, ob->number, ob->xmin, ob->xmax, ob->ymin, ob->ymax, BLACK,
	    	SDF_FILLED_RECTANGLE, NULL);
    DeleteItem(sdf,MarkItem);
    AddCall(sdf,MarkItem,x,y,MarkSymbol);
    EndSymbol(sdf,TopSymbol,vgt);
    MarkX = x;
    MarkY = y;
  }


DoMark(x,y)
    short x, y;
  {
  	/*
	 * position the marker
	 */
    Contain( &x, &y);
    printf("Mark at (%d, %d)\r\n", x, y);
    EditSymbol(sdf,TopSymbol);
    DeleteItem(sdf,MarkItem);
    AddCall(sdf,MarkItem,x,y,MarkSymbol);
    EndSymbol(sdf,TopSymbol,vgt);
    MarkX = x;
    MarkY = y;
  }

DoUnselect()
  {
	/*
	 * Unselect an object - user interface
	 */
    short x, y, buttons;
    LISTTYPE thing, FindSelectedObject();
    register struct Object *ob;
    int num;

    printf("Click at object to unselect B: ");
    GetMouseClick(&x,&y,&buttons);
    thing = FindSelectedObject(sdf,x,y,vgt,All);
    if (thing.NumOfElements==0)
      {
        printf("Could not find anything near there\r\n");
	return;
      }
    num = thing.Header->item;
    if (num>MaxObjects) return;
    ob = Table+num;
    if (!ob->selected) 
      {
    	printf("%s %d is not selected\r\n",
	  ob->kind==Text ? ob->string : "Line", num );
	return;
      }
    UnSelect(num);
    printf("unselected %s %d\r\n",
	  ob->kind==Text ? ob->string : "Line", num );
  }


UnSelect(n)
  {
  	/*
	 * Remove the given object from the select list.
	 * If zero, remove the whole list.
	 */
    register struct Object *ob = SelectList, *last = NULL;

    EditSymbol(sdf,TopSymbol);
    for (;ob;last = ob, ob = ob->next)
      {
        if (n && ob->number != n) continue;
        ob->selected = 0;
	switch (ob->kind)
	  {
	    case Text:
#ifdef INVERT
	        ChangeItem(sdf,ob->number,ob->xmin, ob->xmax, 
		  BaseOf(ob->stringFontMap[ ob->fontNumber ],ob->ymin,ob->ymax),
		   0, FontMap[ ob->fontNumber ], SDF_TEXT, ob->string);
#else INVERT
		DeleteItem(sdf,ob->number+SelectFlag);
#endif INVERT
		break;

	    case Line:
	    	ChangeItem(sdf,ob->number,ob->xmin, ob->xmax, ob->ymin, ob->ymax, 
			BLACK, SDF_FILLED_RECTANGLE, NULL);
		break;
	  }
	if (last) last->next = ob->next;
	if (ob==SelectList) SelectList = ob->next;
      }
    EndSymbol(sdf,TopSymbol,vgt);
  }


DoSelect(x,y,unFlag)
  {
  	/*
	 * Select something at the given point.
	 * "unFlag" means deselect everything else first.
	 */
	 
    LISTTYPE thing, FindSelectedObject();
    int num;
    register struct Object *ob;
    
    if (unFlag) UnSelect(0);
    thing = FindSelectedObject(sdf,x,y,vgt,All);
    if (thing.NumOfElements==0)
      {
        printf("Could not find anything near there\r\n");
	return;
      }
    num = thing.Header->item;
    if (num>MaxObjects) return;    
    ob = Table+num;
    switch (ob->kind)
      {
	case Line:
	    EditSymbol(sdf,TopSymbol);
	    ChangeItem(sdf,num,ob->xmin, ob->xmax, ob->ymin, ob->ymax, GRAY,
	    	SDF_FILLED_RECTANGLE, NULL);
	    EndSymbol(sdf,TopSymbol,vgt);

            if (ob->selected == 0)
	      {
    	        printf("Select line number %d\r\n", num);
	        ob->next = SelectList;
	        SelectList = ob;
		ob->selected = 1;
	      }
	    break;

	case Text:
	    EditSymbol(sdf,TopSymbol);
#ifdef INVERT
	    ChangeItem(sdf,num,ob->xmin, ob->xmax, 
	  BaseOf(ob->string,FontMap[ ob->fontNumber ],ob->ymin,ob->ymax),	
		 0,  FontMap[ ob->fontNumber ], SDF_TEXT, Sel(ob->string));
#else INVERT
	    InquireItem(sdf,num, &(ob->xmin), &(ob->xmax), 
	    	NULL, &(ob->ymax), NULL, NULL, NULL);
	    AddItem(sdf,num+SelectFlag,ob->xmin, ob->xmax, 
	    	ob->ymin - 1, ob->ymax - 1, 
	    	HILIGHT, SDF_FILLED_RECTANGLE, NULL);
#endif INVERT
	    EndSymbol(sdf,TopSymbol,vgt);

            if (ob->selected == 0)
	      {
	        printf("Select text ``%s''\r\n", ob->string);
	        ob->next = SelectList;
	        SelectList = ob;
		ob->selected = 1;
	      }
	    break;
      }
  }

DoDelete()
  {
	/*
	 * Delete items that are selected
	 */
    register struct Object *ob = SelectList;
    int count = 0;

    if (ob==NULL)
      {
        printf("You have to select something first\r\n");
	return;
      }
    printf("Deleted ");
    Changed++;
    EditSymbol(sdf,TopSymbol);
    for (;ob;ob=ob->next)
      {
        DeleteItem(sdf,ob->number);
	DeleteItem(sdf,ob->number+SelectFlag);
	ob->state = Deleted;
	count++;
      }
    EndSymbol(sdf,TopSymbol,vgt);
    printf("%d item%s\r\n", count, count==1 ? "" : "s");
    SelectList = NULL;
  }


DoWrite(name)
  char *name;
  {
  	/*
	 * Write the SIL file out
	 */
    File *f;
    
    f = fopen(name,"w");
    if (f==NULL)
      {
        printf("Unable to open %s for writing!\r\n", name);
	return;
      }
    printf("Writing %sfile %s...",Changed ? "changed " : "", name);
    fflush(f);
    PutSilFile(f);
    fclose(f);
    printf("done.\r\n");
    Changed = 0;
  }

# ifdef Vsystem
GetMilliseconds()
    {
	/*
	 * Return the time in milliseconds on V-System
	 */
      long ticks, seconds;

      seconds = GetTime(&ticks);
      return( seconds*1000 + ticks*10);
    }

# else Vsystem

# include <sys/types.h>
# include <sys/timeb.h>

GetMilliseconds()
    {
	/*
	 * Return the time in milliseconds on Unix
	 */
      struct timeb tp;

      ftime(&tp);
      return( tp.time*1000 + tp.millitm);
    }
# endif Vsystem
