/*
 *  Copyright (C) 1980, Ivor Durham
 *
 *  Translated 1980 from SAIL into C by Tom Rodeheffer (tlr)
 *    with permission of the original author.
 *
 *------------------------------------------------------------
 *
 *  HISTORY
 * 23-Mar-81  Tom Rodeheffer (tlr) at Carnegie-Mellon University
 *	Corrected the byte order of press files.  Press files had
 *	been being written with bytes swapped in pairs.  This was
 *	clearly wrong, but unfortunately it was the way which
 *	existing software expected press files to be written.  Now
 *	that that software has been repaired (notably, cz), we fix
 *	the byte order here.
 *
 * 12-Feb-81  Tom Rodeheffer (tlr) at Carnegie-Mellon University
 *	Added routines to allow the document directory's banner,
 *	creator, and number of copies to be set.
 *
 * 29-Jan-81  Tom Rodeheffer (tlr) at Carnegie-Mellon University
 *	Undid the edit of 25-Jan-81.  Added instead a routine
 *	PressShowRectangleHere which uses the current xpos and ypos.
 *
 * 25-Jan-81  Tom Rodeheffer (tlr) at Carnegie-Mellon University
 *	Changed PressShowRectangle so that it does not demand
 *	xpos and ypos.
 *
 * Dec-80  Tom Rodeheffer (tlr) at Carnegie-Mellon University
 *	Translation from SAIL to C.
 *
 */


#include <stdio.h>
/*#include "string.h"*/
#include <pwd.h>
#include "press.h"

extern char * ctime();
extern char * getenv();
extern long time();
extern struct passwd * getpwuid();
extern char * malloc();



#define bool int
#define TRUE 1
#define FALSE 0

#define local static
#define global




#define PressRecordLength 512

struct PressRecord
{
    struct PressRecord *next;
    unsigned char data [PressRecordLength];
};

struct PressStream
{
    int recordcount;              /* total press records */
    int totalbytes;               /* total bytes written */
    bool immediate;               /* output each record when full */
    unsigned char * bytep;        /* address to deposit next byte */
    int freebytes;                /* bytes left in current record */
    struct PressRecord * first;   /* first unwritten press record */
    struct PressRecord * current; /* current press record */
};

struct PagePart
{
    struct PressStream DL;        /* data list */
    struct PressStream EL;        /* entity list */
    int firstrecord;
};

struct Entity
{
    int type;
    int fontset;
    int beginbyte;
    int bytelength;
    int xposition;
    int yposition;
    int left;
    int bottom;
    int width;
    int height;
    int length;
    int currentfont;
    long showpending;            /* number of characters on DL whose */
                                 /* show command is still pending    */
};




local FILE * presschannel = NULL;

local char pressfilename [100];
local char pressbanner [100];
local char creatorname [100];
local char pressdate [100];
local long creationtimestamp;

local int totalrecords;
local int totalparts;
local int partdirrecord;
local int totalcopies;

local struct PagePart currentpage;
local struct Entity currententity;

local bool pageopen;
local bool entityopen;

local struct PressStream fontdir;
local struct PressStream partdir;
local struct PressStream docdir;

#define _PrintedPagePart 0
#define _FontDirectoryPart 1



/*
 * Attempt to allocate memory and panic if you do not get any.
 */

local char *
PressAlloc (size)

int size;
{
    register char * p;

    p = malloc(size);

    if (p == NULL)
    {
	quit(0200,"PressAlloc: insufficient memory\n");
    }

    return (p);
}


#define Alloc(thing)  (thing *) PressAlloc(sizeof (thing))



/*
 * Convert the UNIX date and time into Alto format.
 */

local long
AltoDateTime (datetime)

long datetime;
{
    long adjustment = (1970-1901)*365.24*24*60*60;

    return (datetime + adjustment);
}



local
InitPressStream (ps,immediate)

struct PressStream * ps;
bool immediate;
{
    ps->immediate = immediate;
    ps->recordcount = 0;
    ps->totalbytes = 0;
    ps->first = NULL;
    ps->current = NULL;
    ps->freebytes = 0;
}



/*
 * Write a press record into the press file.
 */ 

local
WritePressRecord (pr)

struct PressRecord * pr;
{
    fwrite(pr->data,PressRecordLength,1,presschannel);
}



local
FlushPressStream (ps)

struct PressStream * ps;
{
    register struct PressRecord * pr;
    register struct PressRecord * nextpr;

    for (pr = ps->first;  pr != NULL;  pr = nextpr) {

	WritePressRecord(pr);
	ps->recordcount++;
	totalrecords++;

	nextpr = pr->next;
	free(pr);
    }

    ps->first = pr;
    if (pr == NULL)
	ps->current = NULL;
}



local
NextRecord (ps)

struct PressStream * ps;
{
    register int i;
    register struct PressRecord * pr;

    if (ps->immediate)
	FlushPressStream(ps);

    pr = Alloc(struct PressRecord);

    pr->next = NULL;
    for (i = 0;  i < PressRecordLength;  i++)
	pr->data[i] = 0;

    if (ps->current == NULL) {
	ps->first = pr;
        ps->current = pr;
    } else {
	ps->current->next = pr;
	ps->current = pr;
    }

    ps->freebytes = PressRecordLength;
    ps->bytep = &pr->data[0];
}



local
PutBytetoStream (ps,b)

struct PressStream * ps;
unsigned char b;
{
    if (ps->freebytes <= 0)
	NextRecord(ps);

    *ps->bytep++ = b;
    ps->freebytes--;
    ps->totalbytes++;
}



local
PutWordtoStream (ps,w)

struct PressStream * ps;
unsigned short w;
{
    PutBytetoStream(ps,(w>>8)&0377);
    PutBytetoStream(ps,w&0377);
}



local
PutLongtoStream (ps,l)

struct PressStream * ps;
unsigned long l;
{
    PutWordtoStream(ps,(l>>16)&0177777);
    PutWordtoStream(ps,l&0177777);
}



local
PutBCPLStringtoStream (ps,str,maxwordlen)

struct PressStream * ps;
char * str;
int maxwordlen;
{
    int bytepadding;
    int strlength;
    int i;

    bytepadding = maxwordlen * 2;

    strlength = strlen(str);
    if (strlength >= bytepadding) {
	strlength = bytepadding - 1;
    }

    PutBytetoStream(ps,(unsigned char) strlength);

    for (i=0; i<strlength; i++)
	PutBytetoStream(ps,(unsigned char) str[i]);

    for (; i<bytepadding-1; i++)
	PutBytetoStream(ps,0);
}



local
EnterPartinDirectory (type,beginningrecord,lengthinrecords,value)

int type;
int beginningrecord;
int lengthinrecords;
int value;
{
    PutWordtoStream(&partdir,(unsigned short) type);
    PutWordtoStream(&partdir,(unsigned short) beginningrecord);
    PutWordtoStream(&partdir,(unsigned short) lengthinrecords);
    PutWordtoStream(&partdir,(unsigned short) value);

    totalparts++;
}



local
FinishPartDirectory ()
{
    partdirrecord = totalrecords;

    FlushPressStream(&partdir);
}



global
EnterFontinDirectory (fontset,font,firstchar,lastchar,
                      family,face,source,size,rotation)

int fontset;
int font;
int firstchar;
int lastchar;
char * family;
int face;
int source;
int size;
int rotation;
{
    char uppercasefamily [30];

    foldup(uppercasefamily,family);

    PutWordtoStream(&fontdir,(unsigned short) 16);
    PutBytetoStream(&fontdir,(unsigned short) fontset);
    PutBytetoStream(&fontdir,(unsigned short) font);
    PutBytetoStream(&fontdir,(unsigned short) firstchar);
    PutBytetoStream(&fontdir,(unsigned short) lastchar);
    PutBCPLStringtoStream(&fontdir, uppercasefamily, 10);
    PutBytetoStream(&fontdir,(unsigned short) face);
    PutBytetoStream(&fontdir,(unsigned short) source);
    PutWordtoStream(&fontdir,(unsigned short) size);
    PutWordtoStream(&fontdir,(unsigned short) rotation);
}



local
FinishFontDirectory ()
{
    int firstrecord;
    int wordspadding;

    PutWordtoStream(&fontdir,(unsigned short) 0);

    wordspadding = fontdir.freebytes / 2;
    firstrecord = totalrecords;

    FlushPressStream(&fontdir);

    EnterPartinDirectory(_FontDirectoryPart,
	firstrecord,fontdir.recordcount,wordspadding);
}



local
MakeDocumentDirectory ()
{
    int unused;

    PutWordtoStream(&docdir, 27183);
    PutWordtoStream(&docdir, totalrecords+1);
    PutWordtoStream(&docdir, totalparts);
    PutWordtoStream(&docdir, partdirrecord);
    PutWordtoStream(&docdir, partdir.recordcount);
    PutWordtoStream(&docdir, -1);
    PutLongtoStream(&docdir, AltoDateTime(creationtimestamp));
    PutWordtoStream(&docdir, 1);
    PutWordtoStream(&docdir, totalcopies);
    PutWordtoStream(&docdir, -1);
    PutWordtoStream(&docdir, -1);
    PutWordtoStream(&docdir, -1);

    for (unused=13; unused<=0177; unused++)
	PutWordtoStream(&docdir, -1);

    PutBCPLStringtoStream(&docdir, pressbanner, 26);
    PutBCPLStringtoStream(&docdir, creatorname, 16);
    PutBCPLStringtoStream(&docdir, pressdate, 20);

    FlushPressStream(&docdir);
}



local
DLPutByte (value)

int value;
{
    if (!entityopen) {
	quit(0200,"DLPutByte: No entity open!");
    }

    PutBytetoStream(&currentpage.DL,value);

    currententity.bytelength++;
}



local
ELPutByte (value)

int value;
{
    if (!entityopen) {
	quit(0200,"ELPutByte: No entity open!");
    }

    PutBytetoStream(&currentpage.EL,value);

    currententity.length++;
}



local
ELPutWord (value)

int value;
{
    if (!entityopen) {
	quit(0200,"ELPutWord: No entity open!");
    }

    ELPutByte((value>>8)&0377);
    ELPutByte(value&0377);
}



local
ELPutLong (value)

long value;
{
    if (!entityopen) {
	quit(0200,"ELPutLong: No entity open!");
    }

    ELPutWord((value>>16)&0177777);
    ELPutWord(value&0177777);
}





/*
 * Generate all of the pending Show Characters commands
 * which have been accumulating, so that a command can
 * be written to the entity list.
 */

local
FlushShowPending ()
{
    register long count;

    count = currententity.showpending;
    currententity.showpending = 0;

    while (count > 255) {
	ELPutByte(ELShowCharacters);
	ELPutByte(255);
	count -= 255;
    }

    if (count > 32) {
	ELPutByte(ELShowCharacters);
	ELPutByte(count);
    }
    else if (count > 0) {
	ELPutByte(ELShowCharactersShort+count-1);
    }
}



/*
 * Put an character onto the data list and increment the
 * count of Show Characters commands which are pending.
 */

global
PressShowCharacter (c)

char c;
{
    DLPutByte(c);
    currententity.showpending++;
}





global
PressSetX (position)

int position;
{
    FlushShowPending();

    ELPutByte(ELSetX);
    ELPutWord(position);
}



global
PressSetY (position)

int position;
{
    FlushShowPending();

    ELPutByte(ELSetY);
    ELPutWord(position);
}



global
PressFont (fontnumber)

int fontnumber;
{
    fontnumber &= 017;

    if (fontnumber != currententity.currentfont) {
	FlushShowPending();
	currententity.currentfont = fontnumber;
	ELPutByte(ELFont + fontnumber);
    }
}



global
PressShowRectangleHere (width,height)

int width;
int height;
{
    FlushShowPending();

    ELPutByte(ELShowRectangle);
    ELPutWord(width);
    ELPutWord(height);
}



global
PressShowRectangle (xposition,yposition,width,height)

int xposition;
int yposition;
int width;
int height;
{
    PressSetX(xposition);
    PressSetY(yposition);

    ELPutByte(ELShowRectangle);
    ELPutWord(width);
    ELPutWord(height);
}



global
PutStringontoPage (xposition,yposition,text)

int xposition;
int yposition;
char * text;
{
    register int i;
    register int textlength;

    textlength = strlen(text);

    if (textlength == 0)
	return;

    if (textlength > 255) {
	textlength = 255;
    }

    PressSetX(xposition);
    PressSetY(yposition);

    if (textlength <= 32) {
	ELPutByte(ELShowCharactersShort+textlength-1);
    } else {
	ELPutByte(ELShowCharacters);
	ELPutByte(textlength);
    }

    for (i=0; i<textlength; i++)
	DLPutByte(text[i]);
}



global
BeginEntity (type,fontset,xabsolute,yabsolute,left,bottom,width,height)

int type;
int fontset;
int xabsolute;
int yabsolute;
int left;
int bottom;
int width;
int height;
{
    if (entityopen) {
	quit(0200,"BeginEntity: Entity already open!");
    }

    entityopen = TRUE;

    currententity.type = type;
    currententity.fontset = fontset;
    currententity.beginbyte = currentpage.DL.totalbytes;
    currententity.bytelength = 0;
    currententity.xposition = xabsolute;
    currententity.yposition = yabsolute;
    currententity.left = left;
    currententity.bottom = bottom;
    currententity.width = width;
    currententity.height = height;
    currententity.length = 0;
    currententity.currentfont = 0;
    currententity.showpending = 0;
}



global
EndEntity ()
{
    if (!entityopen) {
	quit(0200,"EndEntity: No entity open!");
    }

    FlushShowPending();

    if ((currententity.length & 01) != 0) {
	ELPutByte(ELNop);
    }

    ELPutByte(currententity.type);
    ELPutByte(currententity.fontset);
    ELPutLong(currententity.beginbyte);
    ELPutLong(currententity.bytelength);
    ELPutWord(currententity.xposition);
    ELPutWord(currententity.yposition);
    ELPutWord(currententity.left);
    ELPutWord(currententity.bottom);
    ELPutWord(currententity.width);
    ELPutWord(currententity.height);
    ELPutWord(currententity.length/2 + 1);

    entityopen = FALSE;
}



global
BeginPage ()
{
    if (pageopen) {
	quit(0200,"BeginPage: Page already in progress!");
    }

    pageopen = TRUE;

    InitPressStream(&currentpage.DL,TRUE);
    InitPressStream(&currentpage.EL,FALSE);

    PutWordtoStream(&currentpage.EL,0); /* entity list starts with 0 */
    entityopen = FALSE;

    currentpage.firstrecord = totalrecords;
}



global
EndPage ()
{
    int trailersize;
    int partsize;

    if (!pageopen) {
	quit(0200,"EndPage: No page currently open!");
    }

    if (entityopen) {
	quit(0200,"EndPage: The last entity is still open!");
    }

    trailersize = currentpage.EL.freebytes / 2;

    FlushPressStream(&currentpage.DL);
    FlushPressStream(&currentpage.EL);

    partsize = currentpage.DL.recordcount +
               currentpage.EL.recordcount;

    EnterPartinDirectory(_PrintedPagePart,
        currentpage.firstrecord,partsize,trailersize);

    pageopen = FALSE;
}



global bool
PressStart (filename,copies)

char * filename;
int copies;
{
    register char *p;


    if (presschannel != NULL) {
	quit(0200,
	    "PressStart: Can only have one Press file open at a time");
	return (FALSE);
    }

    presschannel = (FILE *)fopenp("", filename, pressfilename,"w");
    if (presschannel==NULL)
       quit(0200,"PressStart: cannot open %s\n",filename);
/*
    presschannel = fwantwrite("",filename,pressfilename,
	"Press file name: ",0);
 */

    strcpy(pressbanner,pressfilename);


    strcpy(creatorname,getpwuid(getuid())->pw_gecos);

    for (p = creatorname;  *p!=0 && *p!='(';  p++)
	;
    *p = 0;


    creationtimestamp = time(0);


    strcpy(pressdate,ctime(&creationtimestamp));

    for (p = pressdate;  *p!=0 && *p!=012;  p++)
	;
    *p = 0;



    if (copies <= 0)
	copies = 1;

    totalcopies = copies;

    InitPressStream(&docdir,FALSE);
    InitPressStream(&partdir,FALSE);
    InitPressStream(&fontdir,FALSE);

    totalrecords = 0;
    totalparts = 0;

    return (TRUE);
}




global
PressSetCopies (copies)

int copies;
{
    totalcopies = copies;
}


global
PressSetCreator (creator)

char * creator;
{
    strcpy(creatorname,creator);
}


global
PressSetBanner (banner)

char * banner;
{
    strcpy(pressbanner,banner);
}






global bool
PressFinish ()
{
    if (presschannel == NULL) {
	quit(0200,"PressFinish: No Press file was being generated");
    }

    if (pageopen) {
	quit(0200,"PressFinish: The last page is still open.");
    }

    FinishFontDirectory();
    FinishPartDirectory();
    MakeDocumentDirectory();

    fclose(presschannel);
    presschannel = NULL;

    return (TRUE);
}

