#
/* press/fonts.c
 *
 * last edited by shore on Shasta.Stanford : Fri May 20 14:34:10 1983
 *
 * Routines to load and manipulate font information for the press to Postscript converter
 * All font information is local to this module.
 *
 * CONTENTS:
 *	ProcessFontDirectory	- translate font definitions from Press font
 *				- directory to Postscript preamble font defs
 *	DecodeFace		- ascii string from face encoding
 *	SetFont			- set Postscript font state from Press font number
 *	ZapSpace		- effect a Press Set-Space command
 *	Reset			- effect a Press Reset-Space command
 *
 */

#include "show.h"
#include "pressformat.h"

#ifdef FONTDEBUG
extern int Debug;
#define tracem(x) if (Debug) fprintf x
#else
#define tracem(x)
#endif FONTDEBUG



#define BadFont -1	/* fontNum for empty entries in fontTable */

/* font-related globals */

    static long nFonts;	/* total number of fonts in font directory */
    static struct FontEntry
      {
        long fontNum;	/* unique number of font */
        long rotation;	/* in minutes anticlockwise */
      };

    static struct FontEntry fontTable[4096]; /* indexed by (set<<4)+num */

/*    long currentFont;		 index of the current font */
    long currentRotation;	/* it's rotation */
    long	fdStart;	/* start record of font directory */
    long	fdRecs;		/* length of font directory */


/* ProcessFontDirectory();
 *	find the Font Directory Part, process the font definitions and
 *	build the font description table.
 */

ProcessFontDirectory()
  {
    extern long	pdStart;	/* starting record of part directory */
    extern long	nParts;		/* number of parts in part directory */

    long	part;		/* part number of Part Dir entry */
    long 	partType;	/* part type of entry */
    
    register long i;		/* index to initialize fontTable */
    register struct FontEntry *font; /* == fontTable[nFonts] */
    register char *cp;
    char	face[10];	/* face code string for DecodeFace */
    long	entryLength;	/* .. in words, inclusive */
    long	n;		/* last char -- test for bad fonts */
    long	set;		/* font set of entry */
    long	fn;		/* number within set [0..15] */
    long	index;		/* set<<4 + fn */
    long	sizeInMicas;	/* size of the font (+points, -micas) */
    long	faceCode;	/* Xerox Face encoding (see DecodeFace) */
    char	family[20];		/* font family name */

    FileSeek(fdStart*BytesPerRecord, 0);
    /* translate the font directory */
/*   if ... fatal("Improper Press file, no font directory part");*/            

            while ((entryLength = GetUWord()) != 0)
              {
    register FontInfo *fnt = (FontInfo *)(malloc(sizeof(FontInfo)));
    if (!fnt) fatal("Ran out of memory!");
                if (entryLength != 16)
                    crash("Can't handle Shape fonts");
                set = GetUByte();
                fn = GetUByte();
                fnt->fontNum = (set<<4)+fn; /* slot in fontTable */
		fnt->next = FontList; FontList = fnt;
		fnt->checkSum = 0;
/*                font->fontNum = nFonts;*/
                
                fnt->minChar = GetUByte();
                fnt->maxChar = n = GetUByte();
		fnt->width0 = NULL;
		fnt->internalFontMag = -1;
		fnt->internalFont = -1;
                if ((n & 0377) == 0377L)
                  {
                    if (BCPLtoCstring(family, 20) == 0)
                        crash("Can't handle Shape fonts");
                  }
                else BCPLtoCstring(family, 20);
		for (cp = family; *cp != '\0'; cp++)
		    if (*cp >= 'A' && *cp <= 'Z') *cp += 'a' - 'A';
                faceCode = GetUByte();
                GetUByte(); /* source */
                sizeInMicas = GetWord();
		GetWord(); /* rotation */

                tracem((stderr,"Font %d (%d %d) %s",nFonts,set,fn,family));
                tracem((stderr," F:%d S:%d R:%d\n",face,sizeInMicas,font->rotation));
		fnt->fileName = NULL;
                DecodeFace(face, faceCode);
                if (sizeInMicas > 0)
                  {/* convert Points to Micas */
                    sizeInMicas = (long) (sizeInMicas * MicasPerPoint);
                  }
                else
                  {/* absolute value is Micas */
                    sizeInMicas = -sizeInMicas;
                  }
		fnt->designSize = sizeInMicas*PointsPerMica;
		fnt->scaleFactor = sizeInMicas*PointsPerMica*(float)(1<<16);
                /* note that Press font names and faces contain no parens */

                printf("(F%d) (%s-%s) %d",
                    nFonts, family, face, sizeInMicas);

                nFonts++;
		{ char buf[200];
		    sprintf(buf, "%s%s",
		        family,
/* (long)(sizeInMicas/MicasPerPoint + 0.5), */
			face);
	            /* set up the dictionary for fonts and define them */
		    fnt->fontName = (char *)(malloc(strlen(buf) + 1));
	            strcpy(fnt->fontName, buf);
printf("[F:%s@%gpt]\n", buf,fnt->designSize);
		}
       }

  } /* ProcessFontDirectory */


/* DecodeFace(faceString, faceCode)
 *	decode's faceCode into an ascii string (MRR...)
 * 	see the explanation in "Font Representation and Formats"
 * 		by  Sproull, Swinehart, and Ramshaw 
 *		filed on [IFS]<PrintingDocs>FontFormats.Press
 */

static DecodeFace(faceString, faceCode)
    char *faceString;
    long faceCode;
  {
    long	slope, weight, expansion;	/* info in face code */

    if (faceCode >= 54) 
      {/* TEX logical size */
        sprintf(faceString, "%d\000", (254-faceCode)/2);
      }
    else
      {
        faceCode -= slope = faceCode % 2;
        faceCode -= weight = faceCode % 6;
        faceCode -= expansion = faceCode % 6;
        if (faceCode != 0) *faceString++ = 'A';
        switch (weight)
          {
            case  0: /* *faceString++ = 'M'; */ break;
            case  2:	*faceString++ = 'B';	break;
            case  4:	*faceString++ = 'L';	break;
            default:	*faceString++ = 'w';	break;	/* ERROR, PS will cope */
          }
        switch (slope)
          {
            case  0: /*	*faceString++ = 'R'; */	break;
            case  1:	*faceString++ = 'I';	break;
            default:	*faceString++ = 's';	break;	/* ERROR */
          }
        switch (expansion)
          {
            case  0: /*	*faceString++ = 'R'; */	break;
            case  6:	*faceString++ = 'C';	break;
            case 12:	*faceString++ = 'E';	break;
            default:	*faceString++ = 'e';	break;	/* ERROR */
          }
        *faceString++ = 0;
      }

  } /* DecodeFace */


/* SetFont(fontNumber)
 *	called with the Press font number [0..15] of the font to change
 * 	to in the current font set (global FontSet).  Look up the font in the
 *	font table and issue PS commands to set the current font.
 */

SetFont(fontNumber)
    long fontNumber;
  {
    extern int fontSet;		/* current font set for entity */
    register long index;	/* Press unique id */

    index = (fontSet<<4)+fontNumber;
    currentRotation = fontTable[index].rotation/60;
    SelectFont(index);
  } /* SetFont */


/* ZapSpace(spaceWidth, direction):
 *	effects a Press SetSpace command,  
 *	We only allow SetSpace's in the direction of the font, which is
 *	completely consistent with the way Press is USED, although stranger
 *	things are (theoretically) allowed.  BESIDES, anyone who does a 
 *	SetSpaceY for a portrait-oriented font (for example) gets what they
 *	deserve
 */

ZapSpace(spaceWidth, direction)
    register long	spaceWidth;	/* new width (in Micas) of ' ' */
    register int	direction;	/* {SpaceX, SpaceY} :
                                         * must be consistent with current rotation */
  {
    if (((direction == SpaceX) && ((currentRotation % 180) != 0)) ||
        ((direction == SpaceY) && ((currentRotation % 90) != 0)))
      {
        crash("FATAL ERROR: Set-Space not along font baseline direction");
      }
    else
      {
        if (CurFont == NULL) SetFont(0);
/*        Spacing = (spaceWidth + MicasPerPixel/2) / MicasPerPixel;*/
	Spacing = PixelRound(spaceWidth);
      }
  } /* ZapSpace */




/* Reset():
 *	effect a Reset-Space in Postscript
 *	done by simply resetting the font (thereby restoring the default
 *	width of space
 */

Reset()
  {
    if (CurFont == NULL) SetFont(0);
    Spacing = InvalidWidth;
  }
