/*
 * Use the sun as terminal to the serial lines (like a serial telnet).
 *
 * Usage:
 *   serial [ -b bps ] [ device ]
 *
 * The default data rate is 9600 bps.  The default device is [device]serial0.
 *   A single-digit device name '#' will be expanded to [device]serial#.
 *
 * RJN 8/25/83 - Author.
 */
#include <Vioprotocol.h>
#include <Vserial.h>
#include <Vteams.h>
#include <ctype.h>

# define ReaderStack 700  /* 500 probably is enough, but just in case */
# define WriterStack 700
# define OutputStack 700
# define MIN( A, B )	((A) < (B) ? (A) : (B))

extern ProcessId GetPid();

/* Defaults */
#define DEFAULT_DATA_RATE 9600
#define DEFAULT_DEVICE "[device]serial0"
#define DEVICE_NUMBER_INDEX 14	/* location of '0' in above */

/* The circular buffer */
# define BUFF_SIZE	8192
char InBuffer[BUFF_SIZE];
char *InHead = &InBuffer[0], *InTail = &InBuffer[0];
char *InEnd = &InBuffer[BUFF_SIZE];

WriterProcess( Head, Tail, End, To )
    register char **Head, **Tail;
    char *End;
    register File *To;
  /*
   * take the characters from the tail of the queue and output until
   * tail meets the Head or the End.  When the buffer is empty,
   * Delay for a while and try again.
   */
  {
    register int i;
    register char *TmpHead;
    Message msg;
    register unsigned EndMinusBuffSize = (unsigned)End - BUFF_SIZE;
    register unsigned BlockSize = To->blocksize;
    
    do
      {
        while( *Head == *Tail ) Delay( 1, 0 );

	if ( *Head < *Tail )
	  {
	    /* go to end first */
	    if ( *Tail < End )
	      {
		i = fwrite( *Tail, 1, End - *Tail, To );
		fflush( To );
	        if ( i <= 0 ) goto writer_error;
		*Tail += i;
	      }
	    *Tail = (char*)EndMinusBuffSize;
	  }
        /* clean up the rest */
	if ( *Tail < (TmpHead = *Head) )
          {
	    i = fwrite( *Tail, 1, TmpHead - *Tail, To );
	    fflush( To );
	    if ( i <= 0 ) goto writer_error;
	    *Tail += i;
	  }
      }
    while ( Eof(To) == OK );

writer_error:
    PrintError( Eof(To), "serial: Error on standard output" );
    
    msg[0] = 0;
    Send( msg, Creator( 0 ) );
	
  } /* WriterProcess */


ReaderProcess( Head, Tail, End, From, WriterPid )
    register char **Head, **Tail;
    char *End;
    register File *From;
    ProcessId WriterPid;
  /*
   * Read characters from From and put in the circular buffer.
   * After reading some characters, Wakeup the WriterProcess.
   * Have to make sure that there is enough room to read in
   * From->blocksize worth of bytes.  No check for overflow
   * so the buffer will get overwritten in this event.
   */
  {
    register int i;
    register char *TmpTail;
    Message msg;
    register unsigned EndMinusBuffSize = (unsigned)End - BUFF_SIZE;
    register unsigned BlockSize = From->blocksize;
   
    do
      {
	if ( *Tail <= *Head )
	  {
	    /* go to end first */
	    while ( *Head < End )
	      {
		/* Grody -- assumes that even if we read less than BlockSize,
		 * the next read will still start from where we left off.
		 * This is currently true of serial lines, but not guaranteed.
		 */
	        i = Read( From, *Head, MIN( BlockSize, End - *Head ) );
		From->block++;
	        if ( i <= 0 ) goto reader_error;
		*Head += i;
	        Wakeup( WriterPid );
	      }
	    *Head = (char*)EndMinusBuffSize;
	  }
	
        /* Have to account for the start */
	while ( *Head < (TmpTail = *Tail) - 1 )
	  {
	    i = Read( From, *Head, MIN( BlockSize, TmpTail - *Head ) );
	    From->block++;
	    if ( i <= 0 ) goto reader_error;
	    *Head += i;
	    Wakeup( WriterPid );
          }

	/* Try delaying a little while if buffer full */
	while ( *Head == *Tail - 1 )
	  {
	    Wakeup( WriterPid );	/* just in case? */
	    Delay( 0, 2 );
	  }
      }
    while( From->lastexception == OK );
    
reader_error:
    PrintError( From->lastexception, "serial: Error on serial input" );
    
    msg[0] = 0;
    Send( msg, Creator( 0 ) );
	
  } /* ReaderProcess */


/*
 * Output to the serial line is not buffered here; we do NOT want to
 *   read characters from serial's stdin as fast as it can
 *   deliver them and throw them away if the serial line isn't that fast!!
 * The tofile must be VARIABLE_BLOCK; this is not checked.  But a serial
 *   line always should be.
 */
OutputProcess(fromfile, tofile)
    register File *fromfile, *tofile;
  {
    register char *buffer, *p;
    register int incount, outcount, fromblocksize, toblocksize;
    register SystemCode error;
    Message msg;

    fromblocksize = BlockSize(fromfile);
    toblocksize = BlockSize(tofile);
    buffer = (char *) malloc(fromblocksize);
    Resynch(fromfile);
    Resynch(tofile);

    for (;;)
      {
	incount = Read(fromfile, buffer, fromblocksize);
	if (incount == 0) 
	  {
	    PrintError( FileException(fromfile),
	    	"serial: Error on standard input" );
	    msg[0] = 0;
	    Send( msg, Creator( 0 ) );  /* request death */
	  }
	fromfile->block++;
	p = buffer;
	while (incount > 0)
	  {
	    outcount = Write(tofile, p, MIN(incount, toblocksize));
	    if (outcount == 0)
	      {
	        PrintError( FileException(fromfile),
		    "serial: Error on serial output" );
	        msg[0] = 0;
	        Send( msg, Creator( 0 ) );  /* request death */
	      }
	    tofile->block++;
	    incount -= outcount;
	    p += outcount;
	  }
      }
  }

main( argc, argv )
    int argc;
    char **argv; 
 {
    ProcessId InRdr,InWrtr,OutProc;
    File *serialread, *serialwrite;
    SystemCode error;
    Message msg;
    char *name;
    int i, dataRate;
    ModifySerialDeviceRequest *mrequest = (ModifySerialDeviceRequest *) msg;
    ModifySerialDeviceReply *mreply = (ModifySerialDeviceReply *) msg;
    QuerySerialDeviceRequest *qrequest = (QuerySerialDeviceRequest *) msg;
    QuerySerialDeviceReply *qreply = (QuerySerialDeviceReply *) msg;

    /* Set defaults */
    dataRate = DEFAULT_DATA_RATE;
    name = DEFAULT_DEVICE;

    /* Check for flags */
    i = 1;
    while ( i < argc && argv[i][0] == '-' )
      {
	switch ( argv[i][1] )
	  {
	    case 'b':
	        /* Set data rate */
		i++;
		dataRate = atoi(argv[i++]);
		break;

	    default:
	        /* Nothing else implemented so far */
		fprintf(stderr, "%s: Unknown flag %s\n", 
		    argv[0], argv[i] );
		exit(1);
		break;
	  }
      }

    /*
     * Open an instance of the serial device.
     */
  
    if ( i < argc )
      {
	/* Check for single-digit argument */
	if ( argv[i][1] == '\0' && isdigit(argv[i][0]) )
	    name[DEVICE_NUMBER_INDEX] = argv[i][0];
	else
	    name = argv[i];
      }

    serialwrite = Open(name, FCREATE+FBLOCK_MODE, &error);
    if (error != OK || (serialwrite == NULL))
      {
	fprintf(stderr, "%s: Failed to open serial write instance: %s\n", 
		argv[0], ErrorString( error ) );
	exit( error );
      }
    
    serialread = OpenFile( FileServer(serialwrite), 
    		FileId(serialwrite) + 1, FREAD+FBLOCK_MODE, &error );
    if (error != OK || (serialread == NULL))
      {
	fprintf(stderr, "%s: Failed to open serial read instance: %s\n", 
		argv[0], ErrorString( error ) );
	exit( error );
      }
    
    /* Modify the serial line as needed */
    qrequest->requestcode = QUERY_FILE;
    qrequest->fileid = FileId(serialwrite);
    Send(qrequest, FileServer(serialwrite));

    /* At this point the results of the query are sitting in ou
     *   local message buffer, so we now use them as default values
     *   in the modify message */
    mrequest->requestcode = MODIFY_FILE;
    mrequest->fileid = FileId(serialwrite);
    mrequest->dataRate = dataRate;
    Send(mrequest, FileServer(serialwrite));

    /* Initialize processes */
    ChangeTeamPriority( GetPid( ACTIVE_PROCESS, LOCAL_PID ), REAL_TIME1 );
    InRdr = Create( 2, ReaderProcess, ReaderStack );
    InWrtr = Create( 34, WriterProcess, WriterStack );
    OutProc = Create( 33, OutputProcess, OutputStack );

    ModifyPad( stdout,0 );	/* Put pad in "raw" mode */

    Ready( InRdr, 5, &InHead, &InTail, InEnd, serialread, InWrtr );
    Ready( InWrtr, 4, &InHead, &InTail, InEnd, stdout );

    Ready( OutProc, 2, stdin, serialwrite );

    Receive( msg );
    DestroyProcess( InRdr );
    DestroyProcess( InWrtr );
    DestroyProcess( OutProc );
    Flush( stdout );

  } /* main */
