/*
 *  find.c
 *
 *  This file contains the routines used to select an object on the
 *  screen.  The primary routine of interest is FindObject(), which
 *  will locate an object of the given type, based upon the distance
 *  to its sticky points.  The object with the minimum distance to a
 *  control point is selected.
 *
 *  SelectObject() is the standard routine used to have the user pick
 *  an existing object on the screen.
 *
 *  David Kaelbling, May 1983
 */
 
/* Includes */
# ifdef VAX
# include "stdio.h"
# else
# include "Vio.h"
# endif
# include "Vgts.h"
# include "splines.h"
# include "draw.h"
 
 
/* Imports */
extern GetInput();
extern Checkpoint();
extern RevertToCKP();
extern Highlight();
extern HighlightStatic();
extern PermMisc();
extern PermHelp();
extern PermExit();
extern DefNib();
extern DefPattern();
extern DefFilling();
extern DefText();
 
 
/* Exports */
extern ITEM_LIST_PTR *FindObject();
extern FrameObject();
extern UnFrameObject();
extern short SelectObject();
 
 
/* Local Definitions */

/*
 *  This routine will return the square of the distance between two points.
 */
 
int PointDistance( x1, y1, x2, y2 )
	register short x1, y1, x2, y2;
  {
    register short x, y;
    
    x = (x1 - x2);
    y = (y1 - y2);
    return( x * x  +  y * y );
  }

/*
 *  This internal routine is strictly a helper process for SplineDistance.
 *  It will return the proper knot from the simulated knot vector in order
 *  to calculate the sticky points on a spline.  End conditions are dealt
 *  with properly.
 */
 
short Knot( i, order, closed, x, numvert, vert )
	short i, order, closed, x, numvert;
	POINT *vert;
  {
    register short j;			/* Index variable */
    
    /* Compute the proper index into the vertex array. */
    if (closed)
      {
	/* Just circulate the vertices */
	j = i % numvert;
      }
    else
      {
	/* Duplicate beginning and ending knots appropriately. */
	i = i + 3 - order;
	if (i <= 0)
	    j = 0;
	else if (i >= numvert)
	    j = numvert - 1;
	else
	    j = i;
      }
    
    /* Return the value of this control point. */
    if (x)
	return( vert[j].x );
    else
	return( vert[j].y );
  }

/*
 *  This internal routine, along with its helper Knot(), will compute
 *  the square of the distance from a point to the nearest sticky
 *  point on a spline.
 */
 
int SplineDistance( numvert, vert, order, closed, x, y, dx, dy, cp )
	short numvert;		/* Number of control points. */
	POINT *vert;		/* Array of control points. */
	short order;		/* Order of the spline. */
	BOOLEAN closed;		/* End conditions. */
	short x, y, *dx, *dy;	/* Current point, closes point */
	short cp;		/* Measure from control points too? */
  {
    int curdist;
    register int dist;
    register short xval, yval, i;
    short bestx, besty;
    
    /* Initialize */
    order = ((order == 2) ? 3 : order);
    curdist = MAXDIST << 6;
    
    /* Check each sticky point. */
    for (i = -(!closed); i < numvert; i++)
      {
	switch (order)
          {
	    case 3:	/* Sticky point is (p1 + p2) / 2 */
	        xval = (Knot(i, order, closed, 1, numvert, vert) +
			Knot(i+1, order, closed, 1, numvert, vert)) / 2;
		yval = (Knot(i, order, closed, 0, numvert, vert) +
			Knot(i+1, order, closed, 0, numvert, vert)) / 2;
		break;
		
	    case 4:	/* Sticky point is (p1 + 4*p2 + p3) / 6 */
		xval = (Knot(i, order, closed, 1, numvert, vert) +
			4*Knot(i+1, order, closed, 1, numvert, vert) +
			Knot(i+2, order, closed, 1, numvert, vert)) / 6;
		yval = (Knot(i, order, closed, 0, numvert, vert) +
			4*Knot(i+1, order, closed, 0, numvert, vert) +
			Knot(i+2, order, closed, 0, numvert, vert)) / 6;
		break;
		
	    case 5:	/* Sticky point is (p1 + 11*p2 + 11*p3 + p4) / 24 */
		xval = (Knot(i, order, closed, 1, numvert, vert) +
			11*Knot(i+1, order, closed, 1, numvert, vert) +
			11*Knot(i+2, order, closed, 1, numvert, vert) +
			Knot(i+3, order, closed, 1, numvert, vert)) / 24;
		yval = (Knot(i, order, closed, 0, numvert, vert) +
			11*Knot(i+1, order, closed, 0, numvert, vert) +
			11*Knot(i+2, order, closed, 0, numvert, vert) +
			Knot(i+3, order, closed, 0, numvert, vert)) / 24;
		break;
	  }
	
	/* Compute the distance from this sticky point. */
	dist = PointDistance( x, y, xval, yval );
	if (dist < curdist)
	  {
	    bestx = xval;
	    besty = yval;
	    curdist = dist;
	  }
	
	/* Check Control points too? */
	if (cp)
	  {
	    dist = PointDistance( x, y, vert[i].x, vert[i].y );
	    if (dist < curdist)
	      {
		bestx = vert[i].x;
		besty = vert[i].y;
		curdist = dist;
	      }
	  }
      }
    
    /* Return the results */
    *dx = bestx;
    *dy = besty;
    return( curdist );
  }

/*
 *  This procedure will return the coordinates of the minimum distance
 *  sticky points on an object, measured from a given point.  The value
 *  of the procedure is the square of the distance.
 */
 
int ObjectDistance( x, y, px, py, ptr, expand, deltax, deltay )
	short x, y, *px, *py;
	register ITEM_DESCRIPTOR *ptr;
	short expand, deltax, deltay;
  {
    short xtmp, ytmp;
    register int dist, curdist = MAXDIST << 6;
    SPLINE *sptr;
    POINT *pptr;
    ITEM_LIST_PTR *gptr;
    
    /* Subtract offset from target point now, */
    /*  add it in to returned point later.  */
    x -= deltax;  y-= deltay;
    switch (ptr->type)
      {
	case TextObj:
	    /* Sticky points are the corners and midpoints of	*/
	    /* the edges of the bounding box.			*/
	    if ((pptr = (POINT *) malloc( sizeof(POINT) * 4 )) == NULL)
	      {
		printf("Out of memory.  Can't compute text distance.\n\r");
		break;
	      }
	    pptr[0].x = ptr->xmin;
	    pptr[0].y = ptr->ymin;
	    pptr[1].x = ptr->xmin;
	    pptr[1].y = ptr->ymax;
	    pptr[2].x = ptr->xmax;
	    pptr[2].y = ptr->ymax;
	    pptr[3].x = ptr->xmax;
	    pptr[3].y = ptr->ymin;
	    dist = SplineDistance( 4, pptr, 2, 1, x, y, &xtmp, &ytmp, 1 );
	    if (dist < curdist)
	      {
		curdist = dist;
		*px = xtmp;  *py = ytmp;
	      }
	    free( pptr );
	    break;
	
	case OpenSplineObj:
	case ClosedSplineObj:
	case OpenPolygonObj:
	case ClosedPolygonObj:
	case TemplateObj:
	    /* A spline.  Measure distance from sticky points. */
	    sptr = (SPLINE *) ptr->data;
	    pptr = &(sptr->head);
	    if (sptr->numvert == 1)
	      {
		/* Only one possible point.  An easy case. */
		dist = PointDistance( x, y, xtmp=pptr[0].x, ytmp=pptr[0].y );
		if (dist < curdist)
	          {
		    *px = xtmp;  *py = ytmp;
		    curdist = dist;
	          }
	      }
	    else
	      {
		/* A hard case.  Call the more elaborate routines. */
		dist = SplineDistance(
			sptr->numvert, pptr, sptr->order, sptr->closed,
			x, y, &xtmp, &ytmp, (sptr->order == 2) );
		if (dist < curdist)
	          {
		    *px = xtmp;  *py = ytmp;
		    curdist = dist;
	          }
	      }
	    break;
	
	case GroupObj:
	    /* If necessary, check all of the objects in the group */
	    if (expand)
	      {
		gptr = (ITEM_LIST_PTR *) ptr->data;
		while (gptr)
		  {
		    dist = ObjectDistance( x, y, &xtmp, &ytmp,
			gptr->itemdesc, 1, gptr->dx, gptr->dy );
		    if (dist < curdist)
		      {
			*px = xtmp;  *py = ytmp;
			curdist = dist;
		      }
		    gptr = gptr->next;
		  }
	      }
	    break;
	    
	default:
	    printf("Internal Error:  Illegal object type %d in ObjDist.\n\r",
	    		ptr->type);
	    break;
      }
    
    /* Compensate for displacements, and return results */
    *px += deltax;  *py += deltay;
    return( curdist );
  }

/*
 *  This routine will find the closest suitable object to a given point.
 *  Groups are examined only if the 'expand' flag is set.
 */
 
ITEM_LIST_PTR *FindObject( type, x, y, dx, dy, distance, expand )
	short type;
	short x, y;
	short *dx, *dy;
	int *distance;
	short expand;
  {
    register ITEM_LIST_PTR *p, *object;
    int bestDistance, tempDistance;
    short bestx, besty;
    
    /* Initialize for no object found. */
    object = NULL;
    bestDistance = MAXDIST << 6;
    
    /* Scan the active object list */
    p = activelist->prev;
    while (p)
      {
	/* Shall we even try? */
	if ( ((int) p->itemdesc->type == type) || (type == -1) )
	  {
	    /* Right type.  Right place? */
	    tempDistance = ObjectDistance( x, y, dx, dy,
	    		p->itemdesc, expand, p->dx, p->dy );
	    
	    if (Debug)
		printf("    distance = %d, object number = %d\n\r",
			tempDistance, p->itemdesc->number );
		
	    /* Is this the one we want? */
	    if (tempDistance < bestDistance)
	      {
		bestx = *dx;
		besty = *dy;
		bestDistance = tempDistance;
		object = p;
	      }
	  }
	p = p->next;
      }
    
    /* Debugging Information? */
    if (Debug)
      {
	printf("    Best distance = %d, object number = ", bestDistance);
	if (object)
	    printf("%d\n\r", object->itemdesc->number);
	else
	    printf(" 0\n\r");
      }

    /* Report the results */
    *dx = bestx;  *dy = besty;
    *distance = bestDistance;
    return( (bestDistance > MAXDIST) ? NULL : object );
  }

/*
 *  This routine will put a frame of some kind around an object, thus
 *  highlighting it.
 */
 
FrameObject( q, frameno )
	ITEM_LIST_PTR *q;
	short *frameno;
  {
    register ITEM_DESCRIPTOR *id = q->itemdesc;
    ITEM_LIST_PTR *p;
    SPLINE *sptr, *tptr;
    POINT *svert, *tvert;
    short i, saveSymbol, saveVgt;
    
    if (id == NULL)
	return;
    
    switch (id->type)
      {
	case TextObj:
	    /* Show an expanded bounding box. */
	    EditSymbol( sdf, mainSymbol );
	    *frameno = AddItem( sdf, item++,
		id->xmin - 3 + q->dx, id->xmax + 3 + q->dx,
		id->ymin - id->base - 3 + q->dy, id->ymax + 3 + q->dy,
		AllEdges, SDF_OUTLINE, NULL );
	    EndSymbol( sdf, mainSymbol, mainVgt );
	    break;
	
	case GroupObj:
	    /* A group.  Frame all of the objects inside the group. */
	    saveSymbol = mainSymbol;
	    saveVgt = mainVgt;
	    *frameno = DefineSymbol( sdf, item++, "group frame symbol" );
	    EndSymbol( sdf, *frameno, 0 );
	    item++;
	    mainSymbol = *frameno;
	    mainVgt = 0;
	    for (p = (ITEM_LIST_PTR *) id->data;  p;  p = p->next)
		FrameObject( p, &i );
	    mainSymbol = saveSymbol;
	    mainVgt = saveVgt;
	    EditSymbol( sdf, mainSymbol );
	    AddCall( sdf, (*frameno) + 1, q->dx, q->dy, *frameno );
	    EndSymbol( sdf, mainSymbol, mainVgt );
	    *frameno = - (*frameno);
	    break;
	
	default:
	    /* Some form of spline */
	    sptr = (SPLINE *) id->data;
	    
	    /* For degenerate splines, simply show bounding box. */
	    if ((sptr->order == 2) || (sptr->numvert <= 2) ||
	        (id->type == TemplateObj))
	      {
		EditSymbol( sdf, mainSymbol );
		*frameno = AddItem( sdf, item++,
			id->xmin + 1 + q->dx, id->xmax - 1 + q->dx,
			id->ymin + 1 + q->dy, id->ymax - 1 + q->dy,
			AllEdges, SDF_OUTLINE, NULL );
		EndSymbol( sdf, mainSymbol, mainVgt );
		break;
	      }
	      
	    /* For REAL splines, show a frame */
	    tptr = (SPLINE *) malloc ( sizeof(SPLINE) +
		(sptr->numvert - 1) * sizeof(POINT) );
	    if (tptr == NULL)
	      {
		printf("Out of Memory.  Can't frame anything.\n\r");
		break;
	      }
	    tptr->order = 2;
	    tptr->numvert = sptr->numvert;
	    tptr->nib = NibCircle0;
	    tptr->border = 1;
	    tptr->closed = sptr->closed;
	    tptr->filled = 0;
	    tptr->pat = PatWhite;
	    svert = &(sptr->head);
	    tvert = &(tptr->head);
	    for (i = sptr->numvert;  i--;)
	      {
		tvert[i].x = svert[i].x + q->dx;
		tvert[i].y = svert[i].y + q->dy;
	      }
	    EditSymbol( sdf, mainSymbol );
	    *frameno = AddItem( sdf, item++, 0, 0, 0, 0, 0, SDF_SPLINE, tptr);
	    EndSymbol( sdf, mainSymbol, mainVgt );
	    free( tptr );
	    break;
      }
  }

/*
 *  This routine will remove the frame from around an item.
 */
 
UnFrameObject( frameno, refresh )
	short *frameno, refresh;
  {
    /* If we haven't done so already, delete the frame. */
    if (*frameno > 0)
      {
	EditSymbol( sdf, mainSymbol );
	DeleteItem( sdf, *frameno );
	EndSymbol( sdf, mainSymbol, (refresh ? mainVgt : 0) );
	*frameno = 0;
      }
    else if (*frameno < 0)
      {
	/* Unframe a group.  Delete the symbol call and definition. */
	EditSymbol( sdf, mainSymbol );
	DeleteItem( sdf, (- (*frameno)) + 1 );
	EndSymbol( sdf, mainSymbol, (refresh ? mainVgt : 0) );
	DeleteSymbol( sdf, -(*frameno) );
	*frameno = 0;
      }
  }

/*
 *  This routine will permit the user to select and confirm an existing
 *  object on the screen.  It will return 1 if the user confirmed with
 *  CAlmostDone, 0 otherwise.  The variable 'allselected' will contain
 *  1 if all objects were selected, 2 if the user aborted, and 0 otherwise.
 *  The selected and type parameters are returned so that if the user
 *  clicks CAlmostDone, no state will be lost.  They should be initialized
 *  to 0 and -1 respectively.
 */
 
short SelectObject( victim, cmd, selected, type, generalprompt,
		    allselected, refresh, numpnt, pntdat )
	ITEM_LIST_PTR **victim;
	enum MenuOptions cmd;
	short *selected, *type;
	char *generalprompt;		/* Prompt to pick an object type */
	short *allselected, refresh;
	short numpnt;
	POINT_DATA *pntdat;
  {
    short picking;
    short x, y, but, frame;
    short dx, dy;
    short pntfound;
    int dist;
    
    /* Print Debugging Information? */
    if (Debug)
	printf("SelectObject:  selected = %d, type = %d\n\r",
		*selected, *type);
    
    /* Make ready */
    *victim = NULL;
    picking = 1;
    frame = 0;
    pntfound = ((*selected == 1) ? 0 : -999);
    if (*selected == 5)
      {
	cmd = CCurrentObj;
	Highlight( &(Objects[*selected]) );
	*selected = 0;
      }
    else if (*selected == 8)
	cmd = CGroup;
    else if (pntfound == numpnt)
	printf("Confirm with Done or AlmostDone.\n\r");
    else if (pntfound != -999)
	printf("%s\n\r", pntdat[pntfound].prompt );
    else if (*selected == 1)
	printf("Confirm everything selected.\n\r");
    else if (*selected)
	printf("Select an object.\n\r");
    else
	printf("%s\n\r", generalprompt);
    
    if (cmd == CNull)
	GetInput( &cmd, &x, &y, &but );
    
    /* Select some objects */
    while (picking)
      {
	switch (cmd)
	  {
	    case CNull:
		break;
	    
	    case CDataPoint:
		/* Are we expecting a data point? */
		if ((pntfound == numpnt) && numpnt)
		  {
		    printf("No need to specify more points.\n\r");
		    break;
		  }
		else if ((pntfound != -999) && numpnt)
		  {
		    if (Debug)
			printf("Found point %d of %d at (%d, %d)\n\r",
				pntfound, numpnt, x, y);
		    pntdat[pntfound].x = x;
		    pntdat[pntfound].y = y;
		    EditSymbol( sdf, mainSymbol );
		    pntdat[pntfound].flag = AddItem( sdf, item++, 
				x - 2, x + 2, y - 2, y + 2,
				AllEdges, SDF_OUTLINE, NULL );
		    EndSymbol( sdf, mainSymbol, mainVgt );
		    pntfound++;
		    break;
		  }
		
		/* We are expecting an object to be selected. */
		pntfound = -999;
		if (*victim)
		  {
		    printf("Previous object deselected.\n\r");
		    *victim = NULL;
		    UnFrameObject( &frame, 1 );
		  }
		if (*selected == 0)
		  {
		    printf("Select the object TYPE, then the object.\n\r");
		  }
		else if (*selected == 1)
		  {
		    printf("Everything is selected.  Pointing at objects");
		    printf(" is meaningless.\n\r");
		  }
		else if (*type == -1)
		  {
		    printf("You can't point to objects of that type.\n\r");
		    printf("Confirm your previous choice or change type.\n\r");
		  }
		else
		  {
		    /* Try to find something */
		    *victim = FindObject( *type, x, y, &dx, &dy, &dist, 0 );
		    if (*victim)
		      {
		        FrameObject( *victim, &frame );
			pntfound = 0;
		      }
		    else
			printf("Nothing appropriate there.  Try again.\n\r");
		  }
		break;
		
	    case CAllObj:
	    case CText:
	    case COpenCurve:
	    case CClosedCurve:
	    case CCurrentObj:
		if ((cmd == CCurrentObj) && (CurrentObject == NULL))
		  {
		    printf("No current object to select!\n\r");
		    break;
		  }
	    case COpenPolygon:
	    case CClosedPolygon:
	    case CGroup:
	    case CTemplate:
		/* If we were selecting points, discard them */
		EditSymbol( sdf, mainSymbol );
		while (--pntfound >= 0)
		  {
		    DeleteItem( sdf, pntdat[pntfound].flag );
		  }
		EndSymbol( sdf, mainSymbol, mainVgt );
		pntfound = -999;
		
		/* Deselect old stuff, select new object type */
		if (*selected)
		    Highlight( &(Objects[*selected]) );
		*selected = (int) cmd - (int) CAllObj + 1;
		Highlight( &(Objects[*selected]) );
		if (*victim)
		  {
		    printf("Previous object deselected.\n\r");
		    UnFrameObject( &frame, 1 );
		    *victim = NULL;
		  }
		*type = (int) cmd - (int) CText;
		
		/* If we know which object already, act accordingly */
		if (cmd == CAllObj)
		  {
		    pntfound = 0;
		  }
		else if (cmd == CCurrentObj)
		  {
		    *victim = CurrentObject;
		    FrameObject( *victim, &frame );
		    *type = -1;
		    pntfound = 0;
		  }
		else if (cmd == CGroup)
		  {
		    GetGroup( victim );
		    *type = -1;
		    if (*victim)
		      {
			pntfound = 0;
			FrameObject( *victim, &frame );
		      }
		    else
		      {
			Highlight( &(Objects[*selected]) );
			*selected = 0;
			pntfound = -999;
		      }
		  }
		else if ((int) cmd > (int) CCurrentObj)
		    *type -= 1;
		break;
		
	    case CDone:
	    case CAlmostDone:
		/* Is this acceptable? */
		HighlightStatic( &(StaticMenu[0]) );
		if ((*selected == 1) && (!pntfound) && (numpnt))
		  {
		    Highlight( &(Objects[*selected]) );
		    *selected = 0;
		  }
		else if ((numpnt != pntfound) && (*victim || (*selected==1)))
		  {
		    printf("You're not done yet.  Select more points.\n\r");
		    HighlightStatic( &(StaticMenu[0]) );
		    break;
		  }
		
		/* Remove highlighting. */
		picking = 0;
		EditSymbol( sdf, mainSymbol );
		while (--pntfound >= 0)
		  {
		    DeleteItem( sdf, pntdat[pntfound].flag );
		  }
		EndSymbol( sdf, mainSymbol, mainVgt );
		if (*victim)
		  {
		    if ((CurrentCmd != CCopy) && (CurrentCmd != CRaise) &&
		        (CurrentCmd != CLower))
			refresh = 0;
		    if ((*victim)->itemdesc->type == TextObj)
			refresh = 1;
		    UnFrameObject( &frame, refresh );
		  }
		HighlightStatic( &(StaticMenu[0]) );
		break;
	    
	    case CAbort:
		HighlightStatic( &(StaticMenu[1]) );
		picking = 0;
		printf("Abort.\n\r");
		if (frame)
		    UnFrameObject( &frame, 0 );
		EditSymbol( sdf, mainSymbol );
		while (--pntfound >= 0)
		  {
		    DeleteItem( sdf, pntdat[pntfound].flag );
		  }
		EndSymbol( sdf, mainSymbol, 0 );
		RevertToCKP();
		HighlightStatic( &(StaticMenu[1]) );
		break;
	    
	    case CUndo:
		HighlightStatic( &(StaticMenu[2]) );
		/* Undo a data point? */
		if (pntfound > 0)
		  {
		    pntfound--;
		    EditSymbol( sdf, mainSymbol );
		    DeleteItem( sdf, pntdat[pntfound].flag );
		    EndSymbol( sdf, mainSymbol, mainVgt );
		    HighlightStatic( &(StaticMenu[2]) );
		    break;
		  }
		else
		    pntfound = ((*selected == 1) ? 0 : -999);
		
		/* Undo an object selection. */
		if (*victim)
		  {
		    UnFrameObject( &frame, 1 );
		    *victim = NULL;
		  }
		else if (*selected != 1)
		    printf("No objects selected.\n\r");
		HighlightStatic( &(StaticMenu[2]) );
		break;
	    
	    case CMisc:
		HighlightStatic( &(StaticMenu[3]) );
		PermMisc();
		HighlightStatic( &(StaticMenu[3]) );
		break;
		
	    case CHelp:
		HighlightStatic( &(StaticMenu[4]) );
		PermHelp();
		HighlightStatic( &(StaticMenu[4]) );
		break;
	    
	    case CExit:
		HighlightStatic( &(StaticMenu[5]) );
		PermExit();
		HighlightStatic( &(StaticMenu[5]) );
		break;
		
	    case CNib:		DefNib();  break;
	    case CPattern:	DefPattern();  break;
	    case CFilling:	DefFilling();  break;
	    case CTextDefault:	DefText();  break;
	    case CCheckPoint:	Checkpoint();  break;
	    
	    default:
		printf("Sorry, that command doesn't make sense here.\n\r");
		break;
	  }
	
	/* Reprompt the user. */
	if (picking)
	  {
	    if (pntfound == numpnt)
		printf("Confirm with Done or AlmostDone.\n\r");
	    else if (pntfound != -999)
		printf("%s\n\r", pntdat[pntfound].prompt );
	    else if (*selected == 1)
		printf("Confirm everything selected.\n\r");
	    else if (*selected)
		printf("Select an object.\n\r");
	    else
		printf("%s\n\r", generalprompt);
	    GetInput( &cmd, &x, &y, &but );
	  }
      }
    
    *allselected = ((cmd == CAbort) ? 2 : (*selected == 1));
    
    /* Turn off any highlighting or framing */
    if (frame)
	printf("Internal error:  Exit SelectObject() with frame != 0.\n\r");
    if ((*selected) && (cmd != CAlmostDone))
      {
	Highlight( &(Objects[*selected]) );
	*selected = 0;
      }
    
    /* Return results */
    return( (short) (cmd == CAlmostDone) );
  }
