/*
 *  writepsm.c
 *
 *  This file contains the code to write a PostScript Metafile (PSM)
 *
 *  Gustavo Fernandez - 11/26/85 - New today, modified from save.c
 */
 
 
/* Includes */
# ifdef UNIX
# include "stdio.h"
# else
# include "Vio.h"
# endif
# include "Vgts.h"
# include "splines.h"
# include "draw.h"
 
 
/* Imports */
extern Quit();
extern DebugObjectList();
extern GetMisc();
extern char *GetString();
extern RedrawActivelist();
extern EraseObject();
 
/* Exports */
extern short WritePSM();
 
/* Local Definitions */
# define tolower(c)	(((c >= 'A') && (c <= 'Z')) ? (c - 'A' + 'a') : c)
# define abs(a)		((a)>0?(a):-(a))
# define PSMHEADER 	"%! V Draw PSM file version 1.0\n"
# define HEADERFILE	"/usr/V/run/psm.ps"
# define LPR		"/usr/ucb/lpr"
# define PENDSTACKSIZE  30

char *boolstr[] = { "false","true" };
short oldtype;
short oldfont,oldsize,oldjust;
SPLINE oldspline;
short olddx,olddy;

static short PendStackLength;
static ITEM_LIST_PTR *PendStack[PENDSTACKSIZE];
static short groupcount,firstgroup;


/*
 *  This internal routine is a helper process for WritePSM().  It will
 *  scan the document and determine the absolute bounding box.
 */
 
ScanBBox ( fp, p, xmin, xmax, ymin, ymax, dx, dy)
	FILE *fp;
        ITEM_LIST_PTR *p;
	short *xmin,*xmax,*ymin,*ymax;
	short dx,dy;
{
    short   i;
    ITEM_DESCRIPTOR * ptr;
    ITEM_LIST_PTR * q;
    SPLINE * sptr;
    POINT * pptr;

#define XMIN(a) (*xmin = a < *xmin ? a : *xmin)
#define XMAX(a) (*xmax = a > *xmax ? a : *xmax)
#define YMIN(a) (*ymin = a < *ymin ? a : *ymin)
#define YMAX(a) (*ymax = a > *ymax ? a : *ymax)


    if (p == NULL)
	return;

    ptr = p->itemdesc;
    if (ptr == NULL)
	return;
    switch (ptr->type) {
	case TextObj: 
	    /* Text */
	    if (ptr->typedata != 3) { /*Ignore PressEdit*/
		XMIN (ptr->xmin + dx);
		XMAX (ptr->xmax + dx);
		YMIN (ptr->ymin + dy);
		YMAX (ptr->ymax + dy);
	    }
	    break;

	case GroupObj: 
	    for (q = (ITEM_LIST_PTR *) ptr->data; q; q = q->next) {
		ScanBBox (fp, q, xmin, xmax, ymin, ymax, q->dx + dx, q->dy + dy);
	    }
	    break;

	default: 
	/* Splines */
	    sptr = (SPLINE *) ptr->data;
	    pptr = &(sptr->head);
	    for (i = 0; i < sptr->numvert; i++) {
		XMIN (pptr[i].x + dx);
		XMAX (pptr[i].x + dx);
		YMIN (pptr[i].y + dy);
		YMAX (pptr[i].y + dy);
	    }
	    break;
    }
}


ScanPSM ( fp, p, dx, dy, cnt, printer , level)
	FILE *fp;
        ITEM_LIST_PTR *p;
	short *cnt;
	short dx,dy;
	short printer;
	short level;
{
    short   i;
    ITEM_DESCRIPTOR * ptr;
    ITEM_LIST_PTR * q;
    SPLINE * sptr;
    POINT * pptr;
    FONTDATA * fd;
    short   fn,
            fs,
            fj;

 /* Stop when we run out of data. */

    if (p == NULL)
	return;

    ptr = p->itemdesc;
    if (ptr == NULL)
	return;

    if (ptr->type == TextObj && ptr->typedata == 3) 
	return; /*Ignore PressEdit symbols in Postscript */

    if (printer) {
	dx += p->dx;
	dy += p->dy;
    }
    else {
	dx = 0;
	dy = 0;
    }

    if (ptr->filnum == 0 || printer) {
    /* Print new items, numbering them as we go. */
	if (ptr->filnum == 0)
	    ptr->filnum = ++(*cnt);

	if (!printer) {
	    for (i = 1; i<= level; i++)
		fprintf(fp,"\t");
	    fprintf (fp, "*G /G%d ", ptr->filnum);
	}
	switch (ptr->type) {
	    case TextObj: 
	    /* Text.  Print the text string. */
		fprintf (fp, "*T %d ", strlen (ptr->data));
		SlashString (ptr->data, fp);
		fd = &FontData[ptr->subtype];
		if (fd->psmfont == 0) {
		    printf ("font %s is not available in Postscript. Using Helvetica %d.\n\r", fd->fontname, fd->psmsize);
		    fn = 1;
		}
		else
		    fn = fd->psmfont;
		fs = fd->psmsize;
		fj = ptr->typedata;
		if (fn == oldfont && fs == oldsize && fj == oldjust &&
			1 == oldtype) {
		    switch (ptr->typedata) {
			case PositionLeft: 
			case 3: /* PressEdit */
			    fprintf (fp, " %d %d $ [] T ",
				    ptr->xmin + dx, ptr->ymin + dy);
			    break;
			case PositionCenter: 
			    fprintf (fp, " %d %d $ [[/J 1 %d %d]] T ",
				    ptr->xmin + dx, ptr->ymin + dy,
				    (ptr->xmin + ptr->xmax) / 2 + dx, ptr->ymin + dy);
			    break;
			case PositionRight: 
			    fprintf (fp, " %d %d $ [[/J 2 %d %d]] T ",
				    ptr->xmin + dx, ptr->ymin + dy,
				    ptr->xmax + dx, ptr->ymin + dy);
			    break;
		    }
		}
		else {
		    switch (ptr->typedata) {
			case PositionLeft: 
			case 3: /* PressEdit */
			    fprintf (fp, " %d %d %d %d [[/J 0 %d %d ][/F (%s)]] T ",
				    ptr->xmin + dx, ptr->ymin + dy, fn, fs,
				    ptr->xmin + dx, ptr->ymin + dy, fd->fontname);
			    break;
			case PositionCenter: 
			    fprintf (fp, " %d %d %d %d [[/J 1 %d %d ][/F (%s)]] T ",
				    ptr->xmin + dx, ptr->ymin + dy, fn, fs,
				    (ptr->xmin + ptr->xmax) / 2 + dx, ptr->ymin + dy, fd->fontname);
			    break;
			case PositionRight: 
			    fprintf (fp, " %d %d %d %d [[/J 2 %d %d ][/F (%s)]] T ",
				    ptr->xmin + dx, ptr->ymin + dy, fn, fs,
				    ptr->xmax + dx, ptr->ymin + dy, fd->fontname);
			    break;
		    }
		    oldtype = 1;
		    oldfont = fn;
		    oldsize = fs;
		    oldjust = fj;
		}
		break;

	    case GroupObj: 
		for (q = (ITEM_LIST_PTR *) ptr->data; q; q = q->next) {
		    ScanPSM (fp, q, dx, dy, cnt, printer, level+1);
		}
		break;

	    default: 
	    /* Splines.  Print the spline data. */
		sptr = (SPLINE *) ptr->data;
		fprintf (fp, "*S %d [", sptr->numvert);
		pptr = &(sptr->head);
		for (i = 0; i < sptr->numvert; i++)
		    fprintf (fp, "[%d %d]", pptr[i].x + dx, pptr[i].y + dy);
		if (oldtype == 2 &&
			oldspline.order == sptr->order &&
			oldspline.border == sptr->border &&
			oldspline.closed == sptr->closed &&
			oldspline.filled == sptr->filled &&
			(!oldspline.filled ||
			    (oldspline.opaque == sptr->opaque &&
				oldspline.pat == sptr->pat)) &&
			(!oldspline.border || oldspline.nib == sptr->nib))
		    fprintf (fp, "] $ S ");
		else {
		    fprintf (fp, "] %d %s %s %d %s %d %s [[/T %d %d]] S ",
			    sptr->order, boolstr[sptr->closed],
			    boolstr[sptr->border], sptr->nib, boolstr[sptr->filled],
			    sptr->pat, boolstr[sptr->opaque], ptr->type,
			    ptr->subtype);
		    oldtype = 2;
		    oldspline.order = sptr->order;
		    oldspline.border = sptr->border;
		    oldspline.closed = sptr->closed;
		    oldspline.filled = sptr->filled;
		    oldspline.opaque = sptr->opaque;
		    oldspline.pat = sptr->pat;
		    oldspline.nib = sptr->nib;
		}
		break;
	}
	if (!printer)
	    fprintf (fp, "() [] G ");
    }
    if (!printer)
	fprintf (fp, "*C %d %d G%d C\n", p->dx, p->dy, ptr->filnum);
    else
	fprintf (fp, "\n");
    return(1);
}

/*
 *  This routine will write the activelist into a PSM file.
 *  header determines whether the psm header is included at the beginning.
 *  dest =      PRINTER_DEST  	(to be directly printed)
 *		DOCUMENT_DEST 	(to be included as part of a text document)
 *		SAVE_DEST	(to be read back in to Draw or the VGTS)
 */
 
short WritePSM(header,dest,fname,xsize,ysize,pname)
short header,dest;
char *fname, *pname;
float *xsize,*ysize;
  {
    short count;		/* Reachable item count */
    FILE *fp, *fh, *fopen();
    ITEM_DESCRIPTOR *id;
    ITEM_LIST_PTR *p,*q;
    short xmin, xmax, ymin, ymax;
    short c;
    static char template[] = "/tmp/drawvpsmXXXXXXXX";
    static char tempname[] = "                     ";

    if (fname == NULL) {
	strcpy(tempname,template);
	mktemp(tempname);
	fp = fopen(tempname,"w");
        if (fp == NULL)  {
	    mprintf(2,"Error:  Couldn't open temporary file.");
	    return(0);
        }
    }
    else {
	fp = fopen(fname,"w");
        if (fp == NULL)  {
	    mprintf(2,"Error:  Couldn't open file '%s'.", fname);
	    free( fname );
	    return(0);
        }
    }
    xmin = ymin = 32700;
    xmax = ymax = -32700;
    for (p = activelist->prev; p; p = p->next )
      {
	ScanBBox( fp, p, &xmin, &xmax, &ymin, &ymax, p->dx, p->dy);
      }
    if (header) {
	if ((fh = fopen(HEADERFILE, "r")) == NULL) {
	    mprintf(2,"Error: Couldn't open header file '%s'.\n\r", HEADERFILE);
	    if (fname != NULL) free( fname );
	    return(0);
	}
        fputs("%!PS-Adobe-1.0\n", fp );
    }
    else
	fputs("%%PSM - DRAW 1.1 - V PostScript Metafile\n",fp);
	fputs("%%Creator: V system DRAW program\n",fp);
	fputs("%%Title: psm\n",fp);
	fputs("%%CreationDate: ",fp);
	{
	    char *ctime();
	    long clock,ticks;
#ifdef UNIX
	    long time();
	    clock = time(0);
#else
	    long GetTime();
	    clock = GetTime(&ticks);
#endif
	    fputs(ctime(&clock),fp);
	}
	fputs("%%Pages: 1\n",fp);
	fputs("%%DocumentFonts: (atend)\n",fp);
	switch(dest) {
	    case PRINTER_DEST: 
		fprintf(fp,"%%dimensions: %d %d %d %d\n",
			xmin,xmax,ymin,ymax);
		fputs("%%destination: Postscript printer\n",fp);
		break;
	    case DOCUMENT_DEST:
		fprintf(fp,"%%dimensions: %d %d %d %d\n",
			0,0,xmax-xmin,ymax-ymin);
		fputs("%%destination: Inclusion in Postscript document\n",fp);
		break;
	    case SAVE_DEST:
		fprintf(fp,"%%dimensions: %d %d %d %d\n",
			xmin,xmax,ymin,ymax);
		fputs("%%destination: PSM Save file\n",fp);
		break;
	}
	*xsize = (xmax-xmin)/72.0;
	*ysize = (ymax-ymin)/72.0;
	fprintf(fp,"%%Size: %fin %fin\n",*xsize,*ysize);
	fputs("%%EndComments\n",fp);
	if (header) {
	    while ((c=getc(fh)) != EOF) putc(c,fp);
	    fclose(fh);
	    fputs("%%EndProlog\n",fp);
	}
	if (dest==DOCUMENT_DEST)
	    fprintf(fp,"psm begin\n");
    
    /* We have a file to write.  Reset all of the file numbers. */
    if (Debug&DebugIO)
	printf("Resetting file numbers in itemlist.\n\r");
    id = itemlist;
    while (id) {
	id->filnum = 0;
	id = id->prev;
      }
    
    /* Scan and print all of the items referenced by the activelist. */
    count = 1;
    p = activelist->prev;
    oldtype = -1;
    fprintf( fp, "*G /G1\n");
    for ( ; p; p = p->next )
        switch(dest) {
	    case PRINTER_DEST: 
		ScanPSM( fp, p, 0, 0, &count, 1, 0 );
		break;
	    case DOCUMENT_DEST:
	        ScanPSM( fp, p, 306-(xmax+xmin)/2,-ymin, &count, 1, 0 );
		/* 306 = 8.5 (inches) * 72 (dots/inch) / 2 (center) */
		break;
	    case SAVE_DEST:
		ScanPSM( fp, p, 0, 0, &count, 0, 0 );
		break;
        }
    fprintf( fp, "(Top Level Symbol) [] G\n");
    fprintf( fp, "*C 0 0 G1 C\n");
    if (dest == PRINTER_DEST)
	fprintf( fp, "*P P\n");
    if (dest==DOCUMENT_DEST)
	fprintf(fp,"end\n");
    
    /* Close the file and finish up. */
    fclose( fp );
    if (fname != NULL) 
        free( fname );
    else {
	char pname2[50];
	FILE *pfile[2];
#ifdef UNIX
	sprintf(pname2,"-P%s",pname);
	mprintf(1,"spooling print job...");
	if (vfork()==0) { 
	    execlp (LPR, LPR, "-r", pname2, tempname, 0);
            mprintf (2,"Couldn't execute %s.",LPR);
	    _exit();
	}
#else
	static char *argv [] = {
	    "-r",
	    "                           ",
	    "/tmp/drawbruce XXXXXX      ",
	    0};
	sprintf(argv[1],"-P%s",pname);
	mprintf(1,"spooling print job...");
	strcpy(argv[2],tempname);
	if (RemoteExecute(pfile,LPR,argv,FREAD)!=OK) {
            mprintf (2,"Couldn't execute %s on the Vax.",LPR);
	    return(0);
	}
#endif
    }
    return (1);
}

/*
 * This routine writes out a string using slash excapes when necessarry.
 */

SlashString(s,f)
char *s;
FILE *f;
{
   char c;
   fputc('(',f);
   while ((c = *s++)!='\0') {
	switch(c) {
	    case '\n': fputs("\\n",f); break;
	    case '\r': fputs("\\r",f); break;
	    case '\t': fputs("\\t",f); break;
	    case '(' : fputs("\\(",f); break;
	    case ')' : fputs("\\)",f); break;
	    case '\\': fputs("\\\\",f); break;
	    case '\b': fputs("\\b",f); break;
	    case '\f': fputs("\\f",f); break;
	    default:
		if (c>=' ') putc(c,f);
		else        printf("\\o%03o",(int)c);
	}
    }
    fputc(')',f);
}
