/*
 *  backend.c
 *
 *  This is the "back end" of draw.
 *
 *  This file contains the low level data structure manipulation routines
 *  which create, delete, and manipulate objects.
 *
 *  David Kaelbling, April 1983
 *  Split factorx and factory for scale vs stretch in ScaleObject - GAF 5/15/85
 *  Split off menu and quit functions from misc.c - GAF 5/18/85
 */
 
 
/* Includes */
#ifdef UNIX
#include "stdio.h"
#else
#include "Vio.h"
#endif
#include "Vgts.h"
#include "splines.h"
#include "draw.h"

 
/* Imports */
extern DebugItemDescriptor();
extern DebugObjectList();
extern DebugLinks();
extern SetCurrentObject();
extern double sin();
extern double cos();
extern ITEM_LIST_PTR *backcobject[];
extern short GetINum();		/* Get an item number */
extern short GetSeqINum();	/* Get a sequential set of item numbers */
extern short LastINum();	/* Return the last item number gotten
extern short FreeINum();	/* Return an item number to the freelist */
extern InitFrame();		/* Init the main symbol & frame */
 
 
/* Exports */
extern RedrawActivelist();	/* Redisplay the main drawing area. */
extern NewObject();		/* Add a new object to the display list */
extern RevertToCKP();		/* Back off a transaction. */
extern DecrementRef();		/* Decrement reference count on an item */
extern Checkpoint();		/* Make a checkpoint. */
extern NewGroup();		/* Create a new group name */
extern RotateObject();		/* Rotate an object about a point. */
extern ScaleObject();		/* Expand/Contract an object about a point. */
extern CopyObject();		/* Make a copy of an object. */
extern MoveObject();		/* Move an object. */
extern EraseObject();		/* Erase (delete) an object. */
extern LowerObject();		/* Lower an object. */
extern RaiseObject();		/* Raise an object. */
extern AttribObject();		/* set nib, fill, and opaque attributes. */
extern LoadFont();		/* loads a font */
extern short MainSymbolOpen;	/* 1 if main symbol is being edited */
    
/* Local definitions */
static int CopyCount = 0;	/* Unique tag for copied group names. */
#define ArrowChar	'\031'	/* Displays as a right-arrow character. */

/*
 *  This routine opens the main symbol if not already open. This is used
 *  to avoid redundant refreshesof the screen.
 */
OpenMainSymbol()
  {
    if (!MainSymbolOpen) {
	EditSymbol( sdf, mainSymbol);
	MainSymbolOpen = 1;
    }
  }

/*
 * This is the counterpart to OpenMainSymbol. It closes the main symbol if 
 * not already closed. In this way, redundant calls to this routine from 
 * various parts of the program don't hurt.
 */
CloseMainSymbol(refresh)
    short refresh;

  {
    if (MainSymbolOpen) {
        EndSymbol( sdf, mainSymbol, (refresh ? mainVgt : 0) );
	MainSymbolOpen = 0;
    }
  }
	
/*
 *  This internal routine will add a single object to the display.
 */
DisplayObject( q, inum)
	ITEM_LIST_PTR *q;
	short inum;
  {
    /* Add the item to the current display list. */
    OpenMainSymbol();
    if (inum <=0)
	inum = GetINum();
    q->call = AddCall( sdf, inum, q->dx, q->dy, q->itemdesc->symbol );
  }

/*
 *  This internal routine will replace an object in the display with an updated
 *  version
 */
ReplaceObject( q, refresh)
	ITEM_LIST_PTR *q;
	short refresh;
  {
    /* Add the item to the current display list. */
    OpenMainSymbol();
    DeleteItem(sdf,q->call);
    q->call = AddCall( sdf, q->call, q->dx, q->dy, q->itemdesc->symbol );
  }

/*
 *  This internal routine is called each time a part is created.
 *  It will allocate a symbol for the new part.
 */
  
CreatePart( id, refresh )
    register ITEM_DESCRIPTOR *id;	/* Optimize dereferencing. */
    short refresh;
  {
    ITEM_LIST_PTR *g;		/* Pointer for following group chains */
    
    /* Bomb Proofing */
    if (id == NULL)
      {
	printf("Internal Error:  CreatePart on a NULL Item.\n\r");
	return;
      }
    CloseMainSymbol(refresh);
    if (id->type == TextObj)
      {
	/* Text. */
	LoadFont(id->subtype);
	id->symbol = DefineSymbol( sdf, GetINum(), "new text symbol" );
        id->number = AddItem( sdf, GetINum(), id->xmin, id->xmax,
		id->ymin, id->ymax,
		FontData[id->subtype].refnumber, SDF_TEXT, id->data );
	EndSymbol( sdf, id->symbol, 0 );
      }
    else if (id->type != GroupObj)
      {
	/* A spline of some sort. */
	id->symbol = DefineSymbol( sdf, GetINum(), "new spline symbol" );
	
	/* Since this is a new object, get bounding box & number. */
	id->number = AddItem( sdf, GetINum(), 0, 0, 0, 0,
		0, SDF_SPLINE, id->data );
	InquireItem( sdf, id->number, &(id->xmin), &(id->xmax),
	    	&(id->ymin), &(id->ymax), 0, 0, 0 );
	EndSymbol( sdf, id->symbol, 0 );
      }
    else
      {
	/* Construct the group */
	id->symbol = DefineSymbol( sdf, GetINum(), "new group symbol" );
	g = (ITEM_LIST_PTR *) id->data;
	while (g)
	  {
	    AddCall( sdf,GetINum(), g->dx, g->dy, g->itemdesc->symbol );
	    g = g->next;
	  }
	EndSymbol( sdf, id->symbol, 0 );
      }
  }

/*
 * LoadFont loads the specified font into memory. Returns 0 on error
 */

LoadFont(WhichFont)
short WhichFont;

{
    /* load font if necessary */
    if (!(FontData[WhichFont].loaded)) {
	mprintf(1,"Loading font '%s', please wait ..",
	       FontData[WhichFont].fontname);
	FontData[WhichFont].refnumber =
	    DefineFont( FontData[WhichFont].fontname, NULL );
        if ((FontData[WhichFont].refnumber < 0) ||
	    (FontData[WhichFont].refnumber == 255)) {
	    mprintf(2,"Error. Can't load font %s.\n\r",
		FontData[WhichFont].fontname);
	    return(0);
	}
        else {
            FontData[WhichFont].loaded = 1;
            mprintf(1,"Font '%s' loaded.",FontData[WhichFont].fontname);
            return(1);
	}
    }
    else {
	return(1);
    }
}

/*
 *  This routine will rebuild the Groups data structure, which consists
 *  of a list of pointers to all groups reachable throught the activelist.
 */
 
RebuildGroups()
  {
    register ITEM_LIST_PTR *p;
    register int numgr;
    
    /* Scan through the active list. */
    for (numgr = 0, p = activelist->prev;  p;  p = p->next)
      {
	/* Is this a group? */
	if (p->itemdesc->type == GroupObj)
	  {
	    /* Yes.  Attempt to cache the pointer to it. */
	    if (numgr < MAXGROUP)
	      {
		Groups[numgr++] = p;
	      }
	    else
	      {
		printf("Internal Error: Too many groups to rebuild cache.\n\r");
		return;
	      }
	  }
      }
    
    /* Wipe the rest of the entries. */
    while (numgr < MAXGROUP)
      {
	Groups[numgr++] = NULL;
      }
  }

/*
 *  This routine will wipe the drawing area and redisplay the
 *  activelist from scratch.
 */
RedrawActivelist( refresh )
	register short refresh;
  {
    register ITEM_LIST_PTR *p;
    
    /* Delete the old drawing area. */
    CloseMainSymbol(0);
    DeleteSymbol( sdf, mainSymbol );
    
    /* Create the new one. */
    InitFrame(FALSE);
    if (Debug&DebugBackend)
	DebugObjectList( "Redraw activelist", activelist, 0 );
    
    /* Update the Vgts */
    for (p = activelist->prev;  p;  p = p->next)
      {
	DisplayObject( p, p->call );
      }
    CloseMainSymbol(0);
    /* Update the screen */
    if (refresh)
	DisplayItem( sdf, mainSymbol, mainVgt );
  }

/*
 *  Add a new object to the active display list.  Note that there are
 *  several display lists -- those used by RevertToCKP to back off a 
 *  set of updates, and the the current list (activelist).
 */
 
NewObject( dataptr, typeP, subtypeP, td, x1, x2, y1, y2, b, nameP,
	   refresh, deltax, deltay )
	char *dataptr;			/* New itemdesc->data entry. */
	enum ObjTypes typeP;		/* itemdesc->type entry. */
	short subtypeP, td;		/* subtype and typedata fields. */
	short x1, x2, y1, y2, b;	/* Bounding box and baseline */
	char *nameP;			/* name field. */
	short refresh, deltax, deltay;
  {
    ITEM_LIST_PTR *ptr;
    ITEM_DESCRIPTOR *id;
    
    /* We have a new current object.  Make it so. */
    if ((id = (ITEM_DESCRIPTOR *) malloc( sizeof(ITEM_DESCRIPTOR) )) == NULL)
      {
	printf("Ran out of memory -- couldn't create a new object.\n\r");
	if (dataptr)
	    free( dataptr );
	return;
      }
    
    if (typeP == TextObj && subtypeP <0)
	subtypeP=1;
    /* Initialize all the fields, append to main item list */
    if (itemlist)
	itemlist->next = id;
    id->next = NULL;
    id->prev = itemlist;
    itemlist = id;
    id->data = dataptr;
    id->type = typeP;
    id->subtype = subtypeP;
    id->typedata = td;
    id->number = 0;
    id->base = b;
    id->name = nameP;
    id->refs = 1;
    id->xmin = x1;
    id->xmax = x2;
    id->ymin = y1;
    id->ymax = y2;
    
    /* Append to active list */
    if ((ptr = (ITEM_LIST_PTR *) malloc( sizeof(ITEM_LIST_PTR) )) == NULL)
      {
	printf("Out of memory.  Couldn't allocate item list pointer.\n\r");
	if (dataptr)	free( dataptr );
	if (nameP)	free( nameP );
	free( id );
	SetCurrentObject(NULL,1);
	return;
      }
    ptr->prev = activelist->next;
    ptr->itemdesc = id;
    ptr->next = NULL;
    ptr->dx = deltax;
    ptr->dy = deltay;
    if (activelist->prev == NULL)
	activelist->prev = ptr;
    if (activelist->next)
	(activelist->next)->next = ptr;
    activelist->next = ptr;
    CreatePart( id, refresh );
    DisplayObject( ptr, GetINum() );
    SetCurrentObject(ptr,refresh);
    if (Debug&DebugBackend)
      {
	printf("NewObject:  Just created\n\r");
	DebugItemDescriptor( id, 0 );
	DebugLinks( activelist );
      }
  }

/*
 *  This routine will decrement the reference count for an entry
 *  in the main item descriptor list.  It will, if necessary,
 *  also free the storage associated with unused items.
 */
 
DecrementRef( q )
	register ITEM_DESCRIPTOR *q;
  {
    ITEM_LIST_PTR *p, *ptmp;
    
    if (q == NULL)
	return;

    /* Do the easy part.  Decrement the reference count. */
    q->refs -= 1;
    if (q->refs > 0)
	return;
	
    /* Free the storage */
    if (q->type == GroupObj)
      {
	/* Decrement the count for items in the group recursively */
	p = (ITEM_LIST_PTR *) q->data;
	while (p)
	  {
	    DecrementRef( p->itemdesc );
	    ptmp = p;
	    p = p->next;
	    ptmp->next = (ITEM_LIST_PTR *) -9;	/* Crowbar */
	    free( ptmp );
	  }
      }
    else if (q->data)
	free( q->data );
    q->data = (char *) -11;
    if (q->name)
	free( q->name );
    q->name = (char *) -13;
    /* Fix up the links */
    if (q->prev)
	q->prev->next = q->next;
    if (q->next)
	q->next->prev = q->prev;
    if (itemlist == q)
	itemlist = q->prev;
    q->prev = (ITEM_DESCRIPTOR *) -15;
    q->next = (ITEM_DESCRIPTOR *) -17;
    CloseMainSymbol(0);
    DeleteSymbol( sdf, q->symbol );
/* #if 0 */
    OpenMainSymbol();
    DeleteItem( sdf, q->symbol );
    FreeINum(q->symbol);
/* #endif */
    FreeINum(q->number);    
    free( q );
  }

/*
 *  Checkpoint switches backup display lists, making a backup copy of
 *  the activelist, and freeing any unreferenced objects in the central
 *  object list.
 */
    
Checkpoint()
  {
    register ITEM_LIST_PTR *p, *ptmp, *pold;
    
    /* Do the dirty bit */
    modified = 1;
    
    /* Deallocate the old backup list, freeing unused memory. */
    CurrentBackup = (CurrentBackup + 1) % NumBackup;
    if (Debug&DebugBackend)
      {
	printf("Checkpoint:  Deallocating backuplist %d\n\r", CurrentBackup);
	DebugLinks( backuplists[CurrentBackup] );
      }
    p = backuplists[CurrentBackup]->prev;
    while (p)
      {
	ptmp = p->next;
	DecrementRef( p->itemdesc );
	if (p->call > 0)
	    FreeINum(p->call);
	free( p );
	p = ptmp;
      }
    
    if (Debug&DebugBackend)
      {
	printf("Activelist:\n\r");
	DebugLinks( activelist );
      }
    
    /* Build the new backup list */
    backcobject[CurrentBackup] = NULL;
    backuplists[CurrentBackup]->prev = NULL;
    p = activelist->prev;
    pold = NULL;
    while (p)
      {
	if ((ptmp = (ITEM_LIST_PTR *) malloc( sizeof(ITEM_LIST_PTR) ))
		== NULL)
	  {
	    printf("Out of memory.  Can't make a checkpoint.\n\r");
	    backuplists[CurrentBackup]->next = pold;
	    return;
	  }
	if (pold == NULL)
	    backuplists[CurrentBackup]->prev = ptmp;
	ptmp->prev = pold;
	if (pold)
	    pold->next = ptmp;
	
	/* Copy over the data */
	ptmp->itemdesc = p->itemdesc;
	p->itemdesc->refs += 1;
	ptmp->dx = p->dx;
	ptmp->dy = p->dy;
	ptmp->next = NULL;
	ptmp->call = 0;
        if (p == CurrentObject) backcobject[CurrentBackup] = ptmp;
	
	pold = ptmp;
	p = p->next;
      }
    backuplists[CurrentBackup]->next = pold;
  }

/*
 *  RevertToCKP will abort a transaction, reverting back to the previous
 *  backup list.
 */
 
RevertToCKP()
  {
    register ITEM_LIST_PTR *tmp,*tmp2;
    
    /* Set the dirty bit */
    modified = 1;
    
    /* Swap display lists */
    tmp2=CurrentObject;
    SetCurrentObject(NULL,0);
    tmp = activelist;
    activelist = backuplists[CurrentBackup];
    backuplists[CurrentBackup] = tmp;
    /* Display this version of the main drawing area */
    RebuildGroups();
    RedrawActivelist( 1 );
    SetCurrentObject(backcobject[CurrentBackup],1);
    backcobject[CurrentBackup]=tmp2;
    CurrentBackup = (CurrentBackup + NumBackup - 1) % NumBackup;
  };

/*
 *  This procedure will update the object list to reflect the creation
 *  of a new group.  It will also update the active group list.  If
 *  this is an original group (i.e., not a copy of an existing group),
 *  then items are transfrerred off of the activelist.
 */
 
NewGroup( q, original, refresh )
	ITEM_LIST_PTR *q;
	short original, refresh;
  {
    ITEM_LIST_PTR *gptr;
    
    if (Debug&DebugBackend)
	printf("New Group:  %d, original %d, refresh %d\n\r",
		q, original, refresh );
    
    /* Update the active list */
    gptr = (ITEM_LIST_PTR *) q->itemdesc->data;
    if (original)
      {
	/* For an original group, move objects off of the active list. */
        while (gptr)
          {
	    ITEM_LIST_PTR *tmp;
	
	    /* Verify that items are on activelist */
	    tmp = activelist->prev;
	    while ((tmp) && (tmp->itemdesc != gptr->itemdesc))
	        tmp = tmp->next;
	    if (tmp == NULL)
	      {
		/* Only groups can be specified when not top level items */
		if (gptr->itemdesc->type != GroupObj)
		  {
		    printf("Internal Error:  Newgroup item not active.\n\r");
		    DebugItemDescriptor( gptr->itemdesc, 0 );
		  }
		else
		    gptr->itemdesc->refs += 1;
	      }
	    else
	      {
		/* Update pointers within activelist */
		if (tmp->prev)
		    tmp->prev->next = tmp->next;
		else
		    activelist->prev = tmp->next;
		if (tmp->next)
		    tmp->next->prev = tmp->prev;
		else
		    activelist->next = tmp->prev;
        	free( tmp );
	      }
	    gptr = gptr->next;
	  }
      }
    else
      {
	/* Update the reference counts for a non-original group. */
	while (gptr)
	  {
	    gptr->itemdesc->refs += 1;
	    gptr = gptr->next;
	  }
      }
    
    /* Creating a group can raise/lower objects within it. */
    RebuildGroups();
    if (refresh)
	RedrawActivelist( 1 );
  }

/*
 *  This routine will delete an object from the activelist and from
 *  the screen.
 */
 
EraseObject( victim, refresh )
	ITEM_LIST_PTR *victim;
	short refresh;
  {
    register short rebuild;
    
    /* Print Debugging Information */
    if (Debug&DebugBackend)
      {
	printf("Erase Object:\n\r");
	DebugItemDescriptor( victim->itemdesc, 0 );
	DebugLinks( activelist );
      }
    
    /* Update the screen */
    OpenMainSymbol();
    DeleteItem( sdf, victim->call );
    FreeINum(victim->call);

    /* Remove any dangling pointers. */
    SetCurrentObject(NULL,1);
    rebuild = (victim->itemdesc->type == GroupObj);
    
    /* Update the activelist */
    DecrementRef( victim->itemdesc );
    if (victim->prev)
	victim->prev->next = victim->next;
    if (victim->next)
	victim->next->prev = victim->prev;
    if (activelist->prev == victim)
	activelist->prev = victim->next;
    if (activelist->next == victim)
	activelist->next = victim->prev;
    /* The next three lines are crowbars */
    victim->itemdesc = (ITEM_DESCRIPTOR *) -3;
    victim->prev = (ITEM_LIST_PTR *) -5;
    victim->next = (ITEM_LIST_PTR *) -7;
    free( victim );
    if (rebuild)
	RebuildGroups();
  }

/*
 *  This primitive will raise an object to the top of the display.
 *  This operation is useful when opaque filling is used.
 */
 
RaiseObject( id, refresh )
	ITEM_LIST_PTR *id;
	short refresh;
  {
    if (Debug&DebugBackend)
      {
	printf("Raise object\n\r");
	DebugItemDescriptor( id->itemdesc, 0 );
	DebugLinks( activelist );
      }
    
    if (activelist->next == NULL)
      {
	printf("Internal Error:  Null end cell in activelist.\n\r");
	return;
      }
      
    /* If it's already there, don't bother playing with the links. */

    if (activelist->next == id)
      {
	if (refresh) {
	    CloseMainSymbol(0);
	    DisplayItem( sdf, mainSymbol, mainVgt );
 	}
        SetCurrentObject(id,refresh);
	return;
      }
	
    /* Update the links in the activelist */
    if (id->prev)
        id->prev->next = id->next;
    else
	activelist->prev = id->next;
    if (id->next)
	id->next->prev = id->prev;
    else
	printf("Internal Error:  Raise already raised object.\n\r");
    activelist->next->next = id;
    id->prev = activelist->next;
    id->next = NULL;
    activelist->next = id;
    SetCurrentObject(id,1);
    /* Redisplay? */
    if (refresh)
      {
	if (id->itemdesc->type == GroupObj)
	  {
	    /* Lots of work.  Might as well start from scratch. */
	    RedrawActivelist( 1 );
	  }
	else
	  {
	    /* Not so much work.  Delete and add the item back. */
	    ReplaceObject( id, 1 );
	  }
      }
  }

/*
 *  This routine will change the display offset of an object in the
 *  activelist.  It is assumed that a call to RaiseObject will be made
 *  in conjunction with a call to this routine, as the Vgts will
 *  implicitly raise a modified item to the top of its display list.
 */
 
MoveObject( p, deltax, deltay, refresh )
	ITEM_LIST_PTR *p;
	short deltax, deltay;
	short refresh;
  {
    /* Don't die because of bad pointers. */
    if (p == NULL)
      {
	printf("Internal Error:  Move NULL object.\n\r");
	return;
      }
    
    /* Debugging information requested? */
    if (Debug&DebugBackend)
      {
	printf("Move Object %x by (%d, %d)\n\r", p, deltax, deltay );
	DebugItemDescriptor( p->itemdesc, 0 );
	DebugLinks( activelist );
      }
    
    /* Do the actual work */
    p->dx += deltax;
    p->dy += deltay;
    SetCurrentObject(NULL,refresh);
    if (refresh)
      if (p->itemdesc->type != GroupObj)
        {
    	  /* Move a single item.  Try incremental screen updates. */
	  ReplaceObject( p, 1);
        }
      else
        {
	  /* Move a group.  Lots of work. */
	  RedrawActivelist( 1 );
        }
    SetCurrentObject(p,refresh);
  }

/*
 *  This primitive will lower an object to the bottom of the display list.
 *  This operation is useful when opaque filling is used.
 */
 
LowerObject( id, refresh )
	ITEM_LIST_PTR *id;
	short refresh;
  {
    if (Debug&DebugBackend)
      {
	printf("Lower object\n\r");
	DebugItemDescriptor( id->itemdesc, 0 );
	DebugLinks( activelist );
      }
    
    if (activelist->prev == NULL)
      {
	printf("Internal Error:  Null start cell in activelist.\n\r");
	return;
      }
      
    /* If it's already there, don't bother playing with the links. */
    if (activelist->prev == id)
      {
	if (refresh)
	    DisplayItem( sdf, mainSymbol, mainVgt );
        SetCurrentObject(NULL,refresh);
	return;
      }
	
    /* Update the links in the activelist */
    if (id->prev)
        id->prev->next = id->next;
    else
	printf("Internal Error:  Lower already lowered object.\n\r");
    if (id->next)
	id->next->prev = id->prev;
    else
	activelist->next = id->prev;
    activelist->prev->prev = id;
    id->next = activelist->prev;
    id->prev = NULL;
    activelist->prev = id;
    SetCurrentObject(NULL,refresh);
    /* Redisplay? */
    if (refresh)
      {
	/* Lots of work.  Might as well start from scratch. */
	RedrawActivelist( 1 );
      }
  }

/*
 *  This routine will copy an object.  Groups are copied with a shallow
 *  copy, so changing the elements of the root group will change all of
 *  its instances.  Other objects are copied with a deep copy, so each
 *  copy is independent.
 */
 
CopyObject( p, deltax, deltay, refresh )
	ITEM_LIST_PTR *p;
	short deltax, deltay;
	short refresh;
  {
    register ITEM_DESCRIPTOR *id;
    char *string;
    SPLINE *sptr, *newsptr;
    POINT *pptr, *newpptr;
    register short i;
    ITEM_LIST_PTR *idptr;
    
    /* Do a little checking */
    if ((p == NULL) || (p->itemdesc == NULL))
      {
	printf("Internal Error:  Bad pointers to CopyObject.\n\r");
	return;
      }
    
    /* Print debugging information */
    if (Debug&DebugBackend)
      {
	printf("Copy Object %x with translation (%d, %d)\n\r",
		p, deltax, deltay );
	DebugItemDescriptor( p->itemdesc, 0 );
      }
    
    /* For non-groups, just invoke NewObject() properly. */
    id = p->itemdesc;
    if (id->type == TextObj)
      {
	/* Text.  Make a new copy of the string, and use new offsets. */
	if ((string = (char *) malloc( MAXLEN + 1 )) == NULL)
	  {
	    printf("Out of memory.  Can't copy any text.\n\r");
	    return;
	  }
	for (i = MAXLEN + 1;  i--;)
	    string[i] = '\0';
	strcpy( string, id->data );
	NewObject( string, TextObj, id->subtype, id->typedata,
		id->xmin + deltax + p->dx, id->xmax + deltax + p->dx,
		id->ymin + deltay + p->dy, id->ymax + deltay + p->dy,
		id->base, NULL, refresh, 0, 0 );
      }
    else if (id->type != GroupObj)
      {
	/* Splines.  Copy spline data structure. */
	sptr = (SPLINE *) id->data;
	if ((newsptr = (SPLINE *) malloc( sizeof(SPLINE) +
		sptr->numvert * sizeof(POINT) )) == NULL)
	  {
	    printf("Out of memory.  Can't copy a spline-type object.\n\r");
	    return;
	  }
	newsptr->order = sptr->order;
	newsptr->numvert = sptr->numvert;
	newsptr->nib = sptr->nib;
	newsptr->border = sptr->border;
	newsptr->closed = sptr->closed;
	newsptr->filled = sptr->filled;
	newsptr->opaque = sptr->opaque;
	newsptr->pat = sptr->pat;
	
	pptr = &(sptr->head);
	newpptr = &(newsptr->head);
	for (i = 0;  i < sptr->numvert;  i++)
	  {
	    newpptr[i].x = pptr[i].x + deltax + p->dx;
	    newpptr[i].y = pptr[i].y + deltay + p->dy;
	  }
	NewObject( newsptr, id->type, 0, 0, 0, 0, 0, 0, 0, NULL,
		refresh, 0, 0 );
      }
    else
      {
	/* Check that this is possible. */
	if (Groups[MAXGROUP - 1])
	  {
	    printf("Sorry, no room for more groups.\n\r");
	    return;
	  }
	  
	/* A group.  Construct a new group with old group as contents. */
	if ((string = (char *) malloc ( MAXLEN + 1 )) == NULL)
	  {
	    printf("Out of memory.  Can't copy group name.\n\r");
	    return;
	  }
	for (i = MAXLEN + 1;  i--;)
	    string[i] = '\0';
	sprintf( string, "%s, copy %d", id->name, CopyCount++ );
	if ((idptr = (ITEM_LIST_PTR *) malloc(sizeof(ITEM_LIST_PTR))) == NULL)
	  {
	    printf("Out of memory.  Can't copy group contents.\n\r");
	    free( string );
	    return;
	  }
	idptr->prev = NULL;
	idptr->next = NULL;
	idptr->itemdesc = id;
	idptr->flag = 0;
	idptr->dx = p->dx;
	idptr->dy = p->dy;
	idptr->call = 0;
	NewObject( idptr, GroupObj, 0, CopyCount, 0, 0, 0, 0, 0,
		string, 0, deltax, deltay );
	NewGroup( CurrentObject, 0, refresh );
      }
  }

/*
 *  This internal routine is a helper routine for RotateObject().
 *  It will rotate a single point about (x0, y0), given the sine and
 *  cosine of the angle.
 */
 
RotatePoint( x, y, sine, cosine, x0, y0 )
	short *x, *y;
	double sine, cosine;
	short x0, y0;
  {
    register short newx, newy;
    
    if (Debug&DebugBackend)
	printf("    (%d, %d) rotates to ", *x, *y );
    
    /* Compute the distance of the point from the center of rotation. */
    newx = x0 + (cosine * (*x - x0) + sine * (*y - y0));
    newy = y0 + (cosine * (*y - y0) - sine * (*x - x0));
    *x = newx;
    *y = newy;
    
    if (Debug&DebugBackend)
	printf(" (%d, %d)\n\r", *x, *y );
  }

/* 
 *  This internal routine is a helper routine for ScaleObject().
 *  It will scale a single point about (x0, y0), given the scale factor.
 */
 
ScalePoint( x, y, factorx, factory, x0, y0 )
	short *x, *y;
	double factorx, factory;
	short x0, y0;
  {
    register short newx, newy;
    
    if (Debug&DebugBackend)
	printf("    (%d, %d) scales to ", *x, *y );
    
    /* Scale the point. */
    newx = ((*x - x0) * factorx) + x0;
    newy = ((*y - y0) * factory) + y0;
    *x = newx;
    *y = newy;
    
    if (Debug&DebugBackend)
	printf(" (%d, %d)\n\r", *x, *y );
  }

/*
 *  This routine will transform and raise an object.  Currently, the two
 *  transformation are rotation about a point, and scaling.
 */
 
TransformObject( p, fact1, fact2, x0, y0, dx, dy, refresh, erase, ptrans )
	ITEM_LIST_PTR *p;
	double fact1, fact2;	/* Point Transform factors. */
	short x0, y0;		/* Center of transformation. */
	short dx, dy;		/* Cumulative offset. */
	short refresh, erase;
	register (*ptrans)();	/* Point transformation routine. */
  {
    register ITEM_DESCRIPTOR *id;
    char *string;
    SPLINE *sptr, *newsptr;
    POINT *pptr, *newpptr;
    short i, x, y;
    ITEM_LIST_PTR *idptr, *gptr, *newgptr;
    
    
    /* For non groups, transform about the proper point. */
    id = p->itemdesc;
    if (id->type == TextObj)
      {
	/* Text.  Transform about the alignment point. */
	if ((string = (char *) malloc( MAXLEN + 1 )) == NULL)
	  {
	    printf("Out of memory.  Can't transform any text.\n\r");
	    return;
	  }
	for (i = MAXLEN + 1;  i--;)
	    string[i] = '\0';
	strcpy( string, id->data );
	switch (id->typedata)
	  {
	    /* Transform about the positioning point. */
	    case PositionLeft:
		x = id->xmin + p->dx + dx;
		y = id->ymin + p->dy + dy;
		ptrans( &x, &y, fact1, fact2, x0, y0 );
		x = (id->xmin + p->dx + dx) - x;
		y = (id->ymin + p->dy + dy) - y;
		break;
	    
	    case PositionCenter:
		x = ((id->xmin + id->xmax) / 2) + p->dx + dx;
		y = id->ymin + p->dy + dy;
		ptrans( &x, &y, fact1, fact2, x0, y0 );
		x = (((id->xmin + id->xmax) / 2) + p->dx + dx) - x;
		y = (id->ymin + p->dy + dy) - y;
		break;
	    
	    case PositionRight:
		x = id->xmax + p->dx + dx;
		y = id->ymin + p->dy + dy;
		ptrans( &x, &y, fact1, fact2, x0, y0 );
		x = (id->xmax + p->dx + dx) - x;
		y = (id->ymin + p->dy + dy) - y;
		break;
	  }
	dx -= x;
	dy -= y;
	if (erase)
	    EraseObject( p, refresh );
	NewObject( string, TextObj, id->subtype, id->typedata,
		id->xmin + p->dx + dx, id->xmax + p->dx + dx,
		id->ymin + p->dy + dy, id->ymax + p->dy + dy,
		id->base, NULL, refresh, 0, 0 );
      }
    else if (id->type != GroupObj)
      {
	/* Splines.  Transform all of the points. */
	sptr = (SPLINE *) id->data;
	if ((newsptr = (SPLINE *) malloc( sizeof(SPLINE) +
		sptr->numvert * sizeof(POINT) )) == NULL)
	  {
	    printf("Out of memory.  Can't transform a spline object.\n\r");
	    return;
	  }
	newsptr->order = sptr->order;
	newsptr->numvert = sptr->numvert;
	newsptr->nib = sptr->nib;
	newsptr->border = sptr->border;
	newsptr->closed = sptr->closed;
	newsptr->filled = sptr->filled;
	newsptr->opaque = sptr->opaque;
	newsptr->pat = sptr->pat;
	
	pptr = &(sptr->head);
	newpptr = &(newsptr->head);
	for (i = 0;  i < sptr->numvert;  i++)
	  {
	    newpptr[i].x = pptr[i].x + dx + p->dx;
	    newpptr[i].y = pptr[i].y + dy + p->dy;
	    ptrans( &(newpptr[i].x),&(newpptr[i].y),fact1,fact2,x0,y0 );
	  }
	if (erase)
	    EraseObject( p, refresh );
	NewObject( newsptr, id->type, 0, 0, 0, 0, 0, 0, 0, NULL,
		refresh, 0, 0 );
      }
    else
      {
	/* Groups.  Transform all of the objects within the group. */
	if ((string = (char *) malloc ( MAXLEN + 1 )) == NULL)
	  {
	    printf("Out of memory.  Can't transform group name.\n\r");
	    return;
	  }
	for (i = MAXLEN + 1;  i--;)
	    string[i] = '\0';
	strcpy( string, id->name );
	
	/* Transform the contents. */
	gptr = (ITEM_LIST_PTR *) id->data;
	if ((newgptr = (ITEM_LIST_PTR *) malloc( sizeof(ITEM_LIST_PTR) ))
	    == NULL)
	  {
	    printf("Out of memory.  Can't transform group contents.\n\r");
	    free( string );
	    return;
	  }
	newgptr->prev = NULL;
	newgptr->next = NULL;
	dx += p->dx;
	dy += p->dy;
	while (gptr)
	  {
	    TransformObject( gptr, fact1, fact2, x0, y0,
		dx + gptr->dx, dy + gptr->dy, 0, 0, ptrans );
	    if ((idptr = (ITEM_LIST_PTR *) malloc( sizeof(ITEM_LIST_PTR) ))
		== NULL)
	      {
		printf("Out of memory.  Can't transform a group.\n\r");
		return;
	      }
	    idptr->itemdesc = CurrentObject->itemdesc;
	    idptr->dx = 0;
	    idptr->dy = 0;
	    idptr->call = 0;
	    idptr->flag = 0;
	    idptr->next = NULL;
	    if (newgptr->next)
		newgptr->next->next = idptr;
	    else
		newgptr->prev = idptr;
	    idptr->prev = newgptr->next;
	    newgptr->next = idptr;
	    gptr = gptr->next;
	  }

	if (erase)
	    EraseObject( p, 0 );
	RedrawActivelist( 0 );
	NewObject( newgptr->prev, GroupObj, 0, 0, 0, 0, 0, 0, 0,
		string, 0, 0, 0 );
	NewGroup( CurrentObject, 1, 0 );
	if (refresh)
	    RedrawActivelist( 1 );
      }
  }

/*
 *  This routine will rotate and raise an object.
 */
 
RotateObject( p, theta, x0, y0, refresh, erase )
	ITEM_LIST_PTR *p;
	double theta;		/* Angle of rotation. */
	short x0, y0;		/* Center of rotation. */
	short refresh, erase;
  {
    double sine, cosine;
    
    /* Check parameters. */
    if (p == NULL)
	return;
    
    if (Debug&DebugBackend)
      {
	printf("Rotate object %d through %f about (%d, %d)\n\r",
		p, theta, x0, y0);
	DebugItemDescriptor( p->itemdesc, 0 );
      }
    
    /* Transform the object. */
    sine = sin( theta );
    cosine = cos( theta );
    TransformObject( p, sine, cosine, x0, y0, 0, 0,
	refresh, erase, RotatePoint );
  }

/*
 *  This routine will scale and raise an object.  To do this, it invokes
 *  TransformObject(), which will transform an object based on the
 *  point transformation routine passed to it.
 */
 
ScaleObject( p, factorx, factory, x0, y0, refresh, erase )
	ITEM_LIST_PTR *p;
	double factorx,factory;	/* Scale factor */
	short x0, y0;		/* Center of transformation */
	short refresh, erase;
  {
    /* Check parameters. */
    if (p == NULL)
	return;
    
    if (Debug&DebugBackend)
      {
	printf("Scale object %d by %f,%f about (%d, %d)\n\r",
		p, factorx,factory, x0, y0);
	DebugItemDescriptor( p->itemdesc, 0 );
      }
    
    /* Transform the object. */
    TransformObject( p, factorx, factory, x0, y0, 0, 0,
	refresh, erase, ScalePoint );
  }


/*
 *  This internal routine will change the filling, nib, and opaque attributes
 *  of spline objects and the font and centering attributes of text objects.
 *
 *   togopaque - 0 = don't toggle, 1 = toggle opaque bit
 *   nib - 0-15 = nib type, -1 = don't change
 *   pat - -1 = don't change, 0 = clear, 1 = white, 2 = black, etc.
 *   newfont - -1 = don't change, 0,1,.... font number
 *   newcenter - -1 = don't change, 0,1,2 = set centering.
 */
 
AttribObject( p, togopaque, nib, pat, newfont, newcenter)
	ITEM_LIST_PTR *p;
	short togopaque;
	enum Nib nib;
	enum Pattern pat;
	short newfont,newcenter;
  {
    register ITEM_DESCRIPTOR *id;
    char *string;
    SPLINE *sptr, *newsptr;
    POINT *pptr, *newpptr;
    short i, x,x1, x2, y,tf,bmin,bmax;
    ITEM_LIST_PTR *idptr, *gptr, *newgptr;
    short nextfont,nextcenter,slength;
    
    id = p->itemdesc;
    if (id->type == TextObj) {
	if (newfont >0 || newcenter > 0) {
	if ((string = (char *) malloc( MAXLEN + 1 )) == NULL)
	  {
	    printf("Out of memory.  Can't set attributes on any text.\n\r");
	    return;
	  }
	for (i = MAXLEN + 1;  i--;)
	    string[i] = '\0';
	strcpy( string, id->data );
	nextfont   = newfont   < 0 ? id->subtype : newfont;
	nextcenter = (newcenter < 0 || id->typedata == 3) ? id->typedata 
							  : newcenter; 
	if (!LoadFont(nextfont)) nextfont = id->typedata;
	OpenMainSymbol();
	AddItem( sdf, GetINum(), 0, 0, 0, 0,
		FontData[nextfont].refnumber, SDF_TEXT, string );
	if (InquireItem( sdf,LastINum(),0,&slength,&bmin,&bmax,0,0,0 ) == 0) {
	    printf("Internal Error: AttribObject; InquireItem failed (slength).\n\r");
	    slength=id->xmax-id->xmin;
	    bmin=id->ymin;
	    bmax=id->ymax;
	}
	DeleteItem( sdf, LastINum() );
	FreeINum(LastINum());
	y = id->ymin + p->dy;
	switch (id->typedata)
	  {
	    /* old positioning point. */
	    case PositionLeft:
	    case 3:
		x = id->xmin + p->dx;
		break;
	    case PositionCenter:
		x = ((id->xmin + id->xmax) / 2) + p->dx;
		break;
	    
	    case PositionRight:
		x = id->xmax + p->dx;
		break;
	  }
	switch (nextcenter) {
	    case PositionRight:
		x1 = x - slength;
		x2 = x;
		break;
		    
	    case PositionCenter:
		x1 = x - (slength >> 1);
		x2 = x + (slength >> 1);
		break;
			
	    case PositionLeft:
	    case 3:  /* fudge for PressEditSymbol */
		x1 = x;
		x2 = x + slength;
		break;
	}
	EraseObject( p, 1 );
	NewObject( string, TextObj, nextfont, nextcenter,
	    x1, x2, y, y+bmax-bmin, -bmin, NULL, 1, 0, 0 );
    } }
    else if (id->type != GroupObj) {
	if (togopaque || (int) nib >=0 || (int) pat >=0)
      {
	/* Splines.  Transform all of the points. */
	sptr = (SPLINE *) id->data;
	if ((newsptr = (SPLINE *) malloc( sizeof(SPLINE) +
		sptr->numvert * sizeof(POINT) )) == NULL)
	  {
	    printf("Out of memory.  Can't transform a spline object.\n\r");
	    return;
	  }
	newsptr->order = sptr->order;
	newsptr->numvert = sptr->numvert;
	newsptr->nib = (int)nib>=0?nib:sptr->nib;
	newsptr->border = (int)nib>0?1:(int)nib==0?0:sptr->border;
	newsptr->closed = sptr->closed;
	newsptr->filled = (int)pat> 0 ? 1 : (int)pat == 0 ? 0: sptr->filled;
	newsptr->opaque = sptr->opaque^togopaque;
	newsptr->pat = (int)pat>0?(enum Pattern)((int)pat-1):sptr->pat;

	/* Keep the user from shooting his foot! */
	tf=0;
	if (!newsptr->closed && newsptr->filled) {
	    newsptr->filled=0;
	    tf=1;
	    mprintf(2,"Can't fill open curves and polygons.\n\r");
	}
	if (!newsptr->border && (!newsptr->filled || 
			        (newsptr->filled && (int)newsptr->pat==0 &&
				 !newsptr->opaque))) {
	    newsptr->border=1;
	    newsptr->nib=(enum Nib)1;
	    if (!tf) mprintf(2,"Use the erase icon to erase objects!\n\r");
	}
	pptr = &(sptr->head);
	newpptr = &(newsptr->head);
	for (i = 0;  i < sptr->numvert;  i++)
	  {
	    newpptr[i].x = pptr[i].x + p->dx;
	    newpptr[i].y = pptr[i].y + p->dy;
	  }
	EraseObject( p, 0 );
	NewObject( newsptr, id->type, 0, 0, 0, 0, 0, 0, 0, NULL,1, 0, 0 );
      } }
    else      
      {
	/* This is all going away. Keep Jeff Mogul from dying for a week.*/
	mprintf(2,"sorry, can't change groups yet.");
	return;
#if 0
	/* Groups.  Transform all of the objects within the group. */
	if ((string = (char *) malloc ( MAXLEN + 1 )) == NULL)
	  {
	    printf("Out of memory.  Can't transform group name.\n\r");
	    return;
	  }
	for (i = MAXLEN + 1;  i--;)
	    string[i] = '\0';
	strcpy( string, id->name );
	
	/* Transform the contents. */
	gptr = (ITEM_LIST_PTR *) id->data;
	if ((newgptr = (ITEM_LIST_PTR *) malloc( sizeof(ITEM_LIST_PTR) ))
	    == NULL)
	  {
	    printf("Out of memory.  Can't transform group contents.\n\r");
	    free( string );
	    return;
	  }
	newgptr->prev = NULL;
	newgptr->next = NULL;
	while (gptr)
	  {
	    AttribObject( gptr, togopaque,nib,pat,newfont,newcenter);
	    if ((idptr = (ITEM_LIST_PTR *) malloc( sizeof(ITEM_LIST_PTR) ))
		== NULL)
	      {
		printf("Out of memory.  Can't transform a group.\n\r");
		return;
	      }
	    idptr->itemdesc = CurrentObject->itemdesc;
	    idptr->dx = 0;
	    idptr->dy = 0;
	    idptr->call = 0;
	    idptr->flag = 0;
	    idptr->next = NULL;
	    if (newgptr->next)
		newgptr->next->next = idptr;
	    else
		newgptr->prev = idptr;
	    idptr->prev = newgptr->next;
	    newgptr->next = idptr;
	    gptr = gptr->next;
	  }

	EraseObject( p, 0 );
	RedrawActivelist( 0 );
	NewObject( newgptr->prev, GroupObj, 0, 0, 0, 0, 0, 0, 0,
		string, 0, 0, 0 );
	NewGroup( CurrentObject, 1, 0 );
	RedrawActivelist( 1 );
#endif
      }
  }

