/*
 *  debug.c
 *
 *  This file contains routines useful for making debugging printouts.
 *
 *  David Kaelbling, May 1983
 */
 
 
/* Includes */
# ifdef VAX
# include "stdio.h"
# else
# include "Vio.h"
# endif
# include "Vgts.h"
# include "splines.h"
# include "draw.h"
 
 
/* Imports */
 
 
/* Exports */
extern PrintObjType();
extern PrintTemplate();
extern PrintCommand();
extern DebugGroup();
extern DebugSpline();
extern DebugItemDescriptor();
extern DebugObjectList();
extern DebugLinks();
 
 
/* Local Definitions */

/*
 *  This routine will print a ObjTypes variable.
 */
 
PrintObjType( type )
	enum ObjTypes type;
  {
    printf("%d ", (int) type);
    switch (type)
      {
	case TextObj:		printf("(Text)");  break;
	case OpenSplineObj:	printf("(OpenSpline)");  break;
	case ClosedSplineObj:	printf("(ClosedSpline)");  break;
	case OpenPolygonObj:	printf("(OpenPolygon)");  break;
	case ClosedPolygonObj:	printf("(ClosedPolygon)");  break;
	case GroupObj:		printf("(Group)");  break;
	case TemplateObj:	printf("(Template)");  break;
	default:		printf("(unknown)");  break;
      }
  }

/*
 *  This routine will print a TemplateTypes variable.
 */
 
PrintTemplate( type )
	enum TemplateTypes type;
  {
    printf("%d ", (int) type);
    switch (type)
      {
	case NarArrowO:		printf("(Narrow Open Arrowhead)");  break;
	case NarArrowC:		printf("(Narrow Closed Arrowhead)");  break;
	case WideArrowO:	printf("(Wide Open Arrowhead)");  break;
	case WideArrowC:	printf("(Wide Closed Arrowhead)");  break;
	case Oval:		printf("(Oval)");  break;
	case Circle:		printf("(Circle)");  break;
	case Rectangle:		printf("(Rectangle)");  break;
	case NullTemplate:	printf("(Null Template)");  break;
	default:		printf("(Unknown)");  break;
      }
  }

/*
 *  This routine will print a MenuOptions variable.
 */
 
PrintCommand( cmd )
	enum MenuOptions cmd;
  {
    printf("%d (", (int) cmd);
    switch (cmd)
      {
	case CNull:		printf("CNull");  break;
	case CRotate:		printf("CRotate");  break;
	case CScale:		printf("CScale");  break;
	case CMove:		printf("CMove");  break;
	case CCopy:		printf("CCopy");  break;
	case CDraw:		printf("CDraw");  break;
	case CAlter:		printf("CAlter");  break;
	case CErase:		printf("CErase");  break;
	case CLower:		printf("CLower");  break;
	case CRaise:		printf("CRaise");  break;
	case CAllObj:		printf("CAllObj");  break;
	case CText:		printf("CText");  break;
	case COpenCurve:	printf("COpenCurve");  break;
	case CClosedCurve:	printf("CClosedCurve");  break;
	case CCurrentObj:	printf("CCurrentObj");  break;
	case COpenPolygon:	printf("COpenPolygon");  break;
	case CClosedPolygon:	printf("CClosedPolygon");  break;
	case CGroup:		printf("CGroup");  break;
	case CTemplate:		printf("CTemplate");  break;
	case CDone:		printf("CDone");  break;
	case CAbort:		printf("CAbort");  break;
	case CUndo:		printf("CUndo");  break;
	case CMisc:		printf("CMisc");  break;
	case CHelp:		printf("CHelp");  break;
	case CExit:		printf("CExit");  break;
	case CNib:		printf("CNib");  break;
	case CPattern:		printf("CPattern");  break;
	case CFilling:		printf("CFilling");  break;
	case CTextDefault:	printf("CTextDefault");  break;
	case CDataPoint:	printf("CDataPoint");  break;
	case CCheckPoint:	printf("CCheckPoint");  break;
	case CAlmostDone:	printf("CAlmostDone");  break;
	default:		printf("unknown");  break;
      }
    printf(")");
  }

/*
 *  This routine will print a group structure.
 */
 
DebugGroup( g, indent )
	register ITEM_LIST_PTR *g;
	int indent;
  {
    while (g)
      {
	printf("%*s", indent, "");
	printf("  %x:  prev %x, next %x, itemdesc %x, call %d at (%d, %d)\n\r",
		g, g->prev, g->next, g->itemdesc, g->call, g->dx, g->dy );
	DebugItemDescriptor( g->itemdesc, indent );
	g = g->next;
      }
  }

/*
 *  This routine will print a spline data structure.
 */
 
DebugSpline( sptr, indent )
	SPLINE *sptr;
	int indent;
  {
    register POINT *p;
    register short i;
    
    if (sptr == NULL)
      {
	printf("%*s    NULL SPLINE DATA STRUCTURE\n\r", indent, "");
	return;
      }
    printf("%*s", indent, "");
    printf("    order %d, border %d, closed %d, filled %d, opaque %d\n\r",
	sptr->order, sptr->border, sptr->closed, sptr->filled, sptr->opaque);
    printf("%*s", indent, "");
    printf("    numvert %d, nib %d, fill pattern %d",
	sptr->numvert, (int) sptr->nib, (int) sptr->pat);
    p = &(sptr->head);
    for (i = 0;  i < sptr->numvert;  i++)
      {
        if ((i % 5) == 0)
	    printf("\n\r%*s    ", indent, "");
	printf("(%d, %d)  ", p[i].x, p[i].y);
      }
    printf("\n\r");
  }

/*
 *  This routine will print an item descriptor.
 */
 
DebugItemDescriptor( id, indent )
	ITEM_DESCRIPTOR *id;
	int indent;
  {
    if (id == NULL)
      {
	printf("%*s    NULL ITEM DESCRIPTOR\n\r", indent, "");
	return;
      }
    
    printf("%*s", indent, "");
    printf("    number %d, symbol %d, refs %d, prev %x, next %x\n\r",
	id->number, id->symbol, id->refs, id->prev, id->next );
    printf("%*s", indent, "");
    printf("    xmin %d, xmax %d, ymin %d, ymax %d\n\r",
	id->xmin, id->xmax, id->ymin, id->ymax );
    printf("%*s", indent, "");
    printf("    type ");
    PrintObjType( id->type );
    switch (id->type)
      {
	case TextObj:
	    printf(", font number %d, centering %d, descent %d\n\r",
		id->subtype, id->typedata, id->base);
	    printf("%*s", indent, "");
	    if (id->data)
		printf("    data '%s'\n\r", id->data );
	    else
		printf("    data NULL\n\r");
	    break;
	
	case OpenSplineObj:
	case ClosedSplineObj:
	case OpenPolygonObj:
	case ClosedPolygonObj:
	    printf("\n\r");
	    DebugSpline( (SPLINE *) (id->data), indent );
	    break;
	
	case GroupObj:
	    if (id->name)
		printf("  Name = '%s'", id->name);
	    else
		printf("  Name = NULL,");
	    printf("  Items are:\n\r");
	    DebugGroup( (ITEM_LIST_PTR *) (id->data), indent + 4 );
	    break;
	
	case TemplateObj:
	    printf("  subtype ");
	    PrintTemplate( id->subtype );
	    printf("\n\r");
	    DebugSpline( (SPLINE *) (id->data), indent );
	    break;
	
	default:
	    printf("(Unknown)\n\r");
	    break;
      }
  }    

/*
 *  This routine will dump an object list (eg. activelist).
 */
 
DebugObjectList( name, list, indent )
	char *name;
	ITEM_LIST_PTR *list;
	int indent;
  {
    if (list == NULL)
      {
	printf("DebugObjectList:  NULL list specified.\n\r");
	return;
      }
      
    /* Maybe parameters are okay. */
    printf("%*s", indent, "");
    printf("Item list %s:  first = %x, last = %x\n\r",
		name, list->prev, list->next);
    list = list->prev;
    while (list)
      {
	printf("\n\r%*s", indent, "");
	printf("  %x:  prev %x, next %x, itemdesc %x, call %d at (%d, %d)\n\r",
		list, list->prev, list->next, list->itemdesc,
		list->call, list->dx, list->dy );
	if (list->itemdesc)
	    DebugItemDescriptor( list->itemdesc, indent );
	list = list->next;
      }
  }

/*
 *  This routine will print a short form of an ObjectList.
 */
 
DebugLinks( head )
	ITEM_LIST_PTR *head;
  {
    short i;
    
    /* Print the header information */
    if (head == NULL)
	printf("DebugLinks:  NULL head cell.\n\r");
    printf("Head Cell:  first = %x, last = %x", head->prev, head->next);
    
    /* Print the links. */
    i = 0;
    head = head->prev;
    while (head)
      {
	if (i++ % 4 == 0)
	    printf("\n\r    ");
	printf("(%x -- %x)    ", head->prev, head->next);
	head = head->next;
      }
    printf("\n\r");
  }
