/*
 * fexecute.c - run a program on the current fileserver (session host).
 *
 * Control flow:
 * 	1) First check for explicit invocation, i.e.,
 *	      fexecute <remote-program>
 *	   If so, then change argv to point to "<remote-program>".
 *
 *	2) try to run the remote program, failure: print error and exit.
 *
 *	3) Set up "pid1", so that it is reading from stdin and sending to
 *	   remote process. "pid2" reads from the remote process and writes
 *	   to stdout.   Pid1 is ReaderProcess() and pid2 is 
 *	   SimpleReaderProcess().
 *
 *	4) Wait for pid1 to die or send a message.  If a message is
 *	   sent, then close input to remote process and wait for pid2 to die.
 *	   Otherwise kill pid2 and close instances.
 *
 *	Death of pid1 happens if:
 *		a) File exception occurs (pid1 is the break process for stdin).
 *		b) Remote execution terminates and it is killed by pid2.
 *	Pid1 sends a message if:
 *		a) End of file encountered on standard input.
 */
#include <Vio.h>
#include <Vtermagent.h>

#define READER_STACK 		1024	/* small stack for reader process */
#define ROOT_PROCESS_PRIORITY	4	/* should be a system manifest */
#define READER_PRIORITY 	(ROOT_PROCESS_PRIORITY + 1)

extern SystemCode RemoteExecute();
extern SimpleReaderProcess(), ReaderProcess();
extern ProcessId GetPid();
char BannerBuffer[ 100 ];
char ProgName[] = "fexecute";

main( argc, argv )
    int argc;
    char **argv; 
 {
    SystemCode err;
    ProcessId pid1, pid2;
    File *ProcessFAD[2];
    Message msg;
    int checkPosn;

    checkPosn = strlen(argv[0]) - strlen(ProgName);
    if ((checkPosn >= 0) && strcmp(&(argv[0][checkPosn]), ProgName)==0)
      {
      	   /*
	    * Skip our name if we were invoked explicitly.
	    * Note that this test succeeds even if argv[0] is
	    * (e.g.) "[bin]fexecute" or "./fexecute", although it also
	    * succeeds if it is something like "bogusfexecute".
	    */
        argc--;
	argv++;
      }

    if ( argc == 0 )
      {
        fputs(ProgName,stderr);
        fputs(": no arguments\n",stderr);
	exit(1);
      }

    err = RemoteExecute( ProcessFAD, argv[0], &(argv[1]), FCREATE );
    if ( err != OK )
      {
	fputs( argv[0], stderr );
        fputs( ": remote execution failed: ", stderr );
	fputs( ErrorString( err ), stderr );
        putc( '\n', stderr );
	exit( err );
      }

    pid1 = Create( READER_PRIORITY, ReaderProcess, READER_STACK );
    pid2 = Create( READER_PRIORITY + 1, SimpleReaderProcess, READER_STACK );


    if ( GetPid( VGT_SERVER, LOCAL_PID ) == stdin->fileserver )
      {
        ModifyPad( stdin, CR_Input+LF_Output+Echo+LineBuffer+PageOutput );
        strcpy( BannerBuffer, "Remotely executing: " );
        strncat( BannerBuffer, argv[ 0 ]
	         , sizeof( BannerBuffer ) - 1 - strlen( BannerBuffer ) );
        SetVgtBanner( stdin, BannerBuffer );
      }

    /* So kill program works correctly */
    SetBreakProcess( stdin, pid1 );

    Ready( pid1, 4, stdin, ProcessFAD[1], stdout, pid2 );
    Ready( pid2, 4, ProcessFAD[0], stdout, stdout, pid1 );

    /* wait for the reader process to die or send a message */
    if ( ReceiveSpecific( msg, pid1 ) != 0 )
      {
        Close( ProcessFAD[1] );
        ReceiveSpecific( msg, pid2 );
      }
    else
      {
        Close( ProcessFAD[1] );
      }

    DestroyProcess( pid1 );
    DestroyProcess( pid2 );

    Close( ProcessFAD[0] );

    Flush( stdout );

    exit(0);
  } /* main */

SimpleReaderProcess( in, to, err, pid )
    register File *in, *to, *err;
    ProcessId pid;
/*
 * Read characters from in and write to to.
 * Terminate on EOF
 */
  {
    register int c;

    while ( (c = getc( in )) != EOF )
      {
        putc( c, to );
        if ( BufferEmpty( in ) ) 
	    Flush(to);
      }

    if (in->lastexception != END_OF_FILE)
      {
        fputs(ProgName,stderr);
	fputs(": error reading from remote program: ", stderr);
	fputs(ErrorString(in->lastexception), stderr);
	Flush(stderr);
      }

    Destroy( pid );
    Destroy( 0 );
	
  } /* SimpleReaderProcess */

ReaderProcess( in, to, err, pid )
    register File *in, *to, *err;
    ProcessId pid;
  {
/*
 * read from the input pad and send to the remote command.
 * pid is the process going in the other direction, which we
 * kill if we finish.
 */
    register int c;
    Message msg;
    msg[ 0 ] = 0;

    while ( (c=getc(in))!=EOF )
      {
	putc( c, to );
	if ( BufferEmpty( in ) ) 
	    Flush(to);
      }  /* while */

    if (in->lastexception != END_OF_FILE)
      {
        fputs(ProgName,stderr);
	fputs(": error reading from stdin: ", stderr);
	fputs(ErrorString(in->lastexception), stderr);
	Flush(stderr);
	Destroy( pid );
	Destroy( 0 );
      }
    else
      {
	Send( msg, Creator( 0 ) );  /* creator closes the fad & kills us */
      }
	

  } /* ReaderProcess */


