/*
 *  select.c
 *
 *  This file contains routines to deal with the "current object" There are
 *  three main routines.
 *
 *  SelectObject selects the object nearest the given x,y position (if any). 
 *      It unselects any previously selected object(s).
 *  ToggleSelectObject will toggle the selection status of any object without
 *	affecting any other objects.
 *  SetCurrentObject unselects all other objects and selects the given object.
 *
 *  If more than one object is selected, then all selected objects are placed
 *  	in the "fake group" and the group becomes the "real" selected object.
 *	The "Group" command (commands.c) turns the fake group into a real group
 *	and vice versa. This is the primary method by which groups are created
 *	and decomposed back to their component parts.
 *
 *  This file also contains routines to frame the currently selected object(s).
 */
 
/* Includes */
# ifdef UNIX
# include "stdio.h"
# else
# include "Vio.h"
# endif
# include "Vgts.h"
# include "splines.h"
# include "draw.h"
# include <math.h>
 
 
/* Imports */
extern short GetINum();		/* Get an item number */
extern short GetSeqINum();	/* Get a sequential set of item numbers */
extern short LastINum();	/* Return the last item number gotten
extern short FreeINum();	/* Return an item number to the freelist */
extern OBJECT *FindObject();	/* Find an object given a point */
extern BOOLEAN RectInObject();  /* Determine whether an object is in a rect */
extern OBJECT *CreateGroup();	/* Create a new empty group */
extern MoveToGroup();		/* Move an object into a group */
extern short MoveFromGroup();	/* Move an object out of a group */
extern OBJECT *Ungroup();	/* Move up all objects andand delete a group */

 
/* Exports */
extern SetCurrentObject();
extern SelectObject(); 
 
/* Local Definitions */

static BOOLEAN Selecting = FALSE;


/*
 * This routine will set the current object and frame it, or set no
 * current object if NULL
 *
 * The fake flag applies only to groups and specifies that this group is the
 * fake group and should be highlighted accordingly.
 *
 * The keepoldfake flag inhibits the destruction of any previous fake group.
 */


SetCurrentObject(q,keepoldfake,fake,refresh)
	OBJECT *q;
	BOOLEAN fake;
	BOOLEAN keepoldfake;
	BOOLEAN refresh;

  {
    PART *id;
    SPLINE *s;
    OBJECT *TempObject = CurrentObject;
    OBJECT *q2;
    if (q==CurrentObject) return;
    if (Debug&DebugSelect) {
	printf("SetCurrentObject %x: fake=%s; refresh=%s;   previous = %x\n\r",
		q,fake?"TRUE":"FALSE",refresh?"TRUE":"FALSE",CurrentObject);
    }
    if (CurrentObject != NULL) {
	CurrentObject = NULL;
	UnFrameObject(TempObject,refresh);
	/* This hack enables the VGTS frame optimization which only works if 
	    exactly one SDF_OUTLINE is deleted and there is nothing on it 
	    (but not necessarily inside it.) */
	if (Selecting) 
	    CloseAllSymbols();
	if (FakeGroup && !keepoldfake) {
	    Ungroup(TempObject, &q);
	    FakeGroup = FALSE;
	}
    }
    if (q != NULL && refresh) {
	FrameObject(q,fake,TRUE,0,0);
	/* We now set the default menu parameters to reflect the newly selected
	   object */
	q2 = q;
	id = q2->part;
	if (fake && id->type!=GroupObj) {
	    printf("ERROR*** SetCurrentObj. Non-group fake\n\r");
        }
	switch(id->type) {
	    case TextObj:
		if (id->typedata != 3) /* Ignore PressEdit */
		    SetCurrentFont(id->subtype,id->typedata);
		break;
	    case GroupObj:
		break;
	    default:
		s = (SPLINE *) id->data;
		if (s->border)
		    SetCurrentNib(s->nib);
		else
		    SetCurrentNib((enum Nib) 0);
		if (s->filled)
		    SetCurrentPat((int)s->pat + 1);
		else
		    SetCurrentPat(0);
		break;
	}
    }
    CurrentObject = q;
    FakeGroup = fake;
  }

/*
 * This routine will toggle the selection status of any object. If more than
 * one object is caused to be selected, then the fake group is created. If
 * an object is de-selected, leaving only a single object in the fake group,
 * then it is destroyed, and the object brought back to the main item list.
 * last is updated to reflect the (previous) last element in a group. Next
 * will contain the next element after the just deleted item.
 */


ToggleCurrentObject(q,refresh,last,next)
	OBJECT *q;
	short refresh;
	OBJECT **last, **next;

  {
    OBJECT *TempObject=CurrentObject, *NewGroup;
    PART *id;
    SPLINE *s;
    if (q==NULL) return;
    if (Debug&DebugSelect) {
	printf("ToggleCurrentObject %x: refresh=%s;   previous = %x\n\r",
		q,refresh?"TRUE":"FALSE",CurrentObject);
    }
    /*
     * de-select the last object
     */
    if (q==CurrentObject && !FakeGroup) {
	SetCurrentObject(NULL,FALSE,FALSE,refresh);
    }
    /*
     * Select a single object (nothing else is selected)
     */
    else if (CurrentObject==NULL) {
	SetCurrentObject(q,FALSE,FALSE,refresh);
    }
    /*
     * Select the fake group - an error
     */
    else if (q == CurrentObject) {
	printf("ERROR*** ToggleCurrentObject of Fake group.\n\r");
    }
    /*
     * Two items selected. Create the fake group
     */
    else if (!FakeGroup) {
	CurrentObject = NULL;
	NewGroup = CreateGroup(0,0,activelist);
	FrameObject(q,FALSE,refresh,0,0);
	MoveToGroup(TempObject,NewGroup,NULL);
	MoveToGroup(q,NewGroup,NULL);
	NewGroup->frame = -1;
	FakeGroup=TRUE;
	CurrentObject = NewGroup;
    }
    /*
     * Add an item to the fake group
     */
    else if (q->parent == activelist) {
	CurrentObject = NULL;
	FrameObject(q,FALSE,refresh,0,0);
	MoveToGroup(q,TempObject,last);
	CurrentObject = TempObject;
    }
    /* 
     * delete an item from the fake group
     */
    else {
        short count;
	UnFrameObject(q,refresh);
	CurrentObject = NULL;
	count = MoveFromGroup(q,TempObject,last,next);
	if (count == 0) 
	    /*
	     * That was the last item in the group. This is an error since 
	     * groups will always have more than one item inside them
	     */
	    printf("Error*** ToggleObject: Missed the last ungroup\n\r");
	else if (count == 1) {
	    /* 
	     * We have but one item left in the fake group. Move it back into
	     * the activelist and make it a normal selected object
	     */
	    CurrentObject = Ungroup(TempObject,NULL);
	    FakeGroup = FALSE;
	}
	else {
	    /*
 	     * We still have two or more objects
	     */
	    CurrentObject = TempObject;
	}
    }
  }

/*
 *  This routine will put a frame of some kind around an object, thus
 *  highlighting it. If fake=TRUE, then all of the items of a group are
 *  framed individually (one level deep.) Otherwise, the group highlight
 *  is used if q is a group.
 */
 
FrameObject( q, fake , refresh,dx,dy)
	OBJECT *q;
	BOOLEAN fake;
	short refresh;
	short dx,dy;
  {
    register PART *id = q->part;
    OBJECT *p;
    SPLINE *sptr;
    POINT *svert;
    
    if (id == NULL)
	return;
    
    if (Debug&DebugSelect) {
	printf("FrameObject %x: fake=%s; refresh=%s\n\r",
		q,fake?"TRUE":"FALSE",refresh?"TRUE":"FALSE");
    }
    switch (id->type) {
	case TextObj:
	    /* Show an expanded bounding box. */
	    OpenSymbol(mainSymbol);
	    q->frame = AddItem( sdf, GetINum(),
		id->xmin-3+q->dx+dx,id->xmax+3+q->dx+dx,
		id->ymin-3-id->base+q->dy+dy,id->ymax+3+q->dy+dy,
		AllEdges, SDF_OUTLINE, NULL );
	    CloseSymbol(TRUE);
	    break;
	
	case GroupObj:
	    /* A group. */
	    if (fake) {
		/*
		 * This is the fake group. Highlight it as if all the elements
		 * are at the top level
		 */
		q->frame = -1;
		for (p=((OBJECT_HEADER *)q->part->data)->first;p;p=p->next) {
		    if (p->frame == 0)
			FrameObject(p,FALSE,refresh,q->dx+dx,q->dy+dy);
		}
	    }
	    else {
		/* 
		 * This is a real group. Highlight just a large rectangle		 */
	    	OpenSymbol(mainSymbol);
	    	q->frame = AddItem( sdf, GetSeqINum(2),
		id->xmin-3+q->dx+dx, id->xmax+3+q->dx+dx,
		id->ymin-3+q->dy+dy, id->ymax+3+q->dy+dy,
		AllEdges, SDF_OUTLINE, NULL );
	    	           AddItem( sdf, q->frame+1,
		id->xmin-5+q->dx+dx, id->xmax+5+q->dx+dx,
		id->ymin-5+q->dy+dy, id->ymax+5+q->dy+dy,
		AllEdges, SDF_OUTLINE, NULL );
	        CloseSymbol(TRUE);
	    }
	    break;
	
	default:
	    /* Some form of spline */
	    sptr = (SPLINE *) id->data;
		OpenSymbol(mainSymbol);
		if (sptr->order >=5) {
		    double d1,d2;
		    short x0,y0,r,r2;
		    /* HACK assumes that splines order 5 are circles */
	            svert = &(sptr->head);
		    d1=(double)(svert[2].x-svert[0].x);
		    d2=(double)(svert[2].y-svert[0].y);
		    r = (short)(sqrt(d1*d1+d2*d2) * .35);
		    d1=(double)(svert[3].x-svert[1].x);
		    d2=(double)(svert[3].y-svert[1].y);
		    r2 = (short)(sqrt(d1*d1+d2*d2) * .35);
		    if (r2>r) r=r2;
		    x0=(svert[2].x+svert[0].x)/2;
		    y0=(svert[2].y+svert[0].y)/2;
		    q->frame = AddItem(sdf, GetINum(),
			x0 + q->dx - r + dx,
			x0 + q->dx + r + dx,
			y0 + q->dy - r + dy,
			y0 + q->dy + r + dy,
			AllEdges, SDF_OUTLINE, NULL );
		    }
		else
		    q->frame = AddItem( sdf, GetINum(),
			id->xmin - 2 + q->dx+dx, id->xmax + 2 + q->dx+dx,
			id->ymin - 2 + q->dy+dy, id->ymax + 2 + q->dy+dy,
			AllEdges, SDF_OUTLINE, NULL );
	 	CloseSymbol(TRUE);
		break;
      }
  }

/*
 *  This routine will remove the frame from around an item.
 */
 
UnFrameObject( q, refresh )
	OBJECT *q;
	BOOLEAN refresh;
  {

    OBJECT *p;

    if (Debug&DebugSelect) {
	printf("UnFrameObject %x: refresh=%s\n\r",
		q,refresh?"TRUE":"FALSE");
    }
    /* If we haven't done so already, delete the frame. */
    if (q->frame > 0) {
	OpenSymbol(mainSymbol);
	if (q->part->type==GroupObj) {
	    DeleteItem(sdf,q->frame+1);
	    FreeINum(q->frame+1);
	    CloseSymbol(TRUE);
	    CloseAllSymbols();	/* outline optimization hack */
	    OpenSymbol(mainSymbol);
	}
	DeleteItem( sdf, q->frame );
	FreeINum(q->frame);
	q->frame = 0;
	CloseSymbol(TRUE);
    }
    else if (q->frame < 0) {
	/* Unframe a  (fake) group.  Unframe all of the parts */
	MajorChange();
	for (p=((OBJECT_HEADER *)q->part->data)->first;p;p=p->next) {
	    if (p->frame != 0)
		UnFrameObject(p,refresh);
    }
    q->frame = 0;
  }
}

/*
 *  SelectObject is used to select objects in the field given
 *  the button state and the x/y coordinates.
 */

SelectObject(x,y,but)
short x,y,but;
{
    short dx,dy;

    Selecting = TRUE; /* Kluge to make the VGTS outline optimization work */
    SetCurrentObject(FindObject(x,y,FakeGroup),FALSE,FALSE,TRUE);
				/*  ^^^^ */
    /* This allows selecting a single object from a set of selected objects*/
    Selecting = FALSE;
}

/*
 *  SelectObject is used to toggle the select status of objects in the field 
 *  given the button state and the x/y coordinates.
 */

ToggleSelectObject(x,y,but)
short x,y,but;
{
    short dx,dy;

    Selecting = TRUE; /* Kluge to make the VGTS outline optimization work */
    ToggleCurrentObject(FindObject(x,y,TRUE),TRUE,NULL,NULL);
    Selecting = FALSE;
}

/*
 *  RangeSelect is used to toggle the selection status of a rane of objects
 *  within a rectangle. This routine must be very careful since it is
 *  running down the same data structures that are being manipulated by
 *  the ToggleCurrentObject command.
 */

SelectRange(xmin,xmax,ymin,ymax)
short xmin,xmax,ymin,ymax;
{
    OBJECT *ob, *obf, *obl, *obla, *next, *last;
    short nxmin,nxmax,nymin,nymax;
    OBJECT_HEADER *gptr;

    Selecting = TRUE; /* Kluge to make the VGTS outline optimization work */
    ob = activelist->first;
    /* 
     * obf and obl point to the first and last objects already in the fake
     * group. This ensures that no object is check twice and thus double-
     * toggled
     */
    if (FakeGroup) {
	if (!CurrentObject || CurrentObject->part->type!=GroupObj)
	    printf("ERROR*** SelectRange - FakeGroup inconsistency\n\r");
	gptr = (OBJECT_HEADER *)CurrentObject->part->data;
	obf = gptr->first;
	obl = gptr->last;
    }
    else {
	obf = NULL;
	obl = NULL;
    }
    /*
     * First check to select elements not in the fake group. (or de-select a
     * single selected object
     */
    ob = activelist->first;
    obla = activelist->last;
    while (ob) {
	next = ob->next;
	if (!FakeGroup || ob != CurrentObject) {
	    if (Debug&DebugSelect) 
		printf("SelectRange: Trying %x in activelist \n\r",ob);
	    if (RectInObject(ob,xmin,xmax,ymin,ymax))
    		ToggleCurrentObject(ob,TRUE,&obl,NULL);
	}
	if (ob == obla)
	    break;
	ob = next;
    }
    /* 
     * Now we try de-selecting what may already have been in the fake group
     * before the range select. Anything added above would be located AFTER
     * the last element which we are goosing as it will be moved if there
     * are multiple copies of the fake group.
     *
     * obf and obl were initialized above to the original first and last 
     * elements of the group. obf may now be invalid, but obl was goosed
     * and was kept valid. gptr->first can serve as the updated obf.
     * if obf is NULL however, the group was build totally above and we are
     * not interested in re-checking these objects.
     */
    if (obf) {
	ob = ((OBJECT_HEADER *)CurrentObject->part->data)->first;
	nxmin = xmin - CurrentObject->dx;
	nxmax = xmax - CurrentObject->dx;
	nymin = ymin - CurrentObject->dy;
	nymax = ymax - CurrentObject->dy;
	while (ob) {
	    next = ob->next;
	    last = obl;
	    if (Debug&DebugSelect) 
		printf("SelectRange: Trying %x in group \n\r",ob);
	    if (RectInObject(ob,nxmin,nxmax,nymin,nymax))
		ToggleCurrentObject(ob,TRUE,&last,&next);
	    if (!FakeGroup) {
		/* We just de-selected all but the last object. Treat it 
		 * special and quit */
		if (RectInObject(CurrentObject,xmin,xmax,ymin,ymax))
		    ToggleCurrentObject(CurrentObject,TRUE,NULL,NULL);
		break;
	    }
	    if (ob == obl)
		break;
	    obl = last;
	    ob = next;
	}
    }
    Selecting = FALSE;
}
