/*
 *  parts.c
 *
 *  These routines manipulate draw parts. A part contains most of the data
 *  needed to draw an object. Multiple objects may point to the same part.
 *  There are generally three types of parts: text, spline, and group. Text
 *  parts point to a character string. Spline parts point to a spline 
 *  descriptor. Group parts point to an object list.
 *
 *  These roiutines call the lowest level symbol manipulation routines which
 *  actually talk to the VGTS.
 *
 */
 
 
/* Includes */
#ifdef UNIX
#include "stdio.h"
#else
#include "Vio.h"
#endif
#include "Vgts.h"
#include "splines.h"
#include "draw.h"
 
 
/* Imports */
extern DebugItemDescriptor();
extern DebugObjectList();
extern DebugLinks();
extern short GetINum();		/* Get an item number */
extern short GetSeqINum();	/* Get a sequential set of item numbers */
extern short LastINum();	/* Return the last item number gotten
extern short FreeINum();	/* Return an item number to the freelist */
extern char *malloc();
extern OBJECT *AddObject();	/* Add an object to an object list */
 
 
/* Exports */
extern RedrawActivelist();	/* Redisplay the main drawing area. */
extern DecrementRef();		/* Decrement reference count on an item */
extern LoadFont();		/* loads a font */
extern PART *CopyPart();	/* Duplicate a part */
extern TransformPart();		/* Transform a part */



/*
 *  This internal routine is called each time a part is created.
 *  It will allocate a symbol for the new part.
 */
PART *CreatePart( dataptr, typeP, subtypeP, td, x1, x2, y1, y2, b)
	char *dataptr;			/* New part->data entry. */
	enum ObjTypes typeP;		/* itemdesc->type entry. */
	short subtypeP, td;		/* subtype and typedata fields. */
	short x1, x2, y1, y2, b;	/* Bounding box and baseline */
  {
    PART *id;		/* The part we are creating */
    OBJECT *g;		/* Pointer for following group chains */
    
    /* We have a new current object.  Make it so. */
    if ((id = (PART *) malloc( sizeof(PART) )) == NULL)
      {
	printf("Ran out of memory -- couldn't create a new part.\n\r");
	if (dataptr)
	    free( dataptr );
	return(NULL);
      }
    if (Debug&DebugParts) {
	printf("CreatePart: id=%x. Type = ",id);
	PrintObjType(typeP);
	printf("\n\r");
    }
    if (typeP == TextObj && subtypeP <0)
	subtypeP=1;

    /* Initialize all the fields of the part associated with the object */
    id->data = dataptr;
    id->type = typeP;
    id->subtype = subtypeP;
    id->typedata = td;
    id->number = 0;
    id->base = b;
    id->refs = 0;
    id->xmin = x1;
    id->xmax = x2;
    id->ymin = y1;
    id->ymax = y2;
    id->prev = LastPart;
    id->next = NULL;
    if (LastPart) 
        LastPart->next = id;
    LastPart = id;
    if (FirstPart == NULL)
	FirstPart = id;
    id->filnum = 0;
    CloseAllSymbols();
    if (id->type == TextObj)
      {
	/* Text. */
	LoadFont(id->subtype);
	id->symbol = DefineSymbol( sdf, GetINum(), "new text symbol" );
        id->number = AddItem( sdf, GetINum(), id->xmin, id->xmax,
		id->ymin, id->ymax,
		FontData[id->subtype].refnumber, SDF_TEXT, id->data );
	EndSymbol( sdf, id->symbol, 0 );
      }
    else if (id->type != GroupObj)
      {
	/* A spline of some sort. */
	id->symbol = DefineSymbol( sdf, GetINum(), "new spline symbol" );
	
	/* Since this is a new object, get bounding box & number. */
	OptSpline(id);
      }
    else
      {
	/* Construct the group */
	((OBJECT_HEADER *) id->data)->symbol =
	   id->symbol = DefineSymbol( sdf, GetINum(), "new group symbol" );
	g = ((OBJECT_HEADER *) id->data)->first;
	while (g)
	  {
	    g->call = AddCall( sdf,GetINum(), g->dx, g->dy, g->part->symbol );
	    g = g->next;
	  }
	EndSymbol( sdf, id->symbol, 0 );
      }
    return(id);
  }

/*
 *  This routine will update a part so that the VGTS knows of any changes
 *  we may have just made. This routine assumes that anything but the type
 *  of the object may have changed.
 */
UpdatePart( id )
    PART *id;		/* The part we are creating */
  {
    OBJECT *g;		/* Pointer for following group chains */
    
    if (Debug&DebugParts)
	printf("UpdatePart: part = %x\n\r",id);
    CloseAllSymbols();
    if (id->type == TextObj)
      {
	/* Text. */
	LoadFont(id->subtype);
	id->symbol = EditSymbol( sdf, id->symbol );
	DeleteItem(sdf, id->number);
        id->number = AddItem( sdf, id->number, id->xmin, id->xmax,
		id->ymin, id->ymax,
		FontData[id->subtype].refnumber, SDF_TEXT, id->data );
	EndSymbol( sdf, id->symbol, 0 );
      }
    else if (id->type != GroupObj)
      {
	/* A spline of some sort. */
	id->symbol = EditSymbol( sdf, id->symbol );
	
	if (id->number < 0) {
	    DeleteItem(sdf,-(id->number));
	    DeleteItem(sdf,-(id->number)+1);
	    FreeINum(-(id->number));
	    FreeINum(-(id->number)+1);
	}
	else if (id->number > 0) {
	    DeleteItem(sdf,id->number);
	    FreeINum(id->number);
	}
	OptSpline(id);
      }
    else
      {
	/* Construct the group */
	((OBJECT_HEADER *) id->data)->symbol =
	   id->symbol = EditSymbol( sdf, id->symbol );
	g = ((OBJECT_HEADER *) id->data)->first;
	while (g)
	  {
	    if (g->call)  {
		DeleteItem(sdf,g->call);
	    }
	    else
		g->call = GetINum();
	    g->call = AddCall( sdf,GetINum(), g->dx, g->dy, g->part->symbol );
	    g = g->next;
	  }
	EndSymbol( sdf, id->symbol, 0 );
        InquireItem( sdf, id->symbol, &(id->xmin), &(id->xmax),
	    &(id->ymin), &(id->ymax), 0, 0, 0 );
      }
  }

/*
 *  CopyPart creates an identical copy of the given part.
 *  If the part is a group, and goose is one of the objects within the group,
 *  then its value will be updated to the new copy. Same for goose2.
 */
PART *CopyPart(id,goose,goose2)
PART *id;
OBJECT **goose, **goose2;
  {
    PART *newid;
    char *newdata;
    if (Debug&DebugParts)
	printf("CopyPart: old part = %x\n\r",id);
    if (id->type == TextObj)
      {
	register short i;
	if ((newdata = (char *) malloc( MAXLEN + 1 )) == NULL)
	  {
	    printf("Out of memory. Can't copy any text.\n\r");
	    return(NULL);
	  }
	for (i = MAXLEN + 1;  i--;)
	    newdata[i] = '\0';
	strcpy( newdata, id->data );
      }
    else if (id->type != GroupObj)
      {
	register short i;
	/* Splines.  copy all of the points. */
        SPLINE *sptr, *newsptr;
        POINT *pptr, *newpptr;
	sptr = (SPLINE *) id->data;
	if ((newsptr = (SPLINE *) (newdata = (char *)malloc( sizeof(SPLINE) +
		sptr->numvert * sizeof(POINT) ))) == NULL)
	  {
	    printf("Out of memory.  Can't copy a spline object.\n\r");
	    return(NULL);
	  }
	newsptr->order = sptr->order;
	newsptr->numvert = sptr->numvert;
	newsptr->nib = sptr->nib;
	newsptr->border = sptr->border;
	newsptr->closed = sptr->closed;
	newsptr->filled = sptr->filled;
	newsptr->opaque = sptr->opaque;
	newsptr->pat = sptr->pat;
	
	pptr = &(sptr->head);
	newpptr = &(newsptr->head);
	for (i = 0;  i < sptr->numvert;  i++)
	  {
	    newpptr[i].x = pptr[i].x ;
	    newpptr[i].y = pptr[i].y ;
	  }
      }
    else {
	/* Groups.  simply copy the object list and increment the ref count
	   of all of the parts in the list */

	OBJECT_HEADER *gptr, *newgptr;
	OBJECT *temp;

	/* Transform the contents. */
	gptr = (OBJECT_HEADER *) id->data;
	if ((newgptr = (OBJECT_HEADER *) (newdata =
	    (char *)malloc( sizeof(OBJECT_HEADER) ))) == NULL) {
	    printf("Out of memory.  Can't copy group header.\n\r");
	    return(NULL);
	}
	newgptr->first = NULL;
	newgptr->last = NULL;
	temp = gptr->first;
	while (temp) {
	   OBJECT *newob=AddObject(temp->part,temp->dx,temp->dy,FALSE,
		TRUE,newgptr);
	   newob->frame = temp->frame;
	   temp->frame = 0;
	   if (Debug&DebugParts) printf("CopyPart:       %x -> %x\n\r",
			temp,newob);
	   if (goose && temp == *goose) {
		if (Debug&DebugParts) printf("CopyPart: Updating goose %x -> %x\n\r",
			*goose,newob);
		*goose = newob;
	   }
	   if (goose2 && temp == *goose2) {
		if (Debug&DebugParts) printf("CopyPart: Updating goose2 %x -> %x\n\r",
			*goose2,newob);
		*goose2 = newob;
	   }
	   temp = temp->next;
	}
    }

    /* we have copied the data portion. Now make the rest of the part */

    newid = CreatePart( newdata, id->type, id->subtype, id->typedata,
	id->xmin, id->xmax, id->ymin, id->ymax, id->base);
    if (Debug&DebugParts)
	printf("CopyPart: old part = %x; new part = %x\n\r",id,newid);
    return(newid);
  }

/*
 * Transform part will modify the given part by applyinging the given
 * transformation function
 */ 
TransformPart( id, fact1, fact2, x0, y0, ptrans,fake )
	PART *id;
	double fact1, fact2;	/* Point Transform factors. */
	short x0, y0;		/* Center of transformation. */
	register (*ptrans)();	/* Point transformation routine. */
	BOOLEAN fake;		/* the group is the fake group */
  {
    SPLINE *sptr;
    POINT *pptr;
    short i, x, x2, y2;
    OBJECT *gptr;
    
    if (Debug&DebugParts) {
	printf("TransformPart: part=%x; fact1,fact2 = %f,%f; x0,y0 = %d,%d\n\r"
		,id,fact1,fact2,x0,y0);
	printf("TransformPart: id->filnum = %d; DupCheck = %d\n\r",
		id->filnum,DupCheck);
    }
    /* Make sure that, because of multiple references, we have not already
       transformed this part */
    if (id->filnum == DupCheck) return;    
    id->filnum = DupCheck;
    /* For non groups, transform about the proper point. */
    if (id->type == TextObj)
      {
        if (Debug&DebugParts) {
	    printf("TransformPart: transforming text: %s\n\r",id->data);
        }
	x2 = id->xmax-id->xmin;
	y2 = id->ymax-id->ymin;
	switch (id->typedata) {
	    /* Transform about the positioning point. */
	    case PositionLeft:
		ptrans( &id->xmin, &id->ymin, fact1, fact2, x0, y0 );
		break;
	    
	    case PositionCenter:
		x = ((id->xmin + id->xmax) / 2);
		ptrans( &x, &id->ymin, fact1, fact2, x0, y0 );
		id->xmin = x - x2/2;
		break;
	    
	    case PositionRight:
		x = id->xmax;
		ptrans( &x, &id->ymin, fact1, fact2, x0, y0 );
		id->xmin = x - x2;
		break;
	  }
	id->ymax = id->ymin + y2;
	id->xmax = id->xmin + x2;
      }
    else if (id->type != GroupObj)
      {
	/* Splines.  Transform all of the points. */
        if (Debug&DebugParts) {
	    printf("TransformPart: transforming spline: %x\n\r",id->data);
        }
	sptr = (SPLINE *) id->data;
	pptr = &(sptr->head);
	for (i = 0;  i < sptr->numvert;  i++)
	    ptrans( &(pptr[i].x),&(pptr[i].y),fact1,fact2,x0,y0 );
      }
    else
      {
	/* Groups.  Transform all of the objects within the group. */
	BOOLEAN SaveFake = FakeGroup;
        if (Debug&DebugParts) {
	    printf("TransformPart: transforming group: %x\n\r",id->data);
        }
	FakeGroup = FALSE;
	gptr = ((OBJECT_HEADER *)id->data)->first;
	while (gptr)
	  {
	    /* if (!fake) */
	        ptrans( &(gptr->dx),&(gptr->dy),fact1,fact2,0,0 );
	    DupCheck--; /*Do this because TransformObject increments it*/
	    TransformObject( gptr, fact1, fact2, x0+gptr->dx, y0+gptr->dy,
		ptrans, FALSE);
	    gptr = gptr->next;
	  }
	FakeGroup = SaveFake;
      }
    UpdatePart(id);
  }

/*
 *  This internal routine will change the filling, nib, and opaque attributes
 *  of spline parts and the font and centering attributes of text parts.
 *
 *   togopaque - 0 = don't toggle, 1 = toggle opaque bit
 *   nib - 0-15 = nib type, -1 = don't change
 *   pat - -1 = don't change, 0 = clear, 1 = white, 2 = black, etc.
 *   newfont - -1 = don't change, 0,1,.... font number
 *   newcenter - -1 = don't change, 0,1,2 = set centering.
 *
 */
 
AttribPart( id, togopaque, nib, pat, newfont, newcenter)
	PART *id;
	short togopaque;
	enum Nib nib;
	enum Pattern pat;
	short newfont,newcenter;
  {
    SPLINE *sptr;
    short x,tf,bmin,bmax;
    OBJECT  *gptr;
    short nextfont,nextcenter,slength;
    
    if (Debug&DebugParts) {
	printf("AttribPart: Part=%x; togopaque=%d; nib=%d; pat=%d; newfont=%d; newcenter=%d\n\r",
	    id,togopaque,nib,pat,newfont,newcenter);
	printf("AttribPart: id->filnum = %d; DupCheck = %d\n\r",
		id->filnum,DupCheck);
    }
    /* Make sure that, because of multiple references, we have not already
       set attributes on this part */
    if (id->filnum == DupCheck) return;    
    id->filnum = DupCheck;
    if (id->type == TextObj) {
        if (Debug&DebugParts) {
	    printf("AttribPart: attributing text: %s\n\r",id->data);
        }
	if (newfont >0 || newcenter > 0) {
	nextfont   = newfont   < 0 ? id->subtype : newfont;
	nextcenter = (newcenter < 0 || id->typedata == 3) ? id->typedata 
							  : newcenter; 
	if (!LoadFont(nextfont)) nextfont = id->typedata;
	OpenSymbol(mainSymbol);
	AddItem( sdf, GetINum(), 0, 0, 0, 0,
		FontData[nextfont].refnumber, SDF_TEXT, id->data );
	if (InquireItem( sdf,LastINum(),0,&slength,&bmin,&bmax,0,0,0 ) == 0) {
	    printf("Internal Error: InquireItem failed (slength).\n\r");
	    DeleteItem( sdf, LastINum() );
	    FreeINum(LastINum());
	    CloseSymbol(FALSE);
	    return;
	}
	DeleteItem( sdf, LastINum() );
	FreeINum(LastINum());
	CloseSymbol(FALSE);
	switch (id->typedata)
	  {
	    /* old positioning point. */
	    case PositionLeft:
	    case 3:
		x = id->xmin;
		break;
	    case PositionCenter:
		x = ((id->xmin + id->xmax) / 2);
		break;
	    
	    case PositionRight:
		x = id->xmax;
		break;
	  }
	switch (nextcenter) {
	    case PositionLeft:
	    case 3:  /*PressEditSymbol*/
		id->xmin = x;
		break;
		    
	    case PositionCenter:
		id->xmin = x - slength/2;
		break;
			
	    case PositionRight:
		id->xmin = x - slength;
		break;
	}
	id->xmax = id->xmin+slength;
	id->subtype=nextfont;
	id->typedata=nextcenter;
	id->ymax = id->ymin+bmax-bmin;
	id->base = -bmin;
    } }
    else if (id->type != GroupObj) {
	if (togopaque || (int) nib >=0 || (int) pat >=0)
      {
        if (Debug&DebugParts) {
	    printf("AttribPart: attributing spline: %x\n\r",id->data);
        }
	/* Splines.  Attribute filling, border, and opacity */
	sptr = (SPLINE *)id->data;
	sptr->nib = (int)nib>=0?nib:sptr->nib;
	sptr->border = (int)nib>0?1:(int)nib==0?0:sptr->border;
	sptr->filled = (int)pat> 0 ? 1 : (int)pat == 0 ? 0: sptr->filled;
	sptr->opaque = sptr->opaque^togopaque;
	sptr->pat = (int)pat>0?(enum Pattern)((int)pat-1):sptr->pat;

	/* Keep the user from shooting his foot! */
	tf=0;
	if (!sptr->closed && sptr->filled) {
	    sptr->filled=0;
	    tf=1;
	    mprintf(2,"Can't fill open curves and polygons.\n\r");
	}
	if (!sptr->border && (!sptr->filled || 
			        (sptr->filled && (int)sptr->pat==0 &&
				 !sptr->opaque))) {
	    sptr->border=1;
	    sptr->nib=(enum Nib)1;
	    if (!tf) mprintf(2,"Use the erase icon to make objects go away!\n\r");
	}
      } }
    else
      {
	/* Groups.  Attribute all of the objects within the group. */
        if (Debug&DebugParts) {
	    printf("AttribPart: attributing group: %x\n\r",id->data);
        }
	gptr = ((OBJECT_HEADER *)id->data)->first;
	while (gptr)
	  {
	    DupCheck--; /*Do this because AttribObject increments it*/
	    AttribObject( gptr, togopaque,nib,pat,newfont,newcenter,FALSE);
	    gptr = gptr->next;
	  }
      }
    UpdatePart(id);
  }

/*
 *  This routine will decrement the reference count for a part.
 *  It will, if necessary, also free the storage associated with unused parts.
 */
 
DecrementRef( id )
	register PART *id;
  {
    OBJECT *ob, *ob2;
    OBJECT_HEADER *oblist;
    
    if (Debug&DebugParts)
	printf("DecrementRef: part = %x; refs was %d\n\r",id,id->refs);
    if (id == NULL)
	return;

    /* Do the easy part.  Decrement the reference count. */
    id->refs -= 1;
    if (id->refs > 0)
	return;
	
    /* Free the storage */
    if (id->type == GroupObj)
      {
	/* Decrement the count for items in the group recursively */
	oblist = (OBJECT_HEADER *) id->data;
	ob = oblist->first;
	while (ob)
	  {
	    OpenSymbol(oblist->symbol);
	    DeleteItem(sdf,ob->call);
	    FreeINum(ob->call);
	    DecrementRef( ob->part );
	    ob2 = ob;
	    ob = ob->next;
	    ob2->next = (OBJECT *) -9;	/* Crowbars */
	    ob2->prev = (OBJECT *) -17;
	    ob2->parent = (OBJECT_HEADER *) -19;
	    free( ob2 );
	  }
	CloseSymbol(FALSE);
      }
    else if (id->number < 0) {
      FreeINum(-(id->number));    
      FreeINum(-(id->number)+1);    
    }
    else if (id->number > 0) {
      FreeINum(id->number);    
    }
	
    if (id->data)
	free( id->data );
    id->data = (char *) -11;
    CloseAllSymbols();
    DeleteSymbol( sdf, id->symbol );
/*    FreeINum(id->symbol); */
    if (id->next)
	id->next->prev = id->prev;
    else
	LastPart = id->prev;
    if (id->prev)
	id->prev->next = id->next;
    else
	FirstPart = id->next;
    id->prev = (PART *)-13;
    id->next = (PART *)-15;
    free( id );
  }

/*
 * LoadFont loads the specified font into memory. Returns 0 on error
 */

LoadFont(WhichFont)
short WhichFont;

{
    if (Debug&DebugFonts)
	printf("LoadFont: WhichFont = %s; loaded was %s\n\r",
	       	 FontData[WhichFont].fontname,
	 	 FontData[WhichFont].loaded?"TRUE":"FALSE");
    /* load font if necessary */
    if (!(FontData[WhichFont].loaded)) {
	mprintf(1,"Loading font '%s', please wait ..",
	       FontData[WhichFont].fontname);
	FontData[WhichFont].refnumber =
	    DefineFont( FontData[WhichFont].fontname, NULL );
        if ((FontData[WhichFont].refnumber < 0) ||
	    (FontData[WhichFont].refnumber == 255)) {
	    mprintf(2,"Error. Can't load font %s.\n\r",
		FontData[WhichFont].fontname);
	    return(0);
	}
        else {
            FontData[WhichFont].loaded = TRUE;
            mprintf(1,"Font '%s' loaded.",FontData[WhichFont].fontname);
            return(1);
	}
    }
    else {
	return(1);
    }
}

/*
 * This routine applies various opimization heuristics to try to draw a spline
 * object with any number of simpler VGTS primitives
 */

OptSpline(id)
PART *id;

{
    register SPLINE *s = (SPLINE *)id->data;
    register POINT *p = &s->head;
    short x1,y1,x2,y2;
    short shape,size;
    short item;
    BOOLEAN optflag = FALSE;

    /* optimize filled rectangles */
    if ((s->numvert==4 && s->order==2 && s->closed && 
	 s->filled && (!s->opaque || s->pat == PatBlack)) && 
        ((p[0].x==p[1].x && p[0].y==p[3].y && 
	  p[2].x==p[3].x && p[1].y==p[2].y) ||
         (p[0].y==p[1].y && p[0].x==p[3].x && 
	  p[2].y==p[3].y && p[1].x==p[2].x))) {
	/* we have a rectangle */
	if (p[0].x < p[2].x) {
	    x1 = p[0].x;
	    x2 = p[2].x;
	}
	else {
	    x2 = p[0].x;
	    x1 = p[2].x;
	}
	if (p[0].y < p[2].y) {
	    y1 = p[0].y;
	    y2 = p[2].y;
	}
	else {
	    y2 = p[0].y;
	    y1 = p[2].y;
	}
	if (s->border)
	    item = GetSeqINum(2);
	else
	    item = GetINum();
	id->number = AddItem( sdf, item,x1,x2,y1,y2,
	    (short)s->pat+STIPPLE_OFFSET,SDF_FILLED_RECTANGLE,0);
	if (s->border) {
	    s->filled = 0;
	    size = AddItem( sdf, item+1, 0, 0, 0, 0,
		0, SDF_SPLINE, id->data );
	    s->filled = 1;
	    id->number = -id->number;
	}
	/* These are the spline fudge factors in the VGTS */
	id->xmin = x1-10;	
	id->xmax = x2+10;
	id->ymin = y1-10;
	id->ymax = y2+10;
	optflag = TRUE;
    }
    /* Optimize straight lines */
    else if (s->numvert==2) {
	/* we have a line */
	shape = ((int)s->nib)/4;
	size  = ((int)s->nib)%4;
	if (p[0].x < p[1].x) {
	    x1 = p[0].x;
	    x2 = p[1].x;
	}
	else {
	    x2 = p[0].x;
	    x1 = p[1].x;
	}
	if (p[0].y < p[1].y) {
	    y1 = p[0].y;
	    y2 = p[1].y;
	}
	else {
	    y2 = p[0].y;
	    y1 = p[1].y;
	}
	id->xmin = x1-10;	
	id->xmax = x2+10;
	id->ymin = y1-10;
	id->ymax = y2+10;
	optflag = TRUE;	/* this is valid even if we DO take the default */
	if (x1==x2) {  /*Vertical line*/
	    if (size==0 || shape==3) { /* thin line */
	        id->number = AddItem( sdf, GetINum(), x1, x1, y1, y2,
		    0, SDF_VERTICAL_LINE, 0 );
	    }
	    else if (shape != 1) { /* thick line */
		id->number = AddItem( sdf, GetINum(), x1-size,x1+size,y1,y2,
		    BLACK, SDF_FILLED_RECTANGLE, 0);
	    }
	    else { /* default */
        	id->number = AddItem( sdf, GetINum(), 0, 0, 0, 0,
	    	    0, SDF_SPLINE, id->data );
	    }
	}
	else if (y1==y2) { /* Horizontal line */
	    if (size==0 || shape==2) { /* thin line */
	        id->number = AddItem( sdf, GetINum(), x1, x2, y1, y1,
		    0, SDF_HORIZONTAL_LINE, 0 );
	    }
	    else if (shape != 1) { /* thick line */
		id->number = AddItem( sdf, GetINum(), x1,x2,y1-size,y1+size,
		    BLACK, SDF_FILLED_RECTANGLE, 0);
	    }
	    else { /* default */
        	id->number = AddItem( sdf, GetINum(), 0, 0, 0, 0,
	    	    0, SDF_SPLINE, id->data );
	    }
	}
	else if (size==0) { /* thin general line */
	    id->number = AddItem( sdf, GetINum(), p[0].x, p[1].x, p[0].y, p[1].y,
		0, SDF_GENERAL_LINE, 0 );
	}
	else { /* default */
	    id->number = AddItem( sdf, GetINum(), 0, 0, 0, 0,
	        0, SDF_SPLINE, id->data );
	}
    }

    else { 
	/* The default case - the general spline */
        id->number = AddItem( sdf, GetINum(), 0, 0, 0, 0,
	    0, SDF_SPLINE, id->data );
    }
    EndSymbol( sdf, id->symbol, 0 );
    if (!optflag)
	InquireItem( sdf, id->symbol, &(id->xmin), &(id->xmax),
	    	&(id->ymin), &(id->ymax), 0, 0, 0 );
}
