/*
 *  groups.c
 *
 *  This file contains routines that deal solely with groups.
 *
 *  A group is a part whos data field points to an object list. This object
 *  list may in turn contain object that point to other groups, etc. Thus,
 *  an arbitrary single-source DAG may be created using groups and multiple
 *  part references.
 *
 *  Gustavo Fernandez - 2/20/86
 */
 
 
/* Includes */
#ifdef UNIX
#include "stdio.h"
#else
#include "Vio.h"
#endif
#include "Vgts.h"
#include "draw.h"
 
 
/* Imports */
extern DebugItemDescriptor();
extern DebugObjectList();
extern DebugLinks();
extern short GetINum();		/* Get an item number */
extern short GetSeqINum();	/* Get a sequential set of item numbers */
extern short LastINum();	/* Return the last item number gotten
extern short FreeINum();	/* Return an item number to the freelist */
extern RedrawActivelist();	/* Redisplay the main drawing area. */
extern OBJECT *NewObject();	/* Add a new object to the object list */
extern OBJECT *AddObject();	/* Add a new object given a part */
extern DecrementRef();		/* Decrement reference count on an item */
extern PART *CreatePart();	/* Creates a new part */
 
 
/* Exports */

extern PART *CreateGroupPart(); /* Creates a new empty group part */
extern OBJECT *CreateGroup();	/* Create a new empty group */
extern MoveToGroup();		/* Move an object into a group */
extern short MoveFromGroup();	/* Move an object out of a group */
extern OBJECT *Ungroup();	/* Move up all objects andand delete a group */

/*
 *  This procedure will create an empty group part.
 */	
 
PART *CreateGroupPart()
  {
    PART *newpart;
    OBJECT_HEADER *newlist;
    if ((newlist = (OBJECT_HEADER *) malloc( sizeof(OBJECT_HEADER) )) == NULL)
      {
	printf("Out of memory.  Couldn't allocate object header for new group.\n\r");
	return(NULL);
      }
    newlist->first = NULL;
    newlist->last = NULL;
    newlist->symbol = 0;
    newpart = CreatePart(newlist,GroupObj,0,0,0,0,0,0,0);
    if (Debug & DebugGroups)
	printf ("CreateGroupPart: newpart = %x; newlist = %x\n\r",
		newpart,newlist);
    return(newpart);
  }

/*
 *  This procedure will create an empty group in the given object list
 */	
 
OBJECT *CreateGroup(dx,dy,oblist)
    short dx,dy;
    OBJECT_HEADER *oblist;
  {
    OBJECT *newgroup;
    PART *newpart;
    OBJECT_HEADER *newlist;
    if ((newpart = CreateGroupPart())==NULL)
	return(NULL);
    newlist = (OBJECT_HEADER *)newpart->data;
    newgroup = AddObject(newpart,dx,dy,FALSE,FALSE,oblist);
    if (Debug & DebugGroups)
	printf ("CreateGroup: newgroup=%x; newpart=%x; newlist=%x; parent list=%x\n\r",
		newgroup,newpart,newlist,oblist);
    return(newgroup);
  }


/*
 * This procedure will move a single object on an object list into a group.
 * 'object' becomes invalid after this call.
 * Goose is assumed to be an object in the group. It is updated to point to the
 * (possibly copied) version
 */

MoveToGroup( object, group, goose )
	OBJECT *object, *group, **goose;
{
    OBJECT_HEADER *oblist = object->parent;
    OBJECT *newob;
    PART *oldid, *id = group->part;
    short count;

    CloseAllSymbols();
    if (Debug & DebugGroups)
	printf ("MoveToGroup: Object = 0x%x, Group = 0x%x\n\r", object, group);

    if (group->part->type != GroupObj) {
	printf ("Internal Error:  MoveToGroup; 'group' is not really a group.\n\r");
	return;
    }
    if (group->parent != object->parent) {
	printf ("Internal Error:  MoveToGroup; parent mismatch.\n\r");
	return;
    }
    if (group == object) {
	printf ("Internal Error:  MoveToGroup; group == object.\n\r");
	return;
    }
#if 0
    /* This is a very weird (but possible) case. Create a group, copy it,
	take one of the copies, expand it, and try to put the other copy inside
	the expanded copy! */
    if (object->part->type == GroupObj && 
	object->part->data == group->part->data) {
    /* The MajorChange() is a hack to get rid of a strange double artifact
	which goes away when you refresh the screen. (Ye old VGTS DAG
	inconseestensee!!!) */
	MajorChange();
        count = FreshCopy(object,TRUE,&oldid,NULL,NULL);
        BringOver(object,count,FALSE,oldid);
    }
#endif 0
    count = FreshCopy(group,TRUE,&oldid,goose,NULL);
    BringOver(group,count,FALSE,oldid);
    newob = AddObject(object->part,object->dx-group->dx,object->dy-group->dy,
		FALSE,FALSE,(OBJECT_HEADER *)group->part->data);
    newob->frame = object->frame;
    EraseObject(object,FALSE);
    CloseAllSymbols();
    InquireItem( sdf, id->symbol, &(id->xmin), &(id->xmax),
	&(id->ymin), &(id->ymax), 0, 0, 0 );
}

/*
 * This procedure will move a single object in a group back onto the active
 * list. Returns 0, 1 or >1 depending upon whether there are 0, 1 or more than
 * 1 items left in the group after the move. This is NOT a count.
 * 'object' becomes invalid after this call. Next returns the next object
 * in the (possibly copied) group that had been just after the object 
 * that moved. Goose is updated if a points to a copied part.
 */
short MoveFromGroup( object, group, goose, next )
	OBJECT *object, *group, **goose, **next;
{
    OBJECT_HEADER *oblist = object->parent;
    OBJECT_HEADER *glist;
    PART *oldid, *id = group->part;
    OBJECT *newob;
    short count;

    CloseAllSymbols();
    if (Debug & DebugGroups)
	printf ("MoveFromGroup: Object = 0x%x, Group = 0x%x\n\r", object, group);

    if (group->part->type != GroupObj) {
	printf ("Internal Error:  MoveFromGroup; 'group' is not really a group.\n\r");
	return(0);
    }

    glist = (OBJECT_HEADER *)group->part->data;
    if (glist != object->parent) {
	printf ("Internal Error:  MoveFromGroup; parent mismatch.\n\r");
	return(0);
    }
    count = FreshCopy(group,TRUE,&oldid,&object,goose);
    BringOver(group,count,FALSE,oldid);
    if (next)
	*next = object->next;
    newob = AddObject(object->part,object->dx+group->dx,object->dy+group->dy,
		FALSE,FALSE,group->parent);
    newob->frame = object->frame;
    EraseObject(object,FALSE);
    CloseAllSymbols();
    InquireItem( sdf, id->symbol, &(id->xmin), &(id->xmax),
	  &(id->ymin), &(id->ymax), 0, 0, 0 );

    /* The count is returned for the benefit of routines that just want to
	know whether it is woth having this group around. */

    if (glist->first == NULL)
	return(0);
    else if (glist->first->next == NULL)
	return(1);
    else 
	return(2);
}

/*
 * This procedure will move an entire group onto the objec list and delete
 * the group if there are no other references to it. 
 * This routine returns a pointer to the first object in the object list
 * that was added by this routine. The rest can be found by following the
 * "next" chain.
 * There can be a "goosed" object which will be replaced by its new incarnation
 */

OBJECT *Ungroup( group, goose)
	OBJECT *group, **goose;
{
    OBJECT_HEADER *oblist = group->parent;
    OBJECT *ob, *tmp2, *retval = NULL;
    BOOLEAN gooseflag = (goose == NULL);
    
    CloseAllSymbols();
    if (Debug&DebugGroups)
	printf("Ungroup: Group = 0x%x; oblist = 0x%x\n\r",
		group,oblist);
    
    if (group->part->type != GroupObj) {
	printf ("Internal Error:  Ungroup; 'group' is not really a group.\n\r");
	return(0);
    }
    ob = ((OBJECT_HEADER *)group->part->data)->first;
    while (ob) {
	tmp2 = AddObject(ob->part,ob->dx+group->dx,ob->dy+group->dy,
		FALSE,FALSE,oblist);
	if (!gooseflag && *goose==ob) {
	    *goose = tmp2;
	    gooseflag = TRUE;
	}
	tmp2->frame = ob->frame;
	ob->frame = 0;
	if (retval == NULL) retval = tmp2;
	ob = ob->next;
    }
    EraseObject(group,FALSE);
    CloseAllSymbols();
    return(retval);
}
