/*
 *  checkpoint.c
 *
 *  This file contains two routines - Checkpoint and RevertToCKP. These
 *  are the heart of Draw's global UNDO facility. Note that there is also
 *  a local undo which removes single points when either a spline is being
 *  created or a multi-point operation is being performed. UNDO in these
 *  cases is taken care of by the individual functions, as they call 
 *  defpoint and indefpoint (point.c)
 *
 *  These routines handle a more global undo. Before any major operation,
 *  Checkpoint() is called. This makes a snapshot of the activelist and saves
 *  the current object (if any) into a circular stack of NUMBACKUP elements.
 *  This means that executing UNDO NUMBACKUP-1 times is effectively a redo.
 *
 *  RevertToCKP takes the last back-up list and exchanges it with the 
 *  activelist. It does likewise with the current object. Only a shallow copy
 *  of the data structures is performed. If an object is modified before
 *  falling off the end of the undo stack, a fresh copy is made. In this way,
 *  objects are really not deleted until at least NUMBACKUP other operations
 *  have been performed
 *
 *  David Kaelbling, April 1983
 *  Split off from backend.c - GAF 2/23/86
 */
 
 
/* Includes */
#ifdef UNIX
#include "stdio.h"
#else
#include "Vio.h"
#endif
#include "Vgts.h"
#include "draw.h"
 
 
/* Imports */
extern RedrawActivelist();
extern DebugItemDescriptor();
extern DebugObjectList();
extern DebugLinks();
extern SetCurrentObject();
extern short GetINum();		/* Get an item number */
extern short GetSeqINum();	/* Get a sequential set of item numbers */
extern short LastINum();	/* Return the last item number gotten
extern short FreeINum();	/* Return an item number to the freelist */
 
 
/* Exports */
extern Checkpoint();		/* Make a checkpoint. */
extern RevertToCKP();		/* Back off a transaction. */


/*
 *  Checkpoint switches backup object lists, making a backup copy of
 *  the activelist, and freeing any unreferenced parts as the backup stack
 *  overflows
 */
    
Checkpoint()
  {
    register OBJECT *ob, *obtmp, *obold;
    
    /* Do the dirty bit */
    modified = 1;
    
    /* Deallocate the old backup list, freeing unused memory. */
    CurrentBackup = (CurrentBackup + 1) % NumBackup;
    if (Debug&DebugCheckpoint)
      {
	printf("Checkpoint:  Deallocating backuplist %d\n\r", CurrentBackup);
	DebugLinks( backuplists[CurrentBackup] );
      }
    ob = backuplists[CurrentBackup]->first;
    while (ob)
      {
	obtmp = ob->next;
	DecrementRef( ob->part );
	if (ob->call > 0)
	    FreeINum(ob->call);
	free( ob );
	ob = obtmp;
      }
    
    if (Debug&DebugCheckpoint)
      {
	printf("Activelist:\n\r");
	DebugLinks( activelist );
      }
    
    /* Build the new backup list */
    backcobject[CurrentBackup] = NULL;
    backuplists[CurrentBackup]->first = NULL;
    ob = activelist->first;
    obold = NULL;
    while (ob)
      {
	if ((obtmp = (OBJECT *) malloc( sizeof(OBJECT) ))
		== NULL)
	  {
	    printf("Out of memory.  Can't make a checkpoint.\n\r");
	    backuplists[CurrentBackup]->last = obold;
	    return;
	  }
	if (obold == NULL)
	    backuplists[CurrentBackup]->first = obtmp;
	obtmp->prev = obold;
	if (obold)
	    obold->next = obtmp;
	
	/* Copy over the data */
	obtmp->part = ob->part;
	ob->part->refs += 1;
	obtmp->dx = ob->dx;
	obtmp->dy = ob->dy;
	obtmp->frame = 0;
	obtmp->parent = backuplists[CurrentBackup];
	obtmp->next = NULL;
	obtmp->call = 0;
        if (ob == CurrentObject) backcobject[CurrentBackup] = obtmp;
	
	obold = obtmp;
	ob = ob->next;
      }
    backuplists[CurrentBackup]->last = obold;
    backuplists[CurrentBackup]->symbol = mainSymbol;
    backfake[CurrentBackup] = FakeGroup;
  }

/*
 *  RevertToCKP will abort a transaction, reverting back to the previous
 *  backup list.
 */
 
RevertToCKP()
  {
    register OBJECT_HEADER *ob;
    register OBJECT *ob2;
    BOOLEAN fake2;
    
    /* Set the dirty bit */
    modified = 1;
    
    /* Swap display lists */
    ob2=CurrentObject;
    fake2=FakeGroup;
    SetCurrentObject(NULL,TRUE,FALSE,FALSE);
    ob = activelist;
    activelist = backuplists[CurrentBackup];
    backuplists[CurrentBackup] = ob;
    /* Display this version of the main drawing area */
    NewActiveList();
    SetCurrentObject(backcobject[CurrentBackup],FALSE,
	backfake[CurrentBackup],TRUE);
    backcobject[CurrentBackup]=ob2;
    backfake[CurrentBackup]=fake2;
    CurrentBackup = (CurrentBackup + NumBackup - 1) % NumBackup;
  };
