/* PROGRAM      :  dkusage
 * USAGE        :  dkusage {CONTEXT PREFIXES}
 * DESCRIPTION  :  This program will print information on block allocation
 *                 according to the storage system bitmap and fstab.
 *		   If args are given, only information on those storage servers
 *		   is given.
 *		   If no args are given, information on all the storage
 *		   servers in the group of storage servers is given.
 *		   "/fstab" and "/bitmap" in the root directory are used
 *		   for the file system table and bitmap.
 */

#include "Vstorage.h"
#include "Venviron.h"
#include "Vio.h"
#include "Vioprotocol.h"
#include "../../../servers/storage/storagedefs.h"

typedef unsigned short boolean;

#define    MAX_FILENAME_LEN 512		/* current max for unix 4.2 */
#define    DEFAULT_FSTAB_PATHNAME	"/fstab"
#define    DEFAULT_BITMAP_PATHNAME	"/bitmap"
#define    STORAGE_NAME			"[storage]"
#define    DONT_BUILD_NAME		0		/* for print_info */
#define    BUILD_NAME			1		/* for print_info */

#ifndef FALSE
#define FALSE 0
#endif
#ifndef TRUE
#define TRUE 1
#endif


/* NOTE: If things are done in a non-obvious way, it probably has historical
 * reasons: this code used to print out the actual pattern of bits which
 * represented allocated blocks within the bitmap.  Some of the old variables
 * remain, for fear of breaking something that works.  Most of the code that
 * finds out the names of the storage servers in the group of storage
 * servers was stolen from the command "listdir".	CRZ 2/7/86
 */

/* This static array contains in each POSITION the number of "1" bits
 * which are contained in the binary representation of a BYTE of that VALUE.
 * It is only used when we can use the fast method of counting, which occurs
 * when the starting and ending positions are either not specified or are
 * divisible by 8.
 */

static unsigned Bits_on[256] = {
 0,  1,  1,  2,  1,  2,  2,  3,  1,  2,  2,  3,  2,  3,  3,  4,
 1,  2,  2,  3,  2,  3,  3,  4,  2,  3,  3,  4,  3,  4,  4,  5,
 1,  2,  2,  3,  2,  3,  3,  4,  2,  3,  3,  4,  3,  4,  4,  5,
 2,  3,  3,  4,  3,  4,  4,  5,  3,  4,  4,  5,  4,  5,  5,  6,
 1,  2,  2,  3,  2,  3,  3,  4,  2,  3,  3,  4,  3,  4,  4,  5,
 2,  3,  3,  4,  3,  4,  4,  5,  3,  4,  4,  5,  4,  5,  5,  6,
 2,  3,  3,  4,  3,  4,  4,  5,  3,  4,  4,  5,  4,  5,  5,  6,
 3,  4,  4,  5,  4,  5,  5,  6,  4,  5,  5,  6,  5,  6,  6,  7,
 1,  2,  2,  3,  2,  3,  3,  4,  2,  3,  3,  4,  3,  4,  4,  5,
 2,  3,  3,  4,  3,  4,  4,  5,  3,  4,  4,  5,  4,  5,  5,  6,
 2,  3,  3,  4,  3,  4,  4,  5,  3,  4,  4,  5,  4,  5,  5,  6,
 3,  4,  4,  5,  4,  5,  5,  6,  4,  5,  5,  6,  5,  6,  6,  7,
 2,  3,  3,  4,  3,  4,  4,  5,  3,  4,  4,  5,  4,  5,  5,  6,
 3,  4,  4,  5,  4,  5,  5,  6,  4,  5,  5,  6,  5,  6,  6,  7,
 3,  4,  4,  5,  4,  5,  5,  6,  4,  5,  5,  6,  5,  6,  6,  7,
 4,  5,  5,  6,  5,  6,  6,  7,  5,  6,  6,  7,  6,  7,  7,  8 };

/******************************************************************************
 * usage prints the usage of the command and exits with an error indication.  *
 ******************************************************************************/
usage( str )
    char str[];
  {
    fprintf(stderr, "usage: %s {CONTEXT PREFIXES}\n", str );
    fprintf(stderr, "       Default: Print Disk Usage for All Storage Servers\n" );
    fprintf(stderr, "       Example: %s hood\n", str );
    exit( 1 );
  }  /* usage */

/******************************************************************************
 * get_fast uses the static array "Bits_on" to tabulate the number of free    *
 *             blocks and the number allocated blocks.  It prints out the     *
 *             results and then exits normally.  It is faster than the main   *
 *             loop which is executed only when necessary.                    *
 ******************************************************************************/
unsigned get_fast( bitmap_file, end, block )
    FILE *bitmap_file;
    register int end;
    register unsigned block;
  {
    register unsigned short c;
    register unsigned Free_blocks  = 0;
    register unsigned Alloc_blocks = 0;

/* The following line uses a trick: if "end" was never set in the command
 * line, it will retain its initial value of -1, which, when coerced into the
 * type "unsigned" becomes MAXUNSIGNED, and hence, will never affect the test.
 */
    while( ( ( c = getc( bitmap_file )) != EOF ) && ( block < (unsigned) end ) )
      {
        Alloc_blocks += Bits_on[c];
        Free_blocks  += 8;
	block += 8;
      }
    Free_blocks -= Alloc_blocks;
    fclose( bitmap_file );
    return( Alloc_blocks );
  } /* Print_terse */


unsigned get_alloc( bitmap_file, start, end )
    FILE     *bitmap_file;		/* file pointer to bitmap          */
    int      start;			/* starting block number           */
    int      end ;			/* ending block number             */
  {
    register unsigned block        = 0;    /* current block number            */
    register unsigned short c;             /* byte read from bitmap           */
    register unsigned bitsleft     = 0;    /* number of good bits left in byte*/
    register unsigned in_use_seq   = 0;    /* 0 if in free, 1 if in used sect */
    register unsigned Free_blocks  = 0;    /* number of free blocks so far    */
    register unsigned Alloc_blocks = 0;    /* number of allocated blocks      */
    unsigned bytes;                        /* number of bytes to advance      */
    unsigned done                  = FALSE;/* flag for main loop              */
    unsigned bits;                         /* number of bits to advance       */
    unsigned i                     = 0;    /* temporary variable              */

/* This block of code advances to the first byte of interest in the bitmap
 * file and then advances i to the bit within that byte which corresponds
 * to the starting block.
 */
    if( start > 0 )
      {

        /* advance to the proper byte of the bitmap file */
        bytes = (unsigned) ( start / 8 );
        while( ( bytes-- > 0 ) && ( ( c = getc( bitmap_file )) != EOF ) )
            block += 8;

        /* advance to proper bit of the current (or possibly next) byte */
        bits = start - ( (unsigned) ( start / 8 ) ) * 8;
        if( bits > 0 )
            if( ( c = getc( bitmap_file )) != EOF )
                while( bits-- > 0 )
                  {
                    c <<= 1; /* rotate left one position */
                    block++;
                  }

        /* set in_use_seq to the opposite of the current section 
         * (ie. if currently in used section, mark in_use_seq as free,
         *  and if currently in free section, mark in_use_seq as used).
         */
        if( c & 0x80 ) /* bitwise AND with hex 80 yields the leftmost bit */
            in_use_seq = 0;
        else
            in_use_seq = 1;

        /* set bitsleft, the number of unread bits left in the current byte */
        bitsleft = 8 - start + ( (unsigned) ( ( start - 1 ) / 8 ) ) * 8;
      }

    /* if start and end were never set, or they are both multiples of 8,
     * then call the more efficient function "get_fast".
     */
    if( ( start == (-1) ) || ( ( (unsigned) ( start / 8 ) ) * 8 == start ) )
        if((end == (-1) ) || ( ( (unsigned) ( end   / 8 ) ) * 8 == end   ) )
            return( get_fast( bitmap_file, end, block ) );

    while( !done && ( block < (unsigned) end ) )
      {
        if( bitsleft == 0 )
            if( ( c = getc( bitmap_file )) == EOF ) /* fetch next byte */
                done = TRUE;
	    else
                bitsleft = 8;
        if( !done )
          {
            if( c & 0x80 ) /* bitwise AND with hex 80 yields the leftmost bit */
              {
                ++Alloc_blocks;
                in_use_seq = 1;
              }
            else
              {
                ++Free_blocks;
                in_use_seq = 0;
              }
            c <<= 1; /* rotate left one position */
            block++;
            bitsleft--;
          }
      }
    fclose( bitmap_file );
    return( Alloc_blocks );
  }

/******************************************************************************
 * main program                                                               *
 ******************************************************************************/
main( argc, argv )
    int  argc;
    char *argv[];
  {
    char	ss_path[ MAX_FILENAME_LEN ];	/* context prefix of */
    unsigned	i=0;		/* loop counter for args */
    char	storage_name[ 20 ];		/* storage name */

    while( ++i < argc ) {
	if( ( ( argv[i][0] >= 'a' ) && ( argv[i][0] <= 'z' ) ) ||
	    ( ( argv[i][0] >= 'A' ) && ( argv[i][0] <= 'Z' ) ) ) {
	    sscanf( argv[i], "%s", ss_path );
	    print_info( BUILD_NAME, ss_path );
	  }
	else usage( argv[0] );
      }

    if( argc == 1 ) {
	strcpy( storage_name, STORAGE_NAME );
	get_ss( storage_name );
      }
  }

/******************************************************************************
 * print_info tries to open up the fstab in the root directory of the
 * given context prefix.  If it is not able to, it prints an error message.
 * If it is, it prints out the information contained in the fstab, and also
 * tries to open up the bitmap file to determine disk usage.
 ******************************************************************************/
print_info( build_name, ss_name )
    int	     build_name;
    char     *ss_name;
  {
    char     bitmap_path[ MAX_FILENAME_LEN ];	/* pathname of the bitmap */
    char     fstab_path[ MAX_FILENAME_LEN ];	/* pathname of the fstab */
    char     ss_path[ MAX_FILENAME_LEN ];	/* context prefix of */
						/* storage server */

    File     *bitmap_file	= NULL;		/* file pointer to bitmap */
    File     *fstab_file	= NULL;		/* file pointer to fstab */
    unsigned i			= 0;		/* loop counter for args */
    char     temp[ 1024 ];			/* storage for fstab */
    FsysConfigEntry	*fsys;			/* used to read entries */
						/* from fstab */
    int      start;				/* starting block number */
    int      end;				/* ending block number */
    unsigned alloc;				/* number of blocks allocated */
    SystemCode r;

    if( build_name ) {
	strcpy( ss_path, "[storage/" );
	strcat( ss_path, ss_name );
	strcat( ss_path, "]" );
      }
    else {
	strcpy( ss_path, ss_name );
      }

    printf( "%s:\n", ss_path );

    strcpy( fstab_path, ss_path );
    strcat( fstab_path, DEFAULT_FSTAB_PATHNAME );

    strcpy( bitmap_path, ss_path );
    strcat( bitmap_path, DEFAULT_BITMAP_PATHNAME );

    fstab_file = Open( fstab_path, FREAD, &r );		/* open in read mode */
    if( r != OK ) {
	fprintf( stderr, "    Error in opening %s: ", fstab_path );
	fprintf( stderr, "%s\n", ErrorString( r ) );
	if( r == NOT_FOUND )
	    fprintf( stderr, "    May not be a V Storage Server\n" );
	goto leave;
      }

    bitmap_file = Open( bitmap_path, FREAD, &r );	/* open in read mode */
    if( r != OK ) {
	fprintf( stderr, "    Error in opening %s: ", bitmap_path );
	fprintf( stderr, "%s\n", ErrorString( r ) );
	goto leave;
      }

    for( i=0; i<1024; i++ )
	temp[i] = getc( fstab_file );
    fclose( fstab_file );
    fsys = ( FsysConfigEntry * )temp;

    printf( "    %-16.16s" , "Filesystem" );
    printf( "%6s" ,  "drive" );
    printf( "%8s" ,  "kbytes" );
    printf( "%8s" ,  "used" );
    printf( "%8s" ,  "avail" );
    printf( "%9s" ,  "capacity" );
    printf( "  %s\n", "Mounted on");

    do
      {
	printf( "    %-16.16s", ss_path  );
#ifdef LITTLE_ENDIAN
	ByteSwapLongInPlace( &( fsys->drive ),
			sizeof( FsysConfigEntry ) - FSYS_NAME_LEN );
#endif
	printf( "%6d", fsys->drive );
	printf( "%8d", fsys->length );
	start = fsys->start;
	end = fsys->start + fsys->length;
	alloc = get_alloc( bitmap_file, start, end );
	printf( "%8d", alloc );
	printf( "%8d", fsys->length - alloc );
	if( fsys->length - alloc == 0 )
	    printf( "%6.0f%%", 0.0 );
	else
	    printf( "%6.0f%%", (double) alloc / (double) fsys->length * 100.0 );
	printf("    %s\n", fsys->name );
	fsys++;
      }
    while( strlen( fsys->name ) > 0 );

leave:
    fclose( bitmap_file );
  }
