/************************************************************************/
/*									*/
/*                      (C) COPYRIGHT 1985                              */
/*			BOARD OF TRUSTEES				*/
/*			LELAND STANDFORD JUNIOR UNIVERSITY		*/
/*			STANFORD, CA. 94305, U.S.A.			*/
/*									*/
/************************************************************************/

/* This file, stack.c, is part of the V debugger.  It contains the routines
 * necessary to implement the StackDump command.
 */

#include <b.out.h>
#include <Vioprotocol.h>
#include "Vdb.h"

extern PrintAddress();

/*
 * The VAX stack looks like this after a calls instruction:
 *
 *           |--------------------------------|
 *           |           args                 |
 *           |        (longwords)             |
 *           |            .                   |
 *           |--------------------------------|
 *     AP -> |        number of args          |
 *           |--------------------------------|
 *           |       longword padding         |        |
 *           |--------------------------------|        |
 *           |             .                  |        |
 *           |             .                  |        |
 *           |       saved registers          |        |
 *           |     (according to mask)        |        V
 *           |             .                  |      lower
 *           |             .                  |    addresses
 *           |--------------------------------|
 *           |          saved PC              |
 *           |--------------------------------|
 *           |          saved FP              |
 *           |--------------------------------|
 *           |          saved AP              |
 *           |--------------------------------|
 *           | pad length, entry mask, PSW    |
 *           |--------------------------------|
 *     FP -> |             0                  |
 *           |--------------------------------|
 *
 */


/* StackDump prints the sequence of calls that led to the routine in which
 *  execution is halted, giving the address of each call and the current values
 *  of the first MAXNUMPARAMSTOPRINT parameters of each routine.
 */

StackDump( disassembledCode, maxLines )
	char *disassembledCode;         /* Not used */
	int maxLines;
  {
    memaddr Fp;                         /* Frame pointer (register 13) */
    memaddr Ap;                         /* Arg pointer (register 12) */
    memaddr Pc;                         /* Program counter (register 15) */
    unsigned    numArgs;                /* Number of arguments in CALLS */
    unsigned    arg;                    /* Current argument */
    unsigned    argc;                   /* Current argument number */
    SystemCode error;
    long disp;                          /* Displacement in adr mode */
    int offset;                         /* Offset of opcode relative to pc */

    fprintf(stdout, "\nStack trace:\n");
    Ap = (memaddr) SavedRegisters[AP]; /* Start with current ap & fp */
    Fp = (memaddr) SavedRegisters[FP];

    if (! ((unsigned)Fp >= 0x200 && Fp < DebugeeTeamSize) )
      {
	fprintf(stdout, "Invalid frame pointer (%x).\n", Fp);
	return;
      }

    while ((unsigned)Fp >= 0x200 && Fp < DebugeeTeamSize)
      {
	OutputBuffer.pnt= 0;
	OutputBuffer.col= 1;
	cbfp = &OutputBuffer;

	Pc = GetMemLong( Fp + 16, &error);  /* Get saved PC */
	if( error != OK )
	  {
	    strput("can't get pc\n");
	    goto bad_return;
	  }
	/*
	 * This assumes the standard calling sequence generated by
	 * the C compiler: a calls instruction with a literal argument
	 * count and longword relative addressing.  That means the opcode
	 * is at pc-7, and argument count at pc-6, the address mode at pc-5,
	 * and the displacement at pc-4 through pc-1.
	 *
	 * If pc-7 does not contain the opcode for calls or if pc-6
	 * does not specify literal mode or if pc-5  does not
	 * specify longword displacement off the pc, then we
	 * don't really know what address was called.  This routine
	 * could be modified to look for other modes if necessary.
	 */
	offset = 0;     /* In case we don't know adr of calling instruction */
	if (GetInstrWord(Pc - 7, &error) == 0xFB /* calls */ &&
			error == OK &&
		(GetMemByte(Pc - 6, &error) & 0xC0) == 0 &&
			error == OK &&  /* Literal mode */
		(GetMemByte(Pc - 5, &error) & 0xFF) == 0xEF &&
			error == OK &&  /* Longword relative mode */
		((disp = GetMemLong(Pc - 4, &error)), error == OK))
	{
	    PrintSym(Pc + disp, Symfile);
	    offset = -7;    /* calls instruction is at pc-7 */
	}
	else
	{
	    bufput('?');    /* Don't know what address was called */
	}
	bufput('(');        /* Start of argument list */
	/* AP points to word containing number of arguments */
	numArgs = GetMemLong( Ap, &error);
	if( error != OK )
	  {
	    strput("can't get number of arguments\n");
	    goto bad_return;
	  }
	/*
	 * This assumes arguments were pushed on the stack in reverse
	 * order, i.e. the last argument first.  This method, which the
	 * Unix C compiler uses, is recommended in the VAX architecture
	 * manual.
	 */
	for (argc = 0; argc < numArgs; argc++)
	  {
	    if (argc > 0)
		strput(",");    /* Separate arguments with commas */
	    if (argc > MAXNUMPARAMSTOPRINT)  /* Don't print any more */
	    {
		strput("...");  /* Show that some arguments are elided */
		break;
	    }
	    Ap += 4;                    /* Bump AP to next argument */
	    arg = GetMemLong( Ap, &error);
	    if( error != OK )
	      {
		strput("can't get argument\n");
		goto bad_return;
	      }
	    numout(arg);                /* Print in current base */
	  }
	strput(") Called from ");       /* End of argument list */
	PrintSym(Pc + offset, Symfile); /* Address of calls instruction */
	/*
	 * If we don't know how far back the calling instruction was
	 * from the return pc (which should point to the next instruction),
	 * print a minus sign followed by a question mark to show the
	 * uncertainty.
	 */
	if (offset >= 0)
	    strput("-?");
	bufput('\n');
	bufput('\0');
	fputs(OutputBuffer.buf, stdout);  /* Output the string */

	if (Pc > 0x400000)
	    return;
	if (--maxLines == 0)
	    return;

	Ap = GetMemLong( Fp + 8, &error);   /* Get previous ap */
	if( error != OK )
	  {
	    fprintf( stdout, "can't get ap\n");
	    return;
	  }
	Fp = GetMemLong( Fp + 12, &error);   /* Get previous fp */
	if( error != OK )
	  {
	    fprintf( stdout, "can't get fp\n");
	    return;
	  }
      }
    return;

bad_return:             /* Come here on error */
    bufput('\0');
    fputs(OutputBuffer.buf, stdout);
  }

