/************************************************************************/
/*									*/
/*			(C) COPYRIGHT 1983				*/
/*			BOARD OF TRUSTEES				*/
/*			LELAND STANFORD JUNIOR UNIVERSITY		*/
/*			STANFORD, CA. 94305, U.S.A.			*/
/*									*/
/************************************************************************/

/*
 * Marvin Theimer, Eric Berglund,  5/83
 *
 * If Addr24 is defined, mask off top 8 bits of addresses. Per Bothner. Mar 84
 */

/* This file, memoryaccess.c, is part of the V debugger.  It maintains a
 * buffer containing a copy of INSTRBLK bytes of program memory.  It includes:
 * GetFromMemBuffer, GetMemByte, GetMemWord, GetMemLong, GetInstrWord, PutMem,
 * PutMemData (the replacement for PutMemByte, PutMemWord, PutMemLong, and
 * PutMemString), InvalidMemBuffer, ValidInstrPc, ValidAddress, 
 * and QueryMemBuffer.
 */


#include <b.out.h>
#include <Vio.h> 
#include "Vdb.h"


unsigned char MemBuffer[INSTRBLK] = { 0 };
				/* Buffer for program memory. */
memaddr bfrbegpc, bfrendpc;     /* First and last pc's in current MemBuffer. */
int BfrValid = 0;		/* Signals that mem. buffer may be invalid. */


/* GetFromMemBuffer ensures that the byte pointed to by pc is in
 * MemBuffer unless an error condition occurs. It reads the next
 * INSTRBLK bytes starting at that pc if *pc is not already in MemBuffer,
 * returning an error code if, for some reason, the contents of pc can
 * not be obtained.
 * Other routines which must access program memory do so through the
 * MemBuffer--GetFromMemBuffer is the only one with a MoveFrom instruction,
 */

SystemCode GetFromMemBuffer( pc )
	memaddr pc;
  {
    SystemCode error;
    int blocksize, bytestoend;

    DoFixAddress(&pc);
    if (BfrValid && (bfrbegpc <= pc) && (pc <= bfrendpc))
      return(OK);

    if( (pc < LinkOrigin) || (pc >= DebugeeTeamSize-LoadOrigin+LinkOrigin) )
      return(BAD_ADDRESS);

    bytestoend = (int)( DebugeeTeamSize-LoadOrigin+LinkOrigin - (int) pc );
    if( bytestoend < INSTRBLK )
      blocksize = bytestoend;
    else
      blocksize = INSTRBLK;

    error = MoveFrom( CurExceptionPid, MemBuffer,
		    pc-LinkOrigin+LoadOrigin, blocksize );
    if (error != OK)
      {
        BfrValid = 0;
        fprintf( stdout, "Debugger: Error while trying to access debuggee's memory\n %s\n", ErrorString( error ) );
      }
    else
      {
        bfrbegpc = pc;
        bfrendpc = pc + blocksize - 1;
        BfrValid = 1;			/* Valid mem. buffer */
      }
    return(error);
  }



/* Fetch the byte at loc. */

char GetMemByte( loc, error )
	memaddr loc;
	SystemCode *error;
  {
    DoFixAddress(&loc);
    if( (*error = GetFromMemBuffer(loc)) != OK)	return(-2);
    return( MemBuffer[ loc - bfrbegpc ] & 0xFF );
  }


/* Fetch the word at loc. */

short GetMemWord( loc, error )
	memaddr loc;
	SystemCode *error;
  {
    short word;

    DoFixAddress(&loc);
    if( (*error = GetFromMemBuffer(loc)) != OK ) return(-2);
    word = ( (short) MemBuffer[ loc - bfrbegpc] ) & 0xFF;
    if( (*error = GetFromMemBuffer(loc+1)) != OK ) return(-2);
    word |= ( ((short) MemBuffer[ loc+1 - bfrbegpc ]) << 8) & 0xFF00;
    return( word );
  }


/* Fetch a long at loc. */

long GetMemLong( loc, error )
	memaddr loc;
	SystemCode *error;
  {
    long lword;
    DoFixAddress(&loc);
    if( (*error = GetFromMemBuffer(loc) ) != OK ) return(-2);
    lword = ( (long) MemBuffer[ loc - bfrbegpc ] ) & 0xFF;
    if( (*error = GetFromMemBuffer(loc+1) ) != OK ) return(-2);
    lword |= ( ( (long) MemBuffer[ loc+1 - bfrbegpc ] ) << 8 ) & 0xFF00;
    if( (*error = GetFromMemBuffer(loc+2) ) != OK ) return(-2);
    lword |= ( ( (long) MemBuffer[ loc+2 - bfrbegpc ] ) << 16 ) & 0xFF0000;
    if( (*error = GetFromMemBuffer(loc+3) ) != OK ) return(-2);
    lword |= ( ( (long) MemBuffer[ loc+3 - bfrbegpc ] ) << 24 ) & 0xFF000000;
    return( lword );
  }


/*
 * Determines if the instruction is a breakpoint instruction and returns
 * its true value.
 */

int  GetInstrWord( pc, error )
    memaddr pc;
    SystemCode *error;
  {
    int bpno;
    char instrVal;

    instrVal = GetMemByte(pc, error);
    if ( instrVal == BPINST )
      {
	bpno = FindUserBreakpoint(pc);
	if (bpno != 0)
	  instrVal = bp[bpno].oldinst;
      }
    return( (unsigned int)(instrVal & 0xFF) );
  }


long GetMem( type, addr, error )
enum stype type;
char *addr;
SystemCode *error;
  {
    switch(type)
      {
    case NULLTYPE:
    case LONGTYPE:
        return(GetMemLong(addr, error));

    case CHARTYPE:
    case BYTETYPE:
    case INSTTYPE:
    	return(GetMemByte(addr, error));

    case WORDTYPE:
    	return(GetMemWord(addr, error));

    default:
    	return(*error = BAD_ARGS);
      }
  }

/*
 * Here to PutMem data
 */
int PutMem( type, adrs, data )
	short type;
	memaddr adrs;
	long data;
{
    int bpno;
    char byte;
    short word;
    char instruction;

    switch (type)
      {
	case 0:			/* Default PutMem */
	case LONGTYPE:
	    return(PutMemLong(adrs, &data));

	case CHARTYPE:
	case BYTETYPE:
	    byte = (char) data;
	    return(PutMemByte(adrs, &byte));

	case WORDTYPE:
	    return(PutMemWord(adrs, &word));

	case INSTTYPE:
	    bpno = FindUserBreakpoint( adrs);
	    if ( bpno != 0 )
	      {
		bp[ bpno ].oldinst = (char) data;
		return(2);
	      }
	    else
	      {
		instruction = (char) data;
		return(PutMemWord(adrs, &instruction));
	      }

	default:
	    return( -2 );
      }
}

/*
 * Function to deposit data in team space.
 */

PutMemData( address, data, nbytes )
	memaddr address;
	char *data;
	int nbytes;
  {
    SystemCode error;
    int i, indx;

    DoFixAddress(&address);
    if( address < LinkOrigin ||
      address + nbytes - 1 >= DebugeeTeamSize-LoadOrigin+LinkOrigin )
      return( -2 );
    error = MoveTo( CurExceptionPid, address-LinkOrigin+LoadOrigin,
		  data, nbytes );
    if( error != OK ) return( -2 );

    /* Deposit data at loc in MemBuffer. */
    for (i = 0; i < nbytes; i++)
	if (BfrValid && (bfrbegpc <= address) && (address <= bfrendpc))
	  {
	    indx = address++ - bfrbegpc;
            MemBuffer[indx] = *data++;
	  }
     return( nbytes );
  }



/*
 * InvalidMemBuffer:
 * Invalidates the memory buffer cache.
 */

InvalidMemBuffer()
  {
    BfrValid = 0;
  }


/*
 * Checks if pc points to a legal instr.
 */

ValidInstrPc(pc)
    memaddr pc;
  {
    int legal;

    DoFixAddress(&pc);

    if ((pc >= LinkOrigin) &&
	(pc < DebugeeTeamSize-LoadOrigin+LinkOrigin))
      {
        dasm(pc, Symfile, &legal);
	if (legal)
	    return(1);
      }
    return(0);
  }


ValidAddress( address )
	memaddr address;
  {
    DoFixAddress(&address);
    return( address >= LinkOrigin &&
	    address < DebugeeTeamSize-LoadOrigin+LinkOrigin );
  }



QueryMemBuffer()
  {
    printf("BfrValid %d  bfrbegpc %x  bfrendpc %x \n",
	BfrValid, bfrbegpc, bfrendpc );
  }
