/************************************************************************/
/*									*/
/*			(C) COPYRIGHT 1983				*/
/*			BOARD OF TRUSTEES				*/
/*			LELAND STANDFORD JUNIOR UNIVERSITY		*/
/*			STANFORD, CA. 94305, U.S.A.			*/
/*									*/
/************************************************************************/

/* 			68000 Disassembler
 *    			    V.R. Pratt
 *    			    Jan., 1981
 *			Modified by David Cheriton, 
 *			Eric Berglund, Marvin Theimer.
 *                      Modified for VAX, June 1985.
 */


/* January 13, 1983: Eric J. Berglund
 *   I have just taken a copy of Mr. Pratt's and Mr. Cheriton's disassembler
 *   and modified it to search for the symbolic representations of addresses
 *   on a file resident on a VAX, while the disassembler executes on a SUN
 *   workstation.  The fileid for this file must be passed to the dasm routine
 *   and is assumed to be in the usual format of .r files.
 */


/* This is a compact disassembler intended for use with a microvax.
 *
 * This disassembler uses the instruction table from adb, which in turn
 * depends on definitions used in as.  The instruction table lists for
 * each instruction the opcode, the number of operands, and their types
 * (eg read-only shortword, byte branch displacement).
 * 
 * Anyone reading this code should be moderately familiar with the VAX
 * architecture, and in particular with the various addressing modes.
 */


/* This file includes getlong, readfileheader, dasminit, dasm,
 * InitOpcodeTab, translate, translate_arg,
 * hex, bhex.
 */


#include <sys/types.h>
/* This file defines the structure insttab */
/* #include "/usr/src/bin/as/instrs.h"  /* Host may not have Unix sources */
#include "instrs.h"

struct insttab insttab[] = {
/* This file contains the definitions of the opcodes */
/* #include "/usr/src/bin/adb/instrs.adb" /* Host may not have Unix sources */
#include "instrs.adb"
OP(0,0,0,0,0,0,0,0,0,0)         /* Terminate the list */
};

/*
 * The table is not sorted, so we keep arrays of pointers
 * indexed by op code.  Actually there are 3 tables: one for the
 * 1-byte opcodes (most of them), one for the 2-byte opcodes
 * starting with 0xFD, and one for the 2-byte opcodes starting
 * with 0xFF.  A 0 pointer means there is no such instruction.
 */
struct insttab *CORE_Opcode_Tab[256],
	       *ESCD_Opcode_Tab[256],
	       *ESCF_Opcode_Tab[256];

#include <b.out.h>
#include <Vio.h>
#include <Vtermagent.h>
#include "Vdb.h"

#ifdef MC68000
#define MACHINE ".m68k"
#else  MC68000
#define MACHINE ".vax"
#endif MC68000


/*
 * The following table is indexed by the "type" field in the instrs.h file
 * and maps the those values into the ones used internally.
 */
struct operand_type {
    enum stype oper_type;
    int oper_nbytes;
} operand_types[] =
{
    BYTETYPE,   1,      /* TYPB byte integer */
    WORDTYPE,   2,      /* TYPW word integer */
    LONGTYPE,   4,      /* TYPL long integer */
    QUADTYPE,   8,      /* TYPQ quad integer */
    OCTATYPE,   16,     /* TYPO octa integer */
    FFLTTYPE,   4,      /* TYPF F float */
    DFLTTYPE,   8,      /* TYPD D float */
    GFLTTYPE,   8,      /* TYPG G float */
    HFLTTYPE,   16,     /* TYPH H float */
};


memaddr mpc,                    /* Main program counter. */
	spc;                    /* Start program counter. */



extern InitCache(), PrintSym(), numout(), bufput();





/* Getlong reads a long from the a.out file--least significant byte first,
 *   of course--and returns it.
 */

long getlong( fileptr )
  File *fileptr;
{
  long value;

  value = getc( fileptr ) & 0xFF;
  value |= ( getc( fileptr ) << 8 ) & 0xFF00;
  value |= ( getc( fileptr ) << 16 ) & 0xFF0000;
  value |= ( getc( fileptr ) << 24 ) & 0xFF000000;
  return( value );
}






/* Readfileheader reads the 8 longs from the beginning of the provided
 *   file and stores them in the appropriate subfield of the filhdr structure.
 */

readfileheader( fileptr )
  File *fileptr;
{
  filhdr.fmagic = getlong( fileptr );
  filhdr.tsize = getlong( fileptr );
  filhdr.dsize = getlong( fileptr );
  filhdr.bsize = getlong( fileptr );
  filhdr.ssize = getlong( fileptr );
  filhdr.entry = getlong( fileptr );
  filhdr.rtsize = getlong( fileptr );
  filhdr.rdsize = getlong( fileptr );
}





/* Dasminit provides the fileid of the newly created opened a.out file.
 * Exits program on error conditions.
 */

#define DebuggerPadMode		(CR_Input+LF_Output)

File *dasminit(fileServer, fileId, fileName  )
	ProcessId fileServer;
	InstanceId fileId;
	char *fileName;
  {
    File *fileptr;
    SystemCode error;
    char c;

    fileptr = OpenFile(fileServer, fileId, FREAD, &error);

    if( fileptr == NULL  ||  error != OK )
      {
	fprintf( stderr,
	 "Debugger couldn't open file given fileServer and fileId.\n" );
	fprintf( stderr, "Trying fileName %s%s.\n", fileName, MACHINE );
	fflush( stderr );
	fileptr = Open( strcat( fileName, MACHINE ), FREAD, &error );
      }

    while ((fileptr == NULL) || (error != OK))
      {
	char filename[ 256 ];

        fprintf( stderr, "Debugger: couldn't open symbol table file ");
	fprintf( stderr, "for program to be debugged.\n");
        fprintf( stderr, "Enter name of file to use <CR to give up>: ");
	Flush( stdout );
	ModifyPad( stdin, CR_Input+Echo+LineBuffer+LF_Output ); 
	c = getchar();
	if( c == EOF && stdin->lastexception == BAD_BLOCK_NO )
	  {
	    ClearEof( stdin );
	    Resynch( stdin );
	  }
	else
	    ungetc( c, stdin );

	gets( filename );
	ModifyPad( stdin, DebuggerPadMode );
	if (*filename == '\0') ExitDebugger();
	fileptr = Open(filename, FREAD, &error);
      }
    readfileheader( fileptr );
    InitCache();
    InitOpcodeTab();
    return( fileptr );
  }


/* Initialize the opcode tables. */
static InitOpcodeTab()
{
    struct insttab *p;
    for (p = insttab; p->iname; p++)
    {
	switch(p->eopcode)
	{
	  case CORE:
	  case NEW:     /* Are these any different?? */
	    CORE_Opcode_Tab[p->popcode & 0xFF] = p;
	    break;

	  case ESCD:
	    ESCD_Opcode_Tab[p->popcode & 0xFF] = p;
	    break;

	  case ESCF:
	    ESCF_Opcode_Tab[p->popcode & 0xFF] = p;
	    break;
	}
    }
}

/* Finally, the main entry point to this collection of routines.
 * Given a memory address, we interpret the byte(s) at that address
 * as a VAX instruction and disassemble them putting the disassembled
 * "source" text in one output buffer, and the hex ascii representation
 * in another. The number of bytes displayed is returned.  In the case
 * where no legal instruction could be found, 1 byte is displayed as data.
 */
/* Symfile and symfile are treated very strangely in here!!!*/

int dasm ( address , symfile, legal)
    memaddr address;
    File *symfile;
    int *legal;

  {
     SystemCode error;
     int instrVal;

     Symfile = symfile;			/* Make into a local global to avoid
					   having to pass it around. */

     spc = mpc = address;             /* EJB: mpc will be changed by instr */
     instrVal = GetInstrWord( mpc++, &error );
				/* Get the true value of the first word of
				   the instruction, independent of whether a
				   breakpoint has been set at address. */
     if( error != OK )
       {
	 *legal = 0;
	 return( -1 );
       }

     OutputBuffer.pnt = 0;
     OutputBuffer.col = 1;
     cbfp = &OutputBuffer;

     PrintSym((long)spc, Symfile); /* Print address as symbol */
     if (cbfp->col < 8) bufput('\t');
     bufput('\t');
     translate(instrVal);                   /* dis-assemble the instruction */
     bufput('\0');

     HexBuffer.pnt = 0;
     HexBuffer.col = 1;
     cbfp = &HexBuffer;

     hex((long)spc);			/* address in hex */
     bufput('\t');

     bhex(instrVal);
     spc++;
     while (spc < mpc)
       {
	 bhex( GetInstrWord( spc++, &error ) );   /* EJB--output hex, byte by byte */
       }
     while (cbfp->col < 32) bufput('\t');
     bufput('\0');

     if (mpc == address)
       {			/* No legal instr. found.  Treat as data. */
         *legal = 0;
	 return(1);
       }
     else
       {
         *legal = 1;
         return((int) (mpc - address) );
       }
  }





/* The heart of this file is translate(), which returns the assembler form
 * of the instruction presently pointed to by mpc.  As a side effect it 
 * advances mpc to the byte following the instruction unless the instr.
 * is not a legal one.
 */



static
translate(opcode)
    unsigned int opcode;
{
    struct insttab *p;
    int argc;
    SystemCode error = OK;

    /* See if this is an extended (2-byte) opcode */
    switch(opcode)
    {
	case ESCD:  /* Look at next byte to find opcode */
	    opcode = GetInstrWord( mpc++, &error );
	    p = ESCD_Opcode_Tab[opcode];
	    break;

	case ESCF:  /* Look at next byte to find opcode */
	    opcode = GetInstrWord( mpc++, &error );
	    p = ESCF_Opcode_Tab[opcode];
	    break;

	default:        /* Anything else represents the opcode itself */
	    p = CORE_Opcode_Tab[opcode];
	    break;
    }

    if( error != OK )
      {
	return( -1 );       /* Should give a message here */
      }

    if (p == 0)         /* 0 means this opcode undefined */
    {
	mpc = spc;
	return;
    }
    strput(p->iname);   /* Instruction mnemonic */
    bufput('\t');
    for (argc = 0; argc < p->nargs; argc++) /* Print each operand */
    {
	translate_arg(p->argtype[argc], opcode);
	if (argc < p->nargs-1)  /* No comma after last operand */
	    bufput(',');        /* Separate operands with commas */
    }
}


/* Disassemble operands */
static
translate_arg(argAccessAndType, opcode)
    u_char argAccessAndType;    /* Argument access and type */
    int opcode;                 /* Opcode (needed to tell operand type) */
{
    unsigned modebyte;  /* Mode/register byte */
    unsigned mode;      /* Mode number */
    unsigned reg_num;   /* Register number */
    unsigned int argaccess; /* Argument access type (eg read, dispacement) */
    unsigned int argtype; /* Argument type, e.g. byte or word */
    char *reg_name;     /* Register name, e.g. "r2" or "sp" */
    enum stype type;    /* Type of operand */
    int len;            /* Length of operand */
    long disp;          /* Displacement for modes 10-15 */
    char sbuf[100];     /* String buffer for output */
    struct operand_type *ot;
    SystemCode error;
    extern char *DebuggerSymbolNames[];

    sbuf[0] = '\0';     /* Initialize sbuf to null string */

    /* Find out what kind of operand to expect */
    argaccess = A_ACCEXT(argAccessAndType);
    argtype = A_TYPEXT(argAccessAndType);

    /* Now describe these in our terms */
    ot = &operand_types[argtype];
    type = ot->oper_type;   /* Type of operand (eg byte, d_float) */
    len = ot->oper_nbytes;  /* Length of operand (in bytes) */

    switch(argaccess)
    {
      case ACCB:                /* Branch displacement */
	disp = GetMem(type, mpc, &error);   /* Get the displacement */
	mpc += len;                         /* Update pc */
	PrintSym(mpc + disp, Symfile); /* Always instruction adr */
	break;

      default:                  /* Normal addressing modes */
	/* Get addressing mode */
	modebyte = (unsigned) GetMemByte( mpc, &error ) & 0xff;
	mpc++;          /* Advance pc past mode byte */
	if( error != OK )
	    return( -1 );

	mode = modebyte >> 4;       /* High-order four bits of byte */
	reg_num = modebyte & 0x0f;  /* Low-order four bits of byte */
	reg_name = DebuggerSymbolNames[reg_num];    /* Name of register */

	/* These are the addressing modes for the VAX */
	if (mode >= 8 && reg_num == PC)   /* Program counter addressing */
	{
	    switch(mode)
	    {
	      case 8:       /* Immediate mode */
		disp = GetMem(type, mpc, &error);
		mpc += len;
		/* Until GetMem is modified to deal with floating point... */
		/* KLUDGE */ /* the following cast is not really authorized,
				but is only temporary until floating point
				numbers are accomodated - then the comparison
				can be taken out */
		bufput('$');    /* Indicate immediate mode */
		if ((int)type > (int)LONGTYPE)
		    strput("(floating number)");
		else
		    PrintSym(disp, Symfile);
		break;

	      case 9:       /* Absolute mode */
		disp = GetMemLong(mpc, &error);
		mpc += 4;       /* Bump pc past address */
		strput("*$");   /* Indicate absolute mode */
		PrintSym(disp, Symfile);
		break;

	      case 10:      /* Byte relative mode */
	      case 11:      /* Byte relative deferred mode */
		disp = GetMemByte(mpc, &error);
		mpc += 1;       /* Advance pc past displacement */
		goto pcdisp;

	      case 12:      /* Word relative mode */
	      case 13:      /* Word relative deferred mode */
		disp = GetMemWord(mpc, &error);
		mpc += 2;       /* Advance pc past displacement */
		goto pcdisp;

	      case 14:      /* Long relative mode */
	      case 15:      /* Long relative deferred mode */
		disp = GetMemLong(mpc, &error);
		mpc += 4;       /* Advance pc past displacement */

	    pcdisp:
		if (mode & 0x1) /* Odd-numbered modes are deferred */
		    bufput('*');
		PrintSym(disp + mpc, Symfile);
	    }
	}
	else                /* Regular (non-PC) addressing modes */
	{
	    switch(mode)
	    {
	      case 0:           /* Literal modes */
	      case 1:
	      case 2:
	      case 3:
		bufput('$');        /* Show literal data */
		numout((unsigned long)modebyte); /* The operand is the value */
		break;

	      case 4:           /* Index mode */
		translate_arg(argtype, opcode); /* Print the rest */
		sprintf(sbuf, "[%s]", reg_name); /* Then the index */
		break;

	      case 5:           /* Register mode */
		sprintf(sbuf, "%s", reg_name);
		break;

	      case 6:           /* Register deferred mode */
		sprintf(sbuf, "(%s)", reg_name);
		break;

	      case 7:           /* Autodecrement mode */
		sprintf(sbuf, "-(%s)", reg_name);
		break;

	      case 8:           /* Autoincrement mode */
		sprintf(sbuf, "(%s)+", reg_name);
		break;

	      case 9:           /* Autoincrement deferred mode */
		sprintf(sbuf, "*(%s)+", reg_name);
		break;

	      case 10:          /* Byte displacement */
	      case 11:          /* Byte displacement deferred */
		disp = GetMemByte(mpc, &error);
		mpc += 1;       /* Advance pc past displacement */
		goto dispmode;

	      case 12:          /* Word displacement */
	      case 13:          /* Word displacement deferred */
		disp = GetMemWord(mpc, &error);
		mpc += 2;       /* Advance pc past displacement */
		goto dispmode;

	      case 14:          /* Longword displacement */
	      case 15:          /* Longword displacement deferred */
		disp = GetMemLong(mpc, &error);
		mpc += 4;       /* Advance pc past displacement */

	      dispmode:
		/* Put out a * for the odd-numbered (deferred) modes */
		if (mode & 0x1)
		    bufput('*');
		signed_numout(disp);
	    sprintf(sbuf, "(%s)", reg_name);    /* Register number */
	    break;
	    }                   /* End of switch (mode) */
	}                   /* End of else */
    }                   /* End of switch(argaccess) */
    strput(sbuf);       /* Now print it */
}



/* Convert a long to its ascii hex representation regardless of the
 * "current radix", and send it to the output buffer.
 */

hex(n)
  register unsigned long n;
{
    if ( n > 15 ) hex( n >> 4 );
    n &= 15;
    bufput((char)( n + ( n>9 ? '7': '0')));
}





/* Convert a byte to its ascii hex representation in exactly two
 * digits, and send it to the output buffer. Insert leading zeroes if 
 * required.
 */

bhex(n)
  register unsigned int n;
{
  register int i;
  register char c;

    for( i = 4; i >= 0; i -= 4)
    {
	c = n>>i & 15;
	bufput( c + ( c>9? '7': '0'));
    }
}
