/************************************************************************/
/*									*/
/*			(C) COPYRIGHT 1983				*/
/*			BOARD OF TRUSTEES				*/
/*			LELAND STANFORD JUNIOR UNIVERSITY		*/
/*			STANFORD, CA. 94305, U.S.A.			*/
/*									*/
/************************************************************************/

/*
 * Marvin Theimer, Eric Berglund,  5/83
 */


/*
 * Top level routine for Vdb.
 *
 * Vdb is an assembler-level debugger for C programs running under the 
 * V system on MC68000's.  It expects the V exec to provide initial
 * program setup of itself and the program to debug, and to provide file 
 * io to the .r load file of the program to be debugged.  The load file is
 * expected to contain the symbol table produced by ld68.
 *
 * This file includes the routines main, TryToLoadTeam, SetUpArguments,
 * SetDefaults, ExitDebugger, and Vdebug.
 */


#include <b.out.h>
#include <Vexceptions.h>
#include <Vprocess.h>
#include <Vteams.h>
#include <Vtermagent.h>
#include <Vioprotocol.h>
#include <Vdirectory.h>
#include <Vgroupids.h>
#include "Vdb.h"

extern RootMessage *RootMsg;	/* My root message */

ProcessId DebugeePid;		/* Pid of program to debug. */
ProcessId DebugeeRootPid = 0;

int StopForSure;		/* True at start or access error. */
ProcessId VgtsPid;		/* guess */
int VgtsPresent;		/* True if the Vgts is the stdin fileserver */
int saveMode;			/* Used to save the file mode of stdout if
				   the Vgts is being used. */
#define DebuggerPadMode		(CR_Input+LF_Output)

short *CurBrkPc;		/* pc of last breakpoint stopped at. */
int CurBrkIndx = 0;		/* breakpoint index of last breakpoint
				   stopped at. */

char *StackBottom;		/* Used by StackDump to determine how many
				   parameters a process has. */

RegHandlerRequest regReq;
ProcessId ExceptionServerPid;


/*
 * Imported routines
 */

extern File *OpenPad(), *OpenFile();
extern short *StandardExceptionHandler();
extern File *dasminit();
extern char GetMemByte();
extern short GetInstrWord();
extern int dasm();
extern SetBreakpoint(), ResetBreakpoint(), ClearBreakpoint(),
	RemoveBreakpoint(), FindUserBreakpoint(), InvalidMemBuffer(),
	CommandLoop(), ValidAddress();
extern ProcessId LoadProgram();


/*
 * main:
 * Main debug program.
 * This is where all the interactions with the ExceptionServer and the 
 * process's kernel processState are handled.
 */


main(argc, argv)
    int argc;
    char **argv;
  { 
    ProcessId pid, debuggerPid, watcherPid, fePid;
    Message	msg;
    RootMessage dRootMsg;
    ExceptionRequest *req = (ExceptionRequest *) msg;
    CreateInstanceRequest *debugReq = (CreateInstanceRequest *) msg;

    ProcessBlock processBlk;
    int i;
    short error;

    ProcessId symFileServer;	/* Pid of server from which debugee program
				   was loaded. */
    InstanceId symFileId;	/* File instance of file from which debugee
				   program was loaded. */
    char symFileName[ 256 ];	/* Name of file from which debugee was loaded.*/

    int nArg;			/* Current argument to parse. */
    int debugIoPad = FALSE;	/* True if an io pad should be created for
				   the debugger. */
    int postDebug = FALSE;	/* True if doing a postmortem debug. */
    SystemCode loadError;
    SystemCode moveError;

    PerProcessArea debugeePerProcess;
    PerProcessArea *debugeePerProcessAddr;
    char *assumedStackBottom;
    extern DeathWatcher();
    extern FEWatcher(), BreakProcess();

    int teamSchedulingClass;
    int legal;

    debuggerPid =  GetPid(ACTIVE_PROCESS, LOCAL_PID);
    /*
     * Parse command flags.
     */
    nArg = 1;
    LinkOrigin = LoadOrigin = (char *) TEAM_ORIGIN;  /* default value */
    while ( nArg < argc  &&  argv[nArg][0] == '-')
      {
	if (strcmp(argv[nArg], "-d") == 0)
	  {
	    debugIoPad = TRUE;
	    nArg++;
	  }
	else if (strcmp(argv[nArg], "-p") == 0)
	  {
	    postDebug = TRUE;
	    CurExceptionPid = Receive( msg );
	    QueryProcessState(CurExceptionPid, &processBlk);
	    DebugeeRootPid = processBlk.team_root;
	    nArg++;
	  }
	else if (strcmp(argv[nArg], "-o") == 0)
	  {
	    if (argv[++nArg] != NULL)
	      {
	        sscanf(argv[nArg], "%x", &LinkOrigin);
		nArg++;
	      }
          }
	else
	  {
	    fprintf(stderr, "Debugger: Unknown flag '%s'\n", argv[nArg]);
	    exit(1);
	  }
      }

    if ( !postDebug && nArg>=argc ) 
      {
	fprintf( stderr, "Debugger: No program name specified.\n");
	return;
      }

    /*
     * Set the debugee program's io.
     */
    DefaultRootMessage(&dRootMsg);
    
    VgtsPid = GetPid(VGT_SERVER, LOCAL_PID );
    if( VgtsPresent = ( VgtsPid != 0  &&  VgtsPid != DebugeeRootPid ) )
      {
	/* Note that DebuggeeRootPid == 0 if not in postmortem debugger,
	 * so this only allows a pad to be opened if the VGTS is healthy. */
	if (debugIoPad)
	  {
	    stdout = OpenPad("debugger",26,80,&error);
	    if (error != OK)
	      {
		fprintf( stderr, "Debugger: couldn't open new output pad\n%s\n", ErrorString(error));
		ExitDebugger();
	      }
	    stdin = OpenFile(stdout->fileserver, stdout->fileid, FREAD, &error);
	    if ((error != OK) || (stdin == NULL))
	      {
		fprintf( stderr, "Debugger: couldn't open new pad for input.\n%s\n", ErrorString(error));
		ExitDebugger();
	      }
	    SetVgtBanner( stdout, argv[ nArg ] );
	    SelectPad(stdout);
	  }
      }
    saveMode = QueryPad( stdout );
    ModifyPad(stdout, DebuggerPadMode );

    /*
     * If this is a normal debugging session then
     * try to find the program to debug and then load it.
     * If this is a postmortem debugging session then
     * get the current exception pid from the forwarded exception msg
     * and get the program counter from a call to the 
     * StandardExceptionHandler.
     */
    if (!postDebug)
      {
        CurExceptionPid = LoadProgram( &argv[nArg], NULL,
		&dRootMsg, NULL, 0, &loadError );
	if( loadError != OK )
	  fprintf( stderr, "Debugger:  Couldn't load program:  %s\n",
		ErrorString( loadError ) );


	if (CurExceptionPid == NULL)
	    ExitDebugger();

	ProgramCounter = 0;	/* Signals that we should use the value
				   obtained from QueryProcessState. */
      }
    else
      {
	ProgramCounter = StandardExceptionHandler(req, CurExceptionPid, stdout);
      }

    fprintf( stdout,"\n\nWelcome to the V debugger.\n\nType q to quit.\n\n");

    /*
     * Get process information for process which incurred the exception.
     */
    QueryProcessState(CurExceptionPid, &processBlk);
    DebugeeRootPid = processBlk.team_root;
    DebugeeTeamSize = processBlk.team_size;
    for (i = 0; i < 16; i++)
        SavedRegisters[i] = (long) processBlk.ps.proc_state.regs[i];
    if (ProgramCounter == 0)
	ProgramCounter = (short *) processBlk.ps.proc_state.pc;
    StatusRegister = (long) processBlk.ps.proc_state.sr;

    assumedStackBottom = DebugeeTeamSize;

    /*
     * Get access to the symbol table file for the debugee program.
     * Then initialize the disassembler and get to the user symbol table. 
     */
    AccessSymFile( DebugeeRootPid, &symFileServer, &symFileId, symFileName );
    Symfile = dasminit( symFileServer, symFileId, symFileName );

    /*
     * Enter the debugger to allow the user to examine state and set 
     * breakpoints.
     */
    SetDefaults();
    if( postDebug )
      {
	dasm(ProgramCounter, Symfile, &legal);
	StackDump( OutputBuffer.buf, 0 );
      }
    StopForSure = 1;

    Vdebug( NOT_ATBREAKPOINT, NOT_EXCEPTION );

    StopForSure = 0;
    processBlk.ps.proc_state.sr = (short) StatusRegister;
    processBlk.ps.proc_state.pc = (char *) ProgramCounter;
    for (i = 0; i < 16; i++)
        processBlk.ps.proc_state.regs[i] = (char *) SavedRegisters[i];
    WriteProcessState(CurExceptionPid, &processBlk.ps.proc_state);

    /*
     * Register the debug handler with the exception server.
     */
    ExceptionServerPid = GetPid(EXCEPTION_SERVER, LOCAL_PID);
    if (ExceptionServerPid == 0)
      {
        fprintf( stderr, "Debugger: couldn't find the exception server's pid.\n");
        ExitDebugger();
      }

    regReq.requestcode = REGISTER_HANDLER;
    regReq.regFlag = 1;
    regReq.handlerPid = debuggerPid;
    regReq.regPid = DebugeeRootPid;

    Send(&regReq, ExceptionServerPid);
    if ( regReq.requestcode != OK )
      {
        fprintf( stderr, "Debugger: couldn't register the debugger to handle exceptions.\n");
        ExitDebugger();
      }

    /*
     * Create a process which watches for force-exception calls.
     */
    fePid = Create( 1, FEWatcher, 1000 );
    Ready( fePid, 1, DebugeeRootPid );

    /*
     * Set the debugee program running.
     * First must reset the mode of pad correctly, if necessary.
     */
    if ( !debugIoPad )
      {
	ModifyPad(stdout, saveMode);
      }

    Reply(&dRootMsg, CurExceptionPid);

    /*
     * Create process to watch for death of the debugee.
     */
    watcherPid = Create( 1, DeathWatcher, 500 );
    Ready( watcherPid, 1, DebugeeRootPid );

    /*
     * Loop forever on exception event msgs.
     */
    while (1)
      {
	CurExceptionPid = Receive(msg);

	QueryProcessState(CurExceptionPid, &processBlk);
	teamSchedulingClass = GetTeamPriority(CurExceptionPid);
	if( !StopProcessFlag  &&  Creator(CurExceptionPid) != debuggerPid )
	  ChangeTeamPriority( CurExceptionPid, STOP_TEAM_PRIORITY );

	if (req->requestcode == TERMINATE_TEAM)
	    ExitDebugger();

	Resynch( stdin );
	Resynch( stdout );
	Resynch( stderr );

	DebugeeTeamSize = processBlk.team_size;
	for (i = 0; i < 16; i++)
	    SavedRegisters[i] = (long) processBlk.ps.proc_state.regs[i];
	ProgramCounter = (short *) processBlk.ps.proc_state.pc;
	StatusRegister = (long) processBlk.ps.proc_state.sr;
	debugeePerProcessAddr = (PerProcessArea *) processBlk.ps.proc_state.perProcess;
	moveError = MoveFrom( CurExceptionPid, &debugeePerProcess, debugeePerProcessAddr, sizeof( debugeePerProcess ) );
	if( moveError != OK )
	  StackBottom = assumedStackBottom;
	else
	  StackBottom = (char *)debugeePerProcessAddr + debugeePerProcess.stackSize;

	if( !debugIoPad )
	  {
	    saveMode = QueryPad(stdout);
	    ModifyPad(stdout, DebuggerPadMode );
	  }

	if (req->type == ILLINST )
	  {
	    register short *instrptr;
#ifndef Addr24
	    instrptr = (short *) req->errpc;
#else
	    instrptr = (short*)((int)req->errpc & 0xFFFFFF);
#endif Addr24
	    if ((int)instrptr >= TEAM_ORIGIN) /* not in kernel space */
	      {
		/* Load instruction */
		req->instruction = 0;
		MoveFrom( CurExceptionPid,
				(short *)&(req->instruction)+1, instrptr, 2);
	      }
	    if (req->instruction == BPINST)
		Vdebug( ATBREAKPOINT, NOT_EXCEPTION );
	    else
	      {
		ProgramCounter = StandardExceptionHandler(req, CurExceptionPid, stdout);
		Vdebug( NOT_ATBREAKPOINT, EXCEPTION );
	      }
	  }
	else if ( req->type == TRACETRAP )
	    Vdebug( NOT_ATBREAKPOINT, NOT_EXCEPTION );
	else
	  {
	    ProgramCounter = StandardExceptionHandler(req, CurExceptionPid, stdout);
	    Vdebug( NOT_ATBREAKPOINT, EXCEPTION );
	  }
	if( !debugIoPad )
	    ModifyPad(stdout, saveMode);
	processBlk.ps.proc_state.sr = (short) StatusRegister;
	processBlk.ps.proc_state.pc = (char *) ProgramCounter;
        for (i = 0; i < 16; i++)
            processBlk.ps.proc_state.regs[i] = (char *) SavedRegisters[i];
	WriteProcessState(CurExceptionPid, &processBlk.ps.proc_state);
	ChangeTeamPriority( CurExceptionPid, teamSchedulingClass );

	Reply(&dRootMsg, CurExceptionPid);	/* NOTE: We must reply with dRootMsg
				   because RereadMessage in TeamRoot() is
				   expecting this message in spite of
				   exception messages due to e.g. 
				   single-stepping. */
      }
  }

/*
 * Queries the team server for information about the team with root pid rootPid
 * and returns the file server and file id of the file from which the team
 * was loaded.
 */
AccessSymFile(rootPid, fileServer, fileId, fileName)
	ProcessId rootPid;
	ProcessId *fileServer;
	InstanceId *fileId;
	char *fileName;
  {
    Message msg;
    DescriptorRequest *reqMsg = (DescriptorRequest *) msg;
    DescriptorReply *repMsg = (DescriptorReply *) msg;
    char buffer[sizeof(TeamDescriptor)];
    TeamDescriptor *desc = (TeamDescriptor *) buffer;
    ProcessId teamServerPid;

    reqMsg->requestcode = NREAD_DESCRIPTOR;
    reqMsg->namecontextid = rootPid;
    reqMsg->segmentptr = buffer;
    reqMsg->segmentlen = sizeof(TeamDescriptor);
    reqMsg->dataindex = 0;
    reqMsg->nameindex = reqMsg->segmentlen; /* 0-length name */
    teamServerPid = (LOGICAL_HOST_PART & rootPid) + LTEAM_SERVER_GROUP;

    Send(msg, teamServerPid);
    if ( repMsg->replycode != OK )
      {
	fprintf( stderr, "Debugger:  on access to team server: %s\n", 
		ErrorString(repMsg->replycode));
	ExitDebugger();
      }
    *fileServer = desc->fileserver;
    *fileId = desc->fileid;
    strcpy( fileName, desc->fileName );
  }


File *ReOpenFileIfNecessary( symfileptr )
	File *symfileptr;
  {
    ProcessId fileServer;
    InstanceId fileId;
    char fileName[ 256 ];

    fseek( symfileptr, 0, 0 );
    if( getc( symfileptr ) == EOF )
      {
	fprintf( stderr, "Debugger:  Symbol file is unreadable for some reason--attempting to reopen.\n" );
	Flush( stderr );
	AccessSymFile( DebugeeRootPid, &fileServer, &fileId, fileName );
	Symfile = dasminit( fileServer, fileId, fileName );
	return( Symfile );
      }
    else
	return( symfileptr );
  }
	


/*
 * Initialization routine for Vdb--sets default values for debug parameters.
 */
SetDefaults()
  {
    int i;

    SsSbrFlag = 0;		/* Not singlestepping. */
    SsFlag = 0;
    InSubrFlag = 0;
    WatchFlag = 0;
    WatchBrkptFlag = 0;
    for ( i = 0 ; i < BPMAX ; i++ ) 
      {
	cbptr = &bp[ i ];
	cbptr->pc = 0;		/* No breakpoints set. */
	cbptr->count = 0;
      }
    LargestOffset = DEFOFFSET;
    MaxSymLength = DEF_MAXSYMLEN;
    InputRadix = DEFRADIX;
    OutputRadix = DEFRADIX;
    TypeOutMode = 0;		/* No default type out mode yet. */
    TempTypeOutMode = 0;
    ActualTypeOutMode = 0;
    StringLength = 80;		/* Default string length. */
    StopProcessFlag = 0;
  }



/*
 * ExitDebugger:
 * Exits the debugger and debugee program.
 */

ExitDebugger()
  {
    Flush( stdout );
    Flush( stderr );
    if( DebugeeRootPid != NULL ) DestroyProcess(DebugeeRootPid);

    /* Unregister the debugger. */
    regReq.regFlag = 0;
    Send( &regReq, ExceptionServerPid );
    exit();
  }


/*
 * Helper process to detect and report the death of the debugee.
 */

DeathWatcher( debugeePid )
	ProcessId debugeePid;
  {
    Message msg;
    ExceptionRequest *req = (ExceptionRequest *) msg;

    ReceiveSpecific( msg, debugeePid );

    req->requestcode = TERMINATE_TEAM;
    Send( req, Creator(0) );
  }


/*
 * Helper process to detect force-exception calls.
 */

FEWatcher(debugeePid)
    ProcessId debugeePid;
  {
    ProcessId pid;
    Message msg;

    while (1)
      {
	/* Create a break process to kill to signal that a force exception
	   call should be made. */
	pid = Create(1, BreakProcess, 500);
	Ready(pid, 0);
	SetBreakProcess(stdout, pid);
	ReceiveSpecific(msg, pid);
	ForceException(debugeePid);
      }
  }


BreakProcess()
  {
    Message msg;

    Receive(msg);
  }
   



/* This is the prime entry to the debugger.
 */

Vdebug( atBrkpt, trueException )
	int atBrkpt, trueException;
  {
    short curpc;
    short *brkpc;
    int legal;
    int curpcbp;
    short bpno = 0;
    int atSsbp = 0;
    int atTrbp = 0;
    int watchStop = 0;
    int stopAtBrkpt;
    SystemCode error;

    if( !ValidAddress( ProgramCounter ) )
      {
	fprintf( stdout, "\nWarning: PC value is illegal address: %x\n", ProgramCounter );
	fprintf( stdout, "PC value has been changed to LinkOrigin. Can be reset with rr.\n");
	ProgramCounter = (short *) LinkOrigin;
	atBrkpt = 0;
	StopForSure = 1;
	CurBrkIndx = 0;
	SsFlag = 0;
	SsSbrFlag = 0;
	InSubrFlag = 0;
      }

    StatusRegister &= ~TRACEBIT;		/* clear trace bit in sr */

    if (WatchFlag)
      {
	long current;

	current = GetMem(WatchType, WatchAddr, &error);
	if (error != OK)
	  {
	    watchStop = 1;
	    fprintf(stdout, "Watched address couldn't be accessed.\n");
	  }
	if (current != OldWatchValue)
	  {
	    fprintf(stdout, "Watched location changed\n");
	    TypeOut(WatchType, WatchAddr);
	    fprintf(stdout, "\n");
	    watchStop = 1;
	    OldWatchValue = current;
	  }
      }

    if (CurBrkIndx)		/* Replace breakpt. at instr. just executed.
				   */
      {
	ResetBreakpoint(CurBrkIndx, CurBrkPc);
	CurBrkIndx = 0;
	bpno = FindUserBreakpoint(ProgramCounter - 1);
				/* Find out if this is a user-set brkpt, too. */
	/* return now if not single-stepping, no change in watched location,
	 * not watch-tracing, no exception, and not at a breakpoint
	 */
	if ( (!SsFlag || ( SsSbrFlag && InSubrFlag ) )
		&& !watchStop && !(WatchFlag && !WatchBrkptFlag)
		&& !trueException && !bpno )
	    return;
      }

    if (atBrkpt)
      {
	if (ProgramCounter == bp[SSBP].pc)
				/* Subroutine ss */
	  {
	    atSsbp = 1;
	    RemoveBreakpoint(SSBP);
	    InSubrFlag = 0;
	  }
	if (ProgramCounter == bp[TRBP].pc)
				/* Trap ss */
	  {
	    atTrbp = 1;
	    RemoveBreakpoint(TRBP);
	  }

	bpno = FindUserBreakpoint(ProgramCounter);
				/* Find out if this is a user-set brkpt. */
        
	stopAtBrkpt = !( (SsSbrFlag && !atSsbp) || 
			 (atTrbp && !SsFlag && !bpno) ||
			 (WatchFlag && WatchBrkptFlag) ||
			 (!atSsbp && !atTrbp && !bpno) ) || watchStop;
				/* DONT stop if stepping over subroutines and
				   at an interior breakpoint, or if stepping
				   over a trap instr. and not single stepping
				   or at a normal breakpoint, or if watching
				   only at breakpoints,
				   or if at a breakpoint which has been
				   removed earlier.  Always stop if the
				   watched location has changed. */
      }
    else
      {
	stopAtBrkpt = SsFlag || trueException || StopForSure || watchStop;
	bpno = FindUserBreakpoint(ProgramCounter);
				/* Find out if this is a user-set brkpt. */
      }

    if (stopAtBrkpt)
	  {				/* stop at this breakpoint */
	    Flush( stdout );
	    Flush( stderr );
	    dasm(ProgramCounter, Symfile, &legal);
					/* Display the current instr. */
	    fprintf( stdout,"B%d: %s %s\n", bpno, HexBuffer.buf, OutputBuffer.buf);
	    if (SsFlag) SsCount--;	/* count down single step */
	    if (bpno)		 	/* if breakpoint, */
	      {
		cbptr = &bp[bpno];	/* remember appropriate pointer */
		if( cbptr->count > 0 ) cbptr->count--;
					/* count down the breakpoint counter */
	      }
	    else
		cbptr = NULL;
	    Dot = (char *) ProgramCounter;	/* Dot defaults to here */

	    if ( (SsFlag && SsCount == 0) || (cbptr && cbptr->count == 0) ||
			trueException || StopForSure || watchStop)
		CommandLoop(&bpno);		/* process commands */
	    Flush(stdout);
	    Flush( stderr );
	  }

/* we are about to return to the calling program. set breakpoints or
 * turn on tracing.
 */

    curpc = GetInstrWord( ProgramCounter, &error );
    if( error != OK )
      {
	fprintf( stdout, "Debugger: PC is illegal address: cannot resume execution.\n" );
	ExitDebugger();
      }

    if (bpno)			/* If there is a brkpt. at the current pc
				   location then we have to set it AFTER
				   having executed the actual instr.  Do this
				   by setting the trace bit and setting the
				   brkpt at the next instr. cycle. */
      {
	StatusRegister |= TRACEBIT;
	CurBrkIndx = bpno;
	CurBrkPc = ProgramCounter;
      }

    if (SsFlag || (WatchFlag && !WatchBrkptFlag))
      {
	StatusRegister |= TRACEBIT;
	if ((SsSbrFlag && (((curpc & 0xFF00) == BSRINST) ||
		           ((curpc & 0xFFC0) == JSRINST))) ||
	    ((curpc & 0xFFF0) == TRAPINST))
	  {			/* If single-stepping over subroutines or at
				   a trap instr. then use a brkpt to step
				   over them. */
	    InSubrFlag = 1;
	    brkpc = ProgramCounter + dasm( ProgramCounter, Symfile, &legal );
						/* set bp at return */
	    if( brkpc < ProgramCounter )
	      {
		fprintf( stdout, "Breakpoint not at legal instruction. UH OH!\n");
		Flush( stdout );
		brkpc += 3;
	      }
	    if ((curpc & 0xFFF0) == TRAPINST)
		SetBreakpoint(TRBP, brkpc);
	    else
		SetBreakpoint(SSBP, brkpc);
	    StatusRegister &= ~TRACEBIT;/* Don't want the trace bit on when going
				   over traps or subroutines. */
	  }
      }

    if (bpno)
      ClearBreakpoint(bpno);
    InvalidMemBuffer();			/* Invalidate mem. buffer cache
					   since prog. execution may change
					   things. */

}

