/* Converts bitmap images given with '*' (for black) and '.' (or ' ') (for
 * white) into initialized C data structures.
 * Intended to be used as a filter in front of cpp.
 * Takes various switches to cater for machine dependencies etc.
 */
#include <stdio.h>
#include <Vfont.h>
#define MyName "bitcompile"
char line[512];
#define MAXROWS 2000
BitWord *(rowStart[MAXROWS]);
int rowLen[MAXROWS];
#define MAXWORDS 512
BitWord rowBuf[MAXWORDS];
BitWord mask = 0; /* if 0: 1 is black; if BitWordOnes: 0 is black */
#define LittleEndian 1
#define BigEndian 2
int byteOrder = 0;
int columnOrder = 0; /* if 1: store raster as sun-100 style 16-bit columns */
int PutHeader = 1;

main(argc, argv)
    char **argv;
  { FILE *fin = NULL, *fout = NULL;
    register char *cp;
    int lineNo = 0, colNo, rowNo, maxRowLen = 0, i, j, hWords;
    register BitWord *ptr;
    char name[256];
    for (i = 1; i < argc; i++)
	if (strcmp(argv[i], "-DLITTLE_ENDIAN") == 0)
	    byteOrder = LittleEndian;
	else if (strcmp(argv[i], "-DBIG_ENDIAN") == 0)
	    byteOrder = BigEndian;
	else if (strcmp(argv[i], "-DCOLUMN_ORDER") == 0)
	    columnOrder = 1;
	else if (strcmp(argv[i], "-DROW_ORDER") == 0)
	    columnOrder = 0;
	else if (strcmp(argv[i], "-DBLACK_IS_1") == 0)
	    mask = 0;
	else if (strcmp(argv[i], "-DBLACK_IS_0") == 0)
	    mask = BitWordOnes;
	else if (strcmp(argv[i], "-DNOHDR") == 0)
	    PutHeader = 0;
	else if (strcmp(argv[i], "-DSUN100FB") == 0)
	    columnOrder = 1;
	else if (strcmp(argv[i], "-DSUN120FB") == 0)
	    columnOrder = 1;
	else if (strcmp(argv[i], "-DVAX") == 0)
	  { byteOrder = LittleEndian; mask = BitWordOnes; }
	else if (strcmp(argv[i], "-o") == 0 && fout == NULL && i + 1 < argc)
	  {
	    i++;
	    fout = fopen(argv[i], "w");
	    if (fout == NULL)
	      {
		fprintf(stderr, "%s: cannot create %s\n", MyName, argv[i]);
		exit(1);
	      }

	  }
	else if (argv[i][0] != '-' && fin == NULL)
	  {
	    fin = fopen(argv[i], "r");
	    if (fin == NULL)
	      {
		fprintf(stderr, "%s: cannot read %s\n", MyName, argv[i]);
		exit(1);
	      }
	  }
	else
	  {
	    fprintf(stderr, "%s: bad command line argument: %s\n",
		MyName, argv[i]);
	  }
    if (!byteOrder) byteOrder = BigEndian;
    if (fin == NULL) fin = stdin;
    if (fout == NULL) fout = stdout;

    while (fgets(line, 512, fin) != NULL)
      {
	lineNo++;
/*fprintf(fout, "%d: %.60s",lineNo, line);*/
	if (strncmp(line, "#endraster", 10) == 0)
	  {
	    fprintf(stderr,
		"%s: line %d - unmatched #endraster\n", MyName, lineNo);
	    exit(1);
	  }
	if (strncmp(line, "#raster", 7) != 0)
	  {
	    fputs(line, fout);
	    continue;
	  }

	/* found a #raster command; get the name */
	cp = line + 7;
	while (*cp == ' ' || *cp == '\t') cp++;
	strncpy(name, cp, 255);
	for (cp = name; *cp && *cp != ' ' && *cp != '\t' && *cp != '\n'; cp++);
	*cp = '\0';

	for (rowNo = 0;;)
	  { lineNo++;
	    if (fgets(line, 512, fin) == NULL)
	      {
		fprintf(stderr,
		    "%s: line %d - missing #endraster\n", MyName, lineNo);
		exit(1);
	      }
/*fprintf(fout, "#%d: %.60s*",lineNo, line);*/

	    if (strncmp(line, "#endraster", 10) != 0)
	      {
		if (rowNo >= MAXROWS)
		  {
		    fprintf(stderr,
			"%s: line %d - too many rows\n", MyName, lineNo);
		    exit(1);
		  }
		rowLen[rowNo] = 0;
		for (ptr = rowBuf, colNo = 0; colNo < MAXWORDS; colNo++)
		    *ptr++ = 0;
		for (cp = line, colNo = 0; *cp != '\n'; cp++)
		    switch (*cp)
		      {
			case '.': case '_': /* white pixel */
			    rowLen[rowNo] = colNo+1;
			    /* then fall thru to ... */
			case ' ':
			    colNo++;
			    break;
			case '*': case 'X': /* black pixel */
			    rowLen[rowNo] = colNo+1;
			    /* little-endian order! */
			    rowBuf[colNo >> BitWordLog] |=
				1 << (colNo & (BitWordLen -1));
			    colNo++;
			    break;
			case '\t':
			    colNo = (colNo + 8) & 0xFFF8;
			    break;
			case 0:
			    fprintf(stderr,
				"%s: line %d - line too long\n",
				MyName, lineNo);
			    exit(1);
			default:
			    fprintf(stderr,
				"%s: line %d col %d - invalid char (0x%x)\n",
				MyName, lineNo, colNo, *cp);
			    exit(1);
		      }
		i = (rowLen[rowNo] + BitWordLen) >> BitUnitLog;
		rowStart[rowNo] = (BitWord*)malloc(i);
		bcopy(rowBuf, rowStart[rowNo], i);
		if (rowLen[rowNo] > maxRowLen) maxRowLen = rowLen[rowNo];
		rowNo++;
	      }
	    else
	      {
	     /* we've finally read the whole bitmap; now print it */
		hWords = (maxRowLen + BitWordLen - 1) >> BitWordLog;
		if (PutHeader)
		  {
		    fprintf(fout, "extern BitWord %s[];\n", name);
		    fprintf(fout, "static VRaster _%s = {", name);
		    fprintf(fout, "{0,0},{0,0},(BitUnit*)%s, 0,", name);
		    fprintf(fout, "%d,",	/* stride */
			columnOrder ? rowNo*2 : hWords << BitAlignLog);
		    fprintf(fout, "{%d,%d}};\n", rowNo, maxRowLen);
		  }

		fprintf(fout, "BitWord %s[%d] = {\n", name, rowNo*hWords);
		if (columnOrder)
		  {
		    for (j = 0; j < hWords; j++)
		      {
			fputs("    ", fout);
			for (i = 0; i < rowNo; i++)
			  { PutWord(rowStart[i][j], fout);
			  }
			fputs("\n", fout);
		      }
		  }
		else
		  {
		    for (i = 0; i < rowNo; i++)
		      {
			fputs("    ", fout);
			for (j = 0; j < hWords; j++)
			  { PutWord(rowStart[i][j], fout);
			  }
			fputs("\n", fout);
		      }
		  }
		fprintf(fout, "};\n");
		for (i = 0; i < rowNo; i++) free(rowStart[i]);
		break;
	      }
	  }
      }
    if (fin != stdin) fclose(fin);
    if (fout != stdout) fclose(fout);
    exit( 0 );
  }
extern u_char BitReverse[];
PutWord(w, fout)
    FILE *fout;
  {
    w ^= mask;
    if (byteOrder != LittleEndian)
	w = (BitReverse[w & 0xFF] << 8) + BitReverse[(w >> 8) & 0xFF];
    fprintf(fout, "0x%x,", w);
  }
