/* amaze: a distributed multi-person Vkernel game.
 * Copyright (c) 1983  Eric J. Berglund  &&  David R. Cheriton
 *
 * This file, state.c, is the 6th of the 8 files which make up the distributed
 * game amaze.  It contains the data structures which describe the maze,
 * and the routines which initialize and update the state representation of
 * the game.  Since state is updated whenever the timer process sends a
 * GAME_TIMER message to the manager, it also contains the routine which
 * processes timer requests.
 *
 * Included here are: the arrays Corners and Corridors, and the routines
 *  InitState, ProcessTimer, UpdateState, and CheckForMyMurder.
 */


#include "amaze.h"


/* The "Corners" array consists of every corner in the maze
 * arranged in the following manner :
 * x-coordinate, y-coordinate, the indices of the corridors abutting from
 * above, below, left of, and right of the corner, and the number of
 * corridors leading to the corner.  Note that if no corridor leads into
 * the corner from a specific direction, it is entered as WALLED.
 */

Node Corners [ NUM_CORNERS ] = {
	  0, 365, WALLED, WALLED, 67, 1, 2,	/* 0 */
	 15,  15, WALLED, 3, WALLED, 2, 2,
	 15,  85, 3, WALLED, WALLED, 4, 2,
	 15, 155, WALLED, 6, WALLED, 5, 2,
	 15, 295, 6, WALLED, WALLED, 7, 2,
	 15, 365, WALLED, 8, 1, WALLED, 2,	/* 5 */
	 15, 735, 8, WALLED, WALLED, 9, 2,
	 85,  85, WALLED, 10, 4, WALLED, 2,
	 85, 155, 10, WALLED, 5, 11, 3,
	 85, 225, WALLED, 13, WALLED, 12, 2,
	 85, 295, 13, 14, 7, WALLED, 3,		/* 10 */
	 85, 455, 14, 16, WALLED, 15, 3,
	 85, 735, 16, WALLED, 9, 17, 3,
	155,  15, WALLED, 18, 2, WALLED, 2,
	155,  85, 18, WALLED, WALLED, 19, 2,
	155, 225, WALLED, 21, 12, 20, 3,	/* 15 */
	155, 295, 21, 23, WALLED, 22, 3,
	155, 365, 23, 25, WALLED, 24, 3,
	155, 455, 25, WALLED, 15, WALLED, 2,
	155, 525, WALLED, 27, WALLED, 26, 2,
	155, 595, 27, 29, WALLED, 28, 3,	/* 20 */
	155, 735, 29, WALLED, 17, WALLED, 2,
	225,  15, WALLED, 31, WALLED, 30, 2,
	225,  85, 31, WALLED, 19, 32, 3,
	225, 665, WALLED, 34, WALLED, 33, 2,
	225, 735, 34, WALLED, WALLED, 35, 2,	/* 25 */
	295, 155, WALLED, 37, 11, 36, 3,
	295, 225, 37, WALLED, 20, WALLED, 2,
	345, 365, WALLED, WALLED, 24, 38, 2,
	345, 370, 39, WALLED, WALLED, WALLED, 1,
	365,  15, WALLED, 41, 30, 40, 3,	/* 30 */
	365,  85, 41, WALLED, 32, WALLED, 2,
	365, 155, WALLED, 42, 36, WALLED, 2,
	365, 295, 42, WALLED, 22, WALLED, 2,
	375, 595, WALLED, 44, 28, 43, 3,
	375, 665, 44, WALLED, 33, 45, 3,	/* 35 */
	435,  85, WALLED, 47, WALLED, 46, 2,
	435, 155, 47, WALLED, WALLED, 48, 2,
	435, 225, WALLED, 50, WALLED, 49, 2,
	435, 365, 50, WALLED, 38, 51, 3,
	505, 225, WALLED, 53, 49, 52, 3,	/* 40 */
	505, 295, 53, WALLED, WALLED, 54, 2,
	535, 365, WALLED, 56, 51, 55, 3,
	535, 435, 56, 58, WALLED, 57, 3,
	535, 525, 58, WALLED, 26, WALLED, 2,
	535, 665, WALLED, 59, 45, WALLED, 2,	/* 45 */
	535, 735, 59, WALLED, 35, 60, 3,
	585, 295, WALLED, 61, 54, WALLED, 2,
	585, 365, 61, WALLED, 55, WALLED, 2,
	605, 505, WALLED, 63, WALLED, 62, 2,
	605, 595, 63, 64, 43, WALLED, 3,	/* 50 */
	605, 735, 64, WALLED, 60, WALLED, 2,
	655, 295, WALLED, 66, WALLED, 65, 2,
	655, 365, 66, 68, WALLED, 67, 3,
	655, 435, 68, WALLED, 57, 69, 3,
	675, 155, WALLED, 70, 48, WALLED, 2,	/* 55 */
	675, 225, 70, WALLED, 52, WALLED, 2,
	675, 505, WALLED, 71, 62, WALLED, 2,
	675, 735, 71, WALLED, WALLED, 72, 2,
	745,  15, WALLED, 73, 40, WALLED, 2,
	745,  85, 73, 74, 46, WALLED, 3,	/* 60 */
	745, 295, 74, WALLED, 65, WALLED, 2,
	745, 435, WALLED, 75, 69, WALLED, 2,
	745, 735, 75, WALLED, 72, WALLED, 2,
	760, 365, WALLED, WALLED, 67, 1, 2 };	/* 64 */


/* The Corridors array consists of every corridor in the maze arranged in
 * the following manner:  the index of the UP or LEFT corner at the end
 * of the corridor ( the Near corner ), the index of the DOWN or RIGHT
 * corner at the end of the corridor ( the Far corner ), and the orientation
 * ( HORIZONTAL or VERTICAL ) of the corridor.  Corridor 0 is not referred
 * to at all in the code, and indeed its endpoints violate the assumption
 * made by the UpdateState routine:  no two corners may be closer than
 * MAX_VELOCITY pixels to each other.
 * ( I'm still not satisfied with how I handled the wrap around corridor
 * in this program--I'm looking forward to the next maintainer's solution. )
 */

Edge Corridors[ NUM_CORRIDORS ] =
	{ 64, 0, HORIZONTAL,			/* 0--Not used. */
	  0, 5, HORIZONTAL,
	  1, 13, HORIZONTAL,
	  1, 2, VERTICAL,
	  2, 7, HORIZONTAL,
	  3, 8, HORIZONTAL,			/* 5 */
	  3, 4, VERTICAL,
	  4, 10, HORIZONTAL,
	  5, 6, VERTICAL,
	  6, 12, HORIZONTAL,
	  7, 8, VERTICAL,			/* 10 */
	  8, 26, HORIZONTAL,
	  9, 15, HORIZONTAL,
	  9, 10, VERTICAL,
	  10, 11, VERTICAL,
	  11, 18, HORIZONTAL,			/* 15 */
	  11, 12, VERTICAL,
	  12, 21, HORIZONTAL,
	  13, 14, VERTICAL,
	  14, 23, HORIZONTAL,
	  15, 27, HORIZONTAL,			/* 20 */
	  15, 16, VERTICAL,
	  16, 33, HORIZONTAL,
	  16, 17, VERTICAL,
	  17, 28, HORIZONTAL,
	  17, 18, VERTICAL,			/* 25 */
	  19, 44, HORIZONTAL,
	  19, 20, VERTICAL,
	  20, 34, HORIZONTAL,
	  20, 21, VERTICAL,
	  22, 30, HORIZONTAL,			/* 30 */
	  22, 23, VERTICAL,
	  23, 31, HORIZONTAL,
	  24, 35, HORIZONTAL,
	  24, 25, VERTICAL,
	  25, 46, HORIZONTAL,			/* 35 */
	  26, 32, HORIZONTAL,
	  26, 27, VERTICAL,
	  28, 39, HORIZONTAL,
	  28, 29, VERTICAL,
	  30, 59, HORIZONTAL,			/* 40 */
	  30, 31, VERTICAL,
	  32, 33, VERTICAL,
	  34, 50, HORIZONTAL,
	  34, 35, VERTICAL,
	  35, 45, HORIZONTAL,			/* 45 */
	  36, 60, HORIZONTAL,
	  36, 37, VERTICAL,
	  37, 55, HORIZONTAL,
	  38, 40, HORIZONTAL,
	  38, 39, VERTICAL,			/* 50 */
	  39, 42, HORIZONTAL,
	  40, 56, HORIZONTAL,
	  40, 41, VERTICAL,
	  41, 47, HORIZONTAL,
	  42, 48, HORIZONTAL,			/* 55 */
	  42, 43, VERTICAL,
	  43, 54, HORIZONTAL,
	  43, 44, VERTICAL,
	  45, 46, VERTICAL,
	  46, 51, HORIZONTAL,			/* 60 */
	  47, 48, VERTICAL,
	  49, 57, HORIZONTAL,
	  49, 50, VERTICAL,
	  50, 51, VERTICAL,
	  52, 61, HORIZONTAL,			/* 65 */
	  52, 53, VERTICAL,
	  53, 64, HORIZONTAL,
	  53, 54, VERTICAL,
	  54, 62, HORIZONTAL,
	  55, 56, VERTICAL,			/* 70 */
	  57, 58, VERTICAL,
	  58, 63, HORIZONTAL,
	  59, 60, VERTICAL,
	  60, 61, VERTICAL,
	  62, 63, VERTICAL };			/* 75 */


/* MissileRestY contains the Y-coordinates of missiles that are READY.
 * MonsterDisplay contains the Y-coordinates of the monsters at the right.
 * MonsterRestX contains the X-coordinates of monsters that are EMBRYOnic.
 *
 * The number of values in each of these arrays depends on MAX_PLAYERS.
 */

int MissileRestY [] = { 40, 120, 200, 280, 360 };
int MonsterDisplayY [] = { 40, 120, 200, 280, 360 };
int MonsterRestX [] = { 228, 286, 344, 402, 460 };


/* InitState assigns the initial values to each monster--starting my monster
 * ALIVE and all others GLINT, and starting all missiles JUST_READY.
 */

InitState( state )
	GameState *state;
 {
    register MonsterState *monster;
    unsigned i;

    for( i = 0; i < MAX_PLAYERS; i++ )
      {
	monster = &(state->monsterstates[ i ]);
	monster->remoteasker = RESPOND_WHEN_ASKED;
	monster->missile.state = JUST_READY;
	monster->missile.x = MISSILE_REST_X;
	monster->missile.y = MissileRestY[ i ];

	if( i == state->mymonsternumber )
	  {
	    monster->living_status = ALIVE;
	    monster->count = 0;
	    monster->current_dir = UP;
	    monster->next_dir = UP;
	    monster->corridorindex = FIRST_CORRIDORINDEX;
	    monster->cornerindex = FIRST_CORNERINDEX;
	    monster->velocity = START_VELOCITY;
	    monster->x = FIRST_ALIVE_X;
	    monster->old_x = FIRST_ALIVE_X;
	    monster->y = FIRST_ALIVE_Y;
	    monster->old_y = FIRST_ALIVE_Y;
	    NotifyOthers( state );
	  }
	else
	  {
	    monster->living_status = GLINT;
	    monster->old_x = MonsterRestX[ i ];
	    monster->old_y = MONSTER_REST_Y;
	  }
      }
  }



/* PrepareForRedraw sets READY missiles to JUST_READY and EMBRYOs to CONCEIVED
 * so that they'll be redrawn.
 */

PrepareForRedraw( state )
	GameState *state;
 {
    register MonsterState *monster;
    unsigned i;

    for( i = 0; i < MAX_PLAYERS; i++ )
      {
	monster = &(state->monsterstates[ i ]);
	if( monster->missile.state == READY )
	    monster->missile.state = JUST_READY;
	if( monster->living_status == EMBRYO )
	    monster->living_status = CONCEIVED;
      }
  }


/* Respond to a message from the timer process. Update and redraw the
 * monsters, and determine whether any missiles have passed through my
 * monster.
 */


ProcessTimer( state, req, pid )
	register GameState *state;
	GameRequest *req;
	ProcessId   pid;
  {
    UpdateState( state );
    CheckForMyMurder( state );
    DrawPicture( state );
    return( OK );
  }


/* UpdateState is called by ProcessTimer to advance each ALIVE monster
 *  by monster->velocity pixels in whatever direction its going, decrement
 *  monster->count for any monsters which are HIDDEN, DYING, SHOT, STIFF,
 *  DEAD, etc. and determine whether they should be advanced to the next
 *  living_status,  advance WHIZZING missiles by MISSILE_SPEED pixels, and
 *  maintain missile->counts similarly to monster->counts.  It was previously
 *  broken up into several routines but combined into this workhorse for
 *  efficiency's sake.  
 *  Note:  To truly understand this routine, you must understand its 
 *  relationship to DrawPicture; many of the living_status's were added to
 *  make DrawPicture easier and more efficient.
 */


UpdateState( state )
	GameState *state;
  {
    Node *corner;
    int cornerindex, farOrNearCorner, changeflag, corridorindex;
    int distanceToCorner, distanceLeftToGo;
    unsigned i;
    register MonsterState *monster;
    register MissileState *missile;

    for( i = 0; i < MAX_PLAYERS; i++ )
      {

/* The first half of this loop updates the living_status and position
 * of a monster.  GLINT monsters change to CONCEIVED, so that they can
 * be drawn in the "uterus box" at the center of the screen.  CONCEIVED
 * monsters change to EMBRYOs, which are neither erased or redrawn. 
 * EMBRYOs hold their state.  HIDDEN and ALIVE monsters are moved in their
 * next or current directions--HIDDEN ones become ALIVE when their counts
 * reach 0. DrawPicture then displays them (or doesn't) in their new positions.
 * SHOT monsters become STIFF when their counts reach 0--in the mean time,
 * they are drawn with a flashing skull behind them.  Both STIFF and DYING
 * monsters are counted toward JUST_DIED; STIFF monsters are drawn as skulls,
 * DYING ones are flashed.  JUST_DIED monsters are erased and changed to
 * DEAD.  DEAD monsters are counted down to CONCEIVED.
 */

	monster = &( state->monsterstates[ i ] );

	switch( monster->living_status )
	  {
	    case GLINT:
	      /* Reinitialize the monster.
	       */
	      monster->x = MonsterRestX[ i ];
	      monster->old_x = MonsterRestX[ i ];
	      monster->y = MONSTER_REST_Y;
	      monster->old_y = MONSTER_REST_Y;
	      monster->current_dir = UP;
	      monster->next_dir = UP;
	      monster->corridorindex = FIRST_CORRIDORINDEX;
	      monster->cornerindex = FIRST_CORNERINDEX;
	      monster->velocity = START_VELOCITY;
	      monster->living_status = CONCEIVED;
	      break;

	    case CONCEIVED:
	      monster->living_status = EMBRYO;
	      break;

	    case EMBRYO:
	      break;

	    case HIDDEN:
	      if( monster->count == 0  &&  i == state->mymonsternumber )
		{
		  /* Once a monster has returned from being HIDDEN, it must
		   * remain visible for VISIBLE_COUNT calls to this routine.
		   */
		  monster->living_status = ALIVE;
		  monster->count = VISIBLE_COUNT;
		  NotifyOthers( state );
		}
	      /* WARNING!!! WARNING!!! This case falls through to the next!!
	       *  Hidden monsters are still Alive!!!!
	       */

	    case ALIVE:
	      if( monster->count > 0 ) monster->count--;

	      if( monster->next_dir == NONE )
		break;

	      cornerindex = monster->cornerindex;
	      corner = &( Corners[ cornerindex ] );
	      distanceLeftToGo = monster->velocity;
	      distanceToCorner = Abs( monster->x - corner->x )
					     + Abs( monster->y - corner->y );
/*	      changeflag = NO_NOTICE_NEEDED;  */

	      if( distanceLeftToGo >= distanceToCorner )
	        {
		  /* If a monster is about to reach a corner, not only must
		   * its x and y coordinates be updated, but its cornerindex
		   * and corridorindex must be changed as well.  This is
		   * done by advancing the monster to the corner, selecting
		   * the appropriate direction (try next_dir first, then
		   * current_dir, then perhaps attempt to anticipate), and
		   * then find the corridorindex and cornerindex that this
		   * choice of direction implies.
		   * Here again I take advantage of UP == 0, etc. to index
		   * the corridorindex array.
		   */

		  distanceLeftToGo -= distanceToCorner;

		  if( cornerindex == LEFT_WRAP_AROUND_CORNER )
		    cornerindex = RIGHT_WRAP_AROUND_CORNER;
		  else if ( cornerindex == RIGHT_WRAP_AROUND_CORNER )
		    cornerindex = LEFT_WRAP_AROUND_CORNER;
		  corner = &( Corners[ cornerindex ] );

		  monster->x = corner->x;
		  monster->y = corner->y;

		  if( corner->corridorindex[ monster->next_dir ] != WALLED )  
		    {
		      monster->current_dir = monster->next_dir;
/*		      changeflag = NOTIFIABLE_CHANGE;  */
		    }

		  else if( monster->current_dir == NONE )
		    {
		      monster->x = monster->old_x;
		      monster->y = monster->old_y;
		      break;
		    }

		  else if( corner->corridorindex[ monster->current_dir ]
								 != WALLED )
		    ;

/* If you ever put this back in, note the dependence on UP == 0, etc. */
/*		  else if(corner->corridorsadj==2 && i!=state->mymonsternumber )
 *		    {
 *		      i = UP;
 *		      while( corner->corridorindex[i] == WALLED
 *			|| corner->corridorindex[i] == monster->corridorindex )
 *		        i++;
 *		      monster->current_dir = i;
 *		      changeflag = NOTIFIABLE_CHANGE;
 *		      monster->next_dir = monster->current_dir;
 *		    }
 */
		  else
		    {
		      /* No good choice for a direction, so stay in the corner.
		       * Note that if we just sit in the corner for a while,
		       * this routine will run much quicker after the direction
		       * is set to NONE.
		       */
		      monster->next_dir = NONE;
		      break;
		    }


		  corridorindex = corner->corridorindex[ monster->current_dir ];
		  monster->corridorindex = corridorindex;

		  farOrNearCorner = monster->current_dir & 1;
				/* UP and LEFT are even, DOWN and RIGHT, odd. */
		  monster->cornerindex = Corridors[ corridorindex ].
				endptindex[ farOrNearCorner ];
		}		  

	      switch( monster->current_dir )
		{
		  case UP :
		    monster->y -= distanceLeftToGo;
		    break;

		  case DOWN :
		    monster->y += distanceLeftToGo;
		    break;

		  case LEFT :
		    monster->x -= distanceLeftToGo;
		    break;

		  case RIGHT :
		    monster->x += distanceLeftToGo;
		    break;
		}

/*	      if( changeflag && i == state->mymonsternumber )
 *		NotifyOthers( state );
 */
	      break;

	    case SHOT:
	      if( monster->count-- == 0 )
		{
		  monster->living_status = STIFF;
		  monster->count = STIFF_COUNT;
		}
	      break;

	    case STIFF:
	    case DYING:
	      if( monster->count-- == 0 )
		monster->living_status = JUST_DIED;
	      break;

	    case JUST_DIED:
	      monster->living_status = DEAD;
	      monster->count = DEAD_COUNT;
	      break;

	    case DEAD:
	      if( monster->count-- == 0 )
		{
		  monster->living_status = GLINT;
		}
	      break;
	  }

/* The second half of this routine does for missiles what the first half did
 * for monsters.  Missiles which are JUST_LOADING have been erased and can
 * now be set to LOADING.  LOADING missiles are not displayed--just counted 
 * down to JUST_READY.  JUST_READY missiles are drawn at the side of the
 * screen, and then changed to READY.  READY missiles are just left up.
 * TRIGGER_PULLED missiles are changed to JUST_FIRED, which are erased from
 * the right and drawn at the points from which they were fired.  WHIZZING
 * missiles move down the corridors like ALIVE monsters, only faster, until
 * they hit a wall.  Then they become HIT_WALL, and are drawn as explosions
 * until they count down to JUST_LOADING.
 */
	missile = &( monster->missile );

	switch( missile->state )
	  {
	    case JUST_LOADING:
	      missile->state = LOADING;
	      missile->count = LOAD_COUNT;
	      break;

	    case LOADING:
	      if( missile->count-- == 0 )
		missile->state = JUST_READY;
	      break;

	    case JUST_READY:
	      missile->state = READY;
	      break;

	    case READY:
	      break;

	    case TRIGGER_PULLED:
	      missile->state = JUST_FIRED;
	      break;

	    case JUST_FIRED:
	      missile->state = WHIZZING;
	      break;

/* WHIZZING missiles are handled very much like ALIVE monsters, with the
 * following exceptions:  1) Since corners may be much closer than the speed
 * of the missiles, the missile may have to pass through several corners
 * during this one call.  2) Missiles can't go around corners, so there is
 * no need for a clever direction selecting mechanism.
 * Here, as above, the code depends on the UP == 0, etc. representation.
 */
	    case WHIZZING:

	      farOrNearCorner = missile->direction & 1;
	      distanceLeftToGo = MISSILE_SPEED;
	      cornerindex = missile->cornerindex;
	      corner = &( Corners[ cornerindex ] );
	      distanceToCorner = Abs( missile->x - corner->x )
					 + Abs( missile->y - corner->y );

	      while( distanceLeftToGo >= distanceToCorner &&
		corner->corridorindex[ missile->direction ] != WALLED )
		{
		  distanceLeftToGo -= distanceToCorner;

		  if( cornerindex == LEFT_WRAP_AROUND_CORNER )
		    cornerindex = RIGHT_WRAP_AROUND_CORNER;
		  else if( cornerindex == RIGHT_WRAP_AROUND_CORNER )
		    cornerindex = LEFT_WRAP_AROUND_CORNER;
		  corner = &( Corners[ cornerindex ] );

		  missile->x = corner->x;
		  missile->y = corner->y;

		  cornerindex = Corridors[ corner->corridorindex[ missile->
				direction ] ].endptindex[ farOrNearCorner ];

		  missile->cornerindex = cornerindex;
		  corner = &( Corners[ cornerindex ] );
		  distanceToCorner = Abs( missile->x - corner->x )
					 + Abs( missile->y - corner->y );
		}

       	      if( distanceLeftToGo < distanceToCorner )
		{
		  switch( missile->direction )
		    {
		      case UP:
			missile->y -= distanceLeftToGo;
			break;

		      case DOWN:
			missile->y += distanceLeftToGo;
			break;

		      case LEFT:
			missile->x -= distanceLeftToGo;
			break;

		      case RIGHT:
			missile->x += distanceLeftToGo;
			break;
		    }
		}
	      else
		{
		  missile->x = corner->x;
		  missile->y = corner->y;
		  missile->state = HIT_WALL;
		  missile->count = EXPLOSION_COUNT & ~1;
	        }

	      break;

	    case HIT_WALL:
	      if( missile->count-- == 0 )
		{
		  missile->state = JUST_LOADING;
	          missile->x = MISSILE_REST_X;
	          missile->y = MissileRestY[ i ];
		}
	      break;
	  }
      }
  }


/* CheckForMyMurder determines whether my monster has been killed by a
 * missile or explosion.  It is the sole arbiter of whether or not my monster
 * has been SHOT.
 * He has been SHOT if  1) He runs into any explosion ( HIT_WALL ), even 
 * 			   his own, or
 *			2) Some other monster has fired a missile from a
 *			   position almost identical to his own, or
 *			3) A WHIZZING missile passes from one side of him
 *			   to the other.
 */


CheckForMyMurder( state )
	GameState *state;
  {
    unsigned i;
    int mymonsternumber, newXdistancefromme, newYdistancefromme, direction;
    register MonsterState *mymonster;
    MissileState *missile;

    mymonsternumber = state->mymonsternumber;
    mymonster = &( state->monsterstates[ mymonsternumber ] );

    if( mymonster->living_status == ALIVE  ||  mymonster->living_status ==
								HIDDEN )
      for( i = 0; i < MAX_PLAYERS; i++ )
	{
	  missile = &( state->monsterstates[i].missile );

	  /* These first two tests are not really necessary; they were put
	   * in for efficiency's sake--to weed out cases which couldn't 
	   * possibly cause trouble, as soon as possible.  In particular,
	   * missiles which are JUST_LOADING, LOADING, JUST_READY, or READY
	   * will not be examined at all because their ( x, y ) coordinates
	   * put them way outside the maze.
	   */
	  newXdistancefromme = Abs( mymonster->x - missile->x );
	  if( newXdistancefromme <= MISSILE_SPEED )
	    {
	      newYdistancefromme = Abs( mymonster->y - missile->y );
	      if( newYdistancefromme <= MISSILE_SPEED )
		{
		  if( i == state->mymonsternumber )
		    {
		      if( missile->state == HIT_WALL &&
 			 newXdistancefromme <= CLOSE_ENOUGH_TO_CORNER &&
			 newYdistancefromme <= CLOSE_ENOUGH_TO_CORNER )
		        mymonster->living_status = SHOT;
		    }

		  else
		   {
		    direction = missile->direction;
		    switch( missile->state )
		      {
			case JUST_FIRED:
			  if(( direction == LEFT || direction == RIGHT ) &&
			       newYdistancefromme <= CLOSE_ENOUGH_TO_CORNER &&
			       newXdistancefromme <= MAX_VELOCITY    ||
			     ( direction == UP   || direction == DOWN  ) &&
			       newXdistancefromme <= CLOSE_ENOUGH_TO_CORNER &&
			       newYdistancefromme <= MAX_VELOCITY   ) 
			    mymonster->living_status = SHOT;
			  break;

			case WHIZZING:
			  /* This bit of code contains a known bug that I am
			   * too lazy to fix.  The old_x and old_y fields are
			   * not really supposed to used for state purposes,
			   * only for drawing purposes.  They hold the last
			   * position of the monster so that the drawing
			   * routines know where to erase before redrawing
			   * the monster.  Thus, if a monster were an EMBRYO
			   * during the time unit just before this code was
			   * invoked, those old_* fields would contain
			   * coordinates that are inside the "birthing box"
			   * in the center of the screen, making it unlikely
			   * (though not impossible) that a newborn would
			   * be shot even if a missile passes right through
			   * it.  The fix will probably require a third field,
			   * called former_* perhaps, that will be equal to
			   * old_* in every case but this one.  And I say,
			   * who cares?    --Eric
			   */
			  if(( direction == LEFT || direction == RIGHT ) &&
			      newYdistancefromme <= CLOSE_ENOUGH_TO_CORNER &&
			   (( mymonster->x - missile->x >= 0 ) &&
			      ( missile->old_x - mymonster->old_x > 0 ) ||
			    ( mymonster->x - missile->x < 0 ) &&
			      ( missile->old_x - mymonster->old_x <= 0 ) )  ||
			     ( direction == UP   || direction == DOWN  ) &&
 			      newXdistancefromme <= CLOSE_ENOUGH_TO_CORNER &&
			   (( mymonster->y - missile->y >= 0 ) &&
			      ( missile->old_y - mymonster->old_y > 0 ) ||
			    ( mymonster->y - missile->y < 0 ) &&
			      ( missile->old_y - mymonster->old_y <= 0 ) ) )
			    mymonster->living_status = SHOT;
			  break;

			case HIT_WALL:
 			  if( newXdistancefromme <= CLOSE_ENOUGH_TO_CORNER &&
			    newYdistancefromme <= CLOSE_ENOUGH_TO_CORNER )
			    mymonster->living_status = SHOT;
			  break;

			default:
			  break;
		      }
		   }

		  if( mymonster->living_status == SHOT )
		    {
		      mymonster->count = SHOT_COUNT;
		      NotifyOthers( state );
		      if( state->myautopilot != NULL_PROCESS )
			{
			  DestroyProcess( state->myautopilot );
			  state->myautopilot = NULL_PROCESS;
			}
		    }
		}
	    }
	}
  }
