/*  amaze: a distributed multi-person Vkernel game.
 *  Copyright (c) 1983  Eric J. Berglund  &  David R. Cheriton
 *
 *  This file, draw.c, is the 7th of 8 files which make up the distributed
 *  game amaze.  It contains virtually all references to the graphical
 *  display of the maze and the players' states.  These include the
 *  rasters for the monsters, missiles, explosions ( missiles which have
 *  hit a wall ), the door, and the vertical and horizontal rectangles
 *  which make up the maze display.  Most of these graphics, and some of
 *  the initial work on the program, were done by Judy Sybille.
 *  The routines included are DrawMaze, BlinkDoor, and DrawPicture.
 */


#include "amaze.h"
#include "draw.h"
#include <Vquerykernel.h>

short ScreenLimitX, ScreenLimitY;	/* framebuffer limits */

/* Graphical information needed to convert to a mem_raster or a VRaster,
 * the two structures needed by the graphics routines for the various
 * machines.
 */

struct _AmazeRaster
  {
    short height;		/* Number of rows.			*/
    short width;		/* Number of columns.			*/
    char  *ptrtobits;		/* Pointer to the first word of the  	*/
				/* bitmap.				*/
  };

typedef struct _AmazeRaster AmazeRaster;




#ifndef MC68000

/* MicroVAX primitives */

#include <bitbltops.h>
#include "rasterVAX.h"

extern VRaster *WholeScreen;

DetermineScreen()
  {
    /* Assume MicroVAX+QVss */
    UseQvss( WholeScreen );
    ScreenLimitX = 1024;
    ScreenLimitY = 800;
  }


ClearScreen()
  {
    RasterClear(WholeScreen);
  }


BlackenRectangle( rect )
	Rectangle *rect;
  {
    VRaster temp;

    QvssRectangle( &temp + 1, rect );
    RasterClear( &temp + 1 );
  }


WhitenRectangle( rect )
	Rectangle *rect;
  {
    VRaster temp;

    QvssRectangle( &temp + 1, rect );
    RasterSet( &temp + 1 );
  }


InvertRectangle( rect )
	Rectangle *rect;
  {
    VRaster temp;

    QvssRectangle( &temp + 1, rect );
    RasterInvert( &temp + 1 );
  }



PutRectangle(rect, src)
	Rectangle *rect;
        AmazeRaster *src;
  {
    VRaster dst;
    VRaster vsrc;

    QvssRectangle( &dst + 1, rect );
    vsrc.origin.v = 0;
    vsrc.origin.h = 0;
    vsrc.width.v = 0;
    vsrc.width.h = 0;
    vsrc.start = (BitUnit *) src->ptrtobits;
    vsrc.xoffset = 0;
    vsrc.stride = 2 * ( ( src->width + BitWordLen - 1 ) >> BitWordLog );
    vsrc.bBox.v = src->height;
    vsrc.bBox.h = src->width;
    RasterOp( &dst + 1, &vsrc + 1, GXcopyInverted );
  }


#else

/* Sun primitives */

#include <rasterops.h>
#include <framebuf.h>
#include <confreg.h>
#include "rasterSUN.h"

#define SUN3FbBase 0xFE20000	/* Address of Sun-3 framebuffer */

int Sun2FB = 0; 		/* A flag indicating whether output is to a */
				/* Sun-2 or Sun-3 framebuffer.*/
int GXBase;
VRaster Sun120Fb = { {0,0}, {0,0}, (BitUnit *)SUN2FbBase, 0, 144, {900,1152} };

DetermineScreen()
  {
    MachineConfigurationReply mreply; 
   
    QueryKernel( 0, MACHINE_CONFIG, &mreply );
    if( mreply.machine == MACH_SMI_SUN2  ||  mreply.machine == MACH_SMI_SUN3 )
      {
	/* Determine whether we have a Sun-2 framebuffer, a Sun-3, or
	 * even a Sun-1 hiding in Sun2 upgrade clothes.
	 */

	PeripheralConfigurationReply preply;
	register i;

        Sun2FB = 1;

	if (mreply.machine == MACH_SMI_SUN2)
            GXBase = SUN2FbBase;
	else
	  {
	    GXBase = SUN3FbBase;
	    Sun120Fb.start = (BitUnit *) SUN3FbBase;
	  }

	QueryKernel(0, PERIPHERAL_CONFIG, &preply);
	for (i = 0; i < MAX_PERIPHERALS; ++i)
	    if ( preply.peripheral[i] == PRF_FRAMEBUFFER_SUN1 )
	      {
		/* We have a Sun-1 framebuffer after all. */
		GXBase = SUN2upgFbBase;
		Sun2FB = 0;
	        break;
	      }
      }
    else
      {
	/* Assume Sun-1 */
	GXBase = GXDefaultBase;
      }

    if (Sun2FB)
      {
        ScreenLimitX = 1152;
	ScreenLimitY = 900;
      }
    else
      {
	short config = mreply.confreg;
	if ( ( (struct ConfReg*)&config)->FBType )
	  {
	    ScreenLimitX = 1024;
	    ScreenLimitY = 800;
	  }
	else
	  {
	    ScreenLimitX = 800;
	    ScreenLimitY = 1024;
	  }
      }
    GXwidth = 1;
  }


ClearScreen()
  {
    if (Sun2FB)
      {
	struct fb_raster wholeScreen;

	wholeScreen.x = wholeScreen.y = 0;
	wholeScreen.width = ScreenLimitX;
	wholeScreen.height = ScreenLimitY;
	Sun2_BlackenRaster( &wholeScreen );
      }
    else
      {
	ScreenClear(); /* A (Sun-1) graphics library routine. */
      }
  }


BlackenRectangle( rect )
	Rectangle *rect;
  {
    if( Sun2FB )
      {
	VRaster temp;

	Sun120Rectangle( &temp + 1, rect );
	RasterSet( &temp + 1 );
      }
    else
      { 
	struct fb_raster _rect;
	_rect.height = rect->size.v;
	_rect.width = rect->size.h;
	_rect.y = rect->start.v;
	_rect.x = rect->start.h;
	GXfunction = GXclear;
	RasterFill( &_rect );
      }
   }


WhitenRectangle( rect )
	Rectangle *rect;
  {
    if( Sun2FB )
      {
	VRaster temp;

	Sun120Rectangle( &temp + 1, rect );
	RasterClear( &temp + 1 );
      }
    else
      { 
	struct fb_raster _rect;
	_rect.height = rect->size.v;
	_rect.width = rect->size.h;
	_rect.y = rect->start.v;
	_rect.x = rect->start.h;
	GXfunction = GXset;
	RasterFill( &_rect );
      }
   }


InvertRectangle( rect )
	Rectangle *rect;
  {
    if( Sun2FB )
      {
	VRaster temp;

	Sun120Rectangle( &temp + 1, rect );
	RasterInvert( &temp + 1 );
      }
    else
      { 
	struct fb_raster _rect;
	_rect.height = rect->size.v;
	_rect.width = rect->size.h;
	_rect.y = rect->start.v;
	_rect.x = rect->start.h;
	GXfunction = GXinvert;
	RasterFill( &_rect );
      }
   }


PutRectangle( rect, arast )
	Rectangle *rect;
	AmazeRaster *arast;
  {
    struct fb_raster fbrect;
    struct mem_raster src;

    fbrect.x = rect->start.h;
    fbrect.y = rect->start.v;
    fbrect.height = rect->size.v;
    fbrect.width = rect->size.h;
    src.height = arast->height;
    src.width = arast->width;
    src.start = (short *) arast->ptrtobits;

    if ( Sun2FB )
	Sun2_PutRaster( &fbrect, &src );
    else
      {
	GXfunction = GXcopyInverted;
	RasterPut( &fbrect, &src );
      }
  }
#endif



/* The Rectangles for the vertical and horizontal lines of the
 * maze, and for the door.
 */


Rectangle VerticalWalls [ NUM_VERT_WALLS ] = {
	  0,    1, 359, 9,
	409,    1, 380, 9,
	  0,  789, 359, 10,
	409,  789, 380, 10,
	199,   59,  90, 20,
	339,   59, 390, 20,
	 59,  129,  90, 20,
	269,  129, 180, 20,
	499,  129, 230, 20,
	  0,  199,  79, 20,
	409,  199, 110, 20,
	639,  199, 150, 20,
	199,  339,  90, 20,
	 59,  409, 300, 20,
	269,  479,  90, 20,
	409,  509, 110, 20,
	479,  579, 110, 20,
	639,  579,  90, 20,
	269,  629, 160, 20,
	549,  649, 240, 20,
	129,  719, 160, 20,
	479,  719, 250, 20
};

Rectangle HorizontalWalls [ NUM_HORIZ_WALLS ] = {
	  0,    1,  10, 798,
	 59,   59,  20,  90,
	59,   269,  20,  90,
	59,   409,  20, 330,
	129,    1,  20,  78,
	129,  129,  20, 300,
	129,  479,  20, 260,
	199,   59,  20, 230,
	199,  409,  20, 260,
	269,  199,  20, 160,
	269,  549,  20, 190,
	339,    1,  20,  78,
	339,  199,  20, 230,
	339,  479,  20, 100,
	339,  699,  20, 100,
	409,  199,  20, 120,
	409,  409,  20, 120,
	409,  579,  20,  70,
	409,  699,  20, 100,
	479,  579,  20, 160,
	499,  129,  20, 400,
	569,  199,  20, 400,
	639,  199,  20, 170,
	639,  419,  20, 180,
	709,  269,  20, 260,
	779,    1,  10, 798
};



Rectangle Door = { {409, 319}, {20, 90} };


extern int MonsterDisplayY[];



/* Draw the maze and the figures of the players to the right of it. */

DrawMaze()
  {
    unsigned i;
    Rectangle destrect;
    ClearScreen();

    for( i = 0; i < NUM_VERT_WALLS; i++ )
	WhitenRectangle(&VerticalWalls[i]);
    for( i = 0; i <NUM_HORIZ_WALLS ; i++ )
	WhitenRectangle(&HorizontalWalls[i]);

    for( i=0; i<MAX_PLAYERS; i++ )
      {
	destrect.start.h = MONSTER_DISPLAY_X;
	destrect.start.v = MonsterDisplayY[ i ];
	destrect.size.v = MONSTER_HEIGHT;
	destrect.size.h = MONSTER_WIDTH;
	PutRectangle( &destrect, MonsterRaster[ i ] );
      }
  }


BlinkDoor()
  {
    static int modulus;
    register unsigned i;

    if( ( ++modulus & 7 ) == 7 )
      {
	if( modulus & 8 )
	  {
	    PutRectangle( &Door, &DoorPanels );
	  }
	else
	  {
	    BlackenRectangle( &Door );
	  }
      }
  }



/* Draw the graphical representation of the state of each missile and monster.
 */

DrawPicture( state )
	GameState *state;
  {
    register AmazeRaster *memraster;
    register Rectangle destrect;
    short int i;
    register MonsterState *monster;
    register MissileState *missile;

    BlinkDoor();

    for( i=0; i < MAX_PLAYERS; i++ )
      {
	monster = &( state->monsterstates[ i ] );
	missile = &( monster->missile );

	switch( missile->state )
	  {
	    case JUST_LOADING:
		/* Erase from previous position. */
		/* Note the use of the specific values for the directions */
		/* to index the ExplosionRaster array.			  */
 		memraster = ExplosionRaster[ missile->direction ];
		destrect.start.h = missile->old_x;
		destrect.start.v = missile->old_y;
		destrect.size.h = memraster->width;
		destrect.size.v = memraster->height;
		BlackenRectangle( &destrect );
		break;

	    case LOADING:
		/* Don't display. */
		break;

	    case JUST_READY:
		/* Display away from the maze. */
		/* Note the use of the specific values for the directions */
		/* to index the MissileRaster array.			  */
		memraster = MissileRaster[ RIGHT ];
		destrect.start.h = missile->x;
		destrect.start.v = missile->y;
		destrect.size.h = memraster->width;
		destrect.size.v = memraster->height;
		PutRectangle( &destrect, memraster );
		missile->old_x = missile->x;
		missile->old_y = missile->y;
		break;

	    case READY:
		/* Don't erase or redisplay. */
		break;

	    case JUST_FIRED:
	    case WHIZZING:
		/* Erase at former position and draw at new position. */
		/* Note the use of the specific values for the directions */
		/* to index the MissileRaster array.			  */
		memraster = MissileRaster[ missile->direction ];
		destrect.start.h = missile->old_x;
		destrect.start.v = missile->old_y;
		destrect.size.h = memraster->width;
		destrect.size.v = memraster->height;
		BlackenRectangle( &destrect );
		destrect.start.h = missile->x;
		destrect.start.v = missile->y;
		PutRectangle( &destrect, memraster );
		missile->old_x = missile->x;
		missile->old_y = missile->y;
		break;

	    case HIT_WALL:
		/* Alternately erase former image and draw explosion--to
		 * give flashing effect to explosion. */
		/* Note the use of the specific values for the directions */
		/* to index the ExplosionRaster array.			  */
		memraster = ExplosionRaster[ missile->direction ];
		destrect.size.h = memraster->width;
		destrect.size.v = memraster->height;
		if( missile->count & 1 )
		  {
		    destrect.start.h = missile->x;
		    destrect.start.v = missile->y;
		    PutRectangle( &destrect, memraster );
		    missile->old_x = missile->x;
		    missile->old_y = missile->y;
		  }
		else
		  {
		    destrect.start.h = missile->old_x;
		    destrect.start.v = missile->old_y;
		    BlackenRectangle( &destrect );
		  }
	  }

	switch( monster->living_status )
	  {
	    case GLINT:
		break;

	    case CONCEIVED:
		/* Draw in "uterus box" in center of the screen. */
		memraster = MonsterRaster[ i ];
		destrect.start.h = monster->x;
		destrect.start.v = monster->y;
		destrect.size.h = memraster->width;
		destrect.size.v = memraster->height;
		PutRectangle( &destrect, memraster );
		break;

	    case EMBRYO:
		/* Don't erase or redisplay. */
		break;

	    case ALIVE:
		/* Erase at former position and draw at new position. */
		memraster = MonsterRaster[ i ];
		destrect.start.h = monster->old_x;
		destrect.start.v = monster->old_y;
		destrect.size.h = memraster->width;
		destrect.size.v = memraster->height;
		BlackenRectangle( &destrect );
		destrect.start.h = monster->x;
		destrect.start.v = monster->y;
		PutRectangle( &destrect, memraster );
		monster->old_x = monster->x;
		monster->old_y = monster->y;
		break;

	    case HIDDEN:
		/* Erase at former position and display inverted if it's my
		 * own monster, not at all if another's monster. */
		memraster = MonsterRaster[ i ];
		destrect.start.h = monster->old_x;
		destrect.start.v = monster->old_y;
		destrect.size.h = memraster->width;
		destrect.size.v = memraster->height;
		BlackenRectangle(&destrect);
		destrect.start.h = monster->x;
		destrect.start.v = monster->y;

		if( i == state->mymonsternumber )
		  {
		    PutRectangle( &destrect, memraster );
		    InvertRectangle( &destrect );
		  }
		monster->old_x = monster->x;
		monster->old_y = monster->y;
		break;

	    case SHOT:
		/* Alternately display the monster and a skull at the scene
		 * of the murder. */
		memraster = MonsterRaster[ i ];
		destrect.start.h = monster->old_x;
		destrect.start.v = monster->old_y;
		destrect.size.h = memraster->width;
		destrect.size.v = memraster->height;
		BlackenRectangle(&destrect);

		if( monster->count & 1 )
		  { PutRectangle( &destrect, &SkullHead ); }
		else
		  { PutRectangle( &destrect, memraster ); }
		break;

	    case STIFF:
		/* Display a skull at the scene of the murder. */
		memraster = MonsterRaster[ i ];
		destrect.start.h = monster->old_x;
		destrect.start.v = monster->old_y;
		destrect.start.v = monster->old_y;
		destrect.size.h = memraster->width;
		destrect.size.v = memraster->height;
		BlackenRectangle( &destrect );
		PutRectangle( &destrect, &SkullHead );
		break;

	    case DYING:
		/* Alternately display the monster and the monster inverted
		 * inverted at the place of death. */
		memraster = MonsterRaster[ i ];
		destrect.start.h = monster->old_x;
		destrect.start.v = monster->old_y;
		destrect.size.h = memraster->width;
		destrect.size.v = memraster->height;
		if( monster->count & 1 )
		  {
		    InvertRectangle( &destrect );
		  }
		else
		  {
		    PutRectangle( &destrect, memraster );
		  }
		break;

	    case JUST_DIED:
		/* Erase the monster from the point of death. */
		memraster = MonsterRaster[ i ];
		destrect.start.h = monster->old_x;
		destrect.start.v = monster->old_y;
		destrect.size.h = memraster->width;
		destrect.size.v = memraster->height;
		BlackenRectangle( &destrect );
		break;

	    case DEAD:
		/* Don't display. */
		break;

	  }
	monster->old_living_status = monster->living_status;
      }
  }
