/*  amaze : a distributed multi-person Vkernel game.
 *  Copyright (c) 1983  Eric J. Berglund  &&  David R. Cheriton
 *
 *  This file, communicate.c, is the 5th of the 8 files which make up the
 *  distributed game amaze.  Primarily, it contains the routines needed to
 *  communicate the state of my monster and missile, as well as  to learn
 *  the state of other monsters and missiles.  These are the routines which
 *  are called by the game manager ( main ) in response to
 *  REQUEST_MONSTER_STATE, REPORT_MONSTER_STATE, and MISSILE_REPORT messages.
 *  It also contains the routine which responds to JOIN_GAME requests.
 *
 *  The routines are:  ReturnLocalMonsterState, UpdateRemoteMonsterState,
 *   NotifyOthers, JoinGame, NoteMissile, SendMissileMessage, and Debug.
 */

#include "amaze.h"



/* Return the monster state in response to a request, providing it has changed
 * since the last time it was read by this monster.  If there has been no
 * significant change in state, save the pid of the requester in the remoteasker
 * field of his state description.  Later, when a significant change occurs,
 * NotifyOthers will be called to respond to the pending message.
 */

ReturnLocalMonsterState( state, req, pid )
	GameState *state;
	StatusInquirerRequest *req;
	ProcessId   pid;
  {
    extern StatusInquirer();
    register StatusInquirerReply *replymsg = (StatusInquirerReply *) req;
    register MonsterState *mymonster, *monster;
    unsigned  monsternumber;

    if( ( monsternumber = req->monsternumber ) > MAX_PLAYERS )
      {
	printf(" Player number too big in ReturnLocalMonsterState\n");
	return( ILLEGAL_REQUEST );
      }

    monster = &( state->monsterstates[ monsternumber ] );

    if( monster->statusinquirer == NULL_PROCESS )
      {
	/* Just learned of new monster! */
	monster->gamemanager = req->manager;
	monster->remoteasker = RESPOND_WHEN_ASKED;
	monster->statusinquirer = Create( 3, StatusInquirer, 1500 );
	Ready( monster->statusinquirer, 4, state->mymonsternumber,
	 state->mymanager, monsternumber, req->manager );
      }

    if( monster->remoteasker == RESPOND_WHEN_ASKED )
      {
	/* Big news has been waiting for this inquirer! */
	mymonster = &( state->monsterstates[ state->mymonsternumber ] );
	replymsg->xcoord = mymonster->x;
	replymsg->ycoord = mymonster->y;
	replymsg->corridorindex = mymonster->corridorindex;
	replymsg->cornerindex = mymonster->cornerindex;
	replymsg->living_status = mymonster->living_status;
	replymsg->velocity = mymonster->velocity;
	replymsg->direction = mymonster->current_dir;
	replymsg->next_direction = mymonster->next_dir;
	monster->remoteasker = NO_ONE_WAITING;
	return( OK );
      }
    else
      {
	/* No news since the last time he asked--let him wait. */
	monster->remoteasker = pid;
	return( NO_REPLY );
      }
  }



/* Update the game manager's idea of the state of the specified monster using
 * the report of my trusty status inquirer.
 */

UpdateRemoteMonsterState( state, req, pid )
	GameState *state;
	register StatusInquirerRequest *req;
	ProcessId pid;
  {
    register MonsterState *monster;

    if( req->monsternumber > MAX_PLAYERS )
      {
	printf("Player number too big in UpdateRemoteMonsterState\n");
	return( ILLEGAL_REQUEST );
      }

    monster = &( state->monsterstates[ req->monsternumber ]  );

    if( monster->statusinquirer != pid )
      {
	printf("Wrong status inquirer in UpdateRemoteMonsterState\n");
	return( ILLEGAL_REQUEST );
      }

/* As part of the effort to minimize the need for communication of state
 * among the various game managers, the UpdateState routine in state.c
 * automatically effects the SHOT->STIFF->JUST_DIED->DEAD->CONCEIVED->EMBRYO,
 * DYING->JUST_DIED->DEAD->CONCEIVED->EMBRYO, and GLINT->CONCEIVED->EMBRYO
 * sequences of living_status's.  A monster's manager needs only to notify other
 * managers when he becomes ALIVE, SHOT, DYING, or HIDDEN.  It is possible,
 * however, that a manager could receive a status report at any time
 * and discover that the monster it's examining is in the midst of one of
 * these sequences.  In order to display the sequence properly, we set the
 * living_status back to SHOT or DYING.  This is allowable because
 * none of the sequences are very time critical:  should the monster be reborn
 * quickly, the display is not hurt by skipping past DEAD, CONCEIVED, or EMBRYO.
 */

    if( req->living_status == ALIVE || req->living_status == HIDDEN )
      {
	monster->living_status = req->living_status;
	monster->x = req->xcoord;
	monster->y = req->ycoord;
	monster->corridorindex = req->corridorindex;
	monster->cornerindex = req->cornerindex;
	monster->velocity = req->velocity;
	monster->current_dir = req->direction;
	monster->next_dir = req->next_direction;
      }

    else if( req->living_status == SHOT || req->living_status == STIFF )
      {
	if( monster->living_status != SHOT && monster->living_status != STIFF )
	  {
	    /* If we're already in a SHOT/STIFF sequence, just continue it.
	     * Otherwise, start one.  Note that the x,y,corridor,corner info
	     * need not be examined--just make him dead where we last thought
	     * he was.
	     */
	    monster->living_status = SHOT;
	    monster->count = SHOT_COUNT;
	  }
      }

    else if( req->living_status == DYING )
      {
	if( monster->living_status != DYING )
	  {
	    monster->living_status = DYING;
	    monster->count = DYING_COUNT;
	  }
      }

    else if( req->living_status == CREMATED )
      {
	/* We assume that the player has left the game completely.
	 * Thus we need not maintain an inquirer for his monster and
	 * can open up his spot in the game by setting his manager to
	 * NULL_PROCESS.
	 */
	DestroyProcess( pid );
	monster->statusinquirer = NULL_PROCESS;
	monster->gamemanager = NULL_PROCESS;
	monster->remoteasker = NO_ONE_WAITING;
	monster->living_status = DYING;
	monster->count = DYING_COUNT;
      }

    else if(monster->living_status == ALIVE || monster->living_status == HIDDEN)
	{
	  /* The new living status is either GLINT, CONCEIVED, EMBRYO,
	   * JUST_DIED, or DEAD. If we thought he was alive, we better
	   * kill him off.
	   */
	  monster->living_status = DYING;
	  monster->count = DYING_COUNT;
	}

/*  else the new status is one of the five above and we thought he was
 *  SHOT, STIFF, DYING, or one of that five, so we'll just let the progression
 *  take its normal course.  The display will make more sense than if we
 *  did a sudden switch, and the result will be the same--EMBRYO.  Note that
 *  if we get a message saying he's come back alive any time soon, we'll
 *  immediately change him to ALIVE--accuracy is more important than display
 *  in that case.
 */

    if( monster->living_status == ALIVE || monster->living_status == SHOT
			 || monster->living_status == HIDDEN )
      {
	if( monster->x < 0 || monster->x > 799 || monster->y <= 0 || monster->y
	  > 799 )
	  {
	    monster->x = monster->old_x;
	    monster->y = monster->old_y;
	    monster->living_status = DYING;
	    DestroyProcess( state->mykeyboardreader );
	  }
	ConductTests( monster, req->monsternumber );
      }

    return( OK );
  }




/* Respond to any waiting status inquirers, and set the RESPOND_WHEN_ASKED
 * flag for others who haven't yet asked.  This routine is called when
 * my monster has changed his state significantly.
 */

NotifyOthers( state )
	GameState *state;
  {
    unsigned i;
    register MonsterState *monster, *mymonster;
    Message msg;
    register StatusInquirerReply *replymsg = (StatusInquirerReply *) msg;

    for( i = 0 ; i < MAX_PLAYERS ; i++ )
      {
	monster = &( state->monsterstates[i] );
	if( monster->gamemanager != NULL_PROCESS )
	  {
	    switch( monster->remoteasker )
	      {
		case NO_ONE_WAITING :
		  monster->remoteasker = RESPOND_WHEN_ASKED;
		  break;

		case RESPOND_WHEN_ASKED :
		  break;

		default :	/* Must be the pid of the waiting inquirer. */
		  mymonster = &(state->monsterstates[state->mymonsternumber]);
		  replymsg->xcoord = mymonster->x;
		  replymsg->ycoord = mymonster->y;
		  replymsg->corridorindex = mymonster->corridorindex;
		  replymsg->cornerindex = mymonster->cornerindex;
		  replymsg->living_status = mymonster->living_status;
		  replymsg->velocity = mymonster->velocity;
		  replymsg->direction = mymonster->current_dir;
		  replymsg->next_direction = mymonster->next_dir;
		  replymsg->replycode = OK;
		  Reply( replymsg, monster->remoteasker );
		  monster->remoteasker = NO_ONE_WAITING;
		  break;
	      }
	  }
      }
  }



/* Process a JOIN_GAME request by searching through the list of players for
 * an empty space ( or one already occupied by the requester, God forbid )
 * Having found a hole, save his pid there, and pass back the pids of all
 * the managers in the game.  If there is no room available, return BUSY.
 */

JoinGame( state, req, pid )
	GameState *state;
	GameRequest *req;
	ProcessId pid;
  {
    register MonsterState *monster;
    GameReply *replymsg = (GameReply *) req;
    register unsigned i, j;

    for( i = 0; i < MAX_PLAYERS; i++ )
      {
	monster = &(state->monsterstates[i]);
	if(( monster->gamemanager == NULL_PROCESS )
					 || ( monster->gamemanager == pid ))
	  {
	    monster->gamemanager = pid;
	    for( j = 0; j < MAX_PLAYERS; j++ )
	      {
		monster = &(state->monsterstates[j]);
		replymsg->monstermanagers[j] = monster->gamemanager;
	      }
	    return( OK );
	  }
      }
   return( BUSY );
  }



/* Notifies all other managers that my player has just fired a shot,
 * by Creating one MissileMessenger for each other manager in the game.
 */

SendMissileMessage( state )
	GameState *state;
  {
    extern MissileMessenger();
    register MonsterState *monster;
    MissileState *mymissile;
    unsigned i, mymonsternumber;
    ProcessId pid;

    mymonsternumber = state->mymonsternumber;
    mymissile = &( state->monsterstates[ mymonsternumber ].missile );

    for( i = 0; i < MAX_PLAYERS; i++ )
      {
	monster = &( state->monsterstates[i] );
	if( monster->gamemanager != NULL_PROCESS && i != mymonsternumber )
	  {
	    pid = Create( 2, MissileMessenger, 256 );
	    Ready( pid, 3, monster->gamemanager, mymonsternumber, mymissile );
	  }
      }
  }



/* Adjusts the missile data structure of a monster whose manager has just
 * sent a MISSILE_REPORT.
 */

NoteMissile( state, req, pid )
	GameState *state;
	MissileRequest *req;
	ProcessId pid;
  {
    register MissileState *missile;

    missile = &( state->monsterstates[ req->monsternumber ].missile );

    missile->state = TRIGGER_PULLED;
    missile->x = req->x;
    missile->y = req->y;
    missile->direction = req->direction;
    missile->cornerindex = req->cornerindex;

    return( OK );
  }



Debug()
  {
    printf(" Amaze debug" );
  }
