/*
 * This is a slightly (mostly stylistically) modified version of bitcompile,
 * a program written by Per Bothner.  It is intended to be used as a filter
 * in front of cpp.  It takes input which contains regions of the form:
 *
 * #raster somename
 * ......................
 * ....XXX........XXX....
 * ....XXX........XXX....
 * ....XXX........XXX....
 * ..........XX..........
 * ..........XX..........
 * XXX................XXX
 * .XXX..............XXX.
 * ..XXXXX........XXXXX..
 * ....XXXXXXXXXXXXXX....
 * ........XXXXXX........
 * #endraster
 *
 * and converts them to the appropriate hex numbers to be sent to
 * our bitmap displays.  My version of this program also prints the
 * AmazeRaster structs needed by the draw.c file of Amaze.
 *
 * It takes various switches to adapt to machine dependencies.  For those,
 * consult the ParseCommandLine routine.
 *
 * (By the way, dots above are black, X's are white.)
 *
 * Eric J. Berglund, August 1985
 *
 * P.S.  Per would scream if he saw the comments in this code which refer
 *	 to 16 bit words, when he went out of his way to write this very
 *	 generally (although PutWord seems suspiciously ungeneral to me).
 *	 Anyway, to keep him happy read all my 16 bit word comments as if
 *	 they said BitWord.
 */

#include <stdio.h>
#include "draw.h"

#define MY_NAME "bitcompile"

#define MAXCOLS		512
#define MAXROWS		2000
#define MAX_RASTER_NAME	255

#define FALSE	0
#define TRUE	1

int Sun100ColumnsFlag;	/* TRUE if the raster will be stored as Sun-100   */
			/* style 16-bit columns, FALSE otherwise.	  */

#define UNASSIGNED	0
#define LITTLE_ENDIAN	1
#define BIG_ENDIAN	2

int ByteOrder;		/* Possible values are UNASSIGNED, LITTLE_ENDIAN, */
			/* and BIG_ENDIAN.				  */

BitWord Mask;		/* Either all 0's or all 1's, depending on	  */
			/* whether the machine uses 0 or 1 as black. 	  */
			/* (All 0's if 1 is black!)			  */


extern unsigned char BitReverse[];


main( argc, argv )
	int argc;
	char **argv;
  {
    FILE *fin = NULL;
    FILE *fout = NULL;
    int lineNo = 0;
    char line[ MAXCOLS ];

    ParseCommandLine( argc, argv, &fin, &fout );

    while( fgets( line, MAXCOLS, fin ) != NULL )
      {
	lineNo++;
	if( strncmp( line, "#raster", 7 ) == 0 )
	  {
	    ConvertRaster( line, &lineNo, fin, fout );
	  }
	else if( strncmp( line, "#endraster", 10 ) == 0 )
	  {
	    fprintf( stderr, "%s: line %d - unmatched #endraster\n",
							 MY_NAME, lineNo );
	    exit( 1 );
	  }
	else
	  {
	    fputs( line, fout );
	  }
      }
    if( fin != stdin ) fclose( fin );
    if( fout != stdout ) fclose( fout );
  }


ParseCommandLine( argc, argv, fin, fout )
	int argc;
	char **argv;
	FILE **fin;
	FILE **fout;
  {
    int i;

    ByteOrder = UNASSIGNED;
    Sun100ColumnsFlag = FALSE;
    Mask = 0;

    for( i = 1; i < argc; i++ )
      {
	if( strcmp( argv[i], "-DLITTLE_ENDIAN" ) == 0 )
	    ByteOrder = LITTLE_ENDIAN;

	else if( strcmp( argv[i], "-DSUN100FB" ) == 0 )
	  { 
	    Sun100ColumnsFlag = TRUE;
	    Mask = BitWordOnes;
	  }

	else if( strcmp( argv[i], "-DVAX" ) == 0 )
	  { 
	    ByteOrder = LITTLE_ENDIAN;
	    Mask = BitWordOnes;
	  }

	else if( strcmp( argv[i], "-o" ) == 0 && *fout == NULL && i + 1 < argc )
	  {
	    i++;
	    *fout = fopen( argv[i], "w" );
	    if( *fout == NULL )
	      {
		fprintf( stderr, "%s: cannot create %s\n", MY_NAME, argv[i] );
		exit( 1 );
	      }
	  }

	else if( argv[i][0] != '-' && *fin == NULL )
	  {
	    *fin = fopen( argv[i], "r" );
	    if( *fin == NULL )
	      {
		fprintf( stderr, "%s: cannot read %s\n", MY_NAME, argv[i] );
		exit( 1 );
	      }
	  }

	else
	  {
	    fprintf( stderr, "%s: bad command line argument: %s\n",
		MY_NAME, argv[i] );
	  }
      }
 
    if( ByteOrder == UNASSIGNED ) ByteOrder = BIG_ENDIAN;
    if( *fin == NULL ) *fin = stdin;
    if( *fout == NULL ) *fout = stdout;
  }



ConvertRaster( line, lineNo, fin, fout )
	char *line;
	int *lineNo;
	FILE *fin;
	FILE *fout;
  {
    /* Having encountered a #raster line, read all the rows of the raster,
     * saving them in order in memory pointed to by the rowStarts.  When
     * the #endraster line is read, print the AmazeRaster structure and the
     * hex we've calculated.
     */

    char name[ MAX_RASTER_NAME + 1 ];	/* Name of the raster.		  */
    int colNo, rowNo;
    int i, j;
    int hWords;				/* Number of 16 bit words across. */
    int rowLen;
    int maxRowLen = 0;			/* Number of bits across.	  */
    register char *cp;
    register BitWord *ptr;
    char *retcode;

    BitWord *( rowStart[ MAXROWS ] );	/* Pointers to the row beginnings.*/
    BitWord rowBuf[ MAXCOLS ];		/* Contents of the current row.   */


    /* Found a #raster command; get the name.  Terminate it with a \0. */

    cp = line + strlen( "#raster" );
    while( *cp == ' ' || *cp == '\t' ) cp++;
    strncpy( name, cp, MAX_RASTER_NAME );
    for( cp = name; *cp && *cp != ' ' && *cp != '\t' && *cp != '\n'; cp++ );
    *cp = '\0';

    /* Read and process each line of the raster. */

    rowNo = 0;
    retcode = fgets( line, MAXCOLS, fin );
    while( strncmp( line, "#endraster", 10 ) != 0 )
      {
	if( retcode == NULL )
	  {
	    fprintf( stderr, "%s: line %d - missing #endraster\n",
							 MY_NAME, *lineNo );
	    exit( 1 );
	  }

	if( rowNo >= MAXROWS )
	  {
	    fprintf( stderr, "%s: line %d - too many rows\n", MY_NAME, *lineNo );
	    exit( 1 );
	  }

	*lineNo++;

	/* Clear rowLen and rowBuf, getting ready for the next line.*/

	rowLen = 0;
	ptr = rowBuf;
	for(  colNo = 0; colNo < MAXCOLS; colNo++ )
	    *ptr++ = 0;

	/* Go through each character of the line.  Periods and underscores
	 * are 0's; asterisks and 'X's are 1's; blanks and tabs are 0's
	 * unless they're at the end of the line--then they're ignored
	 * (although they'll end up being interpreted as 0's if any line
	 * gets long enough for the raster to need them).
	 */

	colNo = 0;
	for( cp = line; *cp != '\n'; cp++ )
	  {
	    switch( *cp )
	      {
		case '.':	 /* white pixel */
		case '_':
		    rowLen = colNo+1;
		    colNo++;
		    break;

		case ' ':
		    colNo++;
		    break;

		case '*':	 /* black pixel */
		case 'X':
		    rowLen = colNo+1;
		    /* little-endian order! */
		    rowBuf[ colNo >> BitWordLog ] |=
				1 << ( colNo & ( BitWordLen - 1 ) );
		    colNo++;
		    break;

		case '\t':
		    colNo = ( colNo + 8 ) & 0xFFF8;
		    break;

		case 0:
		    fprintf( stderr, "%s: line %d - line too long\n",
				MY_NAME, *lineNo );
		    exit( 1 );

		default:
		    fprintf( stderr, 
				"%s: line %d col %d - invalid char (0x%x )\n",
				MY_NAME, *lineNo, colNo, *cp );
		    exit( 1 );
	      }
	  }

	i = ( rowLen + BitWordLen ) >> BitUnitLog;
	rowStart[ rowNo ] = (BitWord*) malloc( i );
	Copy( rowStart[ rowNo ], rowBuf, i );

	if( rowLen > maxRowLen )
	    maxRowLen = rowLen;

	rowNo++;
	retcode = fgets( line, MAXCOLS, fin );
      }

    /* We've finally read the whole bitmap; now print it.
     * First print the bits themselves, later the AmazeRaster structure.
     * For our machines, hWords is the number of 16 bit words needed to
     * hold one row of the raster.
     */

    hWords = ( maxRowLen + BitWordLen - 1 ) >> BitWordLog;

    fprintf( fout, "static BitWord _%s[%d] = {\n", name, rowNo * hWords );

    if( Sun100ColumnsFlag )
      {
	/* With Sun-100 style columns, we have to store the words knowing
	 * that they'll be mapped to the screen 16 bits at time--with each
	 * group of 16 in one row, but the next group below the previous one.
	 * When the first 16 columns of each row have been mapped, the next
	 * 16 will be mapped--starting at the first row of the raster and
	 * going down again.
	 *
	 * Thus the outer loop here counts across the columns (taking 16 at
	 * a time) and the inner one counts down the rows.
	 */

	for( j = 0; j < hWords; j++ )
	  {
	    fputs( "    ", fout );
	    for( i = 0; i < rowNo; i++ )
	      {
		/* Print the hex for the next 16 bits. */
		PutWord( rowStart[i][j], fout );
	      }
	    fputs( "\n", fout );
	  }
      }

    else
      {
	/* The other Sun framebuffers and the MicroVax behave much more
	 * rationally.  Though we'll write the hex out 16 bits at a time,
	 * the bits will be readable in normal order (at least for the
	 * endianness of the machine), i.e. all the bits for the first row
	 * will be printed first, then all the bits for the second row, etc.
	 */
  
	for( i = 0; i < rowNo; i++ )
	  {
	    fputs( "    ", fout );
	    for( j = 0; j < hWords; j++ )
	      {
		/* Print the hex for the next 16 bits. */
		PutWord( rowStart[i][j], fout );
	      }
	    fputs( "\n", fout );
	  }
      }

    fprintf( fout, "};\n\n" );

    /* Well, we've printed the bits.  Now for the AmazeRaster structure. */

    fprintf( fout, "AmazeRaster %s = { %d, %d, (char *)_%s };\n\n",
		    name, rowNo, maxRowLen, name );

    for( i = 0; i < rowNo; i++ )
	free( rowStart[i] );
  }


PutWord( w, fout )
	BitWord w;
	FILE *fout;
  {
    /* Print out one BitWord, in hex, in the appropriate order for the machine.
     */

    w ^= Mask;
    if( ByteOrder != LITTLE_ENDIAN )
	w = ( BitReverse[ w & 0xFF ] << 8 ) + BitReverse[ ( w >> 8 ) & 0xFF ];
    fprintf( fout, "0x%x,", w );
  }

