/*
 * Machine-dependent timer functions for MicroVAX I, II and Firefly.
 *
 * These machines have an "interval counter" which can interrupt at IPL 16
 *   every 10 ms.  They do not have the Time-Of-Day Register or variable-period
 *   interval counter found on some other VAXen.
 *
 * These machines (not sure about uVAX I) also contain a battery-backed 
 *   clock/RAM chip (not part of the CPU, but on the processor board).  At
 *   present we ignore it except when we want pseudorandom numbers.
 *
 * $Revision: 1.7.1.4 $
 * $Locker:  $
 * $State: Exp $
 */

#include <Vexceptionprotocol.h>
#include "interrupt.h"
#include "asmdefs.h"

#include "uvaxmemory.h"

#ifdef FIREFLY
#include "firefly.h"
#include "config.h"
#endif FIREFLY

/* Assembly language forward */
#ifdef FIREFLY
_BEGIN_EXTERN_C
void Ms_Timer_interrupt _TAKES(());
void Asm_Ms_Timer_interrupt _TAKES(());
_END_EXTERN_C
#else
_BEGIN_EXTERN_C
void Asm_Timer_interrupt _TAKES(());
_END_EXTERN_C
#endif FIREFLY

unsigned ClicksPerTick = 1;
/*
 * Interrupt service routine that calls machine-independent routine
 *   on each interrupt. Assumes no need to disable interrupts because
 *   no higher priority interrupt unblocks processes.  Hmmm.
 *
 * It turns out that on these machines there's no need for machine-specific
 *   actions on a timer interrupt, so we just invoke the machine-independent
 *   Timer_interrupt routine directly.
 */

#ifdef FIREFLY
Call_timer_inthandler(Ms_Timer_interrupt); /* Macro-expands to interrupt-invoked */
				  /*   C call to Timer_interrupt	*/
#else
Call_timer_inthandler(Timer_interrupt); /* Macro-expands to interrupt-invoked */
				  /*   C call to Timer_interrupt	*/
#endif FIREFLY

void
Init_timer()
  {
#ifdef FIREFLY
    if (Multiprocessor)
      ClicksPerTick = CCF_CLICKS_PER_TICK;
#endif FIREFLY
#ifdef FIREFLY
    setexvec(Asm_Ms_Timer_interrupt, VecIntervalTimer);
#else
    setexvec(Asm_Timer_interrupt, VecIntervalTimer);
#endif FIREFLY
    asm("    mtpr    $0x40, $iccs"); 	/* enable timer interrupts */
  }


#ifdef FIREFLY

void
Ms_Timer_interrupt()
  {
    register ProcessorRec *r11;
    register unsigned i;
    extern int ProcessorStopped;
    extern void KForceStop();

    if ( ProcessorStopped )
      {
        KForceStop();
      }
    AsmGetProcessorRecord(r11);
    Timer_interrupt(r11->active);
    /* Now forward tick to secondary processors */
    if (Multiprocessor)
     {
       for (i = 1; i < NProcessors; i++)
         {
           /* The CVAX Firefly clock only interrupts every 20 ms */
           ProcessorArray[i].md.ticks += ClicksPerTick;
           IPCR(i) = FF_INTERRUPT | FF_SET; 
         }
      }
  }
#endif FIREFLY


unsigned long GenerateRandomNumber()
  {
    /*
     * No todr register, so use a pseudo-random number generator.  We seed
     *   the generator with the current time (read from the MicroVAX II clock
     *   chip).  We can't use the clock chip as the random-number generator
     *   because its resolution is one second.
     */
    extern long random();
    return(random());
  }

RandomSeed()
  {
    /*
     * Read stuff from the MicroVAX II's clock chip.  If we really wanted to
     *   know the time we should ensure that the clock chip isn't half way
     *   through an update, but this is just great for randomness.
     * If the clock isn't actually running we'll get anything but random
     *   numbers.  We always start the clock (i.e. clear the "set" bit) in an
     *   attempt to prevent this.  This may, of course, start it with a 
     *   completely bogus time, but we'll live with that.
     */
#define	CLOCK(x) (*(unsigned char *)(uvax2_local_io_v + 0x38000 + (x)))

    CLOCK(0x16) = 0x06;	/* Write Control/Status Reg B to clear "set" bit, */
			/*   and incidentally disable various interrupts, */
			/*   use binary (not BCD) mode, use a 24-hour     */
			/*   clock and disable daylight savings magic.    */
    return CLOCK(0x00)		/* seconds	*/
	^  CLOCK(0x04) << 5	/* minutes	*/
	^  CLOCK(0x08) << 10	/* hours	*/
	^  CLOCK(0x0e) << 14	/* day in month	*/
	^  CLOCK(0x10) << 18	/* month	*/
	^  CLOCK(0x12) << 21;	/* year		*/
  }
