/*
 * V Kernel - Copyright (c) 1985 by Stanford University, All rights reserved.
 *
 * $Revision: 1.5.1.5 $
 * $Locker:  $
 * $State: Exp $
 */
#include <Venviron.h>
#include <Vquerykernel.h>
#include <Vexceptionprotocol.h>

#include "externals.h"
#include "interrupt.h"
#include "memory.h"
#include "process.h"
#include "uvaxmemory.h"

/* Forward */
void MapLocalIntoQBus _TAKES(());
_BEGIN_EXTERN_C
void NoMemOnWrite _TAKES(());
void NoMemOnRead _TAKES(());
_END_EXTERN_C
/* Exports */
char *uvax2_local_io_v = (char *)BOGUS_ADDR;


unsigned 
GetMemorySize(start_addr)
  /*
   * Probe to determine size of memory on a VAX.  Runs in physical memory.
   *  Returns the amount of memory, in bytes.
   */
    unsigned start_addr;
  {
    extern char end;

    register unsigned physaddr;

    /* Test pages until one doesn't work */

    physaddr = MD_UpToPage(start_addr);
    while (Probe((short *)physaddr, PROBE_PRESERVE))
	physaddr += MD_PAGESIZE;

    /*
     * The MicroVAX primary bootstrap probably passed us various magic numbers,
     *   including some that tell us what pages the MicroVAX monitor wants to
     *   itself (way up the top of memory).  We try to use these here, but are
     *   a bit suspicious: if it looks as though the Monitor took more than 64K
     *   (arbitrary figure, pulled out of the air) we ignore it.  In practice,
     *   it should only take two pages plus enough pages for the bitmap (one
     *   page for each two megabytes of memory).  We could check the bitmap for
     *   consistency, or even use the bitmap to initialize our free list, but
     *   that might be just too thorough.
     */
      {
#include <bootparam.h>
	extern RestartParamBlock RPB;
	unsigned rpb_physaddr;

	rpb_physaddr = (unsigned)RPB.pagecount * MD_PAGESIZE;
	if (rpb_physaddr < physaddr && rpb_physaddr + 65536 >= physaddr)
	    physaddr = rpb_physaddr;
      }
    return physaddr;
  }

int
Probe(address, how)
    short *address;
    unsigned how;
  /*
   * Look for memory at address (physical, not virtual).  Should only be used
   *   when memory management is disabled.
   * If how == PROBE_INIT,
   *   write before reading to initialize, then look for writeable memory.
   * If how == PROBE_PRESERVE,
   *   look for writeable memory, but preserve contents of location.
   * If how == PROBE_READ,
   *   just try reading and see whether it causes a bus error.
   */
  {
    register short x;
    int (*oldBusVector)(), (*oldCheckVector)();
    int i;

    /* Set bus error vector */
    oldBusVector = SysControlBlock[VecBusTimeout];
    oldCheckVector = SysControlBlock[VecMachineChk];
    setexvec(NoMemOnWrite, VecBusTimeout);
    setexvec(NoMemOnRead, VecMachineChk);

    if (how == PROBE_INIT) 
	*address = 0x5c5c;	/* Initialize the test location */

    x = ~*address;		/* Read from it */
    
    if (how == PROBE_READ) {
	/* Success if we get here with no bus error */
	setexvec(oldBusVector, VecBusTimeout);
	setexvec(oldCheckVector, VecMachineChk);
	return (1);
    }
    else {
	*address = x;		/* Write back the complement */
	if (*address == x) {
	    /* Success */
	    *address = ~x;
	    setexvec(oldBusVector, VecBusTimeout);
	    setexvec(oldCheckVector, VecMachineChk);
	    return (1);
        }
    	else {
	    /* Failure */
	    asm("ProbeFailed:");
	    setexvec(oldBusVector, VecBusTimeout);
	    setexvec(oldCheckVector, VecMachineChk);
            return (0);
        }
    }

    asm("	.align	2");
    asm("_NoMemOnWrite:");
    asm("	addl2	$8, sp");
    asm("	mtpr	$0x1c, $ipl");
    asm("	jbr	ProbeFailed");

    asm("	.align	2");
    asm("_NoMemOnRead:");
    asm("	addl2	$48, sp");
      {
	int i;

	if (MachineConfig.processor == PROC_UVAX2)
	  {
	    i = *(int *)(UVAX2_LOCAL_IO_P + UVAX2_MSER);
	    *(int *)(UVAX2_LOCAL_IO_P + UVAX2_MSER) = i;
	  }
      }
    ;asm("	mtpr	$0x1c, $ipl");
    asm("	mtpr	$1, $mcesr");
    asm("	jbr	ProbeFailed");

    /* Can't actually get here, but have the return for C++ compiler */
    return(0);

  }

#include "qbusaddresses.h"

	char *qbus_devices_p	= (char *)UVAX_Q22_IO_P;
	char *qbus_devices_v	= (char *)BOGUS_ADDR;
	char *qbus_devices_base	= (char *)UVAX_Q22_IO_BASE;
unsigned long qbus_devices_size = UVAX_Q22_IO_SIZE;

	char *qbus_memory_p	= (char *)BOGUS_ADDR;

	char *qbus_memory_base	= (char *)UVAX_Q22_MEM_BASE;
unsigned long qbus_memory_size	= UVAX_Q22_MEM_SIZE;

/*
 * BasicInit_SpecificVAX(): This is called very early by Init_memory_mapping
 *   to set up any MicroVAX-specific memory-mapping information that may be
 *   needed before we can successfully execute FindConfiguration() and build
 *   the system page table.
 * FindMachineType() has already been executed, so MachineConfig.processor
 *   is valid.
 */
void
BasicInit_SpecificVAX()
  {
    switch (MachineConfig.processor)
      {
	case PROC_UVAX1:
	    qbus_memory_p = (char *)UVAX1_Q22_MEM_P; break;
	case PROC_UVAX2:
	    /* Includes Firefly (on main processor only) */
	    qbus_memory_p = (char *)UVAX2_Q22_MEM_P; break;
	default:
	    /* If we ever have such a machine, someone can take this */
	    /*   annoying message out once it's served its purpose.  */
	    printx("Don't know where Q-Bus memory should be mapped\n");
	    break;
      }
  }

void
BuildSysMap_SpecificVAX()
  {
    if (MachineConfig.processor == PROC_UVAX2 /* or Firefly */)
	uvax2_local_io_v = BuildSysMap_MapPhysPages(
			(char *)UVAX2_LOCAL_IO_P, UVAX2_LOCAL_IO_SIZE);
    qbus_devices_v = (char *)
		BuildSysMap_MapPhysPages((char *)UVAX_Q22_IO_P, UVAX_Q22_IO_SIZE);
  }

/*
 * Do any remaining machine-specific memory mapping that could not be done
 *   until we'd finished building the system page table.  Assumes we're
 *   now running with memory mapping enabled.
 */
void
InitMem_SpecificVAX()
  {
    if (MachineConfig.processor == PROC_UVAX2)
	MapLocalIntoQBus();
  }

/*
 * (MicroVAX II) Map kernel local memory into the Qbus address space so that
 *   it can be DMA'd with.  This is a crock which then lets us forget about
 *   the virtual/physical/q-bus address distinction, since we fudge the
 *   mapping between them so that it's the identity function (more-or-less; we
 *   may do things like ignoring the most significant bits of the address,
 *   because the hardware does this anyway).  Note that this only works
 *   for addresses in the range we map "properly", i.e. kernel text/data/bss.
 * This crock should go away; we should acknowledge the existence of the map
 *   by doing a MapDvma whenever we generate Q-Bus addresses (e.g. in the
 *   DEQNA driver).  If we DO eliminate this, we must be careful to retain the
 *   part of this routine that initializes things for future calls to
 *   AllocateDvmaSpace.
 * Assumes that we are running in virtual memory.
 */

unsigned FreeDvmaSpace;	/* Next Q-Bus DMA address that we can allocate for  */
			/*   some device that needs a slice of the DMA space*/

void
MapLocalIntoQBus()
  {
    register QBusMapEntry *map;
    unsigned first, kernel_limit;
    register int i;

    first = MD_NumPages(vaxMap.kernel.phys);
    kernel_limit = MD_UpToPage(vaxMap.kernel.phys + vaxMap.kernel.bytes);

    map = (QBusMapEntry *)(uvax2_local_io_v + UVAX2_QMAP_START);
    for (i = 0; i < QBMAP_NumEntries; i++, map++)
if (i != 0x1f00)	/* Horrible and temporary hack for Firefly */
	map->word = 0;
    map = &((QBusMapEntry *)(uvax2_local_io_v + UVAX2_QMAP_START))[first];
    for (i = first; i < MD_NumPages(kernel_limit); i++, map++)
	map->word = QBMAP_Valid | i;
    FreeDvmaSpace = kernel_limit;
	/* We mapped DMA address x to physical address x, so this works */
	/*   even though LHS is an address used by Q-bus devices.	*/
    printx("Mapped Local -> QBus\n");
  }

void
AllocateDvmaSpace(size, qbus_mem_address)
    unsigned size;
    unsigned *qbus_mem_address;
  {
    switch (MachineConfig.processor)
      {
	case PROC_UVAX2:
/* Should really check that noone out there uses any part of this space */
	    *qbus_mem_address = FreeDvmaSpace;
	    FreeDvmaSpace += MD_UpToPage(size);
	    if ((unsigned)FreeDvmaSpace > QBMAP_NumEntries * MD_PAGESIZE)
		Kabort("Out of Q-Bus DMA space");
	    break;
	default:
	    printx(
		"Don't know how Q-bus memory access works on this machine -\n"
		  );
	    printx("  assuming it's trivial (as on a MicroVAX I)\n");
	case PROC_UVAX1:
	    *qbus_mem_address = (unsigned)-1;
	    break;
      }
  }

void
AllocateAlignedDvmaSpace(size, qbus_mem_address, alignment)
    unsigned size;
    unsigned *qbus_mem_address;
    unsigned alignment;	/* The starting Q-bus address we allocate should */
			/*   be a multiple of 2**alignment		 */
  {
    register unsigned mask;

    if (alignment > MD_PAGEBITS)
      {
	mask = (1 << alignment) - 1;
	FreeDvmaSpace = (FreeDvmaSpace + mask) & ~mask;
      }
    AllocateDvmaSpace(size, qbus_mem_address);
  }

void
MapDvma( qbus_mem_address, virt_address, length )
    char **qbus_mem_address, *virt_address;
    unsigned length;
  {
    register QBusMapEntry *q;
    register PageTableEntry *p;
    register int numpages;

    if (MachineConfig.processor == PROC_UVAX2)
      {
	if ( !KernelCanRead((unsigned)virt_address, length) )
	    Kabort("Tried to map bogus address to DMA");
	numpages = MD_NumPages(virt_address+length) -
			MD_PageNumber(virt_address);
	q = & ((QBusMapEntry *)(uvax2_local_io_v + UVAX2_QMAP_START))
			[MD_PageNumber(*qbus_mem_address)];
/* Might be nice to bounds-check addresses a little better */
	switch((unsigned)virt_address >> 30)
	  {
	    case 0:
		p = ( (PageTableEntry *)
			(GetAddressableProcess()->team->team_space.p0base) ) +
		    MD_PageNumber(virt_address);
		break;
	    case 1:
		Kabort("MapDvma: Don't know how to map P1 space");
		numpages = 0;
		break;
	    case 2:
		p = &SysPageTable[MD_PageNumber(virt_address-SYS_SEG_START)];
		break;
	    case 3:
		Kabort("MapDvma: got bogus address (in control space)");
		numpages = 0;
		break;
	  }
	while (numpages-- > 0)
	    *(PageTableEntry *)q++ = *p++;
		/* Hack.  Works because Q-Bus map entries have pfn field   */
		/*   and valid in same place as PTEs, and don't care about */
		/*   the other bits.					   */
      }
    else /* assume like uVAX I, where Q-bus memory IS local memory */
	if (virt_address >= vaxMap.kernel.virt &&
	    virt_address + length <= vaxMap.kernel.virt + vaxMap.kernel.bytes )
	    /* For this region of the kernel, virt_addr = phys_addr|80000000 */
	    *qbus_mem_address = virt_address - vaxMap.kernel.virt
					     + vaxMap.kernel.phys;
	else
	    Kabort("Can\'t do DMA mapping properly on a MicroVAX I - sorry");
  }

int
MachineCheckMagic()
  {
    int i;

    if (MachineConfig.processor == PROC_UVAX2)
      {
	/* Read the Memory System Error Register, then clear the */
	/*   error flags there					 */
	i = *(unsigned short *)(uvax2_local_io_v + UVAX2_MSER);
	/* Writing 1's to any bits that were set clears them, so */
	/*   just write the same thing back			 */
	*(unsigned short *)(uvax2_local_io_v + UVAX2_MSER) = i;
	return i;
      }
    else if (MachineConfig.processor == PROC_UVAX1)
      {
	asm("	mtpr	$0, $mcesr");	/* say we handled the Check */
      }
    return 0;
  }
