/*
 * Distributed V Kernel
 * Copyright (c) 1982 by David Cheriton, Tim Mann, Willy Zwaenepoel
 * (Transliterated from Zed and Verex Kernel)
 *
 * Machine-dependent kernel routines
 *
 * $Revision: 4.38.1.23 $
 * $Locker:  $
 * $State: Exp $
 */

#include <Vprocessprotocol.h>
#include <Vquerykernel.h>
#include <Vmtp.h>
#include <Vusercommon.h>
#include <b.out.h>

#include "asmdefs.h"
#include "asmdefines.h"
#include "config.h"
#include "externals.h"
#include "memory.h"
#include "process.h"
#include "processor.h"
#include "timer.h"
#include "ipc.h"
#include "devman.h"

#ifdef FIREFLY
#include "firefly.h"
#endif FIREFLY

unsigned ProcessorAssignment = 0;
unsigned NProcessors = 1;
BooleanInt Multiprocessor = FALSE;

/* Temporary location for storing ipl in the Lockq and Unlockq macros */
long GlobalLockKludge;

/* These would break the optimizer if they appeared inside the function */
asm("	.globl	Switch");
asm("	.globl	ActivateReadyqHead");

/* Basic Process Implementation */
void
dummySwitch()

    /*
     * Save the state of the active process and activate the process
     * specified by readyq.head, loading its state from its process
     * descriptor.  It is assumed to be called with interrupts
     * disabled.  Switch and ActivateReadyqHead are never called as
     * functions; they are jumped to.  The code is written inside this
     * dummy function body to allow C statements as well as asms to be
     * used.
     *
     * Warning: Do not use any automatic variables in this code, and
     * remember not to declare more register variables than there are
     * registers!
     */

  {
    register ProcessorRec *r11;
    register unsigned char  *r10;
    register Team *r9;
    register Process *pd;
    register PFV finish_up;

    /*
     * Switch is jumped to when we know that we are going to do a
     * context switch.  Currently, however, the only difference between
     * it and ActivateReadyqHead is that Switch saves the process context.
     *
     */
    asm("Switch:");
    SaveProcessContext();
  
    asm("1:	");
    asm("ActivateReadyqHead:");
    AsmGetProcessorRecord(r11);

    /*
     * Now test for the case where the active process has been destroyed,
     * either by a remote DestroyProcess or a local suicide, in which
     * case its pd must be added to FreePdq. See comments for 
     * DestroyOneProcess in pm.c.
     */
    if ( r11->active->localPid == INVALID_PROCESS &&
         r11->active != &r11->idle_process )
      {
        SimpleFreePd(r11->active);
      }

newprocess:

    r11->active = pd = r11->readyq.head;  /* Remove new active process readyq */
    r11->readyq.head = pd->link;
    pd->queuePtr = NULL;
    r9 = pd->team;
    SetAddressableProcess(pd);

    /* Call a function to finish up if necessary */
    if (finish_up = pd->finish_up) {
        disable;		/* Ensure running at disable level */
        pd->finish_up = NULL;	/* This has to be in front of the call to
				   finish_up because finish_up may set it to
				   a new value. */

        if( AlienProcess(pd) && !LocalAlien(pd) && pd->srcSegment.size &&
             ( finish_up == VMTPServer || finish_up == ProcessServer ||
               finish_up == MemoryServer || finish_up == DeviceServer) )
          r10 = (unsigned char *)pd->srcSegment.ptr;
        else
          r10 = NULL;
	(*finish_up)(pd);
        if (r10)
          {
            FreeSegmentBuffer(r10);
          }

        /* Check in case we are no longer ready */
        disable;
        if( pd != r11->readyq.head ) goto newprocess;
	r11->readyq.head = pd->link;
	pd->queuePtr = NULL;
    }

    /* HACK: an interrupt while in the kernel may have changed this */
    r10 = (unsigned char *) &r11->md.kstack[STACK_SIZE];
    ; asm("	mtpr r10,$ksp");


    /*
     * Message-register interface goodies.  Check whether we're meant to
     *   return a message in registers r3-r10, and do it if necessary.
     *   Always clears, as well as testing, the MSG_FLAG in pd_flags, so
     *   that next time we don't accidentally think we're meant to return
     *   a message in registers.
     */

    /* pd = r8 */
    asm("       movl    _STATUS(r8),-(sp)");     /* Restore status */
    asm("       movl    _PC(r8),-(sp)");	/* Restore PC */
    /* TLB already invalidated by SetAddressableProcess */

#ifdef FIREFLY
    /* Do this up here so that do not trash registers */
    if ( IPRBuffer.sequenceNo > 0 )
      {
        IPRBuffer.sequenceNo--;
      }
    else
      {
        KUnlock();
      }
#endif FIREFLY

    if ( !(pd->pdFlags & MSG_FLAG) )
      {
        pd->pdFlags &= ~MSG_FLAG;
        goto nomessage;
      }
    ;pd->pdFlags &= ~MSG_FLAG;
   
    ;asm("	movl    r8, r11" );
    asm("	mtpr	_USER_STACK_PTR(r11), $usp");	/* Restore usp */
    asm("	mtpr	_PCONTEXT+_SSP(r11), $ssp");	/* Restore ssp */
    ;asm("	movl    _PACKET_DELIVERY(r11), r0" );	/* pdelivery */
#ifdef LOAD_USING_TRAP_PARAMS
    asm("	movl    _TRAP_PARAMS(r11), r1" );	/* server */
#else
    asm("	movl    _SERVER_PID(r11), r1" );	/* server */
#endif
    asm("	clrl	r2");				/* server low */
    asm("	moval	_MSG(r11), r10");		/* msg... */
    asm("	movq	(r10)+, r3");
    asm("	movq	(r10)+, r5");
    asm("	movq	(r10)+, r7");
    asm("	movq	(r10) , r9");
#ifdef LOAD_USING_TRAP_PARAMS
    asm("	movl	_TRAP_PARAM2(r11), r11");	/* tid */
    asm("	movl    r1, ap");			/* client */
    asm("	clrl	fp");				/* client low */
#else
    asm("	movl	_CLIENT_PID(r11), ap");         /* client */
    asm("	clrl	fp");				/* client low */
    asm("	movl	_ALIEN_TRANSACTID(r11), r11");	/* tid */
#endif

    asm("	rei");

    nomessage:
    ;asm("	movl    r8, r11" );
    ;asm("	movl	_USER_STACK_PTR(r11),fp");
    asm("	mtpr	fp, $usp");	/* Restore user stack pointer */
    asm("	mtpr	_PCONTEXT+_SSP(r11), $ssp");	/* Restore ssp */
    ;asm("	moval	_PCONTEXT(r11),fp");

    asm("	movl	(fp)+,r0");
    asm("	movl	(fp)+,r1");
    asm("	movq	(fp)+,r2");
    asm("	movq	(fp)+,r4");
    asm("	movq	(fp)+,r6");
    asm("	movq	(fp)+,r8");
    asm("	movq	(fp)+,r10");
    asm("	movq	(fp),ap");
    asm("	rei");			/* Switch to user process */

  }

/* Process Management */

void
SetProcessorFields( pd, migrating, entry, stack )
    register Process *pd; unsigned migrating; Unspec entry; Unspec stack;
    /* Set processor-specific fields for the process to execute in team td. */
    /* Handle migrating process is "migrating" is non-zero. */
  {
    register unsigned j;

#ifndef FIREFLY
    pd->readyq = Readyq(); /* this should disappear */
#endif FIREFLY
    if( !migrating )
      {
	pd->userStackPtr = stack;
	pd->status = (Unspec) 0x03c00000;
	pd->PC = entry;
      }
    else
      {
	j = (unsigned) pd->status;
	j &= 0x000000FF; /* Turn off all but user bits. */
	j |= 0x03c00000; /* Ensure in user mode. */
	pd->status = (Unspec) j;
      }
  }

void
AssignProcessToReadyq( pd )
    register Process *pd;
  {
    register ProcessorRec *r10;
    register ProcessorRec *r9;
    AsmGetProcessorRecord(r9);
    
    ProcessorAssignment = (ProcessorAssignment + 1) % NProcessors;
    r10 = &ProcessorArray[ProcessorAssignment]; 
    printx("Processor %x",ProcessorAssignment);
#ifdef FIREFLY_DEBUG
    printx("Creating process pd = 0x%x, pid = 0x%x,...\n",pd,pd->pid);
#endif FIREFLY_DEBUG
    pd->readyq = &r10->readyq;
  }

void
Ms_CreateTeam(td, pd)
    register Team *td;
    Process *pd; /* Eliminate this */

  /* Perform the machine-specific portion of team creation.
   */
  {
    ReclaimMemory(td);	/* Be sure team space is null */
  }


/*** Kernel Initialization ***/


void
Init_root_team()

  /* Create the first or root team and first process
   */
  {
    register ProcessorRec	*r11;
    register CreateProcessRequest *req;
    register unsigned	teamsize;
    register Process	*pd, *active;

    AsmGetProcessorRecord(r11);
    active = &r11->idle_process; /* active = idle process */

    SetAddressableProcess( active );

    /* Assume first team is already loaded at TEAM_START */
    teamsize = (unsigned)( TeamBheader.tsize + TeamBheader.dsize
		+ TeamBheader.bsize + INIT_STACK);

    req = (CreateProcessRequest *) &(active->msg);
    req->entry = TeamBheader.entry;
    req->stack = teamsize-16;			/* Why "-16" ? TMM. */
    req->pidBlockNumber = 0;
    if( (req->requestCode=CreateTeam(active)) != OK )
      {
	printx( "Failed to create first team: %x\n", req->requestCode );
        Kabort( "first team" );
      }
    pd = MapPid( req->pid );
    RemoveQueue( pd );
    SetUpRootPT(pd, teamsize);
    pd->team->team_priority = 0x0100;
    pd->priority = ROOT_PRIORITY | pd->team->team_priority;
    pd->userNumber = (unsigned)SYSTEM_USER;
    if (FirstTeam != pd->team)
	Kabort("TD for First Team isn't where we expected");

    /* We'll use the V_BOOT_ID while checking for pid block collisions, then
     * change our pid block when we find a unique value.
     */
#ifdef LITTLE_ENDIAN
    pd->localPid = pd->pid |= ((V_BOOT_ID << 16) | LITTLE_ENDIAN_HOST);
#else
    pd->localPid = pd->pid |= (V_BOOT_ID << 16);
#endif
    NewPidBlock(pd);
  }

/*
 * We decide at boot time how many process- and team-descriptors we want, and
 *   allocate a suitably sized chunk of system space to hold them.
 * A VAX Process Control Block is embedded in each V Process descriptor.
 *   The VAX PCBB register, which points to this block, holds a physical
 *   address rather than a virtual one.  So... we make sure the Process
 *   descriptors are contiguous in physical as well as virtual memory, and
 *   store the physical address of the PCB in the Process descriptor.
 */

extern unsigned long	MaxProcesses;
extern Team		*TeamDescriptors;
extern unsigned long	MaxTeams;

void
CalcNum_PDs_and_TDs()
  {
    /*
     * Decide how many Process and Team descriptors to allocate.  For each of
     *   them we (pretty arbitrarily) allow 16k per megabyte of memory, with
     *   a limit of 64K.
     * Process descriptors cost about 340 bytes each.  Team descriptors are
     *   about 60 bytes each, but we take into account the fact that each team
     *   uses space in the system page table as well (currently 1K per team,
     *   since TEAM_LIMIT = 16Mb).
     */
    long int pd_memory;

    pd_memory = MachineConfig.memory >> 6;
    if (pd_memory > 0x10000)
	pd_memory = 0x10000;
    MaxProcesses = pd_memory / sizeof(Process);
    MaxTeams     = pd_memory / (sizeof(Team) + TEAM_SYSPT_BYTES);
  }

void
BuildSysMap_TDs()
  {
    TeamDescriptors	= (Team    *)
		BuildSysMap_AllocPages(MaxTeams     * sizeof(Team)   );
  }

void
AllocatePds()
    /* really just here for compatibility with the Sun version */
  {
    printx("Allocating %d team descriptors and %d process descriptors.\n",
			MaxTeams, MaxProcesses);
  }

void
Ms_pd_init(pd)
    register Process *pd;
    
  /*
   * Perform machine specific initialization of process descriptors
   */
  {
    pd->readyq = Readyq();

    pd->status = (Unspec)0x03c00000;
  }


void
Idle()

  /* Idle process function. Wait for an interrupt and
   * loop as soon as redispatched.
   */
  {
    register Unspec r11;
    register ProcessorRec *r10;
    register Process *pd;
    
    Kdisable;
    KLock();
    /* Initialize the Idle Process */
    AsmGetProcessorRecord(r10);
    pd = &r10->idle_process;

#ifdef FIREFLY
    if (GetProcessorNo(r10) != 0)
      {
        NotPrimaryComplete = 1;
        SecondaryProcessorComplete();
        KUnlock();
        WaitForPrimaryProcessor();
        KLock();
      }
    else
      {
        PrimaryProcessorComplete();
      }
#endif FIREFLY

    Ms_pd_init(pd);
    pd->team = NULL;
    if (SubCreateTeam(&r10->idle_process, &r10->idle_process) != OK)
	Kabort("SubCreateTeam failed");
    pd->status = (Unspec)0x00000000;

    SetAddressableProcess(r10->active);

    pd->pdFlags |= MSG_FLAG;
    pd->userStackPtr = (Unspec)&r10->md.kstack[STACK_SIZE];

    asm("    moval	loop,r11");		/* Set up for reschedule */
    pd->PC = r11;

    /* Add to ready queue. */
#ifdef FIREFLY
    if (GetProcessorNo(r10) != 0)
      {
        printx("Scheduling on the secondary processor\n");
      }
#endif FIREFLY
    Addready(pd);

    asm("    jmp	ActivateReadyqHead");	/* Switch to head of readyq */
    asm("loop:");
    while (1);
  }

void
InitSecondary()
{
  InitPerProcessorState();
  Idle();
}

/*   Copy in from external processor state representation.
 *   Also protects the system from attempts to assign invalid states
 *   to processes.
 */
ResponseCode CopyInRegisters(pd, external, modify_self)
    register Process *pd;
    register Processor_state *external;
    register int modify_self;
  {
    if (((unsigned)external->status & PSL_CURMOD) == PSL_CURKERNEL)
      {
	return (NO_PERMISSION);
      }
   
    if(modify_self) return(OK);
    
    pd->pcontext = external->mdcontext.pcontext;
    pd->status = (Unspec)(((int)pd->status & 0xFFFF0000) |
			((int)external->status & 0x0000FFFF));
    return (OK);
  }


void
CopyOutRegisters(external, pd)
    register Processor_state *external;
    register Process *pd;
  {
    external->status = pd->status;
    external->mdcontext.pcontext = pd->pcontext;
  }

void
FindProcessorConfiguration()
/*
 * Reads a location in the EPROM to find out if we are:
 * 1. A vanilla uvax-2.
 * 2. A CCF Firefly (secondary processors are cvaxes).
 *
 */
  {
#ifdef FIREFLY
    unsigned pconfig;
    pconfig = * (unsigned *) PROCESSOR_CONFIG_ADDR;

    switch (pconfig)
      {
        case CONFIG_UVAX2:
	  printx("Vanilla MicroVAX 2\n");
	  break;
        case CONFIG_CCF:
          Multiprocessor = TRUE;
	  printx("CVAX Firefly\n");
	  break;
	default:
	  Kabort("Unknown processor configuration\n");
	  break;
      }
#endif FIREFLY
  }

void
FindMachineType()
/* This function is called BEFORE VM is enabled */
  {
    register unsigned r11;

    /* We KNOW we are a VAX of some sort and this is true of all Vaxen */
    MachineConfig.machine = MACH_VAX;
    MachineConfig.clicksPerSecond = CLICKSPERSEC;
    AddPeripheral(PRF_CONFREG);

    /* find type of processor */
    asm("	mfpr	$sid, r11");
    MachineConfig.confreg = r11;
    switch (r11 >> 24)
      {
        case 1:
	  MachineConfig.processor = PROC_VAX11_780;
	  printx("VAX-11/780 processor\n");
	  break;

	case 2:
	  MachineConfig.processor = PROC_VAX11_750;
	  printx("VAX-11/750 processor\n");
	  break;

	case 3:
	  MachineConfig.processor = PROC_VAX11_730;
	  printx("VAX-11/730 processor\n");
	  break;

	case 7:
	  MachineConfig.processor = PROC_UVAX1;
	  printx("MicroVax I processor");
	  printx(" (KD32-A%c, %c_floating)\n",
	    r11 & 0x10000 ? 'B' : 'A',
	    r11 & 0x10000 ? 'D' : 'G');
	  break;

	case 8:
	  MachineConfig.processor = PROC_UVAX2;
/*	  printx("MicroVax II processor\n");	*/
	  break;

	default:
	  Kabort("Unknown processor type\n");
	  break;
      }
    if ( MachineConfig.processor == PROC_UVAX2 )
      FindProcessorConfiguration();
  }

void
GetStackSizes(reply)
  register KernelStackReply *reply;
  {
    /*
     * A routine to satisfy Mr Boyle's thirst for statistics
     */

     /* We "know" that the kernel and interrupt stacks are both STACK_SIZE */
    /*   bytes long - tsk, tsk		   				   */

    register ProcessorRec *r10;
    register char *ptr;

    /*
     * Start at the beginning of allocated kernel stack area looking for the
     *   first location which does not match the initialized value.
     */

    AsmGetProcessorRecord(r10);
    for (ptr = r10->md.kstack; ptr < &r10->md.kstack[STACK_SIZE]; 
            ptr += sizeof(long))
	if ( *(unsigned long *)ptr != KERNEL_STACK_INIT_VALUE )
	    break;
    if (ptr > &r10->md.kstack[STACK_SIZE])
	/* Only happens if STACK_SIZE is not a multiple of 4 */
	reply->kernelStackUsed = 0;
    else
	reply->kernelStackUsed = &r10->md.kstack[STACK_SIZE] - ptr;
    /*
     * Ditto for the interrupt stack
     */
     for (ptr = r10->md.istack; ptr < &r10->md.istack[STACK_SIZE]; 
              ptr += sizeof(long))
	if ( *(unsigned long *)ptr != KERNEL_STACK_INIT_VALUE )
	    break;
    if (ptr > &r10->md.istack[STACK_SIZE])
	reply->interruptStackUsed = 0;
    else
	reply->interruptStackUsed = &r10->md.istack[STACK_SIZE] - ptr;
  }

#ifdef FIREFLY
Process *
GetAddressableProcess() 
  {
    register ProcessorRec *r11;
    AsmGetProcessorRecord(r11);
    return(r11->md.addressableProcess);
  }

void
SetAddressableProcess(newprocess) 
  register Process *newprocess;
  {									
    register ProcessorRec *r10;
    register unsigned r9; 
    AsmGetProcessorRecord(r10);
    r10->md.addressableProcess = newprocess;
    r9 = newprocess->team->team_space.p0base;
    ;asm("	mtpr	r9, $p0br");	
    r9 = newprocess->team->team_space.p0len;	
    ;asm("	mtpr	r9, $p0lr");
    asm("  mtpr	$0,$tbia");
  }
#endif FIREFLY

#ifdef FIREFLY

short KernelMutex()
  {
    register ProcessorRec *r11;
    register int lockStatus;
    extern short KernelInterrupted;
    if (KernelInterrupted)
      return(1);
    if (HoldKLock == 1)
      return(0);
    else
      {
        KLockNowait(lockStatus); 
        if (lockStatus == 1) 
          {
            HoldKLock = 1;
            return(0);
          }
        else
          return(1);
      }
  }

#endif FIREFLY
int
KernelGetChar()
  {
    register ProcessorRec *r10;
#ifdef FIREFLY
    register IPRBufferType *iprb = &IPRBuffer;
    AsmGetProcessorRecord(r10);
    if ( !IsPrimaryProcessor(r10) )
      {
        iprb->type = DEVICE_SERVICE;
        iprb->device_function = K_GETCHAR;
        SendIPR(0);
        return  (iprb->params[0]);
      }
    else
#endif FIREFLY
      return (K_getchar());
  }

#ifdef FIREFLY
void
KernelPutChar(c)
  register int c;
  {
    register ProcessorRec *r10;
    AsmGetProcessorRecord(r10);
    if ( !IsPrimaryProcessor(r10) )
      {
        register IPRBufferType *iprb = &IPRBuffer;
        iprb->params[0] = (Unspec) c;
        SendDeviceServiceRequestToPrimary(iprb,K_PUTCHAR);
        return;
      }
    else
      K_putchar(c);
  }
#endif FIREFLY

void 
AllocateKernelStacks()
  {
    unsigned i;
    for (i = 1; i < MAX_PROCESSORS; i++)
      {
        ProcessorArray[i].md.kstack = (char *)SysMalloc(STACK_SIZE, 0);
        ProcessorArray[i].md.istack = (char *)SysMalloc(STACK_SIZE, 0);
      }
  }

/* Returns a pointer to the readyq for calling processor */
SyncQueue *Readyq()
  {
    register ProcessorRec *r11;

    AsmGetProcessorRecord(r11);
    return(&r11->readyq);
  }


/* Sets active = &idle_process for calling processor and returns active */
Process *Init_active()
  {
    register ProcessorRec *r11;

    AsmGetProcessorRecord(r11);
    r11->active = &r11->idle_process; 
    return(r11->active);
   }
