/*
 * V Kernel - Copyright (c) 1985 by Stanford University
 *
 * Kernel Framebuffer support routines for DEC SRC's MDC framebuffer etc.
 *
 *
 * $Revision: 1.6.1.6 $
 * $Locker:  $
 * $State: Exp $
 */

#include "qbusaddresses.h"
#include <Venviron.h>
#include <Vioprotocol.h>
#include <Vquerykernel.h>
#include "process.h"
#include "memory.h"
#include "uvaxmemory.h"
#include "devman.h"
#include "externals.h"
#include "mdc.h"

/* Imports */
/* Should be moved */
extern int _rawio_use_serial;	/* Hack in the libc/rawio routines to */
				/*   direct low-level output away from*/
				/*   the framebuffer.		      */

/* Forward */
ResponseCode MdcCreate _TAKES(( Process *, DeviceInstance *));
ResponseCode MdcWrite _TAKES(( Process *, DeviceInstance *));
ResponseCode MdcRead _TAKES(( Process *, DeviceInstance *));
ResponseCode MdcQuery _TAKES(( Process *, DeviceInstance *, unsigned short));
ResponseCode MdcRelease _TAKES(( Process *, DeviceInstance *));
void ResetMdc _TAKES(());

#define	MDC_CSR	( * (MDC_ControlStatusReg_u *)	\
		    qbus_device_to_v(QBUS_MDC_ControlStatusReg_Addr) )

char	*commandMemory_virt_addr,
	*commandMemory_qbus_addr;
unsigned commandMemory_size;

/*
 * FindMdc(): Determines whether there is an MDC at the standard location.
 *	      Uses physical, not virtual, addressing.
 */
unsigned char 
FindMdc(enable_video_func)
    int (**enable_video_func)();
  {
    if (Probe((short *)qbus_device_to_p(QBUS_MDC_ControlStatusReg_Addr), PROBE_READ))
      {
	printx("Found a MDC framebuffer, mouse and keyboard\n");
	AddPeripheral(PRF_MOUSE_MDC);
	AddPeripheral(PRF_FRAMEBUFFER_MDC);
	/*
	 * Leave *enable_video_func alone, since the MDC doesn't give us any
	 *   simple way to turn the display on and off.
	 */
	return PRF_FRAMEBUFFER_MDC;
      }
    else
	return PRF_NONE;
  }

#define	COMMANDMEMORY_BYTES 0x20000	/* Maximum usable commandMemory size */
					/*   (64K words = 0x20000 bytes)     */

/*
 * BuildSysMap_Mdc() must be called at boot time when the system memory map is
 *		     being constructed.  It asks the memory manager to allocate
 *		     128K of (not necessarily contiguous) physical memory, map
 *		     it in the system page table, and tell us the starting
 *		     virtual address we should use to refer to it.
 */
void
BuildSysMap_Mdc()
  {
    commandMemory_size	    = COMMANDMEMORY_BYTES;
    commandMemory_virt_addr = BuildSysMap_AllocPages(commandMemory_size);
  }

/*
 * Map the memory allocated above so that not only the kernel but user
 *   processes and the MDC (by Q-Bus DMA) can read and write it, then do
 *   boot-time initialization of the MDC and add it to the list of kernel
 *   device drivers.
 */
void
MdcPowerup()
  {
    ChangeSysPagesProtection((unsigned)commandMemory_virt_addr, commandMemory_size,
				VM_RWRW);
    AllocateAlignedDvmaSpace(commandMemory_size, (unsigned *)&commandMemory_qbus_addr, 
				MDC_CSR_ADDRESS_SHIFT);
    MapDvma(&commandMemory_qbus_addr, commandMemory_virt_addr,
				commandMemory_size);
    ResetMdc();
    MdcInputInit();
    AddDevice("mdc",  0, MdcCreate,   0, 0);
  }

void
ResetMdc()
  {
#ifdef FIREFLY
    register ProcessorRec *r11;
#endif FIREFLY
    register MDC_HeadRecord *hr;
    register int i;

    /*
     * NOTE:  This routine and the MDC don't get along all that well together.
     *	      Sometimes an attempt to reset the MDC makes it flash randomly for
     *	      a bit, then hangs the whole machine.  The delay loops in the code
     *	      below are a superstitious and unsuccessful attempt to cure this.
     */

#ifdef FIREFLY
    AsmGetProcessorRecord(r11);
    if ( !IsPrimaryProcessor(r11) )
      {
        register IPRBufferType *iprb = &IPRBuffer;
        SendDeviceServiceRequestToPrimary(iprb,RESET_MDC);
        return;
      }
#endif FIREFLY
    if (_rawio_use_serial == 0)
      {
	printnl("Switching console output from framebuffer to serial line\n");
	_rawio_use_serial = 1;
	printnl("Switched console output from framebuffer to serial line\n");
      }
    if (MDC_CSR & MDC_CSR_VALID)
      {
	/*
	 * If the MDC is enabled, disable it (and wait for it to notice) before
	 *   we write the new info to it.
	 */
	MDC_CSR = 0;
	for (i=0; MDC_CSR & MDC_CSR_COMMAND_RUNNING; i++)
	    if (i > 10000 /* another completely random number */ )
		Kabort("Couldn't reset MDC framebuffer controller");
	for (i=0; i<400000; i++)
	    ; /* MDC gets upset if we immediately set the bit again.  Grrr. */
      }

    hr = (MDC_HeadRecord *)commandMemory_virt_addr;
    hr->currentCommand = 0;	/* MDC will wait until this becomes non-zero */
    hr->cursorLoc.x = 0;	/* Cursor is off-screen (don't display it)   */
    hr->cursorLoc.y = 0;

    MDC_CSR = MDC_CSR_VALID |
		((unsigned)commandMemory_qbus_addr >> MDC_CSR_ADDRESS_SHIFT);
    for (i=0; !(MDC_CSR & MDC_CSR_COMMAND_RUNNING); i++)
	if (i > 10000 /* ditto */)
	    Kabort("Couldn't initialize MDC framebuffer controller");
  }

/*
 * Device Routines
 */

InstanceId MdcInstance;

ResponseCode 
MdcCreate( pd, inst )  
    Process *pd;
    register DeviceInstance *inst;
  {
    DeviceInstance *mdcInst;
  /*
   * Create an instance for the MDC.
   */
    CreateInstanceRequest *req = (CreateInstanceRequest *) &pd->msg;

    if (req->filemode != FCREATE)
	return (MODE_NOT_SUPPORTED);
    if ( (mdcInst = GetDeviceInstance(MdcInstance)) != NULL )
      {
	if (MapPid(mdcInst->owner))
	    return BUSY;
	MdcInstance = 0;
	mdcInst->owner = 0;
      }

    ResetMdc();		/* Give the user a nice, clean MDC to start with */

    /*
     * Initialize the device instance descriptor, can assume all fields
     * are zero except for id and first instance's owner.
     */
    inst->readfunc = MdcRead;
    inst->writefunc = MdcWrite;
    inst->modifyfunc = NotSupported;
    inst->queryfunc = MdcQuery;
    inst->releasefunc = MdcRelease;

    inst->id |= (UIO_READABLE|UIO_WRITEABLE|UIO_RANDOM_ACCESS);
    inst->blocksize = sizeof(short);
    inst->lastbytes = inst->blocksize;
    inst->lastblock = 0;

    MdcInstance = inst->id;

    return OK;
  }

ResponseCode
MdcRead( pd, inst )
    Process		*pd;
    DeviceInstance	*inst;

    /* Return the current contents of the control/status register */
  {
#ifdef FIREFLY
    register ProcessorRec *r11;
#endif FIREFLY
    register IoRequest *req;
    register IoReply *reply;
    register unsigned short *buf;

#ifdef FIREFLY
    AsmGetProcessorRecord(r11);
    if ( !IsPrimaryProcessor(r11) )
      {
        register IPRBufferType *iprb = &IPRBuffer;
        iprb->params[0] = (Unspec) pd;
        iprb->params[1] = (Unspec) inst;
        SendDeviceServiceRequestToPrimary(iprb,MDC_READ);
        return ( (ResponseCode) iprb->params[0]);
      }
#endif FIREFLY

    req = (IoRequest *) &(pd->msg);
    reply = (IoReply *) &(pd->msg);
    buf = (unsigned short *)&(reply->shortbuffer[0]);

    if( req->bytecount != inst->blocksize ) return( BAD_BYTE_COUNT );

    *buf = MDC_CSR;
    
    return (OK);
    
  }

ResponseCode
MdcWrite( pd, inst )
    register Process *pd;
    register DeviceInstance	*inst;

    /* Write the audioValid bit in the control/status register, for what */
    /*   little that's worth.  Don't let the user monkey with any of the */
    /*   other bits.							 */
  {
#ifdef FIREFLY
    register ProcessorRec *r9;
#endif FIREFLY
    register IoRequest *req;
    unsigned short buf;

#ifdef FIREFLY
    AsmGetProcessorRecord(r9);
    if ( !IsPrimaryProcessor(r9) )
      {
        register IPRBufferType *iprb = &IPRBuffer;
        iprb->params[0] = (Unspec) pd;
        iprb->params[1] = (Unspec) inst;
        SendDeviceServiceRequestToPrimary(iprb,MDC_WRITE);
        return ( (ResponseCode) iprb->params[0]);
      }
#endif FIREFLY

    req = (IoRequest *) &(pd->msg);
    buf = *(unsigned short *)&((IoReply *)req)->shortbuffer[0];

    if( req->bytecount != inst->blocksize ) return( BAD_BYTE_COUNT );

    MDC_CSR = MDC_CSR_VALID |
		((unsigned)commandMemory_qbus_addr >> MDC_CSR_ADDRESS_SHIFT) |
		(buf & MDC_CSR_AUDIO_VALID);
    return (OK);
    
  } /* MdcWrite */

ResponseCode
MdcQuery( pd, inst, dirIndex ) 
    Process *pd;
    DeviceInstance *inst;
    unsigned short dirIndex;
  {
    register QueryDeviceReply *reply = (QueryDeviceReply *) &(pd->msg);

    reply->addr = (unsigned)commandMemory_virt_addr;
    reply->addr2 = (unsigned)commandMemory_qbus_addr;
    reply->length =	      commandMemory_size;

    return( OK );
  }

ResponseCode
MdcRelease( pd, inst )
    Process *pd;
    DeviceInstance *inst;
  {
    inst->owner = 0;
    return( OK );
  }
