/*
 * V Kernel - Copyright (c) 1985 by Stanford University
 *
 * $Revision: 1.4.1.3 $
 * $Locker:  $
 * $State: Exp $
 *
 * Routines for DEC's LK201 keyboard, as used on the QVSS and (I believe)
 *   QDSS, SM and SC.
 *
 * The actual reading and writing of bytes to the keyboard is done in
 *   qvssserial.c
 */

#include <prototypes.h>
#include "console.h"
#include "qvss.h"
#include "externals.h"

/* Exports */
extern void LK201_StarTrek _TAKES((void));

/* Forward */
static void		Bell		_TAKES((void));
static ResponseCode	SetTranslation	_TAKES((int));
static ResponseCode	SetClick	_TAKES((int));
static ResponseCode	GetShift	_TAKES((int*, int*, int*));

static void		NoPutchar	_TAKES((int));

/* Private */
static void	(*KeyboardPutchar) _TAKES((int)) = NoPutchar;
static int	KeyboardTranslation		 = -1;
static int	Lock = 0, Shift = 0, Cntrl = 0;

/*
 * In the absence of useful documentation, what we know about the LK201 has
 *   been garnered from Jay's adb of /vmunix on a microvax running ultrix,
 *   from the Ultrix sources, and from the sources for the X window system
 *   (in X.V11R3, see server/ddx/dec/lk201/*.[ch]).
 *
 * The keyboard does not generate ASCII; it sends 8-bit codes which represent
 *   key transitions, sundry key magic (e.g. autorepeat or "all keys are up
 *   now") and error-codes.  A whole tribe of commands ("turn on LED 3",
 *   "set key-click volume") can be sent to the keyboard.
 *   
 * The keys are divided into fourteen "divisions"; for each one, we can tell
 *   the keyboard to notify us of
 *		- key-down transitions only,
 *	    or	- key-down transitions with auto-repeat,
 *	    or	- key-down and key-up transitions.
 *   See KeyboardDefaults[] below for a list of the divisions.
 *
 * In up/down mode, the keyboard tells us which key changed but not whether
 *   it went up or down, so someone has to keep track in software.  The
 *   keyboard seems to think that telling us "all keys are up now" at
 *   appropriate times will help.
 *
 * Did the hardware people think they were making life easier for us, or
 *   taking a load off the CPU, or just using up ROM on their 8051?
 */


/* Commands to keyboard */

#define LK_DEFAULTS 	0xd3		/* reset (some) default settings */
#define LK_AR_DISABLE	0xe1		/* global auto repeat disable	 */
#define LK_AR_ENABLE 	0xe3		/*   ... and enable		 */
#define LK_KBD_ENABLE 	0x8b		/* keyboard enable		 */
#define LK_CL_DISABLE	0x99		/* keyclick disable (?)		 */
#define LK_CL_ENABLE 	0x1b		/* keyclick  )  Followed by	 */
#define LK_BELL_ENABLE 	0x23		/* the bell  )  volume (0-255??) */
#define LK_RING_BELL 	0xa7		/* ring keyboard bell		 */
#define LK_LED_ENABLE 	0x13		/* light led )  Followed by	 */
#define LK_LED_DISABLE 	0x11		/* darken led)    LED code...    */
#define LED_1 		0x81
#define LED_2 		0x82
#define LED_3 		0x84
#define LED_4 		0x88
#define LED_ALL 	0x8f
#define LK_UPDOWN 	0x86		/* Division (1-14) is left-shifted */
#define LK_AUTODOWN 	0x82		/*   three bits and ORed with the  */
#define LK_DOWN 	0x80		/*   appropriate code.		   */

#define KEYCLICK_VOLUME	0x01	/* ? */

/* Input from the keyboard */

#define LK_KDOWN_ERROR	0x3d		/* key down on powerup		 */
#define LK_POWER_ERROR 	0x3e		/* keyboard failure on powerup	 */
#define LK_OUTPUT_ERROR	0xb5		/* keystrokes lost during inhibit*/
#define LK_INPUT_ERROR 	0xb6		/* garbage command to keyboard	 */

#define LK_KEY_SHIFT	0xae		/* Shift (left or right)	 */
#define LK_KEY_CNTRL	0xaf		/* Control			 */
#define LK_KEY_LOCK	0xb0		/* Caps lock (toggling and Lock  */
					/*   light are done in software) */
#define LK_KEY_ALLUP	0xb3		/* No keys down at present	 */
#define LK_KEY_REPEAT	0xb4		/* Autorepeat last real key	 */

/*
 * Liberated from Ultrix driver -- you might want to retype it all in
 * so that we can't be sued.
 */
char KeyTable[]={
	 0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 /*   0 */
	,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 /*   8 */
	,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00	/*  16 */
	,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 /*  24 */
	,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 /*  32 */
	,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 /*  40 */ 
	,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 /*  48 */
	,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 /*  56 */ 
	,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 /*  64 */
	,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 /*  72 */ 
#ifdef FUNCTIONS
	,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x81 ,0x82 /*  80 */
	,0x83 ,0x84 ,0x85 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 /*  88 */ 
	,0x00 ,0x00 ,0x00 ,0x00 ,0x86 ,0x87 ,0x88 ,0x89 /*  96 */
	,0x8a ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 /* 104 */ 
#else /* Make all first group of function keys into ESCAPE */
	,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x1b ,0x1b /*  80 */
	,0x1b ,0x1b ,0x1b ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 /*  88 */ 
	,0x00 ,0x00 ,0x00 ,0x00 ,0x1b ,0x1b ,0x1b ,0x1b /*  96 */
	,0x1b ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 /* 104 */ 
#endif
	,0x00 ,0x1b ,0x08 ,0x0a ,0x8b ,0x00 ,0x00 ,0x00 /* 112 */
	,0x00 ,0x00 ,0x00 ,0x00 ,0x8c ,0x8d ,0x00 ,0x00 /* 120 */
	,0x8e ,0x8f ,0x90 ,0x91 ,0x00 ,0x00 ,0x00 ,0x00 /* 128 */
	,0x00 ,0x00 ,0x92 ,0x93 ,0x94 ,0x95 ,0x96 ,0x97 /* 136 */
	,0x00 ,0x00 ,0xa0 ,0x00 ,0xa1 ,0xa2 ,0xa3 ,0xa4 /* 144 */
	,0xa5 ,0xa6 ,0xa7 ,0xa8 ,0xa9 ,0xaa ,0xab ,0xac /* 152 */
	,0xad ,0x98 ,0x99 ,0x9a ,0x9b ,0x00 ,0x00 ,0x9c /* 160 */
	,0x9d ,0x9e ,0x9f ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 /* 168 */
	,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 /* 176 */
	,0x00 ,0x00 ,0x00 ,0x00 ,0x7f ,'\r' ,0x09 ,'`'  /* 184 */
	,'1'  ,'q'  ,'a'  ,'z'  ,0x00 ,'2'  ,'w'  ,'s'  /* 192 */
	,'x'  ,'<'  ,0x00 ,'3'  ,'e'  ,'d'  ,'c'  ,0x00 /* 200 */
	,'4'  ,'r'  ,'f'  ,'v'  ,' '  ,0x00 ,'5'  ,'t'  /* 208 */
	,'g'  ,'b'  ,0x00 ,'6'  ,'y'  ,'h'  ,'n'  ,0x00 /* 216 */
	,'7'  ,'u'  ,'j'  ,'m'  ,0x00 ,'8'  ,'i'  ,'k'  /* 224 */
	,','  ,0x00 ,'9'  ,'o'  ,'l'  ,'.'  ,0x00 ,'0'  /* 232 */
	,'p'  ,0x00 ,';'  ,'/'  ,0x00 ,'='  ,']'  ,'\\' /* 240 */
	,0x00 ,'-'  ,'['  ,'\'' ,0x00 ,0x00 ,0x00 ,0x00 /* 248 */
};

char ShiftKeyTable[]={
	 0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 /*   0 */
	,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 /*   8 */
	,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00	/*  16 */
	,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 /*  24 */
	,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 /*  32 */
	,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 /*  40 */ 
	,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 /*  48 */
	,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 /*  56 */ 
	,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 /*  64 */
	,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 /*  72 */ 
	,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x81 ,0x82 /*  80 */
	,0x83 ,0x84 ,0x85 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 /*  88 */ 
	,0x00 ,0x00 ,0x00 ,0x00 ,0x86 ,0x87 ,0x88 ,0x89 /*  96 */
	,0x8a ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 /* 104 */ 
	,0x00 ,0x1b ,0x08 ,0x0a ,0x8b ,0x00 ,0x00 ,0x00 /* 112 */
	,0x00 ,0x00 ,0x00 ,0x00 ,0x8c ,0x8d ,0x00 ,0x00 /* 120 */
	,0x8e ,0x8f ,0x90 ,0x91 ,0x00 ,0x00 ,0x00 ,0x00 /* 128 */
	,0x00 ,0x00 ,0x92 ,0x93 ,0x94 ,0x95 ,0x96 ,0x97 /* 136 */
	,0x00 ,0x00 ,0xa0 ,0x00 ,0xa1 ,0xa2 ,0xa3 ,0xa4 /* 144 */
	,0xa5 ,0xa6 ,0xa7 ,0xa8 ,0xa9 ,0xaa ,0xab ,0xac /* 152 */
	,0xad ,0x98 ,0x99 ,0x9a ,0x9b ,0x00 ,0x00 ,0x9c /* 160 */
	,0x9d ,0x9e ,0x9f ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 /* 168 */
	,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 /* 176 */
	,0x00 ,0x00 ,0x00 ,0x00 ,0x7f ,'\r' ,0x09 ,'~'  /* 184 */
	,'!'  ,'Q'  ,'A'  ,'Z'  ,0x00 ,'@'  ,'W'  ,'S'  /* 192 */
	,'X'  ,'>'  ,0x00 ,'#'  ,'E'  ,'D'  ,'C'  ,0x00 /* 200 */
	,'$'  ,'R'  ,'F'  ,'V'  ,' '  ,0x00 ,'%'  ,'T'  /* 208 */
	,'G'  ,'B'  ,0x00 ,'^'  ,'Y'  ,'H'  ,'N'  ,0x00 /* 216 */
	,'&'  ,'U'  ,'J'  ,'M'  ,0x00 ,'*'  ,'I'  ,'K'  /* 224 */
	,'<'  ,0x00 ,'('  ,'O'  ,'L'  ,'>'  ,0x00 ,')'  /* 232 */
	,'P'  ,0x00 ,':'  ,'?'  ,0x00 ,'+'  ,'}'  ,'|'  /* 240 */
	,0x00 ,'_'  ,'{'  ,'"'  ,0x00 ,0x00 ,0x00 ,0x00 /* 248 */
};

/*
 * LK201 special purpose keys. Translations from the tables above
 * includes codes for the function keys and other goodies. They can
 * be determined by the presence of the 8th bit being set.  The 
 * following table is accessed by removing that bit and using the 
 * result as the index to the following table. Note that table begins
 * with a null entry.
 */
char *SpecialKeyTable[] = { 0,
	"\33[11~",			/* f1 */
	"\33[12~",			/* f2 */
	"\33[13~",			/* f3 */
	"\33[14~",			/* f4 */
	"\33[15~",			/* f5 */
	"\33[17~",			/* f6 */
	"\33[18~",			/* f7 */
	"\33[19~",			/* f8 */
	"\33[20~",			/* f9 */
	"\33[21~",			/* f10 */
	"\33[26~",			/* f14 */
	"\33[28~",			/* f15 */
	"\33[29~",			/* f16 */
	"\33[31~",			/* f17 */
	"\33[32~",			/* f18 */
	"\33[33~",			/* f19 */
	"\33[34~",			/* f20 */
	"\33[1~",			/* find */
	"\33[2~",			/* insert */
	"\33[3~",			/* remove */
	"\33[4~",			/* select */
	"\33[5~",			/* prev */
	"\33[6~",			/* next */
	"\33OP",			/* pf1 */
	"\33OQ",			/* pf2 */
	"\33OR",			/* pf3 */
	"\33OS",			/* pf4 */
	"\33[D",			/* left */
	"\33[C",			/* right */
	"\33[B",			/* down */
	"\33[A",			/* up */
	"\33Op",			/* key pad 0 */
	"\33On",			/* key pad . */
	"\33OM",			/* key pad enter */
	"\33Oq",			/* key pad 1 */
	"\33Or",			/* key pad 2 */
	"\33Os",			/* key pad 3 */
	"\33Ot",			/* key pad 4 */
	"\33Ou",			/* key pad 5 */
	"\33Ov",			/* key pad 6 */
	"\33O/*",			/* key pad , */
	"\33Ow",			/* key pad 7 */
	"\33Ox",			/* key pad 8 */
	"\33Oy",			/* key pad 9 */
	"\33Om",			/* key pad - */
	/*
	 * The following strings are to allow a numeric keypad
	 * mode and still use the same translation tables
	 */
	"0",
	".",
	"\r",
	"1",
	"2",
	"3",
	"4",
	"5",
	"6",
	",",
	"7",
	"8",
	"9",
	"-"
};

/*
 * Key-transition specifications for the keyboard:
 *     - "Ultrx" is used when we're doing ASCII encoding and attempting to
 *	 be compatible with what the Ultrix kernel does,
 *     - "UpDnK" is used when we're providing unencoded up/down information
 *	 (normally for a window system),
 *     - "AutoK" is there in case the X11 server wants it; we may not need it.
 * The key-code ranges and key-names for each keyboard division are listed
 * below; "normal" means all the printable keys, including Space, in the main
 * part of the keyboard.
 */

#define Auto LK_AUTODOWN
#define Down LK_DOWN
#define UpDn LK_UPDOWN

typedef unsigned char KeyModes[14];

static KeyModes
Ultrx={Auto,Auto,Auto,Down,UpDn,UpDn,Auto,Auto,Auto,Auto,Auto,Auto,Down,Auto},
#ifdef undef
AutoK={Auto,Auto,Auto,UpDn,UpDn,UpDn,Auto,Auto,UpDn,UpDn,UpDn,UpDn,UpDn,UpDn},
#endif undef
UpDnK={UpDn,UpDn,UpDn,UpDn,UpDn,UpDn,UpDn,UpDn,UpDn,UpDn,UpDn,UpDn,UpDn,UpDn};
/*        |    |    |    |    |    |    |    |    |    |    |    |    |    |
191-255:normal |    |    |    |    |    |    |    |    |    |    |    |    |
145-165:numeric pad |    |    |    |    |    |    |    |    |    |    |    |
    188:         Delete  |    |    |    |    |    |    |    |    |    |    |
189-190:           Return,Tab |    |    |    |    |    |    |    |    |    |
176-178:              Lock,Compose |    |    |    |    |    |    |    |    |
173-175:                     Shift,Ctrl |    |    |    |    |    |    |    |
166-168:                    Left/Right Arrow |    |    |    |    |    |    |
169-172:                            Down/Up Arrow |    |    |    |    |    |
136-144:           Find,Insert,Remove,Select,Prev,Next |    |    |    |    |
 86- 98:                                             F1-F5  |    |    |    |
 99-110:                                                  F6-F10 |    |    |
111-122:                                                      F11-F14 |    |
123-125:                                                           Help,Do |
126-135:                                                               F17-F20
 */

#undef Auto
#undef Down
#undef UpDn

static short	KeyboardInitList[] = {
	    LK_KBD_ENABLE,	/* keyboard itself */
	    LK_BELL_ENABLE,	/* Bell */
	    0x84,		/* Bell volume! */
	    LK_LED_DISABLE,	/* keyboard LEDs */
	    LED_ALL
	};
#define LKINITLENGTH (sizeof(KeyboardInitList)/sizeof(KeyboardInitList[0]))


static void 
SetKeyModes(modes, autorepeat)
    KeyModes modes;
    int autorepeat;
  {
    register int i;
    for (i=1; i<=14; i++)
      {
	(*KeyboardPutchar)( modes[i-1] | (i<<3) );
      }
    /*
     * Dunno whether we really have to globally disable autorepeat when we've
     *   already disabled it for all 14 divisions, but Ultrix and X seem to.
     */
    (*KeyboardPutchar)(autorepeat ? LK_AR_ENABLE : LK_AR_DISABLE);
  }

void
LK201_Init(putchar_proc)
    register void (*putchar_proc)_TAKES((int));
  {
    register int i;
    
    KeyboardPutchar = putchar_proc;

    (*putchar_proc)( LK_DEFAULTS );

    SetTranslation(TR_ASCII);
    SetClick(0);

    for (i = 0; i < LKINITLENGTH; i++)
	(*putchar_proc)( KeyboardInitList[i] );

    ConsoleBellRoutine		= Bell;
    ConsoleKeyboardGetShift	= GetShift;
    ConsoleKeyboardSetClick	= SetClick;
    ConsoleKeyboardSetTrans	= SetTranslation;
  }

static void
Bell _TAKES((void))
  {
    (*KeyboardPutchar)( LK_RING_BELL );
  }

static ResponseCode 
GetShift(shift, ctrl, caps_lock)
    int *shift, *ctrl, *caps_lock;
  {
    if (shift != 0)
	*shift = (Shift != 0);
    if (ctrl  != 0)
	*ctrl  = (Cntrl != 0);
    if (caps_lock != 0)
	*caps_lock = (Lock != 0);
    return OK;
  }

static 
ResponseCode SetClick(click_percent)
    int click_percent;
  {
    if (click_percent <= 0)
      {
	(*KeyboardPutchar)(LK_CL_DISABLE);
      }
    else
      {
	(*KeyboardPutchar)(LK_CL_ENABLE);
	(*KeyboardPutchar)(7 - ((click_percent/14) & 7));
			/* ^ formula comes from X11 sources, hope */
			/*   *they* know what they're doing.      */
      }
    return OK;
  }

static ResponseCode
SetTranslation(keyboard_tr)
    int keyboard_tr;
  {
    switch(keyboard_tr)
      {
	case TR_ASCII:
	    SetKeyModes(Ultrx, 1);
	    break;
	case TR_NONE:
	    SetKeyModes(UpDnK, 0);
	    break;
	default:
	    return BAD_ARGS;
      }
    KeyboardTranslation = keyboard_tr;
    return OK;
  }


void 
LK201_GotKey(key)
    int key;
  {
    register unsigned int c;
    register unsigned char *string;
    int special_key;

    static short Lastchar = 0;

    /*
     * Process key.  If we're in mode TR_NONE, then we basically just hand it
     *   back to the client, whereas in TR_ASCII we map it appropriately.  In
     *   both modes, though, we keep track of Shift, Ctrl and CapsLock and
     *   manage the CapsLock LED (we assume that any KeyModes we're using
     *   specify LK_UPDOWN for these keys).
     */

    special_key = 1;

    switch (key)
      {
	case LK_KEY_LOCK:
	    Lock = !Lock;
	    if (Lock)
		(*KeyboardPutchar)( LK_LED_ENABLE );
	    else
		(*KeyboardPutchar)( LK_LED_DISABLE );
	    (*KeyboardPutchar)( LED_3 );
	    break;
	case LK_KEY_SHIFT:
	    Shift = !Shift;
	    break;
	case LK_KEY_CNTRL:
	    Cntrl = !Cntrl;
	    break;
	case LK_KEY_ALLUP:
	    Cntrl = Shift = 0;
	    break;
	default:
	    special_key = 0;
	    break;
      }

    string = 0;

    switch (KeyboardTranslation)
      {
	case TR_NONE:
	    c = key;
	    Lastchar = 0;
	    break;
	case TR_ASCII:
	    if (special_key)	/* Shift/Ctrl/Lock */
		return;
	    if (key == LK_KEY_REPEAT)
	      {
		c = Lastchar;
	      }
	    else
	      {
		c = Shift ? ShiftKeyTable[key]:KeyTable[key];
		if (Cntrl && c >= ' ' && c <= '~' )
	            c &= 0x1f;
		if (Lock && c >= 'a' && c <= 'z')	/* Caps lock */
		    c -= 'a'-'A';
		Lastchar = c;
	      }
	    if (c & 0x80)
	      {
		string = (unsigned char *) SpecialKeyTable[c&0x7f];
		if (string == 0)
		    return;
		c = *string;
	      }
	    break;
	default:
	    Kabort("Keyboard software confused");
	    break;
      }

    ConsoleGotChar(c);
    if (string != 0)
	while (*++string)
	    ConsoleGotChar(*string);
  }

static void 
NoPutchar(ch)
    int ch;
  {
    /* Somebody didn't call LK201_Init() before trying to use	*/
    /*   (*KeyboardPutchar)().  What can we do but hang?	*/
    while (1)
	ch;	/* Keeps lint happy? */
  }

/*
 * Sequence back and forth across the four LEDs on the keyboard; it's the
 *   kernel's way of saying "I'm not dead yet".  Called 60 times/second by
 *   IntQvssMousePoll() (well, at present the call is commented out).
 */
void 
LK201_StarTrek _TAKES((void))
  {
    static int count = 0;

    count++;
    count = count % 120;
    
    if (count % 20 == 0)
      {
	if (count == 0)
	  {
	    (*KeyboardPutchar)( LK_LED_ENABLE );
	    (*KeyboardPutchar)( LED_1 );
	    (*KeyboardPutchar)( LK_LED_DISABLE );
	    (*KeyboardPutchar)( LED_2 );
	  }
	else if (count == 20)
	  {
	    (*KeyboardPutchar)( LK_LED_ENABLE );
	    (*KeyboardPutchar)( LED_2 );
	    (*KeyboardPutchar)( LK_LED_DISABLE );
	    (*KeyboardPutchar)( LED_1 );
	  }
	else if (count == 40)
	  {
	    (*KeyboardPutchar)( LK_LED_ENABLE );
	    (*KeyboardPutchar)( LED_3 );
	    (*KeyboardPutchar)( LK_LED_DISABLE );
	    (*KeyboardPutchar)( LED_2 );
	  }
	else if (count == 60)
	  {
	    (*KeyboardPutchar)( LK_LED_ENABLE );
	    (*KeyboardPutchar)( LED_4 );
	    (*KeyboardPutchar)( LK_LED_DISABLE );
	    (*KeyboardPutchar)( LED_3 );
	  }
	else if (count == 80)
	  {
	    (*KeyboardPutchar)( LK_LED_ENABLE );
	    (*KeyboardPutchar)( LED_3 );
	    (*KeyboardPutchar)( LK_LED_DISABLE );
	    (*KeyboardPutchar)( LED_4 );
	  }
	else if (count == 100)
	  {
	    (*KeyboardPutchar)( LK_LED_ENABLE );
	    (*KeyboardPutchar)( LED_2 );
	    (*KeyboardPutchar)( LK_LED_DISABLE );
	    (*KeyboardPutchar)( LED_3 );
	  }
      }
  }
