/*
 * Distributed V Kernel 
 * Copyright (c) 1982 by Stanford University.
 *
 * Kernel Ethernet driver
 * Deqna Ethernet Controller ("qna")
 *
 * $Revision: 1.55.1.14 $
 * $Locker:  $
 * $State: Exp $
 *
 * Reference:
 *   [UG] EK-DEQNA-UG-001, DEQNA Ethernet User's Guide, 1st ed., August 1984.
 *
 * Byte-swapping:
 *   We send and receive [x]V packets in big-endian order, even between
 *   pairs of little-endian machines.  V process-IDs also have a flag
 *   which says whether the process lives on a little-endian host.
 *
 *   To deal with all these, we use:
 *     #ifdef LITTLE_ENDIAN for swapping packets and for setting the flag
 *         in process-IDs.
 *     #ifdef LITTLE_ENDIAN also for putting ethernet addresses in big-endian
 *	   order before building setup packets for the DEQNA.
 *   LITTLE_ENDIAN is (had better be) defined in the standard V buildfiles
 *   for anything that runs on a MicroVAX.
 *
 * Addressing - where the DEQNA data lives
 *   The addresses which we write into the DEQNA's Tx/Rx BDL registers, and
 *   the buffer/chain addresses which we put in the buffer descriptor lists,
 *   are really Q-bus addresses, not processor (physical or virtual) addresses.
 *   The functions or macros virt_to_QbusLO() and virt_to_QbusHI should perform
 *   any necessary mapping from a processor address to the low 16 and high 6
 *   bits, respectively, of the Q-bus address.
 */
#include <Venviron.h>
#include <Vnetwork.h>
#include <ether.h>
#include <Vmtp.h>
#include <Vquerykernel.h>
#include "deqnalocal.h"
#include "deqna.h"
#include "interrupt.h"
#include "devman.h"
#include "ipc.h"
#include "externals.h"
#include <Vinstrument.h>

#ifdef DEBUG
#define debug(c) KernelPutChar(c)
#else
#define debug(c)
#endif

#define ENET_MAX_PACKET ( ETHER_HEADER_LENGTH + ETHER_MAX_DATA_LENGTH )

typedef Process_id (*PFPID)();

/* Assembly language Forward */
_BEGIN_EXTERN_C
void EnetInterrupt _TAKES(());
void Asm_EnetInterrupt _TAKES(());
_END_EXTERN_C

/* Forward */
void EnetReset _TAKES(( int ));
void XmitBuffer _TAKES((unsigned, int, int));
void Debug_Deqna _TAKES((unsigned short));
void GoToNextBD _TAKES(());
void Debug_BDL _TAKES(( int, unsigned short *, char *));
void ReadIntoProcess _TAKES(( Process *, DeviceInstance *, int, char *));
void EnetReadPacket _TAKES(());
ResponseCode AddProcessGroupToLogicalHostGroupFilterDeqna  _TAKES(());
ResponseCode DeleteProcessGroupFromLogicalHostGroupFilterDeqna  _TAKES(());

/* Used internally */
extern int	LastPacketWasMulticast;
EtherAddress	EnetHostNumber;		/* physical ethernet address */
int		EnetReceiveMask = NET_RCV_DEFAULT;
unsigned short	EnetCollisions = 0;	/* Number of collision errors */
unsigned short	EnetOverflows = 0;	/* Queue overflow errors */
unsigned short	EnetCRCErrors = 0;	/* Packets with bad CRC's */
unsigned short	EnetSyncErrors = 0;	/* Receiver out of sync */
unsigned short	EnetTimeouts = 0;	/* Transmitter timeouts */
int		EnetValidPackets = 0;

/* 
 * NetAlarm (defined in ../mi/timeman.c) is  set to nonzero when we receive an
 * unclaimed packet.  If it times out, the kernel flushes the packet using
 * NetCheck.
 */

unsigned short  EnetPromiscuous = 0;	/* Are we in promiscuous mode. */
EtherPacket 	*EnetPacket;	/* Points to current incoming ethernet pkt. */

Qna_t		Buffers;	/* The Enet buffer structure */
int		RcvIndex;	/* The receive buffer we are currently using */
int		LastValidBuffer;/* The last valid buffer in the list */


EtherAddress	VMulticastAddr = ETHER_V_MULTICAST_MIN;
char		VMultiCast[6];
char		Broadcast[] = {0xff, 0xff, 0xff, 0xff, 0xff, 0xff};


void
BuildSysMap_Deqna()
  {
    /*
     * Could be used to give us a slice of virtual address space if we wanted
     *   if for packet buffers.  At present we just declare an array in the
     *   kernel bss area.
     */
  }

/* Macro expansion to interrupt-invoked C call to Ethernetinterrupt */
Call_inthandler(EnetInterrupt);


void
EnetPowerupDeqna()
  /* Powerup and initialize the Ethernet Interface Board.
   *  Done just once, from kernel Powerup() routine. 
   */
  {
    /* Indicate we have a deqna board */
    AddPeripheral(PRF_ENET_DEQNA);
    IkpOutq.tail = (Process *) &(IkpOutq.head);
    IkpOutq.type = IKP_OUT_QUEUE;

    /* get my host address */

#define qna_adr(x) (*Deqna_enet_addr(x))

    EnetHostNumber.high = ((qna_adr(0) & 0xFF)<<8) | (qna_adr(1) & 0xFF);
    EnetHostNumber.mid  = ((qna_adr(2) & 0xFF)<<8) | (qna_adr(3) & 0xFF);
    EnetHostNumber.low  = ((qna_adr(4) & 0xFF)<<8) | (qna_adr(5) & 0xFF);

    printx("Ethernet Address is %x%x%x%x.%x%x%x%x.%x%x%x%x\n",
    	0xf&(qna_adr(0)>>4), qna_adr(0)&0xf,
    	0xf&(qna_adr(1)>>4), qna_adr(1)&0xf,
    	0xf&(qna_adr(2)>>4), qna_adr(2)&0xf,
    	0xf&(qna_adr(3)>>4), qna_adr(3)&0xf,
    	0xf&(qna_adr(4)>>4), qna_adr(4)&0xf,
    	0xf&(qna_adr(5)>>4), qna_adr(5)&0xf);

    /* initialize VMulticast */
#ifdef LITTLE_ENDIAN
    ByteSwapShortCopy(&MulticastDefault, VMultiCast, sizeof(VMulticastAddr));
#else
    Copy(VMultiCast, &MulticastDefault, sizeof(VMulticastAddr));
#endif
/*    srandom((EnetHostNumber.mid<<16)+EnetHostNumber.low); */
    
    /* Plug interrupt location and deqna's interrupt vector */
    *Deqna_vector = VecDeqna << 2;
    setexvec(Asm_EnetInterrupt, VecDeqna);

    EnetReset( NET_RCV_DEFAULT );

    AddDevice("enet10",  0, EnetCreate, 0, EnetQuery);
    AddDevice("ethernet",0, EnetCreate, 0, EnetQuery);
  }


void
EnetReset( receiveMask )
    int receiveMask;
  {
#ifdef FIREFLY
    register ProcessorRec *r11;
#endif FIREFLY
    register int i, j, time;
    register QNABD_t *bdl;
    short	rcvMode;
    int numTimesRetry;
    char *bug, *lastbug;
    int save_csr;
    static int numGoodResets = 0;

    EnetReceiveMask = receiveMask;
#ifdef FIREFLY
    /*
     * There is a bug in the compiler which causes it to generate erroneous
     * code if a function  has a static variable and the first instruction
     * is an asm.  Thus make sure that there is some code before the 
     * AsmGetProcessorRecord().
     */
    AsmGetProcessorRecord(r11);
    if ( !IsPrimaryProcessor(r11) )
      {
        register IPRBufferType *iprb = &IPRBuffer;
        iprb->params[0] = (Unspec) receiveMask;
        SendDeviceServiceRequestToPrimary(iprb,ENET_RESET);
        return;
      }
#endif FIREFLY
    /* We need to convert receiveMask into a rcvMode the deqna understands. */
    rcvMode = (receiveMask & NET_RCV_BROADCAST ? QNA_AllMultiCast : 0) |
    		(receiveMask & NET_RCV_PROMISCUOUS ? QNA_Promiscuous : 0);
    rcvMode += sizeof(Buffers.setupbuff);

#ifdef DEBUG
    printx("EnetReset(%d); rcvMode = 0x%x...\n", receiveMask, rcvMode);
#endif DEBUG

    /* Main retry loop.  If all goes well we'll only execute this once. */
    numTimesRetry = 0;
    lastbug = NULL;
    while (1)
      {    
	/*
	 * Assert and hold the RESET bit in the DEQNA; we clear it when we're
	 *   ready, after we've initialized lots of data.  Dunno whether we
	 *   really need to hold the DEQNA reset all this time, but it seems
	 *   like a reasonable idea.
	 */
	*Deqna_csr |= QNARESET;

	/* initialize the setup packet */
	for (i = 0; i < 16; i++)
            for (j = 0; j < 8; j++)
		Buffers.setupbuff[i][j] = 0;
	for (i = 0; i < 6; i++)
	  {
            if (receiveMask & NET_RCV_SELF)
	      {
		Buffers.setupbuff[i][1] = *Deqna_enet_addr(i) & 0xFF;
	      }
	    /* is this even needed? */
	    if (receiveMask & NET_RCV_BROADCAST)
	      {
		Buffers.setupbuff[i][2] = Broadcast[i];
		Buffers.setupbuff[i][3] = VMultiCast[i];
	      }
	  }

	/* reset and flush the buffers */
	UserPacketLength = 0;
	RcvIndex = 0;
	LastValidBuffer = RXLISTLEN-2; /* RXLISTLEN-1 is marked invalid,    */
				       /* RXLISTLEN is the chain descriptor */
    
	/* Initialize descriptors 0..TXLISTLEN */
	for (bdl= &Buffers.txlist[0]; bdl<= &Buffers.txlist[TXLISTLEN]; bdl++)
	  {
	    bdl->bd1	= QNA_BDLUnused;
	    bdl->bd2	= 0; /* sets VALID bit = 0 */
	    bdl->addrlo	= 0;
	    bdl->size	= 0;
	    bdl->status1= QNA_BDLUnused;
	    bdl->status2= 0;
	  }
    
	   /* Initialize descriptors 0..RXLISTLEN-1 */
	for (i = 0, bdl = &Buffers.rxlist[0]; i < RXLISTLEN; i++, bdl++)
	  {
            bdl->bd1	= QNA_BDLUnused;
	    bdl->bd2	= (i==RXLISTLEN-1 ? 0 : QNA_BD2_VALID) | 
			    ( virt_to_QbusHI(&Buffers.rxbuff[i])
				<< QNA_BD2_ADDRHIpos & QNA_BD2_ADDRHImask );
	    bdl->addrlo	= virt_to_QbusLO(&Buffers.rxbuff[i]);
	    bdl->size	= -( (sizeof(Buffers.rxbuff[i])+1) / sizeof(short) );
	    bdl->status1= QNA_BDLUnused;
	    bdl->status2= 0;
	  }
      
	/* set up last BD */
	bdl->bd1    = QNA_BDLUnused;
	bdl->bd2    = QNA_BD2_VALID | QNA_BD2_CHAIN |
			   ( virt_to_QbusHI(Buffers.rxlist) 
			      << QNA_BD2_ADDRHIpos & QNA_BD2_ADDRHImask );
	bdl->addrlo = virt_to_QbusLO(Buffers.rxlist);
	bdl->size   = 0;/* Size and status fields aren't used if chain is   */
			/* true; we initialize them to make debugging easier*/
	bdl->status1= QNA_BDLUnused;
	bdl->status2= 0;
    
	/*
	 * Take the DEQNA out of the RESET state.  Most other DEQNA
	 *   drivers delay for quite a while before clearing RESET, then read
	 *   from the DEQNA's boot ROM (and ignore what they read).  Wish I
	 *   knew whether we really have to do those things.  In the meantime,
	 *   we don't bother and just hope things work.
	 */
	*Deqna_csr &= ~QNARESET;
	i = *Deqna_csr;
	i |= QNARCVINT | QNAILOOP | QNAXMTINT;
	i &= ~(QNARCVENABLE | QNAINTENABLE | QNATIMERENABLE | QNAELOOP);
	*Deqna_csr = i;
	if ( (*Deqna_csr & QNARXBDLBAD) == 0 )
	  {
	    save_csr = *Deqna_csr;
	    bug = "Receiver BDL should be invalid, isn't";
	    goto reset_bug;
	  }
	*Deqna_rxBDLlo = virt_to_QbusLO(Buffers.rxlist);
	*Deqna_rxBDLhi = virt_to_QbusHI(Buffers.rxlist);

	/* send a setup packet (receiver is disabled, so we know that the
	*  packet we are receiving is the looped back setup packet.)
	*/
	XmitBuffer((unsigned)rcvMode, (int)Buffers.setupbuff, 1);
    
	/* wait for it to be looped back */
	time = 0;
	/*
	 * Wait on the Tx, not Rx, buffer because Tx status is written after
	 * Rx status for a setup packet. [UG] section 1.3.4 (LoopBack), p. 1-9
	 */
	while ((Buffers.txlist[0].status1 & QNA_TX1_USEmask)==QNA_BDLUnused &&
		time <= QNAWAITTIME)
	    time++;
	if (time > QNAWAITTIME)
	  {
	    /* we timed out waiting for a looped back packet, Kabort */
	    save_csr = *Deqna_csr;
	    bug = "Timed out waiting for setup packet to loop back";
	    goto reset_bug;
	  }
	if ((Buffers.txlist[0].status1 & QNA_TX1_USEmask) != QNA_BDLLastNoErr)
	  {
	    save_csr = *Deqna_csr;
	    bug = "Got bad transmit status when looping back a setup packet";
	    goto reset_bug;
	  }

	/* Did all the setup stuff successfully; get out of the retry loop */
	break;

reset_bug:

	/* numTimesRetry, bug, save_csr and lastbug should all be defined */
#ifndef VERBOSE
	/*
	 * If we just get one or two DEQNA problems and then everything
	 *   works again, don't bother telling anybody
	 */
    if (numTimesRetry >= 2)
      {
#endif  VERBOSE
	if (lastbug == NULL)
	  {
	    printx(
	    "Previously did %d (decimal) consecutive successful EnetResets\n",
	            numGoodResets);
	    numGoodResets = 0;
	  }
	if (bug != lastbug ||
	    (numTimesRetry & (-numTimesRetry)) == numTimesRetry )
	    /* ^ i.e. it's a power or 2 (or zero), so print something just */
	    /*   to let the user know what's going on.  We use powers of 2 */
	    /*   so the messages become less frequent if the bug persists  */
	  {
	    printx("DEQNA EnetReset: %s\n  (have already done %d retries)\n",
			bug, numTimesRetry);
	  }
	if (numTimesRetry > 1000000)
			    /* ^ Very arbitrary limit */
	  {
	    Debug_Deqna(save_csr);
	    printx("DEQNA EnetReset: %s\n  (have done %d retries)\n",
			bug, numTimesRetry);
	    Kabort("DEQNA EnetReset: Couldn't reset DEQNA despite retrying");
	  }
	lastbug = bug;
#ifndef VERBOSE
      }	/* Finishes the conditionally-compiled if-statement above */
#endif  VERBOSE
	numTimesRetry++;
      } /* End of the retry loop */

    if (numTimesRetry >
#ifdef VERBOSE
		0
#else  VERBOSE
		2
#endif VERBOSE
		 )
      {
	printx("DEQNA EnetReset - finally succeeded after %d retries\n",
		numTimesRetry);
      }

    numGoodResets++;
    Buffers.rxlist[RcvIndex].bd1	= QNA_BDLUnused;
    Buffers.rxlist[RcvIndex].status1	= QNA_BDLUnused;
    GoToNextBD();
    i = *Deqna_csr;
    i |= QNAINTENABLE | QNARCVINT | QNAILOOP | QNAXMTINT | QNARCVENABLE;
    i &= ~(QNATIMERENABLE | QNAELOOP);
    *Deqna_csr = i;
    
    EnetWriter = NO_PROCESS;
    NetAlarm = ENET_READER_TIMEOUT;
  }


void
XmitBuffer( msize, buff, setup )
  /*
   * Loads the Transmit and sends the packet
   * descriptor according to the given information
   */
    unsigned msize;	/* the length of the message */
    int buff;		/* the buffer containing the message */
    int setup;		/* 1 if this is a setup packet */
  {
    ProcessId saveEnetWriter;
    /*
     * We write more than is really necessary.  Slows things down a bit, but it
     *   it helps when debugging if we set everything possible to 0, so we can
     *   see what the deqna has changed since we wrote it.
     */
    
    Buffers.txlist[0].bd1	= QNA_BDLUnused;
    Buffers.txlist[0].bd2	= QNA_BD2_VALID | QNA_BD2_ENDOFMSG |
		(setup ? QNA_BD2_SETUP : 0) |
		((msize & 1) ? QNA_BD2_LBOEND : 0) |
		(virt_to_QbusHI(buff) 
		   << QNA_BD2_ADDRHIpos & QNA_BD2_ADDRHImask);
	/*   chain and HBOStart are zero, as required.  */
    Buffers.txlist[0].addrlo	= virt_to_QbusLO(buff);
    Buffers.txlist[0].size	= -( (msize + 1) / sizeof(short) );
    Buffers.txlist[0].status1	= QNA_BDLUnused;
    Buffers.txlist[0].status2	= 0; /* Unnecessary but nice for debugging */
#ifdef LITTLE_ENDIAN
    if (!setup)
        ByteSwapShortCopy((char *)buff, (char *)buff, ETHER_HEADER_LENGTH);
#endif LITTLE_ENDIAN
    saveEnetWriter = EnetWriter;
    if ( EnetWriter == NO_PROCESS && !setup )
      {
	/*
	 * Debugging code only:
	 *
	 * If EnetWriter == 0 and this is a real packet that we're trying to
	 *   send, then someone screwed up the management of EnetWriter.
	 *   May provoke real problems (like the BDL Kabort below)
	 */
#ifdef VERBOSE
	printx("XmitBuffer: EnetWriter == 0 (Please tell Thomas)\n");
#endif VERBOSE
	EnetWriter = 1; /* probably too late, but... */
      }
    if ( (*Deqna_csr & QNATXBDLBAD) == 0 )
      {
	int i;

	for (i=0; i<1000000 && (*Deqna_csr & QNATXBDLBAD) == 0; i++)
	    ;    /* ^ arbitrary.  Worth retrying for a long time */
	if ( (*Deqna_csr & QNATXBDLBAD) == 0)
	  {
#ifdef undef
	    Debug_Deqna(*Deqna_csr);
#endif undef
	    if (saveEnetWriter == 0)
		printx("XmitBuffer: EnetWriter was 0 (bad)\n");
#ifdef undef
	    Kabort("XmitBuffer: Transmitter BDL should be invalid, isn't\n",
saveEnetWriter	/* Kabort just takes one argument; we include this here */
		/*   only so that it will appear in stack dumps.        */
		);
#else  undef
	    printx(
"DEQNA BDL silliness again: Joe doesn't like Kaborts, so we'll keep going...\n"
		   );
#endif undef
	  }
#ifdef VERBOSE
	else
	    printx("XmitBuffer: Tx BDL got it right after %d tries\n", i);
#endif VERBOSE
      }
    
    *Deqna_txBDLlo = virt_to_QbusLO(Buffers.txlist);
    *Deqna_txBDLhi = virt_to_QbusHI(Buffers.txlist);
  }
    

void
EnetInterrupt()
   /* Handle an interrupt from the ethernet interface.
    */
  {
    register Process *pd;
    register QNABD_t *bdl;
    register int csr;
 
#ifdef VAX
    /* Bring our IPL down to a reasonable value (allow timer interrupts). */
    /*   Q-Bus devices on the MicroVAX II arbitrate at IPLs from 14 to 17,*/
    /*   but the IPL is always set to 17 when a Q-Bus interrupt is taken. */
    asm("	mtpr	$0x15, $ipl");
#endif VAX
    
    /*
     * Determine type of interrupt.
     * We acknowledge the interrupts now, to prevent problems if a packet
     * comes in while we are processing.
     */
    csr = *Deqna_csr;
    *Deqna_csr = csr | QNARCVINT | QNAXMTINT;
    if ( csr & QNANXMINT )
      {
	Debug_Deqna(csr);
        Kabort("Deqna accessed nonexistent memory...");
      }
    if ((Buffers.txlist[0].status1 & QNA_TX1_USEmask) != QNA_BDLUnused
         || (csr & QNAXMTINT) )
      {
debug('x');
        /* check for errors on last transmit */
	bdl = &Buffers.txlist[0];
	EnetCollisions += (bdl->status1 & QNA_TX1_COUNTmask)
				>> QNA_TX1_COUNTpos;
	if ((bdl->status1 & QNA_TX1_USEmask) != QNA_BDLLastNoErr)
	  {
	    /* we had a transmit error */
	    if (bdl->status1 & QNA_TX1_ABORT)
	        EnetCollisions += 16;
	    if (bdl->status1 & (QNA_TX1_LOSS | QNA_TX1_NOCAR))
	      {
	        printx("Bad packet transmit due to malfunctioning hardware\n");
	      }
	  }

	/* If IkpOutq is non-empty, transmit next interkernel packet. */
	Lockq( &IkpOutq );
	if( (pd=IkpOutq.head) != NULL )
	  {
	    if( (IkpOutq.head = pd->link) == NULL )
		IkpOutq.tail = (Process *) &(IkpOutq.head);
	    pd->queuePtr = NULL;
	    Unlockq( &IkpOutq );
	    WriteNextKPacket( pd );
	  }
	else
	  {
	    Unlockq( &IkpOutq );
	    EnetWriter = 0; /* No transmission in progress now */
	  }
        
	/* Mark this buffer as unused */
/*
 * Doesn't this stomp on the buffer descriptor that WriteNextKPacket has
 *   just set up, and the DEQNA is probably using?
 */
	bdl->status1 = QNA_BDLUnused;
	bdl->status2 = 0;		/* Unnecessary, nice for debug*/
	bdl->bd1     = QNA_BDLUnused;
      }
    if ( (csr & QNARCVINT) || 
	(Buffers.rxlist[RcvIndex].status1 & QNA_RX1_USEmask) != QNA_BDLUnused )
      {
debug('r');
	EnetReadPacket();
      }
    if (EnetWriter == 0) NetAlarm = ENET_READER_TIMEOUT;
  }


void 
EnetReadPacket()
   /*
    *  Physically transfer a packet from the Ethernet
    *  Interface Board into the reading process's buffer.
    */
  {
    Process		*pd;
    register DeviceInstance *inst;
    register VmtpPacket	*kp;
    register unsigned	bytes;
    QNABD_t		*bdl;
    extern int		DeliverToEnetReader;

    /* Find a valid error-free packet */
    bdl = &Buffers.rxlist[RcvIndex];
    while ((bdl->status1 & QNA_RX1_USEmask) != QNA_BDLUnused)
      {
	if ((bdl->status1 & QNA_RX1_USEmask) == QNA_BDLLastNoErr)
	  {
	    /* A valid packet */
	    EnetPacket = &Buffers.rxbuff[RcvIndex];
#ifdef LITTLE_ENDIAN
	    ByteSwapShortCopy((char *)EnetPacket, (char *)EnetPacket,
			      ETHER_HEADER_LENGTH);
#endif LITTLE_ENDIAN
	    bytes  = (bdl->status2 & QNA_RX2_RBLLOmask)
			>> QNA_RX2_RBLLOpos;
	    bytes |= (bdl->status1 & QNA_RX1_RBLHImask)
			>> QNA_RX1_RBLHIpos << 8;
	    /* But QNA_RX1_RBLHIpos = 8; wonder whether the */
	    /* compiler is smart enough to optimize this?   */
	    bytes += 60;	/* The deqna gives you the length - 60 */

	    DeliverToEnetReader = 0;
	    kp = (VmtpPacket *) ((unsigned long)EnetPacket + ETHER_HEADER_LENGTH);
#ifdef LITTLE_ENDIAN
	    if (EnetPacket->packettype == ETHER_TYPE_VMTP )
	      {
		SwapVmtpPacket(kp);
	      }
#endif LITTLE_ENDIAN

	    LastPacketWasMulticast =
			(EnetPacket->dstaddress.high & ETHER_MULTICAST_BIT);

	    if ( (EnetPacket->packettype == ETHER_TYPE_VMTP) &&
		 !(kp->serverPid & REMOTE_ALIAS_PROCESS) )
	      {
#ifdef INSTRUMENT
    if (LastPacketWasMulticast)
	KS.seenMulticasts++;
#endif INSTRUMENT
		HandleVmtpPacket((VmtpPacketWithSegment *)kp, bytes-ETHER_HEADER_LENGTH, &(EnetPacket->srcaddress));
	      }

	    if( (EnetPacket->packettype != ETHER_TYPE_VMTP)
		|| (kp->serverPid & REMOTE_ALIAS_PROCESS)
		|| DeliverToEnetReader
		|| EnetPromiscuous )
	      {
		/* it is not a kernel packet */
		/* and not to a remote alias process */
		/* If in promiscuous mode, let the enet reader get too. */

		if( (inst = EthernetInstance) == NULL )
		  {
		    /* Ethernet device not open */
		  }
		else if( !MAP_TO_RPD(pd, inst->reader) )
		  {
		    /* No reader process */
		    if( !MAP_TO_RPD(pd, inst->owner) )
		      {
			/* Ethernet device not open */
			inst->owner = 0; /* Free the instance */
			EthernetInstance = NULL;
			inst = NULL;
			EnetReset( NET_RCV_DEFAULT );
		      }
		    else if (!EnetPromiscuous ||
				EnetPacket->packettype != ETHER_TYPE_VMTP ||
				!(kp->serverPid & REMOTE_ALIAS_PROCESS) )
		      /* If in promiscuous mode, don't save packet if
		       * it is a kernel packet - too risky.*/
		      {
		        /* No one is waiting for this packet, so copy it into
		         * the UserPacketBuffer */
			if( bytes > ENET_MAX_PACKET )
			  {
			    printx("Oversized packet: %d bytes\n", bytes );
			    bytes = ENET_MAX_PACKET;
			  }
		        UserPacketLength = bytes;
		        Copy(UserPacketBuffer, (char *)EnetPacket, bytes);
		      }
		  }
		else
		  {
		    /* give the reader this packet */
		    ReadIntoProcess(pd, inst, (int)bytes, (char *)EnetPacket);
		  }
	      }

	    /* Return packet to receiver */
	    ++EnetValidPackets;
	  }
	else
	  {
	    /* Otherwise we discard this packet */
debug('#');
	    if ( bdl->status1 & QNA_RX1_OVF    ) EnetOverflows++;
	    if ( bdl->status1 & QNA_RX1_CRCERR ) EnetCRCErrors++;
	    if ( bdl->status1 & QNA_RX1_FRAME  ) EnetSyncErrors++;
	  }

	bdl->bd1	= QNA_BDLUnused;
	bdl->status1	= QNA_BDLUnused;
	bdl->status2	= 0;			/* Again, unnecessary */
	GoToNextBD();
	bdl = &Buffers.rxlist[RcvIndex];
      }

  }


void
NetworkWriteDeqna( bufferptr )
register BufferList *bufferptr;
  {
#ifdef FIREFLY
    register ProcessorRec *r10;
#endif FIREFLY
    register char *bp;
    register VmtpPacket	*kp;
    EtherHeader *enetPacket;

#ifdef FIREFLY
    AsmGetProcessorRecord(r10);
    if ( !IsPrimaryProcessor( ( ProcessorRec *) r10) )
      {
        register IPRBufferType *iprb = &IPRBuffer;
        iprb->params[0] = (Unspec) bufferptr;
        SendDeviceServiceRequestToPrimary(iprb,NETWORK_WRITE_DEQNA);
        return;
      }
#endif FIREFLY

    if (bufferptr->bytes < ETHER_HEADER_LENGTH) Kabort("Bad network header");
  
    /* Transfer a packet to the Ethernet Interface Board. */
    bp = (char *)&Buffers.txbuff;
    while (bufferptr->ptr)
      {
	if (bufferptr->bytes)
	  {
	    Copy(bp, bufferptr->ptr, (unsigned)bufferptr->bytes);
	    bp += bufferptr->bytes;
	  }
	bufferptr++;
      }
    ((EtherHeader *) &Buffers.txbuff)->srcaddress = EnetHostNumber;
    enetPacket = (EtherHeader *) &Buffers.txbuff;
#ifdef LITTLE_ENDIAN
    if( enetPacket->packettype == ETHER_TYPE_VMTP )
      {
	kp = (VmtpPacket *) ((unsigned long)enetPacket + ETHER_HEADER_LENGTH);
	SwapVmtpPacket(kp);
      }
#endif LITTLE_ENDIAN
    XmitBuffer(bp - (char *)&Buffers.txbuff, (int)&Buffers.txbuff, 0);
    NetAlarm = ENET_WRITER_TIMEOUT;
  }

ResponseCode 
EnetModifyDeqna( pd, inst, dirIndex )
    register Process *pd;
    DeviceInstance *inst;
    unsigned short dirIndex;
  {
    register QueryNetReply *reply = (QueryNetReply *) &(pd->msg);

    if( reply->NumCollisions != -1 )
	EnetCollisions = reply->NumCollisions;
    if( reply->NumOverflows != -1 )
	EnetOverflows = reply->NumOverflows;
    if( reply->NumCRCErrors != -1 )
	EnetCRCErrors = reply->NumCRCErrors;
    if( reply->NumSyncErrors != -1 )
	EnetSyncErrors = reply->NumSyncErrors;
    if( reply->NumTimeOuts != -1 )
	EnetTimeouts = reply->NumTimeOuts;
    if( reply->NumValidPackets != -1 )
	EnetValidPackets = reply->NumValidPackets;
    if( reply->ReceiveMask != EnetReceiveMask &&
        reply->ReceiveMask != -1 ) EnetReset( reply->ReceiveMask );

   return( OK );
 }    

ResponseCode 
EnetQueryDeqna( pd, inst, dirIndex ) 
    Process *pd;
    DeviceInstance *inst;
    unsigned short dirIndex;
  {
    register QueryNetReply *reply = (QueryNetReply *) &(pd->msg);

    reply->NetworkType = NET_TYPE_ETHER;
    reply->NumCollisions = EnetCollisions;
    reply->NumOverflows = EnetOverflows;
    reply->NumCRCErrors = EnetCRCErrors;
    reply->NumSyncErrors = EnetSyncErrors;
    reply->NumTimeOuts = EnetTimeouts;
    reply->ReceiveMask = EnetReceiveMask;
    reply->NumValidPackets = EnetValidPackets;
    reply->HostAddress.ether = EnetHostNumber;
    if( DifferentByteOrder(pd->pid) )
      {
	/* Swap ethernet address to right byte order for client's use */
	ByteSwapLongInPlace(&(reply->HostAddress.ether), sizeof(NetworkAddress));
	ByteSwapShortInPlace(&(reply->HostAddress.ether), sizeof(NetworkAddress));
      }

    return( OK );
  } 


void
NetCheckDeqna()
  {
    register Process *pd;

    EnetReset( NET_RCV_DEFAULT );
    /* If IkpOutq is non-empty, transmit next interkernel packet. */
    Lockq( &IkpOutq );
    if( (pd = IkpOutq.head) != NULL )
      {
        if( (IkpOutq.head = pd->link) == NULL )
	    IkpOutq.tail = (Process *) &(IkpOutq.head);
	pd->queuePtr = NULL;
	if (EnetWriter == NO_PROCESS)	/* Not really sure this is right, but it */
	    EnetWriter = 1;	/*   seems more right than nothing at all*/
	Unlockq( &IkpOutq );
	WriteNextKPacket( pd );
      }
    else
      {
        EnetWriter = 0; /* No transmission in progress now */
        Unlockq( &IkpOutq );
	NetAlarm = ENET_READER_TIMEOUT;
      }
  }	  

void
ReadIntoProcess(pd, inst, size, buf)
register Process *pd;
register DeviceInstance *inst;
register int size;
register char *buf;
  {
    register IoRequest *req = (IoRequest *) &pd->msg;
    register char *p;
    Process *oldPd = GetAddressableProcess();


    /* First, fill in the bytecount in the reply */
    if ( size < req->bytecount )
      req->bytecount = size;
    /* Now decide where to put the data */
    if( req->bytecount <= IO_MSG_BUFFER ) 
      p = (char *) req->shortbuffer;
    else
      p = (char *) pd->dstSegment.ptr;

    /* Copy data to correct location */
    SetAddressableProcess( pd );
    Copy( p, buf, req->bytecount );
    SetAddressableProcess( oldPd );
    EthernetInstance->reader = 0;

    req->requestcode = OK;
    Addready( pd );
  }


void
GoToNextBD()
/*
 * This function handles going to the next Buffer descriptor, reinitializing
 * the list if it is invalid.
 */
  {
    register QNABD_t *bdl;
    
    if ( RcvIndex == LastValidBuffer )
      {
	Buffers.rxlist[LastValidBuffer].bd2 &= ~QNA_BD2_VALID;
        RcvIndex = (RcvIndex+1) % RXLISTLEN;
	bdl = &Buffers.rxlist[RcvIndex];
	bdl->bd2 |= QNA_BD2_VALID;
	if (*Deqna_csr & QNARXBDLBAD)
	  {
	    *Deqna_rxBDLlo = virt_to_QbusLO(bdl);
	    *Deqna_rxBDLhi = virt_to_QbusHI(bdl);
	  }
	LastValidBuffer = (LastValidBuffer+RXLISTLEN-1) % RXLISTLEN;
      }
    else
      {
        RcvIndex = (RcvIndex+1) % RXLISTLEN;
      }
  }

/*
 * Print debugging info for this DEQNA driver.  Must (1) not occupy too many
 *   lines on the display, and (2) tell as near as we can get to the whole 
 *   truth.  Might also be nice if it (3) made some attempt to pretty-print
 *   things, but this may conflict with (1) and (2) and might even require
 *   some work (gasp).  Prints stuff backwards so that the most important stuff
 *   appears last, and so doesn't vanish from sight.
 * If the DEQNA does anything creative while we're printing this, then what we
 *   print won't be very helpful.  Partly for this reason, we print whatever
 *   csr value the calling routine decides to hand us, and also print the csr's
 *   value when we're almost done.
 */
void
Debug_Deqna(csr)
    unsigned short csr;
  {
    int i;
    printx("\nRx:");
    for (i = RXLISTLEN; i >= 0; i--)
	Debug_BDL(i, (unsigned short *)&Buffers.rxlist[i], (i&1)?"  " : "\r\n");
    printx("Tx:");
    for (i = TXLISTLEN; i >= 0; i--)
	Debug_BDL(i, (unsigned short *)&Buffers.txlist[i], (i&1)?"  " : "\r\n");
    printx(
       "RcvIndex = 0x%x, LastValidBuffer = 0x%x, Deqna CSR was 0x%x, is 0x%x",
	    RcvIndex, LastValidBuffer, csr, *Deqna_csr);
  }

/* The calls to this function use casting where it's probably not necessary. */

void
Debug_BDL(index, address, string)
    int index;
    unsigned short *address;
    char *string;
  {
    printx("%x: %x %x %x %x %x %x%s", index,
	address[0], address[1], address[2], address[3], address[4], address[5],
	string);
  }

/* Interface between Kabort's sort-of-debug routines in exception.c and */
/*   the Debug_Deqna routine.  We don't actually use any arguments	*/
void
KabortDeqna(pd, req, str)
    Process	     *pd;
    ExceptionRequest *req;
    char	     *str;
  {
#ifdef FIREFLY
    register ProcessorRec *r11;
    AsmGetProcessorRecord(r11);
    if ( !IsPrimaryProcessor(r11) )
      {
        register IPRBufferType *iprb = &IPRBuffer;
        iprb->params[0] = (Unspec) pd;
        iprb->params[1] = (Unspec) req;
        iprb->params[2] = (Unspec) str;
        SendDeviceServiceRequestToPrimary(iprb,KABORT_DEQNA);
        return;
      }
#endif FIREFLY
    Debug_Deqna(*Deqna_csr);
    printx("\n");
  }

ResponseCode 
AddProcessGroupToLogicalHostGroupFilterDeqna(gid) 
  GroupId gid;
  {
    gid = gid; /* To shut the C++ compiler up */
    return( OK );
  }

ResponseCode DeleteProcessGroupFromLogicalHostGroupFilterDeqna(gid) 
  GroupId gid;
  {
    gid = gid; /* To shut the C++ compiler up */
    return( OK );
  }


void
InitDeqna()
  {
    NetworkWrite = NetworkWriteDeqna;
    NetCheck = NetCheckDeqna;
    EnetQuery = EnetQueryDeqna;
    EnetModify = EnetModifyDeqna;
    EnetPowerup = EnetPowerupDeqna;
    DeleteProcessGroupFromLogicalHostGroupFilter
	= DeleteProcessGroupFromLogicalHostGroupFilterDeqna;
    AddProcessGroupToLogicalHostGroupFilter
	= AddProcessGroupToLogicalHostGroupFilterDeqna;
  }
