/*
 * V Kernel - Copyright (c) 1982 by Stanford University
 *
 * Kernel Console support routines for the device server
 *
 * The "console" provides access to the default keyboard and display
 *  for standalone programs.  The association between this logical
 *  device and physical device(s) depends on the workstation's
 *  type and configuration.
 *
 * If the MicroVAX console is just a terminal on the serial line, all is well.
 * If, however, we have a framebuffer (currently QVSS or Firefly MDC), things
 * get weird:
 *     - output is sent to an ersatz terminal-emulator for the framebuffer
 *	 (it's in the DEC-supplied console ROM) rather than to the serial line,
 *     - we merge input from the serial line and from any framebuffer keyboards
 *	 into a single stream,
 *     - the client can request unencoded keystrokes, rather than ASCII, from
 *	 the keyboard; to accomplish this we change modes in the keyboard.
 *	 If we have multiple readers (other clients, or the kernel itself [in
 *	 Kabort]) then things will be very sadly confused.  Anyone have a
 *	 better approach?
 *
 * $Revision: 1.15.1.15 $
 * $Locker:  $
 * $State: Exp $
 */

#include <prototypes.h>
#include <Venviron.h>		/* reply codes */
#include <Vprogramprotocol.h>	/* RootIOBlock */
#include <Vioprotocol.h>	/* I/O protocol support, msg structs */
#include <Vprocessprotocol.h>	/* Pd states, via Vprocesscommon.h */
#include <Vquerykernel.h>	/* PRF_* peripheral manifests */
#include "console.h"
#include "devman.h"		/* DeviceInstance */
#include "externals.h"
#include "process.h"		/* Process */
#include "interrupt.h"		/* Interrupt handling */
#include "processor.h"
#include "cbuff.h"		/* Circular buffer, Cbuff macros */
#include "meminstance.h"


/* Private */
static DeviceInstance *ConsoleInInstance, *ConsoleOutInstance;
static CircularBuffer ConsoleInBuff, ConsoleOutBuff;
static unsigned ConsoleXmitReady;
static int KeyboardTranslation = -1;
static int KeyClick = 0;

/* Assembly Language Forward */
_BEGIN_EXTERN_C
void ConsoleRcv _TAKES(()); 
void ConsoleXmit _TAKES(());
void Asm_ConsoleRcv _TAKES(()); 
void Asm_ConsoleXmit _TAKES(());
_END_EXTERN_C

/* Forward */
static void DefaultBellRoutine _TAKES(());
static ResponseCode KeyboardModify _TAKES(( Process *, DeviceInstance *, unsigned short));
static ResponseCode KeyboardQuery _TAKES(( Process *, DeviceInstance *, unsigned short));
static ResponseCode KeyboardRelease _TAKES(( Process *, DeviceInstance *));
static ResponseCode ScreenRelease _TAKES(( Process *, DeviceInstance *));


ResponseCode (*ConsoleKeyboardSetTrans)_TAKES((int)) = NotSupported;
ResponseCode (*ConsoleKeyboardSetClick)_TAKES((int)) = NotSupported;
ResponseCode (*ConsoleKeyboardGetShift)_TAKES((int*,int*,int*))
						   = NotSupported;
void	   (*ConsoleBellRoutine   )()		   = DefaultBellRoutine;

/* Routines */
Call_inthandler(ConsoleRcv);	/* Macro expansion to interrupt-invoked
				 * C call to ConsoleRcv */

Call_inthandler(ConsoleXmit);	/* Macro expansion to interrupt-invoked
				 * C call to ConsoleXmit */

void
InitConsole()
    /* Initialize the Transmit and Receive registers of the console,
    *  and plug in the interrupt handlers for them */
  {
    setexvec(Asm_ConsoleRcv, VecRxConsole);
    setexvec(Asm_ConsoleXmit, VecTxConsole);
    asm("	mtpr	$0x40, $rxcs");
    asm("	mtpr	$0x40, $txcs");
  }


/* Routines that interact directly with the console device */

static void ConsolePutchar(ch)
    int ch;
  {
  /*
   * Write the character out to the VAX console serial line.  Assumes that
   *   txdb is ready for another character now.
   */
    asm( "	movzbl	4(ap), r0" ); /* Guarantee that 3 high bytes are 0 */
    asm("	mtpr	r0, $txdb");  /* write the character */
  } 

/* INTERRUPT HANDLERS */
void
ConsoleRcv() 
  /*
   * Interrupt handler for receiving characters from the console serial line
   */
  {
    register int r11;

    /* Machine specific read of character */
    asm("	mfpr	$rxdb, r11");	/* Read character from 32-bit reg. */
    asm("	movzbl	r11, r11");	/*   and zero top 24 bits	   */

#ifdef undef				/* Still a little unsafe */
    if (FramebufferType == PRF_NONE ||
	KeyboardTranslation == TR_ASCII || KeyboardTranslation == -1)
					/* ^ probably unnecesary */
#endif undef
	ConsoleGotChar(r11);
  }

void
ConsoleGotChar(c)
    int c;
  {
    register Process *pd;
    register DeviceInstance *inst;

    if ( (inst = ConsoleInInstance ) != NULL )
      {
	if (MAP_TO_RPD( pd, inst->reader) != NULL )
	  {
	    /* Someone waiting, so give it to them directly */
	    
	    ((IoReply *)&pd->msg)->replycode = OK;
	    ((IoReply *)&pd->msg)->bytecount = 1;
	    ((IoReply *)&pd->msg)->shortbuffer[0] = c;
	    inst->nextblock++;
	    inst->reader = 0;

	    Addready( pd );
	  }
	else
	  { 
	    /* no one is waiting so queue the character. */
	    CbuffPut( &ConsoleInBuff, c );
	    /* Overflow behaviour is pretty bizarre from a */
	    /*   user-interface point of view.  Too bad.   */
	  }
      }
  }

void
ConsoleXmit()
  /* 
   * Console transmit buffer empty interrupt handler
   */
  {
    register DeviceInstance *inst = ConsoleOutInstance;
    register Process *pd;
    int outch, c;

    if ( inst != NULL && MAP_TO_RPD(pd, inst->writer) != NULL )
      {
	outch = ((IoRequest *)&pd->msg)->shortbuffer[0];

	inst->lastblock++;
	((IoReply *)&pd->msg)->replycode = OK;
	((IoReply *)&pd->msg)->bytecount = 1;
	inst->writer = NULL;
	Addready( pd );

	if ( (c = CbuffGet(&ConsoleOutBuff)) != CBUFF_EMPTY )
	  {
	    CbuffPut(&ConsoleOutBuff, outch);
	    outch = c;
	  }
      }
    else
      {
	if ( (c = CbuffGet(&ConsoleOutBuff)) != CBUFF_EMPTY )
	  {
	    outch = c;
	  }
	else
	  {
	    ConsoleXmitReady = 1;
	    return;
	  }
      }
    ConsolePutchar(outch);
  }

static void
DefaultBellRoutine()
  {
    /* Yup, do nothing */
  }


/* Device Routines */
ResponseCode
ConsoleCreate( pd, inst )  
    Process *pd;
    DeviceInstance *inst;
  {
  /*
   * Create instances for the console.  The instance returned is
   *  used for console output.  The parameters for output
   *  can be obtained by doing a QueryRelatedInstance.
   */
    CreateInstanceRequest *req = (CreateInstanceRequest *) &pd->msg;
    register DeviceInstance *oldInst;

    if (req->filemode != FCREATE) return (MODE_NOT_SUPPORTED);

    if( (oldInst = ConsoleInInstance) != NULL )
      {
	if( MapPid( oldInst->owner ) )  return (BUSY);
	ConsoleInInstance = 0;
	oldInst->owner = 0;
      }

    if( (oldInst = ConsoleOutInstance) != NULL )
      {
	if( MapPid( oldInst->owner ) )  return (BUSY);
	ConsoleOutInstance = 0;
	oldInst->owner = 0;
      }

    /*
     * Initialize the device instance descriptors, can assume all fields
     * are zero except for id and first instance's owner.
     */

    /* Output descriptor */
    inst->readfunc = NotSupported;
    inst->writefunc = ConsoleWrite;
    inst->modifyfunc = NotSupported;
    inst->queryfunc = NotSupported;
    inst->releasefunc = ScreenRelease;

    inst->id |= UIO_WRITEABLE;		/* | STREAM */
    inst->blocksize = sizeof(char);  /* character by character for now */
    inst->lastblock = (unsigned)-1;	     /* indicate file is empty */
    inst->lastbytes = inst->blocksize;
    ConsoleOutInstance = inst;

    /* Input descriptor */
    inst++;
    inst->owner = pd->pid;
    inst->readfunc = ConsoleRead;
    inst->writefunc = ConsoleWrite;
    inst->modifyfunc = KeyboardModify;
    inst->queryfunc = KeyboardQuery;
    inst->releasefunc = KeyboardRelease;

    inst->id |= (UIO_READABLE);	/* STREAM */
    inst->blocksize = sizeof(char);  /* assume character by character for now */
    inst->lastblock = (unsigned)-1;	     /* indicate file is empty */
    inst->lastbytes = inst->blocksize;
    ConsoleInInstance = inst; /* Record the instance for interrupt routine */

    (*ConsoleKeyboardSetTrans)(KeyboardTranslation = TR_ASCII);
    (*ConsoleKeyboardSetClick)(KeyClick            = 0       );
    CbuffFlush(&ConsoleInBuff);	/* Flush any buffered characters */
    CbuffFlush(&ConsoleOutBuff);
    ConsoleXmitReady = 1;

    return( OK );

  }

ResponseCode 
ConsoleRead( pd, inst )
    Process		*pd;
    DeviceInstance	*inst;

   /* Handle a read instance request for the console
    */
  {
    register IoRequest *req = (IoRequest *) &(pd->msg);
    register IoReply *reply = (IoReply *) &(pd->msg);
    register unsigned bytecount;
    register int c;
    Process *reader;

    bytecount = req->bytecount;
    reply->bytecount = 0;
    /*
     * Reply for RETRY if already a waiting reader process.
     * This effectively provides busy-wait queuing for reading. 
     */
    if( inst->reader && MAP_TO_RPD(reader, inst->reader) != NULL) 
	return( RETRY );

    if( bytecount > inst->blocksize ) return( BAD_BYTE_COUNT );
    if( bytecount == 0 ) return ( OK );
    /*
    * Disabled to enable multiple processes to access the console
    * if( req->blocknumber != inst->nextblock ) return( BAD_BLOCK_NO );
    */

    disable;
    /* Try to get characters from the buffer */
    if ( (c = CbuffGet(&ConsoleInBuff)) != CBUFF_EMPTY ) {
	reply->replycode = OK;
	reply->shortbuffer[0] = c;
	reply->bytecount = 1;
	inst->nextblock++;
	inst->reader = 0;
	return( OK );
    }
    else {
	inst->reader = pd->pid;
	pd->state = AWAITING_INT;
	return( NO_REPLY );
    }
  }

ResponseCode
ConsoleWrite( pd, inst )
    register Process *pd;
    register DeviceInstance	*inst;

   /* Handle a write instance request to a serial line
    */
  {
#ifdef FIREFLY
    register ProcessorRec *r9;
#endif FIREFLY

    extern unsigned short FramebufferType;
    register IoRequest *req = (IoRequest *) &(pd->msg);
    register IoReply *reply = (IoReply *) &(pd->msg);
    register int c = req->shortbuffer[0];
    unsigned bytecount;
    unsigned bufsize;
    unsigned newline;

#ifdef FIREFLY
    AsmGetProcessorRecord(r9);
    if ( !IsPrimaryProcessor(r9) )
      {
        register IPRBufferType *iprb = &IPRBuffer;
        iprb->params[0] = (Unspec) pd;
        iprb->params[1] = (Unspec) inst;
        SendDeviceServiceRequestToPrimary(iprb,CONSOLE_WRITE);
        return ( (ResponseCode) iprb->params[0] );
      }
#endif FIREFLY
    bytecount = req->bytecount;
    reply->bytecount = 0;

    if( bytecount > inst->blocksize ) return( BAD_BYTE_COUNT );
    if( bytecount == 0 ) return ( OK );
/*  Disabled to simplify multi-process access to the console.
    if( req->blocknumber != inst->lastblock+1 ) return( BAD_BLOCK_NO );
*/
    disable;
    
    /*
     * Here is where we deal with the framebuffer output routine which lives
     *   in ROM on MicroVAXen and Fireflies; it is not interrupt driven, and
     *   cannot beep with ^G.  (KernelPutChar() calls the ROM routine).
     */
    if (FramebufferType == PRF_FRAMEBUFFER_QVSS ||
	FramebufferType == PRF_FRAMEBUFFER_MDC)
      {
	if (c == 7)
	    (*ConsoleBellRoutine)();
	else
	    KernelPutChar( c );
	inst->lastblock++;
	reply->bytecount = 1;
	reply->delivery = CreateDeliveryMask(reply->bytecount);
	reply->replycode = OK;
	return( OK );
      }
    
    bufsize = CbuffSize(&ConsoleOutBuff);
    newline = c == '\n';
    /* falling through this if => return OK */
    if ( ConsoleXmitReady ) {
        /* bufsize is zero */
        if ( newline ) {
	    ConsolePutchar( '\r' );
	    CbuffPut( &ConsoleOutBuff, '\n' );
	}
	else {
	    ConsolePutchar( c );
	}
	ConsoleXmitReady = 0;
    }
    else if ( bufsize < CBUFF_SIZE-2 || (bufsize == CBUFF_SIZE-2 && !newline))
      {
        if ( newline ) {
	    CbuffPut( &ConsoleOutBuff, '\r' );
	}
	CbuffPut( &ConsoleOutBuff, c );
      }
    else if ( bufsize == CBUFF_SIZE-2 ) {
	CbuffPut( &ConsoleOutBuff, '\r' );
	inst->writer = pd->pid;
	pd->state = AWAITING_INT;
	return( NO_REPLY );
    }
    else if (inst->writer == NULL) { /* cbuff full, wait for int before xmit */
	if ( newline ) {
	    return( RETRY );
	}
	else {
	    inst->writer = pd->pid;
	    pd->state = AWAITING_INT;
	    return( NO_REPLY );
	}
    }
    else {
	return( RETRY );
    }
    inst->lastblock++;
    reply->bytecount = 1;
    reply->replycode = OK;
    return( OK );
  } /* ConsoleWrite */

static ResponseCode
KeyboardModify(pd, inst, dirIndex)
    register Process *pd;
    DeviceInstance *inst;
    unsigned short dirIndex;
#define req ((ModifyKeyboardRequest *)&pd->msg)
  {
    ResponseCode ret;

    if ( (ret = (*ConsoleKeyboardSetTrans)(req->translation)) != OK )
      {
	return ret;
      }
    if ( (ret = (*ConsoleKeyboardSetClick)(req->keyclick ? 100 : 0)) != OK )
      {
	(*ConsoleKeyboardSetTrans)(KeyboardTranslation);
	return ret;
      }
    if (req->translation != KeyboardTranslation)
      {
	CbuffFlush(&ConsoleInBuff);	/* Stuff in buffer is useless if it */
					/*   was generated by the wrong mode*/
	KeyboardTranslation = req->translation;
      }
    KeyClick = (req->keyclick != 0);
    return OK;
  }
#undef req

static ResponseCode KeyboardQuery(pd, inst, dirIndex)
    Process *pd;
    DeviceInstance *inst;
    unsigned short dirIndex;
#define reply ((QueryKeyboardReply *)&pd->msg)
  {
    reply->keyclick    = KeyClick;
    reply->translation = KeyboardTranslation;
    return (*ConsoleKeyboardGetShift)((int *)&reply->fill2[0], (int *)&reply->fill2[1],
				      (int *)&reply->fill2[2]);
  }
#undef reply

static ResponseCode
ScreenRelease( pd, inst )
    Process *pd;
    DeviceInstance *inst;
  /*
   * Release VAX screen (console) instance
   */
  {
    inst->owner = 0;
    return( OK );
  }

static ResponseCode
KeyboardRelease( pd, inst )
    Process *pd;
    DeviceInstance *inst;
  /*
   * Release VAX keyboard (console) instance
   */
  {
    IoReply *reply;
    register Process *rwpd;

    if( rwpd = MapPid( inst->reader) ) 
				/* Unblock reader process */
				/* Why doesn't ScreenRelease() have to check */
				/*   similarly for a blocked writer??  TMM.  */
      {
	inst->reader = 0;
	reply = (IoReply *) &rwpd->msg;
	reply->replycode = END_OF_FILE;
	reply->bytecount = 0;
	Addready( rwpd );
      }
    inst->owner = 0;
    return( OK );
  }
#ifdef VM
#include "vm.h"
#endif VM

void
CreateStdIo( pd )
    Process *pd;
  /* Create a console device instance for the root team's standard i/o */
  {
    register CreateInstanceRequest *req = (CreateInstanceRequest *) &pd->msg;
#define  reply ( (CreateInstanceReply *) req)
#define stsreq ( (SetTeamSizeRequest *) req)
#define INIT_STACK_SIZE 8096
/* Align pointer p to the next b-byte boundary, assuming b is a power of 2. */
#define align(p, b) (  (p) + ( -(int)(p) & ((b)-1) )  )

    RootIOBlock rootIOB;
    char *curptr;
    unsigned int  size;
    Process * oldPd;
    ResponseCode	r;

    req->requestcode = CREATE_INSTANCE;
    req->filemode = FCREATE;
    
    req->namestart = 0;
    req->nameindex = 0;
    req->contextid = DEFAULT_CONTEXT;
    req->filenamelen = 7;

    pd->srcSegment.ptr = "console";
    pd->srcSegment.size = 7;

    if ( (r = CreateDeviceInstance( pd )) != OK) 
	Kabort("Error creating std i/o");

    ConsoleInInstance->owner = pd->pid;
    ConsoleOutInstance->owner = pd->pid;

    /* Set up the rootIOBlock */
    curptr = (char *) align(pd->team->team_size + INIT_STACK_SIZE, 4); 
    size = sizeof(long) + sizeof(RootIOBlock);

    rootIOB.rootflags = STDOUT_APPEND | STDERR_APPEND;
    rootIOB.stdinserver = rootIOB.stdoutserver = rootIOB.stderrserver = 
    	reply->fileserver;
    rootIOB.stdoutfile = rootIOB.stderrfile = reply->fileid;

    pd->requestCode = QUERY_RELATED_INSTANCE;
    if ( (r = QueryRelatedDeviceInstance(pd)) != OK)
	Kabort("Error querying stdin");
    rootIOB.stdinfile = reply->fileid;

    stsreq->requestcode = SET_TEAM_SIZE;
    stsreq->pid = pd->pid;
    stsreq->newbreak = (unsigned) curptr + size;
    
    SetTeamSize(pd);
    oldPd = GetAddressableProcess();
    SetAddressableProcess(pd);
#ifdef VM
    CheckSegWrite(pd, curptr, ((char *)pd->team->team_size - curptr));
#endif VM
    * ( (unsigned  *) curptr) = size;
    *( (RootIOBlock *)( curptr + sizeof(unsigned)) ) = rootIOB;
    SetAddressableProcess(oldPd);

    /* Set up the pointer to the root IO block */ 
    pd->userStackPtr = (Unspec) curptr;

#undef reply
#undef stsreq
  }
