/********************************************************/
/*							*/
/*	  Virtual Graphics Terminal Server		*/
/*							*/
/*		(C) COPYRIGHT 1982			*/
/*		BOARD OF TRUSTEES			*/
/*	LELAND STANFORD JUNIOR UNIVERSITY		*/
/*	  STANFORD, CA. 94305, U. S. A.			*/
/*							*/
/********************************************************/


/*
 * FILE: tty.c
 *
 * Bill Nowicki September 1982
 *
 * Left-over routines from Yale's I/O package
 */


# include "Vgts.h"
# include "sdf.h"
# include "pad.h"
# include <chars.h>


short TtyVGT;	/* Index of the "command" tty VGT.  This
		   VGT is used by the vgts for communications
		   with the user. */

extern short TtySDF;		/* SDF used for internal communication */
extern short Debug;		/* print extra debugging information */
extern short Unique;		/* symbol identifier */

#define MaxInputStringLength 75
#define PauseTime 15			/* Number of clock clicks to delay
		   			   when inverting text line. */

static struct TtyPadRec *TtyPad;


/*******************************************************************/


/*
 * InitTtyVGT:
 * Creates the vgts' tty window.  This window is used by the
 * Vgts to communicate with the user.
 * It is positioned near the bottom, since only the last few lines
 * are visible.  The user can move the top edge up to see more
 * if he so desires.
 */

InitTtyVGT(xmin,ymin,xmax,ymax)
  {
    TtySDF = CreateSDF();
    TtyVGT = PadInit( TtySDF, Unique++, "VGTS", 20, 64 );
    CreateView( TtyVGT, xmin, ymin, xmax, ymax, 2, -2, 0, 0);
    TtyPad = TtyPadList + TtyVGT;
    PadPosition(TtyPad, 24, 0);
  }

/*******************************************************************/


/*
 * TtyGetPromptedString:
 * Clears the vgts ttywindow, puts up the prompt, and
 * accepts a string of characters terminated by a newline.  The string
 * is left in the local character array, so the caller must copy it out if
 * he wishes to keep it.  It also flashes the prompt a couple of times.
 */

char *TtyGetPromptedString(prompt, echo)
 char *prompt;
  BOOLEAN echo;
{
    short i;
    char *TtyGetString();

    while (*prompt)
        PadPutChar(*(prompt++), TtyPad);	/* dump the prompt */
    PadPutChar(' ', TtyPad);
    PadClearToEOL(TtyPad);
    
    RedrawVGT( TtyVGT, 0, 3000, 0, 3000 );
    for (i = 0; i < 2; i++)
      {
        PadBlinkLine(TtyVGT);
        Pause(PauseTime);
      } 
    
    return(TtyGetString(echo));	/* TRD */
}


/*******************************************************************/


/*
 * TtyGetNumber:
 * Displays the prompt in the vgts ttywindow
 * and returns the number which is typed in.
 */

short TtyGetNumber(prompt)
  char *prompt;
{
    char *string, *TtyGetPromptedString();
    short i;
    
    string = TtyGetPromptedString(prompt, TRUE);
    i = 0;
    while (*string)
        i = i*10 + (*string++) - '0';
    return(i);
}


/*******************************************************************/


/*
 * TtyBlinkError:
 * Puts up the error string in the top line of the vgts ttywindow,
 * and blinks it a couple of times.
 */

TtyBlinkError(errorString)
char *errorString;
{
    TtyBlinkHelper(errorString, 6);
}


/*******************************************************************/


/*
 * TtyMessage:
 * Displays the designated string in the vgts ttywindow.
 */

TtyMessage(string)
char *string;
{
    TtyBlinkHelper(string, 2);
}


/*******************************************************************/


/*
 * TtyBlinkHelper:
 * Primary support routine for displaying text in the vgts ttywindow.
 * Displays the indicated text string and flashes it a few times.
 */
static TtyBlinkHelper(errorString, blinkNumber)
  short blinkNumber;
  char *errorString;
{
    short lineSave, columnSave, blinkCount;

    PadClearToEOL(TtyPad);
    while (*errorString)
        PadPutChar(*errorString++, TtyPad);
    
    for (blinkCount = 0; blinkCount < blinkNumber; blinkCount++)
      {
        PadBlinkLine(TtyVGT);
        Pause(PauseTime);
      }
    PadReturn(TtyPad);
    PadNewLine(TtyPad);    
    RedrawVGT( TtyVGT, 0, 3000, 0, 3000 );
}

/* The following two routines added by TRD to fix yale */

/* TtyPutString: dumps a string to the tty.  It does not bother with
 * a final newline or any blinking as in TtyMessage.
 */

TtyPutString(string)
  char *string;
{
  while (*string)
      PadPutChar(*string++, TtyPad);
  RedrawVGT(TtyVGT, 0, 3000, 0, 3000);
}


/* TtyGetString: just gets a string from the user, echoing it in the
 * tty window.  It is like TtyGetPromptedString, except that it does not put out a
 * prompt first.
 */

char *TtyGetString(echo)
  BOOLEAN echo;
{
    static char locString[MaxInputStringLength + 1];
    char GetInputChar();
    char c;
    short locStringPtr = 0;
    short i;

    while (1)
     {
      switch (c = GetInputChar())
       {
	case LF:
	case CR:
	    locString[locStringPtr] = 0;
	    PadReturn(TtyPad);
	    PadNewLine(TtyPad);
            RedrawVGT( TtyVGT, 0, 3000, 0, 3000 );
	    return(locString);

	case BS:
	case DEL:
	    if (locStringPtr)
	      {
		locStringPtr--;
		if (echo)
		  {
		    PadPutChar( BS, TtyPad);
		    PadPutChar( ' ', TtyPad);
  		    PadPutChar( BS, TtyPad);
                    PadRedrawLine( TtyVGT );
		  }
	      }
	    break;

	default:
	    locString[locStringPtr++] = c;
	    if (locStringPtr == MaxInputStringLength)
	      {
		locString[locStringPtr] = 0;
		return(locString);
	      }
	    if (echo)
	      {
		PadPutChar(c, TtyPad);
                PadRedrawLine(TtyVGT);
	      }
	    break;
	}
    }

}

TtyTrace()
  {
  	/*
	 * Debugging routine
	 */
    printf( "Tracing TTy vgt\n");
    SdfTrace(VGTtable[TtyVGT].sdfPtr, 0 );
  }

